% DevelopmentAccounting    3/6/15
%
%  Basic development accounting using PWT 8.0
%   http://citaotest01.housing.rug.nl/FebPwt/Dmn/AggregateXs.mvc/PivotShow
%
%    cgdpo = Current PPP GDP output measure
%     emp  = employment (# of persons engaged)
%     ck   = current PPP capital stock
%     hc   = human capital per person, from Barro-Lee and Psacharopoulos via PWT
%            (e.g. mean(delta(log(hc(:,USA))))=.005 ==> 1/2 a percent per year of growth
%
%     Y = K^a (AhL)^(1-a) ==> Y/L = (K/Y)^(a/(1-a)) * h * A
%
%      so TFP := A = Y/L / [ (K/Y)^(a/(1-a))*h ]
%
%  Note: PWT 8.0 have a TFP measure, but this is Harrod-neutral and uses
%  varying labor shares. For my measure, I use Hicks-Neutral and a constant
%  labor share of 2/3.

diarychad('DevelopmentAccounting');
clear all;
definecolors;


load pwt80;
Y=cgdpo;
K=ck;
h=hc;
KY=K./Y;
ypw=div(Y,emp);       % per capita  

a=1/3
aa=a/(1-a);

% USA = 1
yAll=div(ypw,ypw(:,USA)); % US=1
kyAll=div(KY,KY(:,USA));
hAll=div(h,h(:,USA));
tfpAll=yAll./( kyAll.^aa .* hAll);

yr0=1949;
yr=2010

load regions80;
tfpAll(yr-yr0,outliersAfter1980)=NaN;

% Plot shares of gap accounted for by TFP over time for a select group of countries.
yRatio=1./yAll;
tfpRatio=1./tfpAll;
inputRatio=yRatio./tfpRatio;

tfpshare=tfpRatio./(tfpRatio+inputRatio);

yrs=(1960:2011)';
tfpshare=tfpshare(yrs-yr0,:);
figure(3); figsetup;
plot(yrs,tfpshare(:,BRA),'-','Color',mypurp,'LineWidth',4);
plot(yrs,tfpshare(:,CHN),'-','Color',myred*.9,'LineWidth',4);
plot(yrs,tfpshare(:,IND),'--','Color',mygreen,'LineWidth',4);
plot(yrs,tfpshare(:,KEN),'-','Color',myblue*.9,'LineWidth',4);
%plot(yrs,tfpshare(:,IDN),'--','Color',myblue*1.1,'LineWidth',4);
%plot(yrs,tfpshare(:,ITA),'-','Color',myred*1.1,'LineWidth',4);
plot(yrs,tfpshare(:,GBR),'-','Color',myblue*1.1,'LineWidth',4);
plot(yrs,tfpshare(:,FRA),'-','Color',mygreen*1.1,'LineWidth',4);
chadfig2('Year','Share accounted for by TFP',1,0);
ax=axis; ax(3)=0; ax(2)=2013; axis(ax);
%nums=(0:10:90)'/100;
nums=(10:20:90)'/100;
%labs='0% 10% 20% 30% 40% 50% 60% 70% 80% 90%';
labs='10% 30% 50% 70% 90%';
relabelaxis(nums,strmat(labs),'y');
makefigwide;
text(1999,.6,'Brazil');
text(1980,.85,'China');
text(2005,.65,'India');
text(2003,.91,'Kenya');
text(1970,.30,'France');
text(2000,.29,'U.K.');
%text(1988,.475,'Indonesia');
print DevelopmentAccounting2.eps


tfp=tfpAll(yr-yr0,:)';
smpl=~isnan(tfp);
y=yAll(yr-yr0,:)';
ky=kyAll(yr-yr0,:)';
h=hAll(yr-yr0,:)';

sharetfp=tfpshare(yr-1959,:)'*100;



data=[y ky.^aa h tfp];
fmt='%8.3f %8.3f %8.3f %8.3f %8.1f';
tle='y ky^aa h tfp sh.tfp';

ctys=[USA DEU FRA GBR JPN KOR SGP HKG ARG BRA MEX THA IND CHN IDN ZAF BWA KEN MWI]';
[blah,indx]=sort(-y(ctys));
cshow(namesSTR(ctys(indx),:),[data(ctys(indx),:) sharetfp(ctys(indx),:)],fmt,tle);

% Geometric average
N=sum(smpl);
avg=(prod(data(smpl,:))).^(1/N);
savg=(prod(sharetfp(smpl,:))).^(1/N);

invavg=1./avg;
invshare=invavg(4)/(invavg(4)+invavg(3)*invavg(2))*100;
disp ' ';
cshow('Geometric average    ',[avg savg],fmt,[],[],1);
cshow('1/(Geometric average)',[1./avg invshare],fmt,[],[],1);


disp 'Inverse...'
cshow(namesSTR(ctys(indx),:),[1./data(ctys(indx),:) sharetfp(ctys(indx),:)],fmt,tle);

% All countries
disp ' ';
disp '----------------------';
disp 'DATA FOR ALL COUNTRIES';
disp '----------------------';
ctys=(1:length(y));
ctys=ctys(smpl);
[blah,indx]=sort(-y(ctys));
cshow(namesSTR(ctys(indx),:),[data(ctys(indx),:) sharetfp(ctys(indx),:)],fmt,tle);



namethese=zeros(length(y),1);
namethese(ctys)=1;

labs=strmat('1/256#1/128# 1/64 # 1/32 # 1/16 #  1/8  #  1/4  #  1/2  #   1   #   2   ','#');

figure(1); figsetup;
%plotlog(log(y),tfp,namesSTR,labs,10,[],.7,.2,namethese);
plotlog(log(y),tfp,namesSTR,labs,10,[],.6,.3,namethese);
set(gca,'XTick',log([1/256 1/128 1/64 1/32 1/16 1/8 1/4 1/2 1 2]));
set(gca,'XTickLabel',labs);
ax=axis; ax(1)=log(1/80); ax(3)=log(1/45); ax(4)=log(2.5); axis(ax);
hold on;
gg=[log(1/256) log(2)];
plot(gg,gg,'b-','LineWidth',1);
chadfig2('GDP per worker (US=1)','TFP (labor-augmenting, US=1)',1,0);
makefigwide;
print DevelopmentAccounting.eps

disp ' ';
rho=corrcoef([log(y(smpl)) log(tfp(smpl))]);
fprintf('The correlation between log(y) and log(tfp) is %6.3f\n',rho(1,2));


figure(2); figsetup;
plotnamesym2(log(y),sharetfp,namesSTR,10,[],1,5,namethese);
set(gca,'XTick',log([1/256 1/128 1/64 1/32 1/16 1/8 1/4 1/2 1 2]));
set(gca,'XTickLabel',labs);
ax=axis; ax(1)=log(1/128);  axis(ax);
hold on;
chadfig2('GDP per worker, 2010 (US=1)','Percent',1,0);
makefigwide;
print DevelopmentAccountingShare.eps


% 95th to 5th percentiles
disp ' ';
N=sum(smpl);
fprintf('Number of Countries: %4.0f\n',N);
num=round(.05*N);
disp 'Sorting countries...';
fprintf('95th and 5th percentiles include this many countries: %4.0f\n',num);
y=y(smpl); ky=ky(smpl); h=h(smpl); tfp=tfp(smpl);
[blah,indx]=sort(-y);
y=y(indx);
ky=ky(indx);
h=h(indx);
tfp=tfp(indx);
i95=(1:num);
i05=((N-num+1):N);
yr=(prod(y(i95)./y(i05)))^(1/num);
kyr=(prod(ky(i95)./ky(i05)))^(1/num);
hr=(prod(h(i95)./h(i05)))^(1/num);
tfpr=(prod(tfp(i95)./tfp(i05)))^(1/num);

cshow('95th/5th',[yr kyr.^aa hr tfpr],fmt,tle);


% % 95th to 5th percentiles for various years
% disp ' ';
% disp 'Ratio of 95th / 5th (geometric average in each group)';
% %Years = [1960 1970 1980 1990 1995 2000 2005 2010 2011];
% Years=(1960:2011)';
% tle='Year y ky^aa h tfp NumCtys num95';
% fmt='%6.0f %8.3f %8.3f %8.3f %8.3f %8.0f %8.0f';
% for t=1:length(Years);
%     tyr=Years(t);
%     tfp=tfpAll(tyr-yr0,:)';
%     smpl=~isnan(tfp);
%     y=yAll(tyr-yr0,:)';
%     ky=kyAll(tyr-yr0,:)';
%     h=hAll(tyr-yr0,:)';

%     N=sum(smpl);
%     num=round(.05*N);
%     y=y(smpl); ky=ky(smpl); h=h(smpl); tfp=tfp(smpl);
%     [blah,indx]=sort(-y);
%     y=y(indx);
%     ky=ky(indx);
%     h=h(indx);
%     tfp=tfp(indx);
%     i95=(1:num);
%     i05=((N-num+1):N);
%     yr=(prod(y(i95)./y(i05)))^(1/num);
%     kyr=(prod(ky(i95)./ky(i05)))^(1/num);
%     hr=(prod(h(i95)./h(i05)))^(1/num);
%     tfpr=(prod(tfp(i95)./tfp(i05)))^(1/num);

%     rowtle=sprintf('%4.0f 95th/5th:',tyr);
%     if t==1;
%         cshow(' ',[tyr yr kyr.^aa hr tfpr N num],fmt,tle);
%     else;
%         cshow(' ',[tyr yr kyr.^aa hr tfpr N num],fmt,[],[],1);
%     end;
% end;


% % Rich to Poor averages
% disp ' ';
% disp 'Ratio of Rich to Poor geometric averages for constant sample countries';
% disp 'Countries chosen as 95th and 5th cutoffs in initial year';
% %Years = [1960 1970 1980 1990 1995 2000 2005 2010 2011];
% Years=(1990:2011)';
% tle='Year y ky^aa h tfp';
% fmt='%6.0f %8.3f %8.3f %8.3f %8.3f';

% for t=1:length(Years);
%     tyr=Years(t);
    
%     if t==1; % Get sample
%         tfp=tfpAll(tyr-yr0,:)';
%         y=yAll(tyr-yr0,:)';
%         ky=kyAll(tyr-yr0,:)';
%         h=hAll(tyr-yr0,:)';

%         smpl = ~isnan(tfp);
%         N=sum(smpl)
%         num=round(.05*N)
%         ctyindx=(1:size(yAll,2))';

%         y=y(smpl); ky=ky(smpl); h=h(smpl); tfp=tfp(smpl); ctyindx=ctyindx(smpl);
%         [blah,indx]=sort(-y);
%         y=y(indx);
%         ky=ky(indx);
%         h=h(indx);
%         tfp=tfp(indx);
%         ctyindx=ctyindx(indx);
%         i95=(1:num);
%         i05=((N-num+1):N);
%         rich=ctyindx(1:num);
%         poor=ctyindx((N-num+1):N);
%         fprintf('Rich: '); say(namesSTR(rich,:));
%         fprintf('Poor: '); say(namesSTR(poor,:));        
%     end;
    
%     tfp=tfpAll(tyr-yr0,:)';
%     y=yAll(tyr-yr0,:)';
%     ky=kyAll(tyr-yr0,:)';
%     h=hAll(tyr-yr0,:)';


%     yr=(prod(y(rich)./y(poor)))^(1/num);
%     kyr=(prod(ky(rich)./ky(poor)))^(1/num);
%     hr=(prod(h(rich)./h(poor)))^(1/num);
%     tfpr=(prod(tfp(rich)./tfp(poor)))^(1/num);

%     if t==1;
%         cshow(' ',[tyr yr kyr.^aa hr tfpr],fmt,tle);
%     else;
%         cshow(' ',[tyr yr kyr.^aa hr tfpr],fmt,[],[],1);
%     end;
% end;


% % 95th to 5th percentiles for various years
% disp ' ';
% disp 'Ratio of 95th / 5th -- exactly those percentiles not Top 7 to Bottom 7';
% %Years = [1960 1970 1980 1990 1995 2000 2005 2010 2011];
% Years=(1960:2011)';
% tle='Year y ky^aa h tfp NumCtys num95';
% fmt='%6.0f %8.3f %8.3f %8.3f %8.3f %8.0f %8.0f';
% for t=1:length(Years);
%     tyr=Years(t);
%     tfp=tfpAll(tyr-yr0,:)';
%     smpl=~isnan(tfp);
%     y=yAll(tyr-yr0,:)';
%     ky=kyAll(tyr-yr0,:)';
%     h=hAll(tyr-yr0,:)';

%     N=sum(smpl);
%     num=round(.05*N);
%     y=y(smpl); ky=ky(smpl); h=h(smpl); tfp=tfp(smpl);
%     [blah,indx]=sort(-y);
%     y=y(indx);
%     ky=ky(indx);
%     h=h(indx);
%     tfp=tfp(indx);
%     i95=num;
%     i05=N-num+1;
%     yr=y(i95)/y(i05);
%     kyr=ky(i95)/ky(i05);
%     hr=h(i95)/h(i05);
%     tfpr=tfp(i95)/tfp(i05);

%     if t==1;
%         cshow(' ',[tyr yr kyr.^aa hr tfpr N num],fmt,tle);
%     else;
%         cshow(' ',[tyr yr kyr.^aa hr tfpr N num],fmt,[],[],1);
%     end;
% end;


% % 90th to 10th percentiles for various years
% disp ' ';
% disp 'Ratio of 90th / 10th -- exactly those percentiles not inclusive';
% %Years = [1960 1970 1980 1990 1995 2000 2005 2010 2011];
% Years=(1960:2011)';
% tle='Year y ky^aa h tfp NumCtys num95';
% fmt='%6.0f %8.3f %8.3f %8.3f %8.3f %8.0f %8.0f';
% for t=1:length(Years);
%     tyr=Years(t);
%     tfp=tfpAll(tyr-yr0,:)';
%     smpl=~isnan(tfp);
%     y=yAll(tyr-yr0,:)';
%     ky=kyAll(tyr-yr0,:)';
%     h=hAll(tyr-yr0,:)';

%     N=sum(smpl);
%     num=round(.10*N);
%     y=y(smpl); ky=ky(smpl); h=h(smpl); tfp=tfp(smpl);
%     [blah,indx]=sort(-y);
%     y=y(indx);
%     ky=ky(indx);
%     h=h(indx);
%     tfp=tfp(indx);
%     i90=num;
%     i10=N-num+1;
%     yr=y(i90)/y(i10);
%     kyr=ky(i90)/ky(i10);
%     hr=h(i90)/h(i10);
%     tfpr=tfp(i90)/tfp(i10);

%     if t==1;
%         cshow(' ',[tyr yr kyr.^aa hr tfpr N num],fmt,tle);
%     else;
%         cshow(' ',[tyr yr kyr.^aa hr tfpr N num],fmt,[],[],1);
%     end;
% end;



diary off;