% GuvenenEta.m
%
%   Loads the moments from Guvenen.m and computes eta
%   from smoothed values in key years.
%
%   Key variables:
%
%     GrowthTop1Earners {p99,p95,sd} = change log earnings top 1% earners (table Nyears)
%     GrowthTop5Earners {p99,p95,sd} = change log earnings top 5% earners (table Nyears)
%
%     ProbStayHighGrowth = Nyears x 4 matrix
%      -- Recall Fatih calculates Prob(top 5% growth in t+1 | top X% growth in t)
%         That is, his "final" high growth state is always top 5% growth    
%
%         ==> A good benchmark for growth is 97.5, corresponding to Top 5% growth
%             (these are Top125 and Top525 below).
% 
%       Top 1%  => repct=100 and average rank1yr=100      ==> Top115
%       Top 1%  => repct=100 and average rank1yr=97.5     ==> Top125
%       Top 5%  => repct=96-100 and average rank1yr=100   ==> Top515
%       Top 5%  => repct=96-100 and average rank1yr=97.5 >    Top525
%
%     Delta_e=zeros(Nyears,3); % For Top1, Top2, and Top5


clear;
diarychad('GuvenenEta');


load Guvenen;
yr0=1980;
deltaCD=0.06 % 0.02
PermTransVar=1/4; % Fraction of variance that is due to the permanent shock
fprintf('Fraction of variance due to permanent shock:  PermTransVar = %8.3f\n',PermTransVar);
xi=[-1 -1.5]';  % Moments for half-life calculations
disp ' '; disp ' ';


%ProbStayHighGrowth=ProbStayHighGrowth*2;

%StDevMeasure="High";   % Use .sdH from p99 - p97.5. Doesn't look like it matters much?
% replace ".sd" below with ".sdH" to use the "High" measure.

disp '======================================================================';
disp '            ONCE A DECADE...';
disp '======================================================================';
    
MyYears=[1981 1990 2000 2010]'; % Years for saving smoothed data.
disp ' -------- Top 1%/2.5%/5% ---------';
guveneneta(ProbStayHighGrowth(:,2),GrowthTop1Earners.p97_5,GrowthTop1Earners.sdH,Delta_e(:,1),deltaCD,MyYears,yr0,PermTransVar,HPsmoother);
disp ' -------- Top 5%/2.5%/5% ---------';
guveneneta(ProbStayHighGrowth(:,4),GrowthTop5Earners.p97_5,GrowthTop5Earners.sdH,Delta_e(:,3),deltaCD,MyYears,yr0,PermTransVar,HPsmoother);

disp '======================================================================';
disp '                         ANNUAL DATA';
disp '======================================================================';

EndingYear=2006   % Stop there so that the financial crisis does not overly influence smoothing
MyYears=(1981:EndingYear)';
YearsToSmooth=(1981:EndingYear)-yr0;
oo=ones(length(YearsToSmooth),1);

disp ' ';
disp ' -------- Top 1%/2.5%/5% ---------';
disp ' ';
%Eta01=guveneneta(ProbStayHighGrowth(:,2),GrowthTop1Earners.p99,GrowthTop1Earners.sdH,Delta_e(:,1),deltaCD,MyYears,yr0,PermTransVar,HPsmoother);
[Eta125,SpeedOfConvergence,mutilde,sigma,pbar,delta_e,smootheddata]=guveneneta(ProbStayHighGrowth(:,2),GrowthTop1Earners.p97_5,GrowthTop1Earners.sdH,Delta_e(:,1),deltaCD,MyYears,yr0,PermTransVar,HPsmoother,xi,YearsToSmooth);

Constant_ProbStayHigh=oo*smootheddata(1,4);
Constant_MuTilde = oo*smootheddata(1,1);
Constant_Sigma = oo*smootheddata(1,2)/sqrt(PermTransVar);
Constant_Delta = oo*smootheddata(1,3);

EtaMutilde=guveneneta(Constant_ProbStayHigh,GrowthTop1Earners.p97_5,Constant_Sigma,Constant_Delta,deltaCD,MyYears,yr0,PermTransVar,HPsmoother,[],YearsToSmooth);
EtaSigma=guveneneta(Constant_ProbStayHigh,Constant_MuTilde,GrowthTop1Earners.sdH,Constant_Delta,deltaCD,MyYears,yr0,PermTransVar,HPsmoother,[],YearsToSmooth);
EtaDelta=guveneneta(Constant_ProbStayHigh,Constant_MuTilde,Constant_Sigma,Delta_e(:,1),deltaCD,MyYears,yr0,PermTransVar,HPsmoother,[],YearsToSmooth);
EtaPbar=guveneneta(ProbStayHighGrowth(:,2),Constant_MuTilde,Constant_Sigma,Constant_Delta,deltaCD,MyYears,yr0,PermTransVar,HPsmoother,[],YearsToSmooth);

figure(2); figsetup; makefigwide;
plot(MyYears,Eta125','-','Color',myblack,'LineWidth',LW);
LW=3
plot(MyYears,EtaDelta','-','Color',myorng,'LineWidth',LW);
plot(MyYears,EtaMutilde','-','Color',myblue,'LineWidth',LW);
plot(MyYears,EtaSigma','-','Color',mypurp,'LineWidth',LW);
plot(MyYears,EtaPbar','-','Color',mygreen,'LineWidth',LW);
text(1988,Eta125(1995-yr0),'All together');
text(1999,.475,'$\tilde{\mu}_H$ only','interpreter','latex');
text(1993,.463,'$\sigma_H$ only','interpreter','latex');
text(2006.5,.455,'\delta only');
text(2001,.51,'$\bar{p}$ only','interpreter','latex');
chadfig2('Year','Pareto inequality,\eta',1,0);
print('-depsc','GuvenenEta125-Decomposition');


disp ' ';
disp ' ROLE OF LUCK: Setting sigmaH=0 throughout...'
disp ' ';
guveneneta(ProbStayHighGrowth(:,2),GrowthTop1Earners.p97_5,zeros(size(GrowthTop1Earners.sdH)),Delta_e(:,1),deltaCD,MyYears,yr0,PermTransVar,HPsmoother,[],YearsToSmooth);
disp ' ' 


disp ' ';
disp ' -------- Top 5%/2.5%/5% ---------';
disp ' ';
[Eta525,SpeedOfConvergence,mutilde,sigma,pbar,delta_e,smootheddata]=guveneneta(ProbStayHighGrowth(:,4),GrowthTop5Earners.p97_5,GrowthTop5Earners.sdH,Delta_e(:,3),deltaCD,MyYears,yr0,PermTransVar,HPsmoother,xi,YearsToSmooth);

Constant_ProbStayHigh=oo*smootheddata(1,4);
Constant_MuTilde = oo*smootheddata(1,1);
Constant_Sigma = oo*smootheddata(1,2)/sqrt(PermTransVar);
Constant_Delta = oo*smootheddata(1,3);

EtaMutilde=guveneneta(Constant_ProbStayHigh,GrowthTop5Earners.p97_5,Constant_Sigma,Constant_Delta,deltaCD,MyYears,yr0,PermTransVar,HPsmoother,[],YearsToSmooth);
EtaSigma=guveneneta(Constant_ProbStayHigh,Constant_MuTilde,GrowthTop5Earners.sdH,Constant_Delta,deltaCD,MyYears,yr0,PermTransVar,HPsmoother,[],YearsToSmooth);
EtaDelta=guveneneta(Constant_ProbStayHigh,Constant_MuTilde,Constant_Sigma,Delta_e(:,3),deltaCD,MyYears,yr0,PermTransVar,HPsmoother,[],YearsToSmooth);
EtaPbar=guveneneta(ProbStayHighGrowth(:,4),Constant_MuTilde,Constant_Sigma,Constant_Delta,deltaCD,MyYears,yr0,PermTransVar,HPsmoother,[],YearsToSmooth);

figure(3); figsetup; makefigwide;
LW=3
plot(MyYears,EtaDelta','-','Color',myorng,'LineWidth',LW);
plot(MyYears,EtaMutilde','-','Color',myblue,'LineWidth',LW);
plot(MyYears,EtaSigma','-','Color',mypurp,'LineWidth',LW);
plot(MyYears,EtaPbar','-','Color',mygreen,'LineWidth',LW);
plot(MyYears,Eta525','-','Color',myblack,'LineWidth',4);
text(1986,Eta525(1995-yr0),'All together');
text(2004,EtaMutilde(2006-yr0),'$\tilde{\mu}$ only','interpreter','latex');
text(2009,EtaSigma(2006-yr0)-.007,'\sigma only');
text(1987,EtaDelta(1987-yr0)-.005,'\delta only');
text(2008,EtaPbar(2006-yr0)-.005,'$\bar{p}$ only','interpreter','latex');
chadfig2('Year','Pareto inequality,\eta',1,0);
print('-depsc','GuvenenEta525-Decomposition');



figure(1); figsetup; makefigwide; LW=4;
plot(MyYears,Eta525','-','Color',mygreen,'LineWidth',LW);
plot(MyYears,Eta125','-','Color',myblue,'LineWidth',LW);
%plot(MyYears,Eta115','-','Color',myorng,'LineWidth',LW);
chadfig2('Year','Pareto inequality, \eta',1,0);
print('-depsc','GuvenenEta');


diary off;