% TransitionDynamicsIRS.m
%
%   -- Called from TransitionDynamicsMaster with "CaseName", 
%   -- Considers full set of shocks from NBER/PanelMuH or SSInequalitySolving.m, 
%      rather than just one shock at a time.
%
% IRSCase={1,2,3}:  Default is 1.
%  1. Wages/Salaries + Entr. Income       95       40     0.34     0.43
%  2. Wages and Salaries                  95       40     0.32     0.42
%  3. Entrepreneurial Income only         95       40     0.37     0.51
%
%
%  Simulate the transition dynamics of the inequality model
%  assuming entrepreneurial effort is fixed at its steady state value.
%  That is, we shut off the transition dynamics associated with entreprenuerial effort 
%  choice in the HJB equation (because solving that as well is very hard, while this is easy).
%
%  For the same reason, assume that s (research allocation) jumps to its steady state value.
%  So the only source of transition dynamics is in the evolution of the distribution of x.
%
%  Discretize: 200,000 entrepreneurs, with initial x drawn from initial SS distribution.
%   - Shock the economy (e.g. increase phiH) ==> new SS effort choice e* and new s* (==> delta* and g*)
%   - Let the discretized GBM evolve, so that the distribution of {x} evolves.
%   - Keep track of eta and y:=Y/Lbar and show how they evolve over time.
%
%  This is a rough guide to transition dynamics in the model. Goal is to illustrate two things:
%   1. Transition dynamics are fast for our estimated pbar.
%   2. Observed growth over 10 or 20 years looks different from SS growth bc of the level effect.
%
%  Graphs: show the initial steady-state for 20 years, then the shock hits in 1975.
%     How quickly does eta rise to the new steady state?
%     What does GDP look like?

diarychad(['TransitionDynamicsMaster_' CaseName]);

IncomeTypeStr={'Wages/Salaries + Entr. Income',
               'Wages and Salaries           ',
               'Entrepreneurial Income only  '};


disp ' '; disp ' ';
disp '==========================================================================';
disp(['                     CaseName: ' CaseName]);
fprintf('                NumGridPoints: %6.0f\n',NumGridPoints);
fprintf('                PL_percentile: %8.5f\n',PL_percentile);
fprintf('                           dt:    1 /%3.0f\n',1/dt);
disp(['                  Income Type:  ' IncomeTypeStr{IRSCase}]);
disp '==========================================================================';
disp ' '; disp ' ';

% Create etaUS -- See FractalInequality.m
load WorldTopInequalityDatabase2016;
Tdata=2015;
yrs_data=(1970:Tdata)';
yy=yrs_data-yr0;
s3=is001(:,USA)./is01(:,USA);
etaUS=1+log10(s3(yy));


% Code from SSInequalitySolving.m for key cases:
%load NBER/PanelMuHResults;  % From PanelMuH.m
load PanelMuHResults;  % From PanelMuH.m
deltaCD=0.02 % 0.02             
%pBar=0.8  % To get the starting level right at around 0.40 for eta

% Value for pBar changes between the two types of income
% to get the baseline around 0.4
if IRSCase==3; % Entreprenueur;
    pBar_old=1
else;
    pBar_old=0.8
end;

mutildeH=Results(:,2,IRSCase); 
sigmaH=Results(:,3,IRSCase);
deltaBoth=Results(:,4,IRSCase);
deltabar=deltaBoth-deltaCD;

if ~isempty(HPsmoother);
    mutildeH=hpfilter(mutildeH,HPsmoother);
    sigmaH=hpfilter(sigmaH,HPsmoother);
    deltabar=hpfilter(deltabar,HPsmoother);
end;
sigmaL=2/3*sigmaH;

% Setup parameters
MyYears=(1980:1990)';
oo=ones(length(MyYears),1);
figname='TransitionDynamicsMaster';


% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Time path of inputs
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Initial conditions and exogenous s
eta0=0.39; % Initial value to fit...
s_old=.1;
s_new=.1 ;
g_old=.02*100;
g_new=.02*100;
muL_old=0.004;
mutildeL=muL_old-.5*sigmaL(1)^2;
pbar=pBar_old*oo;
Year0=1965;
YearT=2050;

% Now run the routine for transitiondynamics:
[eta,gdp,yrs,ytrend]=transitiondynamics(mutildeH,mutildeL,sigmaH,sigmaL,pbar,qbar,deltabar,NumGridPoints,alpha,s_old,s_new,g_old,yrs_data,etaUS,MyYears,Year0,YearT,figname,CaseName,dt,PL_percentile,eta0);
chadtimer

% Save the results for later analysis
eval(['data_' CaseName '=table(yrs,eta,gdp,ytrend);']);
diary off
