% MakeData.m 7/7/04 
%
% Chad's program to take the original data from our
%    sources and compute the basic 5-year interval data we will be using.
%
% Designed to replicate Bob's numbers in Bob7-3-04.xls to be sure we
% have it right.

if exist('MakeData.log'); delete('MakeData.log'); end;
diary MakeData.log;
fprintf(['MakeData                 ' date]);
disp ' ';
disp ' ';
help MakeData

clear all;

disp 'Basic parameters for the setup...';
MaxAge=100  % The oldest age we will consider
period=5
Nage=MaxAge/period
years=(1950:5:2000)'  % Our years
Tdata=length(years)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% POPULATION 
%  
% Population Data by age group:
% From Bob8-17-04.xls: HS_3 -- Census data for 1950
%		all 	Und 5 	5-14 	15-24 	25-34 	35-44 	45-54 	55-64 	65 years 
%Year		years 	years 	years 	years 	years 	years 	years 	years 	and over 

popages=[0 5; 5 14; 15 24; 25 34; 35 44; 45 54; 55 64; 65 MaxAge];
popdata=[
1950		151684	16331	24477	22260	23932	21557	17400	13364	12362;
1951		154287	17252	25055	22018	24085	21833	17623	13654	12768;
1952		156954	17228	26656	21796	24197	22109	17881	13918	13169;
1953		159565	17548	27880	21658	24233	22359	18171	14135	13582;
1954		162391	17962	29092	21641	24233	22571	18501	14350	14040;
1955		165275	18467	30248	21667	24175	22818	18824	14586	14489;
1956		168221	18895	31423	21869	24015	23160	19143	14815	14902;
1957		171274	19379	32515	22311	23737	23496	19513	14973	15353;
1958		174141	19768	33322	23162	23430	23693	19857	15139	15771;
1959		177073	20055	34390	23890	23062	23917	20189	15357	16213;
1960		177830	20175	34564	23988	23169	24023	20262	15401	16248;
1961		180671	20341	35735	24576	22919	24221	20578	15625	16675;
1962		183691	20522	37031	25242	22692	24392	20875	15847	17089;
1963		186538	20469	37435	26909	22494	24519	21124	16131	17457;
1964		189242	20342	38124	28223	22410	24584	21346	16436	17778;
1965		191889	20165	38783	29519	22396	24562	21580	16758	18127;
1966		194303	19824	39426	30773	22465	24447	21839	17077	18451;
1967		196560	19208	40051	32012	22725	24276	22125	17408	18755;
1968		198712	18563	40496	33196	23156	24038	22440	17752	19071;
1969		200706	17913	40772	34090	23990	23731	22758	18088	19365;
1970		202677	17376	40884	35236	24681	23383	23047	18390	19680;
1971		205052	17166	40772	36535	25323	23150	23317	18682	20107;
1972		207661	17244	40490	37948	25958	22978	23519	18963	20561;
1973		209896	17101	39946	38449	27623	22859	23687	19211	21020;
1974		211909	16851	39309	39240	28939	22810	23807	19428	21525;
1975		213854	16487	38716	40017	30225	22826	23809	19713	22061;
1976		215973	16121	38240	40812	31471	22831	23756	20045	22696;
1977		218035	15617	37759	41520	32759	23093	23622	20386	23278;
1978		220239	15564	37034	42039	33998	23563	23370	20780	23892;
1979		222585	15735	36220	42442	34963	24437	23174	21112	24502;
1980		225055	16063	35392	42699	36203	25176	22942	21448	25134;
1981		227225	16451	34839	42497	37429	25805	22743	21754	25707;
1982		229466	16893	34360	42204	38900	26394	22580	21913	26221;
1983		231664	17228	34103	41644	39418	28030	22425	22030	26787;
1984		233792	17547	33922	41020	40208	29267	22356	22112	27361;
1985		235825	17695	33788	40460	40962	30503	22386	22155	27877;
1986		237924	17842	33692	39992	41696	31691	22460	22135	28415;
1987		240133	17963	33572	39557	42372	33009	22659	21994	29008;
1988		242289	18052	33807	38890	42841	34227	23096	21751	29625;
1989		244499	18195	34255	38151	43130	35181	23949	21513	30123;
1990		246819	18508	34714	37391	43236	36415	24633	21241	30682;
1991		249623	18856	35290	36922	43216	37783	25207	21102	31247;
1992		252981	19208	36037	36504	43224	39329	25820	21046	31812;
1993		256514	19528	36677	36434	42997	39976	27548	20998	32356;
1994		259919	19729	37367	36464	42653	40912	28864	21028	32903;
1995		263126	19777	38026	36532	42330	41819	30153	21159	33331;
1996		266278	19627	38645	36674	42052	42711	31480	21320	33769;
1997		269394	19408	39296	36797	41809	43552	32800	21590	34143;
1998		272647	19233	39855	37308	41345	44229	34178	22099	34401;
1999		275854	19145	40335	38007	40757	44748	35232	23011	34619;
2000		279040	19136	40819	38676	40178	45077	36578	23778	34798;
2001		282224	19212	41077	39357	39857	45154	38052	24436	35081;
2002		285318	19364	41118	39982	39815	45141	39228	25315	35352;
2003		288369	19609	41037	40590	39928	44917	40084	26602	35602];

popyrs=popdata(:,1);
popdata(:,1:2)=[];  % cut off both the years and the total pop.
pop=popdata(years-1949,:);

% To get the 65-74 and 75+ categories, Bob assumes 60 percent in the former...
pop65up=pop(:,size(pop,2));
pop6574=.6*pop65up;
pop75up=pop65up-pop6574;
pop(:,size(pop,2))=[];
pop=[pop pop6574 pop75up];

disp ' '; disp ' ';
disp 'POPULATION DATA';
tle='Year 0-5 5-14 15-24 25-34 34-44 45-54 55-64 65-74 75+';
fmt='%5.0f %8.0f';
cshow(' ',[years pop],fmt,tle);

% Finally, load Bob's data for 1950 and Births
%  See HS_3 and Pop.  Notice that he uses Census data here
%  instead of the July data and to get higher age categories
%  he uses the mortality rates to distribute the population.

%  Age distribution of the population in 1950 (for 0-4 5-9 10-14 15-19 etc)
N1950=[16.164,12.159,12.159,11.049,11.049,11.880,11.880,10.725,10.725,8.671,8.671,6.647,6.647,4.056,3.289,2.403,1.473,0.728,0.236,0.076]'
%  Births
Births19502000=[16.164,18.242,20.321,18.738,17.154,16.751,16.348,17.351,18.354,18.765,19.176]



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% AGGREGATES (NIPA)
%
% From Bob7-3-04.xls Aggspend (checked from the NIPA tables he copies)
% Price is the pce deflator.  

%	Health		Total	        Price     Population
%	cons	gov	cons	gov              
nipa=[
1950	9.40	3.10	192.20	46.80   16.675     151326;
1951	10.20	3.10	208.50	68.10   17.805     154191;
1952	11.20	3.20	219.50	83.90   18.169     157057;
1953	12.20	3.20	233.10	90.80   18.416     159922;
1954	13.30	3.10	240.00	86.50   18.585     162787;
1955	14.20	3.30	258.80	86.80   18.676     165653;
1956	15.50	3.50	271.70	91.80   19.059     168518;
1957	17.10	3.90	286.90	100.10  19.639     171383;
1958	18.70	4.10	296.20	106.50  20.117     174248;
1959	20.60	4.30	317.60	112.50  20.432     177114;
1960	22.20	4.60	331.70	113.80  20.767     179979;
1961	23.90	5.00	342.10	121.50  20.985     182992;
1962	26.50	5.20	363.30	132.20  21.232     185771;
1963	28.70	5.60	382.70	138.50  21.479     188483;
1964	32.30	6.10	411.40	145.10  21.786     191141;
1965	34.70	6.40	443.80	153.70  22.103     193526;
1966	38.00	7.00	480.90	174.30  22.662     195576;
1967	41.40	7.80	507.80	195.30  23.237     197457;
1968	47.70	8.70	558.00	212.80  24.151     199399;
1969	54.20	9.80	605.20	224.60  25.255     201385;
1970	61.30	11.40	648.50	237.10  26.448     203984;
1971	68.50	12.80	701.90	251.00  27.574     206827;
1972	76.70	14.40	770.60	270.10  28.528     209284;
1973	85.30	16.00	852.40	287.90  30.081     211357;
1974	95.50	18.40	933.40	322.40  33.191     213342;
1975	109.90	21.10	1034.40	361.10  35.955     215465;
1976	124.70	22.30	1151.90	384.50  37.948     217563;
1977	142.10	24.30	1278.60	415.30  40.410     219760;
1978	162.30	26.90	1428.50	455.60  43.248     222095;
1979	183.30	28.50	1592.20	503.50  47.059     224567;
1980	209.60	33.00	1757.10	569.70  52.078     227225;
1981	245.20	35.40	1941.10	631.40  56.720     229466;
1982	274.80	36.40	2077.30	684.40  59.859     231664;
1983	310.00	38.20	2290.60	735.90  62.436     233792;
1984	343.70	41.40	2503.30	800.80  64.795     235825;
1985	376.40	44.10	2720.30	878.30  66.936     237924;
1986	407.40	46.20	2899.70	942.30  68.569     240133;
1987	447.60	50.00	3100.20	997.90  70.947     242289;
1988	505.00	53.80	3353.60	1036.90 73.755     244499;
1989	561.90	57.50	3598.50	1100.20 76.972     246819;
1990	635.10	62.90	3839.90	1181.40 80.498     249623;
1991	692.90	66.20	3986.10	1235.50 83.419     252981;
1992	761.10	67.90	4235.30	1270.50 85.824     256514;
1993	809.00	69.80	4477.90	1293.00 87.804     259919;
1994	853.30	72.80	4743.30	1327.90 89.654     263126;
1995	905.00	72.60	4975.80	1372.00 91.577     266278;
1996	950.70	74.90	5256.80	1421.90 93.547     269394;
1997	1002.80	74.60	5547.40	1487.90 95.124     272647;
1998	1069.40	76.90	5879.50	1538.50 95.978     275854;
1999	1130.80	82.00	6282.50	1641.00 97.575     279040;
2000	1218.30	92.70	6739.40	1751.00 100.000    282125];
%2001	1322.80	99.30	7045.40	1858.00 102.039	   284797];

nipayrs=nipa(:,1);
h=nipa(:,2)+nipa(:,3);
aggCG=nipa(:,4)+nipa(:,5);
pC=nipa(:,6);
poptot=nipa(:,7);
datahealth=h./aggCG;
y=aggCG./poptot./pC*100*10^6;   % Real resources per person
H=h./pC*100;   % Real Aggregate Health spending in millions of dollars

tle='Year H/Y y yindx Hagg';
data=[nipayrs datahealth y y/y(2000-1949) H];
cshow(' ',data(years-1949,:),'%5.0f %7.3f %7.0f %7.3f %7.1f',tle);

c=(1-datahealth).*y;
fprintf('Average Cons growth rate, 1950-2000: %8.4f\n',1/50*log(c(2000-1949)/c(1)));
disp ' ';


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% HEALTH SPENDING BY AGE
%
% The Meera, Whited, Cutler data on health spending
% From Bob7-3-04.xls: MWCdata
%Year   0-5	5-14	15-24	25-34	34-44	45-54	55-64	65-74	75+
mwc=[
1963   0.3160	0.3150	0.7530	0.9760	1.0000	1.2350	1.2380	1.4990	2.4840;
1970   0.7160	0.4370	1.3620	1.2140	1.0000	1.4150	2.2650	2.9020	5.6100;
1977   0.5020	0.3840	0.7230	0.9620	1.0000	1.3200	1.9130	2.7420	5.9110;
1987   0.8160	0.4790	0.8280	0.9590	1.0000	1.6030	2.2180	4.1350	7.8610;
1996   0.7260	0.4460	1.0480	0.7750	1.0000	1.3500	2.2510	3.1530	7.2130;
2000   0.5270	0.5080	0.5800	0.8260	1.0000	1.4210	2.0150	3.1440	5.4460];
   
mwcages=[0 5; 5 14; 15 24; 25 34; 34 44; 45 54; 55 64; 65 74; 75 MaxAge];
mwcyears=mwc(:,1);
mwc(:,1)=[];

% First, interpolate to get our years
[TT,KK]=size(mwc);
mwcinterp=zeros(length(years),KK);
useclosest=1;
for i=1:KK;
  mwcinterp(:,i)=interplin5(mwc(:,i),mwcyears,years,useclosest);
end;

% Show the data
disp ' '; disp ' ';
disp 'MWC DATA, Interpolated: mcwinterp';
tle='Year 0-5 5-14 15-24 25-34 34-44 45-54 55-64 65-74 75+';
fmt='%5.0f %8.3f';
cshow(' ',[years mwcinterp],fmt,tle);


% Now, we want to allocate Hagg using this distribution from MWC
% Eventually we are looking for per capita spending in each age group.
%  Key equations:  H = sum_a N(a)*h(a)   and mwc(a)=h(a)/h3544.  Can solve!
h3544=H(years-1949)./sum(pop'.*mwcinterp')';
hdollars=mult(mwcinterp,h3544)*10^6;

disp ' '; disp ' ';
disp 'REAL PER CAPITA HEALTH SPENDING: hdollars';
cshow(' ',[years hdollars],'%5.0f %8.0f',tle);

h10=hdollars/y(2000-1949);  % In units s.t. y(2000)==1
disp ' '; disp ' ';
disp 'In units s.t. y(2000)==1: h10';
cshow(' ',[years h10],'%5.0f %8.4f',tle);

% Finally, for splitting into the 5 year age groups.
% Bob follows the simple strategy of assuming 5-9 and 10-14 = 5-14
% and 15-19 and 20-24 = 15-24, etc.  
% Then, for all age groups greater than 75, we assume is the 75+ level.
% A simple, convenient split.

h=zeros(Tdata,Nage);   % Tdata x Nage for now...
h(:,1)=h10(:,1);  % 0-4 = 0-5
szh10=size(h10,2);
for i=2:szh10;
  j=2*(i-1);  % indx into the new h matrix
  h(:,j)=h10(:,i);
  h(:,j+1)=h10(:,i);
end;
for i=j+2:Nage;  % Fill the remaining ages with the oldest
  h(:,i)=h10(:,szh10);
end;
disp ' '; disp ' ';
disp 'By 5 year age categories: h';
tle=['Year ' vec2str(0:5:MaxAge-5)];
cshow(' ',[years h],'%5.0f %8.4f',tle);

disp 'Check:  This completes the check of the "Spending" spreadsheet.';


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% MORTALITY Now construct the mortality data
%
% %   Load and explore the age-specific mortality data  from
%   the NCHS Health United States 2003 publication.
%     Table 35  (see the xls file).
%
%  The basic approach here is the following:
%
%   1.  Assign the mortality rates to every (annual) age in the category
%   2.  Compute a survival probability from one age group to the next
%   3.  Apply annually to aggregate to the 5-year time categories
%   4.  The mortality rate for the age category is 1-survprob.


%  Age 1950 1960 1970 1980 1987 1990 1996 1997 1998 1999 2000 2001
mort=[
1	3299.2	2696.4	2142.4	1288.3	1037.2	971.9	760.7	747.6	754.0	736.0	736.7	683.4;
3	139.4	109.1	84.5	63.9	52.1	46.8	38.0	35.5	34.1	34.2	32.4	33.3;
10	60.1	46.6	41.3	30.6	25.9	24.0	21.2	20.2	19.3	18.6	18.0	17.3;
20	128.1	106.3	127.7	115.4	97.8	99.2	88.2	84.6	80.6	79.3	79.9	80.7;
30	178.7	146.4	157.4	135.5	134.7	139.2	122.3	110.1	104.3	102.2	101.4	105.2;
40	358.7	299.4	314.5	227.9	214.6	223.2	220.5	202.2	198.6	198.0	198.9	203.6;
50	853.9	756.0	730.0	584.0	501.9	473.4	440.0	423.9	415.8	418.2	425.6	428.9;
60	1901.0	1735.1	1658.8	1346.3	1256.6	1196.9	1082.6	1049.8	1015.7	1005.0	992.2	964.6;
70	4104.3	3822.1	3582.7	2994.9	2789.4	2648.6	2517.4	2484.9	2471.6	2457.3	2399.1	2353.3;
80	9331.1	8745.2	8004.4	6692.6	6232.2	6007.2	5755.9	5676.6	5672.8	5714.5	5666.5	5582.4;
90	20196.9	19857.5	16344.9	15980.3	15559.6	15327.4	15192.6	15211.2	15190.8	15554.6	15524.4	15112.8];

mage=[0 1; 1 4; 5 14; 15 24; 25 34; 35 44; 45 54; 55 64; 65 74; 75 84; 85 MaxAge];
mort(:,1)=[];
myrs=[1950 1960 1970 1980 1987 1990 1996 1997 1998 1999 2000 2001];

% First show the raw data
disp ' '; disp ' ';
disp '%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%';
disp 'MORTALITY DATA';
disp '%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%';
tle=['Age Age ' vec2str(myrs)];
cshow(' ',[mage mort],'%5.0f %5.0f %8.1f',tle);

mort=mort./100000;   % To turn into a mortality rate

% Just take 1950:10:2000:
myrs=(1950:10:2000);
mort=mort(:,[1:4 6 11]);

ages=(0:MaxAge-1)';
mannaul=zeros(MaxAge,length(myrs));
for i=1:MaxAge;
  age=ages(i);
  indx=find(mage(:,1)<=age); 
  indx=indx(length(indx));
  mannual(i,:)=mort(indx,:);
end;
tle=['Age ' vec2str(myrs)];
cshow(' ',[ages mannual],'%5.0f %9.6f',tle);

newages=(0:5:MaxAge-5)';
testages=[newages; MaxAge];
m10=zeros(Nage,length(myrs));  % Nage x (1950 1960 ... 2000)
for i=1:Nage;
  indx=find((ages>=testages(i)) & (ages<testages(i+1)));
  p=1-mannual(indx,:);  % Survival probabilities
  p5=cumprod(p);
  m10(i,:)=1-p5(5,:);
end;
disp ' '; disp ' ';
disp 'Mortality rates by age category';
cshow(' ',[newages m10],'%5.0f %9.6f',tle);
  

% Finally, interpolate over time by averaging: 1955 = .5*(1950+1960) etc.
newyrs=(1950:5:2000)';
m=zeros(Nage,length(newyrs));
m(:,[1 3 5 7 9 11])=m10;  % The decadal numbers
m(:,[2 4 6 8 10])=0.5*(m10(:,1:5)+m10(:,2:6));
m=m';  % Transpose to match up with Bob's presentation
tle=['Age ' vec2str(newages)];
cshow(' ',[newyrs m],'%5.0f %8.4f',tle);

disp ' '; disp ' ';
disp 'Inverse mortality rates: x';
x=1./m;
cshow(' ',[newyrs x],'%5.0f %8.2f',tle);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% MATCH to Bob's results
% Confirm that h and x match up with Bob's data

% Copied from Bob7-7-04.xls
hbob=[
0.00579	0.00577	0.00577	0.01379	0.01379	0.01788	0.01788	0.01832	0.01832	0.02262	0.02262	0.02268	0.02268	0.02746	0.02746	0.04550	0.04550	0.04550	0.04550	0.04550;
0.00670	0.00668	0.00668	0.01597	0.01597	0.02070	0.02070	0.02121	0.02121	0.02619	0.02619	0.02625	0.02625	0.03179	0.03179	0.05268	0.05268	0.05268	0.05268	0.05268;
0.00864	0.00861	0.00861	0.02059	0.02059	0.02669	0.02669	0.02735	0.02735	0.03377	0.03377	0.03386	0.03386	0.04099	0.04099	0.06793	0.06793	0.06793	0.06793	0.06793;
0.01356	0.01102	0.01102	0.02920	0.02920	0.03289	0.03289	0.03150	0.03150	0.04053	0.04053	0.04825	0.04825	0.05985	0.05985	0.10639	0.10639	0.10639	0.10639	0.10639;
0.02300	0.01404	0.01404	0.04375	0.04375	0.03900	0.03900	0.03212	0.03212	0.04545	0.04545	0.07276	0.07276	0.09322	0.09322	0.18021	0.18021	0.18021	0.18021	0.18021;
0.02478	0.01756	0.01756	0.03985	0.03985	0.04550	0.04550	0.04400	0.04400	0.05928	0.05928	0.08861	0.08861	0.12267	0.12267	0.25633	0.25633	0.25633	0.25633	0.25633;
0.03030	0.02097	0.02097	0.03835	0.03835	0.04885	0.04885	0.05083	0.05083	0.07141	0.07141	0.10188	0.10188	0.16061	0.16061	0.33017	0.33017	0.33017	0.33017	0.33017;
0.04349	0.02656	0.02656	0.04659	0.04659	0.05540	0.05540	0.05774	0.05774	0.08928	0.08928	0.12454	0.12454	0.22266	0.22266	0.43135	0.43135	0.43135	0.43135	0.43135;
0.05831	0.03472	0.03472	0.06686	0.06686	0.06659	0.06659	0.07418	0.07418	0.11266	0.11266	0.16535	0.16535	0.28247	0.28247	0.56713	0.56713	0.56713	0.56713	0.56713;
0.06593	0.04028	0.04028	0.09168	0.09168	0.07125	0.07125	0.08957	0.08957	0.12344	0.12344	0.20130	0.20130	0.29220	0.29220	0.65255	0.65255	0.65255	0.65255	0.65255;
0.06157	0.05935	0.05935	0.06777	0.06777	0.09651	0.09651	0.11684	0.11684	0.16602	0.16602	0.23543	0.23543	0.36733	0.36733	0.63629	0.63629	0.63629	0.63629	0.63629];

xbob=[
26.06014	333.17894	333.17894	156.52854	156.52854	112.32013	112.32013	56.15834	56.15834	23.82538	23.82538	10.92845	10.92845	5.28969	5.28969	2.58239	2.58239	1.47856	1.47856	1.47856;
28.74548	375.28947	375.28947	171.05240	171.05240	123.44367	123.44367	61.18561	61.18561	25.25099	25.25099	11.40898	11.40898	5.46312	5.46312	2.65104	2.65104	1.48619	1.48619	1.48619;
32.04781	429.58474	429.58474	188.54718	188.54718	137.01261	137.01261	67.20147	67.20147	26.85806	26.85806	11.93371	11.93371	5.64830	5.64830	2.72344	2.72344	1.49389	1.49389	1.49389;
35.75848	455.46420	455.46420	171.34399	171.34399	132.06670	132.06670	65.55866	65.55866	27.32101	27.32101	12.19290	12.19290	5.81742	5.81742	2.82383	2.82383	1.58769	1.58769	1.58769;
40.44097	484.66167	484.66167	157.01758	157.01758	127.46543	127.46543	63.99426	63.99426	27.80019	27.80019	12.46360	12.46360	5.99697	5.99697	2.93190	2.93190	1.69404	1.69404	1.69404;
49.83672	556.73724	556.73724	164.94285	164.94285	136.96821	136.96821	74.15762	74.15762	30.84901	30.84901	13.72115	13.72115	6.49791	6.49791	3.15550	3.15550	1.70707	1.70707	1.70707;
64.91966	653.99489	653.99489	173.71069	173.71069	148.00202	148.00202	88.15870	88.15870	34.64892	34.64892	15.26095	15.26095	7.09018	7.09018	3.41603	3.41603	1.72029	1.72029	1.72029;
74.14229	733.00669	733.00669	186.79600	186.79600	146.01402	146.01402	89.07308	89.07308	38.23516	38.23516	16.13473	16.13473	7.50079	7.50079	3.57708	3.57708	1.74509	1.74509	1.74509;
86.41918	833.73343	833.73343	202.01330	202.01330	144.07872	144.07872	90.00663	90.00663	42.64947	42.64947	17.11465	17.11465	7.96189	7.96189	3.75408	3.75408	1.77062	1.77062	1.77062;
98.89078	952.78920	952.78920	223.74392	223.74392	166.66141	166.66141	95.16645	95.16645	44.89682	44.89682	18.68029	18.68029	8.33562	8.33562	3.85088	3.85088	1.76275	1.76275	1.76275;
115.56916	1111.51118	1111.51118	250.71321	250.71321	197.63906	197.63906	100.95384	100.95384	47.39419	47.39419	20.56121	20.56121	8.74617	8.74617	3.95281	3.95281	1.75496	1.75496	1.75496];


herr=(h-hbob)./h;
xerr=(x-xbob)./x;
disp ' '; disp ' ';
disp '%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%';
disp 'CONFIRMING THAT THESE DATA MATCH BOBs RESULTS';
disp '%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%';
disp ' '; disp ' '; disp 'Percentage differences in h';
cshow(' ',[newyrs herr],'%5.0f %8.3f',tle);
disp ' '; disp ' '; disp 'Percentage differences in x';
cshow(' ',[newyrs xerr],'%5.0f %8.3f',tle);

if any(any(abs(herr)>.001)); disp 'herr problem'; end;
if any(any(abs(xerr)>.001)); disp 'xerr problem'; end;

% Life Expectancy data
%   Data from Table 12 of National Vital Statistics Report Volume 51, Number 3 "United
%   States Life Tables, 2000", December 19, 2002. Center for Disease Control.

lifedata=[
 2000         76.9;
 1999         76.7;
 1998         76.7;
 1997         76.5;
 1996         76.1;
 1995         75.8;
 1994         75.7;
 1993         75.5;
 1992         75.8;
 1991         75.5;
 1990         75.4;
 1989         75.1;
 1988         74.9;
 1987         74.9;
 1986         74.7;
 1985         74.7;
 1984         74.7;
 1983         74.6;
 1982         74.5;
 1981         74.1;
 1980         73.7;
 1979         73.9;
 1978         73.5;
 1977         73.3;
 1976         72.9;
 1975         72.6;
 1974         72.0;
 1973         71.4;
 1972         71.2;
 1971         71.1;
 1970         70.8;
 1969         70.5;
 1968         70.2;
 1967         70.5;
 1966         70.2;
 1965         70.2;
 1964         70.2;
 1963         69.9;
 1962         70.1;
 1961         70.2;
 1960         69.7;
 1959         69.9;
 1958         69.6;
 1957         69.5;
 1956         69.7;
 1955         69.6;
 1954         69.6;
 1953         68.8;
 1952         68.6;
 1951         68.4;
 1950         68.2];

lifedata=flipud(lifedata);
years=lifedata(:,1);
life=lifedata(:,2);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ACCIDENTS AND HOMICIDE MORTALITY
%   Chad:  July 18, 2005
%   See accidents.xls for the data, from VSUS and Health United States 2004
% Use the same code as above to manipulate the data (see mort section).

mAccidents=[
%	1950	1960	1970	1980	1990	2000
1	114.9	93.2	65.8	33.0	23.6	22.9;
3	38.3	31.6	31.5	25.9	17.3	12.1;
10	25.7	19.3	20.1	15.0	10.4	7.5;
20	77.1	56.0	68.7	61.7	43.9	36.8;
30	61.9	43.0	51.6	46.3	37.0	31.4;
40	57.9	40.9	48.2	37.3	31.3	34.3;
50	64.5	49.8	53.5	39.0	29.4	33.0;
60	85.1	59.0	63.2	42.6	34.3	31.3;
70	136.2	88.1	85.6	57.7	46.6	42.4;
80	348.0	214.5	174.7	120.3	100.3	95.5;
90	1005.6	624.7	392.6	292.5	257.1	269.9];

mHomicides=[
%	1950	1960	1970	1980	1990	2000
1	4.4	4.8	4.3	5.9	8.4	9.2;
3	0.6	0.7	1.9	2.5	2.5	2.3;
10	0.5	0.5	0.9	1.2	1.5	0.9;
20	5.8	5.6	11.3	15.4	19.7	12.6;
30	9.3	9.2	16.2	19.3	17.4	10.4;
40	8.4	7.8	13.5	14.9	11.6	7.1;
50	5.9	6.1	10.0	11.0	7.5	4.7;
60	3.9	4.1	7.1	7.0	5.0	3.0;
70	3.2	2.8	4.9	5.7	3.8	2.4;
80	2.5	2.3	4.0	5.2	4.3	2.4;
90	2.3	2.4	4.2	5.3	4.6	2.4];
    
% The basic adjusted mortality series:  subtract off
%  the accidents and homicides.
mAccidents(:,1)=[];
mHomicides(:,1)=[];
mortAdj=mort*100000-mAccidents-mHomicides;

mage=[0 1; 1 4; 5 14; 15 24; 25 34; 35 44; 45 54; 55 64; 65 74; 75 84; 85 MaxAge];
myrs=[1950 1960 1970 1980 1990 2000];

% First show the raw data
disp ' '; disp ' ';
disp '%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%';
disp 'MORTALITY DATA (Adjusted):  Subtracts Accidents and Homicides';
disp '%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%';
tle=['Age Age ' vec2str(myrs)];
cshow(' ',[mage mortAdj],'%5.0f %5.0f %8.1f',tle);

mortAdj=mortAdj./100000;   % To turn into a mortality rate
blah=(mAccidents+mHomicides)/100000;
disp ' '; disp 'Mortality Rates per 100000 for Accidents/Homicides';
tle=['Age Age ' vec2str(myrs)];
cshow(' ',[mage blah*100000],'%5.0f %5.0f %8.1f',tle);


% Just take 1950:10:2000:
myrs=(1950:10:2000);

ages=(0:MaxAge-1)';
mannaul=zeros(MaxAge,length(myrs));
for i=1:MaxAge;
  age=ages(i);
  indx=find(mage(:,1)<=age); 
  indx=indx(length(indx));
  mannual(i,:)=mortAdj(indx,:);
end;
tle=['Age ' vec2str(myrs)];
cshow(' ',[ages mannual],'%5.0f %9.6f',tle);

newages=(0:5:MaxAge-5)';
testages=[newages; MaxAge];
m10=zeros(Nage,length(myrs));  % Nage x (1950 1960 ... 2000)
for i=1:Nage;
  indx=find((ages>=testages(i)) & (ages<testages(i+1)));
  p=1-mannual(indx,:);  % Survival probabilities
  p5=cumprod(p);
  m10(i,:)=1-p5(5,:);
end;
disp ' '; disp ' ';
disp 'Mortality rates by age category (Adjusted for Accidents and Homicides)';
cshow(' ',[newages m10],'%5.0f %9.6f',tle);
  

% Finally, interpolate over time by averaging: 1955 = .5*(1950+1960) etc.
newyrs=(1950:5:2000)';
mAdj=zeros(Nage,length(newyrs));
mAdj(:,[1 3 5 7 9 11])=m10;  % The decadal numbers
mAdj(:,[2 4 6 8 10])=0.5*(m10(:,1:5)+m10(:,2:6));
mAdj=mAdj';  % Transpose to match up with Bob's presentation
tle=['Age ' vec2str(newages)];
cshow(' ',[newyrs mAdj],'%5.0f %8.4f',tle);

disp ' '; disp ' ';
disp 'Inverse mortality rates (adjusted): xAdj';
xAdj=1./mAdj;
cshow(' ',[newyrs xAdj],'%5.0f %8.2f',tle);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%  COMPUTING Life Expectancy
%
%  For fun, let's compute life expectancy using both m and mAdj
%  Life expectancy calculation:
%  kVS = vital statistics version, i.e. use year t mortality rates
%m=1-p;
kVS=zeros(Nage,Tdata);
for a=(Nage-1):(-1):1;
   kVS(a,:)=(1-m(:,a)').*(kVS(a+1,:)+5);
end;
LifeExp=kVS(1,:)+3;
disp ' '; disp ' ';
disp 'Life Expectancy for base mortality data (for at birth, add 3 years)';
cshow(' ',[newyrs kVS'],'%5.0f %8.1f',tle);

kVSAdj=zeros(Nage,Tdata);
for a=(Nage-1):(-1):1;
   kVSAdj(a,:)=(1-mAdj(:,a)').*(kVSAdj(a+1,:)+5);
end;
LifeExpAdj=kVSAdj(1,:)+3;
disp ' '; disp ' ';
disp 'Life Expectancy absent Accidents and Homicides (for at birth, add 3)';
cshow(' ',[newyrs kVSAdj'],'%5.0f %8.1f',tle);



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%5
% CLEANUP 

hdata=h;
mAlldata=m;
xAlldata=x;
ydata=y;
mAdjdata=mAdj;
xAdjdata=xAdj;
mAccHom=mAlldata-mAdjdata;  % The accidents/homicides mortality rate.
popdata=pop;
datalife=life;

% datahealth already the health share
save HealthData hdata xAlldata mAlldata xAdjdata mAdjdata mAccHom popdata datalife datahealth aggCG ydata years N1950 Births19502000

diary off;



