% Simulations.m   2/1/06
%
%  The meta-program to run a bunch of simulations of the 
%  Hall-Jones health model.%  
%
%  Choices:  VSL2000 = Value of life of 35-39 year old in 2000 (e.g. 2*10^6)
%            gamma = CRRA
%            FracFromH = Fraction of age-specific trend in mortality that
%                        is due to health spending (e.g. .60)
%            alpha, sigma = Parameters that govern quality of life
%                        (e.g. alpha=0 -- shuts it off, or
%                              alpha=1.3, sigma=1.1
%
%  See examples below...

if exist('Simulations.log'); delete('Simulations.log'); end;
diary Simulations.log;
fprintf(['Simulations                 ' date]);
disp ' ';
disp ' ';
help Simulations

clear all;

global b gamma alpha sigma Phi Theta gTFP gy bdata vprimedata
global tfp beta dlta Nage Tsim Tdata N0 y y2000 periodlength MaxAge agecat % Created by solvesetup...

params=[
    % VSL   gamma FracFromHz zgrowth alphasigma
     3*10^6   2     2/3         0       0     ;   % Baseline scenario
     3*10^6   2.5   2/3         0       0     ;   % gamma=2.5
     3*10^6   1.5   2/3         0       0     ;   % gamma=1.5
     3*10^6   1.01  2/3         0       0     ;   % near log
     4*10^6   2     2/3         0       0     ;   % $4 million VSL
     5*10^6   2     2/3         0       0     ;   % $5 million VSL
     3*10^6   2     2/3         0       1     ;   % Estimate alpha and sigma as well
     3*10^6   2     2/3        0.01     0     ;   %    zgrowth=.01
     3*10^6   2     1/2         0       0     ;   %    50% from hz
]; % End of params

Baseline=1
numScen=size(params,1);
Years=(1950:5:2095)';
Nage=20;
yy=zeros(length(Years),numScen)*NaN;
HealthShare=yy; LifeExp=yy; 
Bvalues=zeros(numScen,1)*NaN; Gamma=zeros(numScen,1)*NaN;
Alpha=zeros(numScen,1)*NaN; Sigma=zeros(numScen,1)*NaN;
Bat=zeros(Nage,numScen)*NaN;
VTech2000=zeros(Nage,numScen)*NaN;  % MC of saving a life in 2000


for i=1:numScen;
  VSL=params(i,1);
  gamma=params(i,2);
  FracFromHz=params(i,3);
  zgrowth=params(i,4);
  alphasigma=params(i,5);
  alpha=0; sigma=1.5;  % defaults...
  disp ' '; disp ' ';
  disp '_________________________________________________________________';
  disp ' ';
  disp '%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%';
  disp '%';
  fprintf(' Scenario %2.0f\n',i);
  fprintf('   VSL=%10.0f gamma=%6.2f alphasigma=%6.4f\n',params(i,[1 2 5]));
  fprintf('   FracFromHz=%4.2f  zgrowth=%5.3f\n',params(i,[3 4]));
  disp '%';
  disp '%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%';
  disp '_________________________________________________________________';
  disp ' '; disp ' ';

  % Setup the data and estimate health production functions
  solvesetup([]);
  [Phi,Theta,gTFP,tfp,se,vv,xfit]=estimatetheta(FracFromHz,zgrowth);
 
  % Estimate b,alpha,sigma to match VSL2000 (and QALY weights)
  [b,alpha,sigma]=estimateb(VSL,alphasigma)
  [hshare,lifeexp,h]=solvemodel([]);  % No need to pass anything; globals;
  Bvalues(i)=b;
  if i==Baseline; hSim=h; PhiSim=Phi; ThetaSim=Theta; seSim=se; VTech=vv; Xfit=xfit; end;  % Save age-specific spending for this run for plotting

  VTech2000(:,i)=vv(:,11);
  HealthShare(:,i)=hshare;
  LifeExp(:,i)=lifeexp;
  Gamma(i)=gamma;
  Alpha(i)=alpha;
  Sigma(i)=sigma;
end;

disp ' '; disp ' '; disp ' ';
disp '**************************************************************';
disp '**************************************************************';
disp '**************************************************************';
disp '***                                                        ***';
disp '***                                                        ***';
disp '***           SIMULATION RESULTS SUMMARY                   ***';
disp '***                                                        ***';
disp '***                                                        ***';
disp '**************************************************************';
disp '**************************************************************';
disp '**************************************************************';
disp ' ';

disp ' ';
tle='Scenario VSL2000 gamma FracFromHz zgrowth Alpha Sigma bsingle';
fmt='%7.0f %12.0f %11.2f %8.3f %8.3f %8.4f %8.4f %9.3f';
cshow(' ',[(1:numScen)' params(:,1:4) Alpha Sigma Bvalues],fmt,tle);

disp ' '; disp 'Marginal Cost of Saving a Life';
disp 'Technology Side:  h*x/theta';
tle='Age 1950 1980 2000 Range';
fmt='%5.0f %8.0f %8.0f %8.0f (%5.0f, %5.0f)';
vmin=min(VTech2000')'; 
vmax=max(VTech2000')'; 
agecat=(1:Nage)'*5;
cshow(' ',[agecat VTech(:,[1 7 11]) vmin vmax],fmt,tle);

disp ' '; disp 'Health Shares H/Y for the Scenarios';
tle=['Year ' num2str(1:numScen,'%2.0f')];
cshow(' ',[Years HealthShare*100],'%7.0f %8.2f',tle);

disp ' '; disp 'Life Expectancy for the Scenarios';
tle=['Year ' num2str(1:numScen,'%2.0f')];
cshow(' ',[Years LifeExp],'%7.0f %8.2f',tle);

clear h Phi Theta vv xfit se gamma alpha sigma zgrowth VSL
save Simulations;
diary off;


