function [b,alpha,sigma]=estimateb(VSL2000,alphasigma);

% function [b,gamma]=estimateb(VSL2000,alphasigma);
%
%   Estimates the constant in flow utility to match Value of Life:
%      VSL2000 = Value of life for 35-39 yr old in 2000, e.g. 2 million
%      alphasigma=1 if we need to estimate these as well...
%      Theta, Phi should be globals, already set.

global Nage Tsim Tdata y2000 tfp gamma beta dlta Theta Phi N0 gy y  % Inputs
global Vyear Vlife MatchAge p c cc x h y % Outputs


disp ' '; disp 'Running estimateb.m...';
if exist('alphasigma')~=1; alphasigma=0; end;

%  These are the Value of Life data we will match:  Year Age Value
data=[2000 40  VSL2000/1000]; % VSL in thousands
Vyear=(data(:,1)-1945)/5   % 11
MatchAge=(data(:,2)/5)
Vlife=data(:,3)


% Extrapolate age-specific data at the age-specific growth rate.
% Actually, we'll do this until 2020, then no growth afterwards in shareT=19 % go out 100 years to 2095 (consistent w/ Tsim=30)
load HealthData;  % hdata is 11x20
T=19   % Out to 2095
t=(1:T)';
gh=mean(delta(log(hdata)))  % The growth rate for each age group
gh=mean(gh')                % The average across all age groups.
keyyr=(2020-2000)/5;
gh=gh*ones(T,1);
gh(t>keyyr)=gy*5;   % only income growth after that
newh=zeros(T,Nage);
for a=1:Nage;
  newh(:,a)=hdata(Tdata,a)*exp(cumsum(gh));  % No need to multiply by 5
end;
h=[hdata; newh];  % This contains the full 1950-2100 age-specific data
h=h';  % Now the data are Nage x 30
newh=newh';


% Population setup
N=zeros(Nage,Tsim);
N=N0; % 1950 and fertility

%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%  Getting N(a,t) and c(t)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%  Given the h(a,t) fake data, do the following:
%    1.  x=f(h), p=1-1./x
%    2.  Construct N(a,t) by iterating forward
%    3.  Construct c(t) 
%    4.  Finally, given a gamma guess, we can iterate backwards to get V'

xnew=zeros(Nage,T);
for t=1:T;
  xnew(:,t)=1./(dlta(:,t)+1./exp(Phi+Theta.*log(tfp(:,t+Tdata).*newh(:,t))));
end;
x=[xAlldata' xnew];
p=1-1./x;


% Iterate forward to get the population
for t=1:(Tsim-1);
  for a=1:(Nage-1);
    N(a+1,t+1)=p(a,t)*N(a,t);
  end;
end;

% Compute consumption
Nt=sum(N);   %  1xT vector of population at each date
Hat=N.*h;    %  Total health spending at each age and date
Ht=sum(Hat);  % aggregate health spending at each date.
hbar=Ht./Nt;  % Per capita spending at each date.
c=y-hbar;
cc=kron(ones(Nage,1),c);  % The Nage x Tsim version of c

%  Life expectancy calculation to make sure x is not crazy
%  kVS = vital statistics version, i.e. use year t mortality rates
m=1-p;
kVS=zeros(Nage,Tsim);
for a=(Nage-1):(-1):1; 
   kVS(a,:)=(1-m(a,:)).*(kVS(a+1,:)+5); 
end;
LifeExp=kVS(1,:)+3;

disp 'Forecast health share and life expectancy paths';
yrs=(1950:5:2095)';
cshow(' ',[yrs hbar'./y' LifeExp'],'%5.0f %8.4f %8.2f','Year hbar/y LifeExp');


%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%  Estimation
%%%%%%%%%%%%%%%%%%%%%%%%%%%%

bguess=10
coeff0=[bguess]'
if alphasigma;
  coeff0=[coeff0; 3; 3]
end;

uestimateb(coeff0)

options=optimset('fsolve');
options=optimset(options,'MaxIter',1000,'MaxFunEvals',2000,'Display','iter','TolFun',.001);%,'TolX',.001);
tic;
coeff=fsolve(@uestimateb,coeff0,options);
toc


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Checking the results

b=coeff(1)
if alphasigma;
  alpha=coeff(2)
  sigma=coeff(3)
else;
  alpha=0;
  sigma=1.5;
end;

%agegroups=[5 14 18];  % 20-24, 65-69, 85-89
x20=xAlldata(9,5); x65=xAlldata(9,14); x85=xAlldata(9,18);
c1990=c(9);
frackeep=(1-alpha*(1-gamma)/(1-sigma)*c1990^(gamma-1)*(x20^(1-sigma)-x65^(1-sigma)))^(1/(1-gamma));
fprintf('A 65-69 year old would give up this frac to be 20-24: %8.4f\n',1-frackeep);
frackeep=(1-alpha*(1-gamma)/(1-sigma)*c1990^(gamma-1)*(x20^(1-sigma)-x85^(1-sigma)))^(1/(1-gamma));
fprintf('A 85-89 year old would give up this frac to be 20-24: %8.4f\n',1-frackeep);

u=b+1/(1-gamma)*cc.^(1-gamma)+alpha*1/(1-sigma)*x.^(1-sigma);
vprime=zeros(Nage,Tsim);  % Assume vprime(Nage,:)=0 (death)
% Guesses for end of simulation values:
for a=(Nage-1):-1:1;
  vprime(a,Tsim)=b+beta*vprime(a+1,Tsim);
end;

% Now solve iterating backwards for vprime:
for t=Tsim:-1:2; 
  for a=2:Nage;
    vprime(a-1,t-1)=u(a,t)+beta*p(a,t)*vprime(a,t) + c(t)^(-gamma)*(y(t)-c(t)-h(a,t));
  end;
end;


% Some value of life calculations
Quan=div(beta*vprime,c.^(-gamma))*y2000/1000;

myyears=yrs;
yrs=[1 3 5 7 9 11 13 15];
ages=[1 2 4 6 8 10 12 14 16 18 20]';

%%%%  Report some results...
disp ' ';
disp 'Health spending';
tle=['Age ' vec2str(myyears(yrs))];
fmt='%6.0f %8.3f';
cshow(' ',[ages*5 h(ages,yrs)],fmt,tle);

disp ' ';
disp 'Utility:  u';
cshow(' ',[ages*5 u(ages,yrs)],fmt,tle);

disp ' ';
disp 'vprime';
cshow(' ',[ages*5 vprime(ages,yrs)],fmt,tle);

disp ' ';
disp 'Survival Probability:  p';
cshow(' ',[ages*5 p(ages,yrs)],fmt,tle);

disp ' ';
disp 'Health Status:  x';
cshow(' ',[ages*5 x(ages,yrs)],fmt,tle);

disp ' ';
disp 'Value of Life (Quantity) in thousands of 2000$';
fmt='%6.0f %8.0f';
cshow(' ',[ages*5 Quan(ages,yrs)],fmt,tle);
disp ' ';
fprintf('At age 40 in the year 2000, Quan = %8.0f\n',Quan(8,11));



%------------------------------------------------%
% uestimateb -- the estimating function
%------------------------------------------------%
function resid=uestimateb(coeff);  % 2/1/06

%  Estimate b to match value of life for age category MatchAge 

global Tsim Vyear Vlife MatchAge y2000 gamma
global p beta c cc x Nage Tsim Vguess h y

b=coeff(1);
if length(coeff)>2;
  alpha=coeff(2);
  sigma=coeff(3);
else;
  alpha=0; sigma=1.5;  % shut off quality term
end;

u=b+1/(1-gamma)*cc.^(1-gamma)+alpha*1/(1-sigma)*x.^(1-sigma);
vprime=zeros(Nage,Tsim);  % Assume vprime(Nage,:)=0 (death)
% Guesses for end of simulation values:
for a=(Nage-1):-1:1;
  vprime(a,Tsim)=b+beta*vprime(a+1,Tsim);
end;

% Now solve iterating backwards for vprime:
for t=Tsim:-1:2;   % Iterate backwards
  for a=2:Nage;
    vprime(a-1,t-1)=u(a,t)+beta*p(a,t)*vprime(a,t) + c(t)^(-gamma)*(y(t)-c(t)-h(a,t));
  end;
end;

Quan=zeros(length(MatchAge),1);
for i=1:length(MatchAge);  % How many points we are matching
  Quan(i)=beta*vprime(MatchAge(i),Vyear(i))./c(Vyear(i)).^(-gamma)*y2000/1000;
  % Note: multiplying by y2000/1000 puts it in thousands of dollars.
end;
  
resid=Vlife-Quan;   %  So that u is Kx1

if alpha~=0;  % If we are estimating alpha and sigma...
  %agegroups=[5 14 18];  % 20-24, 65-69, 85-89
  u20=u(5,9); u65=u(14,9); u85=u(18,9);
  e1=u20/.94-u65/.73;
  e2=u65/.73-u85/.62;
  resid=[resid; e1*1000; e2*1000];
end;

