function [hshare,LifeExp,h]=solvemodel(emptymatrix);

% 2/1/06: Basic solve routine.  Take key parameter values as globals
%   Approach:  1. Guess a time path for c(t).
%              2. Use "solvefoc" function to compute the implied discrepancies
%                 in the resource constraint.
%              3. Find the c(t) path that eliminates these discrepancies.

global b gamma Phi Theta tfp beta dlta Nage Tsim N0 y y2000 periodlength opts

% Now make a guess for consumption and call the solver:
cguess=.90*y;
opts=optimset('fzero');
optsolve=optimset('fsolve');
optsolve=optimset(optsolve,'Display','iter','LargeScale','off','MaxFunEvals',1000,'MaxIter',100);
tic;
[c,fval,OK]=fsolve(@solvediscrep,cguess,optsolve);
toc

% Now recover the full solution:
[h,p,x,vprime,u]=solveh(c);

% Population
N=N0; % Baseline population for 1950 and fertility.
for t=1:(Tsim-1);
  for a=1:(Nage-1);
    N(a+1,t+1)=p(a,t)*N(a,t);
  end;
end;

%%%%%%%%%%%%%%%%%%%%%
% Report the results
%%%%%%%%%%%%%%%%%%%%%

yrs=[1 3 5 7 9 11 13 15];
ages=[1 2 4 6 8 10 12 14 16 18 20]';
myyears=(1950:periodlength:2095);

disp ' ';
disp 'Health spending';
fmt='%6.0f %9.3f';
tle=['Age ' vec2str(myyears(yrs))];
cshow(' ',[ages*5 h(ages,yrs)],fmt,tle);

disp ' ';
disp 'Utility:  u';
fmt='%6.0f %9.2f';
cshow(' ',[ages*5 u(ages,yrs)],fmt,tle);

disp ' ';
disp 'vprime';
cshow(' ',[ages*5 vprime(ages,yrs)],fmt,tle);

disp ' ';
disp 'Survival Probability:  p';
fmt='%6.0f %9.3f';
cshow(' ',[ages*5 p(ages,yrs)],fmt,tle);

disp ' ';
disp 'Health Status:  x';
fmt='%6.0f %9.1f';
cshow(' ',[ages*5 x(ages,yrs)],fmt,tle);

% Some value of life calculations
Quan=div(beta*vprime,c.^(-gamma))*y2000/1000;
disp ' ';
disp 'Value of Life (Quantity) in thousands of 2000$';
fmt='%6.0f %9.0f';
cshow(' ',[ages*5 Quan(ages,yrs)],fmt,tle);
disp ' ';
fprintf('At age 40 in the year 2000, Quan = %8.0f\n',Quan(8,11));


%  Life expectancy calculation:
%  kVS = vital statistics version, i.e. use year t mortality rates
m=1-p;
kVS=zeros(Nage,Tsim);
for a=(Nage-1):(-1):1; 
   kVS(a,:)=(1-m(a,:)).*(kVS(a+1,:)+5); 
end;
LifeExp=kVS(1,:)+3;  % Add 3 since initial age is 0-4.
LifeExp=LifeExp';
hshare=(1-c./y)';

disp ' ';
disp 'VSL/LE in thousands of 2000$';
fmt='%6.0f %9.0f';
cshow(' ',[ages*5 Quan(ages,yrs)./kVS(ages,yrs)],fmt,tle);

% Health Share and Life expectancy
disp ' '; disp ' ';
tle='Year h/y LifeExp';
cshow(' ',[myyears' hshare LifeExp],'%6.0f %8.3f %8.2f',tle);




%-----------------------------------------------------------%
%   SOLVEDISCREP function
%-----------------------------------------------------------%


function d=solvediscrep(c);

% function d=solvediscrep(c);   % 1/31/06
%
% Given a guess for consumption over time, c, solve the model
% and return the discrepancy, d, in the resource constraint.
%
% The nonlinear equation solver will call this function in order to
% find the consumption path that satisfies the resource constraint at
% each point in time.
%
%  N = N(a,t) contains the implied population distribution
%  vprime = v'= v(a+1,t+1) contains the value functions
%
% The routine currently assumes no quality of life effects

global b gamma Phi Theta tfp beta dlta Nage Tsim N0 y opts

[h,p,x,vprime]=solveh(c);  % Call the routine to solve the FOC.

% Now update the population, going forward
N=N0; % Baseline population for 1950 and fertility.
for t=1:(Tsim-1);
  for a=1:(Nage-1);
    N(a+1,t+1)=p(a,t)*N(a,t);
  end;
end;

% Finally, check the resource constraint:
Hat=h.*N; % Aggregate health spending by age x time.
Ht=sum(Hat); % Aggregate health spending over time.
Nt=sum(N); % Aggregate population
ht=Ht./Nt;    % Per capita health spending;
d=y-c-ht;     % The 1xTsim vector of discrepancies in the resource constraint.




%-----------------------------------------------------------%
% SOLVEH function
%-----------------------------------------------------------%

function [h,p,x,vprime,u]=solveh(c);

% function [h,p,x,vprime,u]=solveh(c);
%
%    Given a time path for consumption c(t), solves the FOC
%    for health spending and the implied p,x,vprime.

global b gamma Phi Theta tfp beta dlta Nage Tsim N0 y alpha sigma opts

% First, let's use the consumption path to solve for h(a,t)
% Everything solves backward, from date Tsim back to date 1.

h=zeros(Nage,Tsim);
vprime=zeros(Nage,Tsim);
p=zeros(Nage,Tsim);
x=zeros(Nage,Tsim);
xbar=zeros(Nage,Tsim);
u=zeros(Nage,Tsim);

% Our guess for vprime(a,Tsim).  Assign zero to the oldest age
% and then recurse backward by age to increase the value at younger ages.
% Just a guess at the end of the simulation.  Should disappear...
vprime(Nage,Tsim)=0;  % Value at the oldest age is zero.
for a=(Nage-1):-1:1;
  vprime(a,Tsim)=b+beta*vprime(a+1,Tsim);
end;

for t=Tsim:-1:1;
  for a=Nage:-1:1;
    A=exp(Phi(a));
    theta=Theta(a);

    h(a,t)=(beta*vprime(a,t)*c(t)^gamma*theta/A/tfp(a,t)^theta)^(1/(1+theta));

    if alpha>0;  % Use h(a,t) above as a starting guess and solve
      % Note that having to solve this way slows things down a lot!
      hlow=0.8*h(a,t);
      hhigh=1.2*h(a,t);
       
      for it=1:50
        xbartry=A*(tfp(a,t)*hhigh)^theta; % With dlta, we need to distinguish x and xbar
        if xbartry<1; xbartry=1; end;
        xtry=1/(dlta(a,t)+1/xbartry);
        dtry=hhigh.*xbartry/theta*c(t)^(-gamma)-alpha*xtry^(2-sigma)-beta*vprime(a,t);
        if dtry>=0; break; end
        hhigh=hhigh*1.2;
      end
      
      for it=1:50
        hmid=.5*(hlow+hhigh);
        xbarmid=A*(tfp(a,t)*hmid)^theta; if xbarmid<1; xbarmid=1; end;
        xmid=1/(dlta(a,t)+1/xbarmid);
        dmid=hmid.*xbarmid/theta*c(t)^(-gamma)-alpha*xmid^(2-sigma)-beta*vprime(a,t);
        if dmid<0
          hlow=hmid;
        else
          hhigh=hmid;
        end
      end
      h(a,t)=hmid;
    end;

    xbar(a,t)=A*(tfp(a,t)*h(a,t))^theta;
    if xbar(a,t)<1; xbar(a,t)=1; end; % again for h=0 case at end of life.
    x(a,t)=1/(dlta(a,t)+1/xbar(a,t));
    if x(a,t)<1; x(a,t)=1; end;  % e.g. at end of life, h=0 ==> x=0 with our functional form
    u(a,t)=b+c(t)^(1-gamma)/(1-gamma)+alpha*x(a,t)^(1-sigma)/(1-sigma);          
    p(a,t)=1-1/x(a,t);
    
    if t>1 & a>1;
      vprime(a-1,t-1) = u(a,t)+beta*p(a,t)*vprime(a,t) + c(t)^(-gamma)*(y(t)-c(t)-h(a,t));
      if vprime(a-1,t-1)<1; vprime(a-1,t-1)=1; end; % To catch errors
    end;
  end;
end;
