/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jgit.diff;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.List;
import org.eclipse.jgit.JGitText;
import org.eclipse.jgit.diff.DiffEntry;
import org.eclipse.jgit.diff.Edit;
import org.eclipse.jgit.diff.EditList;
import org.eclipse.jgit.diff.MyersDiff;
import org.eclipse.jgit.diff.RawText;
import org.eclipse.jgit.errors.CorruptObjectException;
import org.eclipse.jgit.errors.LargeObjectException;
import org.eclipse.jgit.errors.MissingObjectException;
import org.eclipse.jgit.lib.AbbreviatedObjectId;
import org.eclipse.jgit.lib.Constants;
import org.eclipse.jgit.lib.CoreConfig;
import org.eclipse.jgit.lib.FileMode;
import org.eclipse.jgit.lib.ObjectLoader;
import org.eclipse.jgit.lib.ObjectStream;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.patch.FileHeader;
import org.eclipse.jgit.util.IO;
import org.eclipse.jgit.util.QuotedString;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class DiffFormatter {
    private static final byte[] noNewLine = Constants.encodeASCII("\\ No newline at end of file\n");
    private final OutputStream out;
    private Repository db;
    private int context;
    private int abbreviationLength;
    private RawText.Factory rawTextFactory = RawText.FACTORY;
    private long bigFileThreshold = 0x3200000L;

    public DiffFormatter(OutputStream out) {
        this.out = out;
        this.setContext(3);
        this.setAbbreviationLength(8);
    }

    protected OutputStream getOutputStream() {
        return this.out;
    }

    public void setRepository(Repository repository) {
        this.db = repository;
        CoreConfig cfg = this.db.getConfig().get(CoreConfig.KEY);
        this.bigFileThreshold = cfg.getStreamFileThreshold();
    }

    public void setContext(int lineCount) {
        if (lineCount < 0) {
            throw new IllegalArgumentException(JGitText.get().contextMustBeNonNegative);
        }
        this.context = lineCount;
    }

    public void setAbbreviationLength(int count) {
        if (count < 0) {
            throw new IllegalArgumentException(JGitText.get().abbreviationLengthMustBeNonNegative);
        }
        this.abbreviationLength = count;
    }

    public void setRawTextFactory(RawText.Factory type) {
        this.rawTextFactory = type;
    }

    public void setBigFileThreshold(long bigFileThreshold) {
        this.bigFileThreshold = bigFileThreshold;
    }

    public void flush() throws IOException {
        this.out.flush();
    }

    public void format(List<? extends DiffEntry> entries) throws IOException {
        for (DiffEntry diffEntry : entries) {
            this.format(diffEntry);
        }
    }

    public void format(DiffEntry ent) throws IOException {
        this.writeDiffHeader(this.out, ent);
        if (ent.getOldMode() == FileMode.GITLINK || ent.getNewMode() == FileMode.GITLINK) {
            this.writeGitLinkDiffText(this.out, ent);
        } else {
            byte[] aRaw = this.open(ent.getOldMode(), ent.getOldId());
            byte[] bRaw = this.open(ent.getNewMode(), ent.getNewId());
            if (RawText.isBinary(aRaw) || RawText.isBinary(bRaw)) {
                this.out.write(Constants.encodeASCII("Binary files differ\n"));
            } else {
                RawText a = this.rawTextFactory.create(aRaw);
                RawText b = this.rawTextFactory.create(bRaw);
                this.formatEdits(a, b, new MyersDiff(a, b).getEdits());
            }
        }
    }

    private void writeGitLinkDiffText(OutputStream o, DiffEntry ent) throws IOException {
        if (ent.getOldMode() == FileMode.GITLINK) {
            o.write(Constants.encodeASCII("-Subproject commit " + ent.getOldId().name() + "\n"));
        }
        if (ent.getNewMode() == FileMode.GITLINK) {
            o.write(Constants.encodeASCII("+Subproject commit " + ent.getNewId().name() + "\n"));
        }
    }

    private void writeDiffHeader(OutputStream o, DiffEntry ent) throws IOException {
        String oldName = DiffFormatter.quotePath("a/" + ent.getOldName());
        String newName = DiffFormatter.quotePath("b/" + ent.getNewName());
        o.write(Constants.encode("diff --git " + oldName + " " + newName + "\n"));
        switch (ent.getChangeType()) {
            case ADD: {
                o.write(Constants.encodeASCII("new file mode "));
                ent.getNewMode().copyTo(o);
                o.write(10);
                break;
            }
            case DELETE: {
                o.write(Constants.encodeASCII("deleted file mode "));
                ent.getOldMode().copyTo(o);
                o.write(10);
                break;
            }
            case RENAME: {
                o.write(Constants.encodeASCII("similarity index " + ent.getScore() + "%"));
                o.write(10);
                o.write(Constants.encode("rename from " + DiffFormatter.quotePath(ent.getOldName())));
                o.write(10);
                o.write(Constants.encode("rename to " + DiffFormatter.quotePath(ent.getNewName())));
                o.write(10);
                break;
            }
            case COPY: {
                o.write(Constants.encodeASCII("similarity index " + ent.getScore() + "%"));
                o.write(10);
                o.write(Constants.encode("copy from " + DiffFormatter.quotePath(ent.getOldName())));
                o.write(10);
                o.write(Constants.encode("copy to " + DiffFormatter.quotePath(ent.getNewName())));
                o.write(10);
                if (ent.getOldMode().equals(ent.getNewMode())) break;
                o.write(Constants.encodeASCII("new file mode "));
                ent.getNewMode().copyTo(o);
                o.write(10);
                break;
            }
            case MODIFY: {
                int score = ent.getScore();
                if (score <= 0 || score > 100) break;
                o.write(Constants.encodeASCII("dissimilarity index " + (100 - score) + "%"));
                o.write(10);
            }
        }
        switch (ent.getChangeType()) {
            case MODIFY: 
            case RENAME: {
                if (ent.getOldMode().equals(ent.getNewMode())) break;
                o.write(Constants.encodeASCII("old mode "));
                ent.getOldMode().copyTo(o);
                o.write(10);
                o.write(Constants.encodeASCII("new mode "));
                ent.getNewMode().copyTo(o);
                o.write(10);
            }
        }
        o.write(Constants.encodeASCII("index " + this.format(ent.getOldId()) + ".." + this.format(ent.getNewId())));
        if (ent.getOldMode().equals(ent.getNewMode())) {
            o.write(32);
            ent.getNewMode().copyTo(o);
        }
        o.write(10);
        o.write(Constants.encode("--- " + oldName + '\n'));
        o.write(Constants.encode("+++ " + newName + '\n'));
    }

    private String format(AbbreviatedObjectId oldId) {
        if (oldId.isComplete() && this.db != null) {
            oldId = oldId.toObjectId().abbreviate(this.db, this.abbreviationLength);
        }
        return oldId.name();
    }

    private static String quotePath(String name) {
        String q = QuotedString.GIT_PATH.quote(name);
        return (String.valueOf('\"') + name + '\"').equals(q) ? name : q;
    }

    private byte[] open(FileMode mode, AbbreviatedObjectId id) throws IOException {
        if (mode == FileMode.MISSING) {
            return new byte[0];
        }
        if (mode.getObjectType() != 3) {
            return new byte[0];
        }
        if (this.db == null) {
            throw new IllegalStateException(JGitText.get().repositoryIsRequired);
        }
        if (id.isComplete()) {
            ObjectLoader ldr = this.db.open(id.toObjectId());
            if (!ldr.isLarge()) {
                return ldr.getCachedBytes();
            }
            long sz = ldr.getSize();
            if (sz < this.bigFileThreshold && sz < Integer.MAX_VALUE) {
                byte[] buf;
                try {
                    buf = new byte[(int)sz];
                }
                catch (OutOfMemoryError noMemory) {
                    LargeObjectException e = new LargeObjectException(id.toObjectId());
                    e.initCause(noMemory);
                    throw e;
                }
                ObjectStream in = ldr.openStream();
                try {
                    IO.readFully(in, buf, 0, buf.length);
                }
                finally {
                    in.close();
                }
                return buf;
            }
        }
        return new byte[0];
    }

    public void format(FileHeader head, RawText a, RawText b) throws IOException {
        int start = head.getStartOffset();
        int end = head.getEndOffset();
        if (!head.getHunks().isEmpty()) {
            end = head.getHunks().get(0).getStartOffset();
        }
        this.out.write(head.getBuffer(), start, end - start);
        this.formatEdits(a, b, head.toEditList());
    }

    public void formatEdits(RawText a, RawText b, EditList edits) throws IOException {
        int curIdx = 0;
        while (curIdx < edits.size()) {
            Edit curEdit = edits.get(curIdx);
            int endIdx = this.findCombinedEnd(edits, curIdx);
            Edit endEdit = edits.get(endIdx);
            int aCur = Math.max(0, curEdit.getBeginA() - this.context);
            int bCur = Math.max(0, curEdit.getBeginB() - this.context);
            int aEnd = Math.min(a.size(), endEdit.getEndA() + this.context);
            int bEnd = Math.min(b.size(), endEdit.getEndB() + this.context);
            this.writeHunkHeader(aCur, aEnd, bCur, bEnd);
            while (aCur < aEnd || bCur < bEnd) {
                if (aCur < curEdit.getBeginA() || endIdx + 1 < curIdx) {
                    this.writeContextLine(a, aCur);
                    if (this.isEndOfLineMissing(a, aCur)) {
                        this.out.write(noNewLine);
                    }
                    ++aCur;
                    ++bCur;
                } else if (aCur < curEdit.getEndA()) {
                    this.writeRemovedLine(a, aCur);
                    if (this.isEndOfLineMissing(a, aCur)) {
                        this.out.write(noNewLine);
                    }
                    ++aCur;
                } else if (bCur < curEdit.getEndB()) {
                    this.writeAddedLine(b, bCur);
                    if (this.isEndOfLineMissing(b, bCur)) {
                        this.out.write(noNewLine);
                    }
                    ++bCur;
                }
                if (!DiffFormatter.end(curEdit, aCur, bCur) || ++curIdx >= edits.size()) continue;
                curEdit = edits.get(curIdx);
            }
        }
    }

    protected void writeContextLine(RawText text, int line) throws IOException {
        this.writeLine(' ', text, line);
    }

    private boolean isEndOfLineMissing(RawText text, int line) {
        return line + 1 == text.size() && text.isMissingNewlineAtEnd();
    }

    protected void writeAddedLine(RawText text, int line) throws IOException {
        this.writeLine('+', text, line);
    }

    protected void writeRemovedLine(RawText text, int line) throws IOException {
        this.writeLine('-', text, line);
    }

    protected void writeHunkHeader(int aStartLine, int aEndLine, int bStartLine, int bEndLine) throws IOException {
        this.out.write(64);
        this.out.write(64);
        this.writeRange('-', aStartLine + 1, aEndLine - aStartLine);
        this.writeRange('+', bStartLine + 1, bEndLine - bStartLine);
        this.out.write(32);
        this.out.write(64);
        this.out.write(64);
        this.out.write(10);
    }

    private void writeRange(char prefix, int begin, int cnt) throws IOException {
        this.out.write(32);
        this.out.write(prefix);
        switch (cnt) {
            case 0: {
                this.out.write(Constants.encodeASCII(begin - 1));
                this.out.write(44);
                this.out.write(48);
                break;
            }
            case 1: {
                this.out.write(Constants.encodeASCII(begin));
                break;
            }
            default: {
                this.out.write(Constants.encodeASCII(begin));
                this.out.write(44);
                this.out.write(Constants.encodeASCII(cnt));
            }
        }
    }

    protected void writeLine(char prefix, RawText text, int cur) throws IOException {
        this.out.write(prefix);
        text.writeLine(this.out, cur);
        this.out.write(10);
    }

    public FileHeader createFileHeader(DiffEntry ent) throws IOException, CorruptObjectException, MissingObjectException {
        FileHeader.PatchType type;
        EditList editList;
        ByteArrayOutputStream buf = new ByteArrayOutputStream();
        this.writeDiffHeader(buf, ent);
        if (ent.getOldMode() == FileMode.GITLINK || ent.getNewMode() == FileMode.GITLINK) {
            this.writeGitLinkDiffText(buf, ent);
            editList = new EditList();
            type = FileHeader.PatchType.UNIFIED;
        } else {
            byte[] aRaw = this.open(ent.getOldMode(), ent.getOldId());
            byte[] bRaw = this.open(ent.getNewMode(), ent.getNewId());
            if (RawText.isBinary(aRaw) || RawText.isBinary(bRaw)) {
                buf.write(Constants.encodeASCII("Binary files differ\n"));
                editList = new EditList();
                type = FileHeader.PatchType.BINARY;
            } else {
                RawText a = this.rawTextFactory.create(aRaw);
                RawText b = this.rawTextFactory.create(bRaw);
                editList = new MyersDiff(a, b).getEdits();
                type = FileHeader.PatchType.UNIFIED;
            }
        }
        return new FileHeader(buf.toByteArray(), editList, type);
    }

    private int findCombinedEnd(List<Edit> edits, int i) {
        int end = i + 1;
        while (end < edits.size() && (this.combineA(edits, end) || this.combineB(edits, end))) {
            ++end;
        }
        return end - 1;
    }

    private boolean combineA(List<Edit> e, int i) {
        return e.get(i).getBeginA() - e.get(i - 1).getEndA() <= 2 * this.context;
    }

    private boolean combineB(List<Edit> e, int i) {
        return e.get(i).getBeginB() - e.get(i - 1).getEndB() <= 2 * this.context;
    }

    private static boolean end(Edit edit, int a, int b) {
        return edit.getEndA() <= a && edit.getEndB() <= b;
    }
}

