#ifndef _sparse_string_array_
#define _sparse_string_array_

#include <string>;
#include "vector.h"

class SparseStringArray {
public:
    /**
     * Constructor: SparseStringArray
     * ------------------------------
     * Constructs the space set aside for a newborn SparseStringArray, settings
     * its length to be that specified, and defaulting all entries to be the
     * empty string.  The groupSize parameter is supplied as a hueristic and should
     * be set to be the number of empty strings we expect to see for every
     * nonempty string once the SparseStringArray has been fully populated with
     * most or all of its data.  It's assumed that length % groupSize is 0, else
     * the behavior of the entire SparseStringArray is undefined.
     */
    SparseStringArray(int length, int groupSize);

    /**
     * Destructor: ~SparseStringArray
     * ------------------------------
     * Disposes of all internal resources (typically any dynamically allocated memory)
     * associated with a SparseStringArray that is itself being disposed of.
     */
    ~SparseStringArray();

    /**
     * Function: size
     * --------------
     * Returns the length of the SparseStringArray.
     */
    int size() const;

    /**
     * Function: operator[]
     * Usage: ssa[10] = "nonempty";            // customary
     *        ssa.operator[](10) = "nonempty"  // unorthodox, but legal
     * ----------------------------
     * Returns a non-const refernce to the string at the supplied index.
     * If the supplied index is out of range, the method's behavior is undefined.
     */
    std::string& operator[](int index);

    /**
     * Function: operator[]
     * Usage: cout << ssa[10];             // customary
     *        cout << ssa.operator[](10);  // unorthodox, but legal
     * ----------------------------
     * Returns a const refernce to the string at the supplied index.
     * If the supplied index is out of range, the method's behavior is undefined.
     */
    const std::string& operator[](int index) const;

private:
    struct group {
        bool *bitmap;
        Vector<std::string> strings;
    };
    group *groups;
    int numGroups;
    int length;
    int groupSize;
    int getVectorIndex(int groupIndex, int bitmapIndex) const;
};

#endif
