/**
 * File: hash-map-impl.h
 * ---------------------
 * Presents the templatized implementation of the HashMap.  Because
 * of the way C++ compilers process templates, this is a cpp-like
 * file that is technically a header file, because client code needs to
 * see the implementation in order to expand it for the specific Key and
 * Value types they choose.
 */

#include <string>
#include <functional>
#include <iostream>
#include "error.h"

template <typename Key, typename Value>
HashMap<Key, Value>::HashMap(int sizeHint) {
    if (sizeHint <= 0)
        error("size hint passed to HashMap  constructor must be positive.");

    count = 0;
    numBuckets = sizeHint;
    buckets = new node *[numBuckets];
    for (int i = 0; i < numBuckets; i++)
        buckets[i] = NULL;
}

template <typename Key, typename Value>
HashMap<Key, Value>::~HashMap () {
    for (int i = 0; i < numBuckets; i++) {
        node *curr = buckets[i];
        while (curr != NULL) {
            node *next = curr->next;
            delete curr;
            curr = next;
        }
    }

    delete[] buckets;
}

template <typename Key, typename Value>
bool HashMap<Key, Value>::containsKey(const Key& key) const {
    return findNode(key) != NULL;
}

template <typename Key, typename Value>
void HashMap<Key, Value>::put(const Key& key, const Value& value) {
    ensureNodeExists(key)->value = value;
}

template <typename Key, typename Value>
Value HashMap<Key, Value>::get(const Key& key) const {
    const node *found = findNode(key);
    return found == NULL ? Value() : found->value;
}

template <typename Key, typename Value>
Value& HashMap<Key, Value>::operator[](const Key& key) {
    return ensureNodeExists(key)->value;
}

/**
 * Method: hash
 * ------------
 * Figures out how to hash the supplied key to some number
 * between 0 and numBuckets - 1.
 */
template <typename Key, typename Value>
int HashMap <Key, Value>::hash(const Key& key) const {
    return std::hash<Key>()(key) % numBuckets;
}

template <typename Key, typename Value>
typename HashMap<Key, Value>::node *HashMap <Key, Value>::ensureNodeExists(const Key& key) {
    node *found = const_cast<node *>(findNode(key));
    if (found == NULL) {
        found = new node;
        found->key = key;
        found->value = Value();
        int hashcode = hash(key);
        found->next = buckets[hashcode];
        buckets[hashcode] = found;
        count++;
    }
    
    return found;
}

template <typename Key, typename Value>
const typename HashMap<Key, Value>::node *HashMap <Key, Value>::findNode(const Key& key) const {
    int hashcode = hash(key);
    const node *curr = buckets[hashcode];
    while (curr != NULL && !(curr->key == key)) {
        curr = curr->next;
    }
    return curr;
}

