/**
 * File: hash-map.h
 * ----------------
 * Defines a generic HashMap (which has the same interface
 * as the Map class, but is backed by a collection of
 * linked lists and is more accessible to us at this point
 * of the course).
 */

#pragma once
template <typename Key, typename Value>
class HashMap {

public:
    /**
     * Constructor: HashMap
     * --------------------
     * Constructs an empty HashMap, although it sets aside
     * enough space internally to accommodate approximately
     * sizeHint key-value pairs.
     */
    HashMap(int sizeHint = 10001);

    /**
     * Destructor: ~HashMap
     * --------------------
     * Disposes of all resources currently held by the HashMap.
     */
    ~HashMap();
    
    /**
     * Method: isEmpty
     * ---------------
     * Returns true if and only if the HashMap is empty.
     */
    bool isEmpty() const { return size() == 0; }

    /**
     * Method: size
     * ------------
     * Returns the number of key/value pairs stored in the
     * HashMap. :)
     */
    int size() const { return count; }
    
    /**
     * Method: containsKey
     * -------------------
     * Returns true if and only if the supplied key
     * is present in the HashMap.
     */
    bool containsKey(const Key& key) const;

    /**
     * Method: put
     * -----------
     * Ensures that the supplied key/value pair is in the map.
     * Any value previously associated with the same key is overwritten.
     */
    void put(const Key& key, const Value& value);

    /**
     * Method: get
     * -----------
     * Returns a copy of the value attached to the provided key.
     * If the key isn't present, get returns a defaultly constructed
     * value (although the client should never rely on this behavior,
     * and instead should guard against such calls by first calling
     * containsKey.
     */
    Value get(const Key& key) const;

    /**
     * Method: operator[]
     * ------------------
     * Returns a reference to the value attached to the supplied
     * key.  If the key isn't present at the time of the call, then
     * the key is inserted, attached to a defaultly constructed
     * value, and a reference to that new value is returned.
     */
    Value& operator[](const Key& key);
    
private:
    struct node {
        Key key;
        Value value;
        node *next;
    };
    
    node **buckets;
    int numBuckets;
    int count;
    
    /**
     * Private Method: hash
     * --------------------
     * Hashes the supplied key, no matter what the type, to some
     * number between 0 and numBuckets - 1, inclusive.
     */
    int hash(const Key& key) const;

    /**
     * Private Method: ensureNodeExists
     * --------------------------------
     * If necessary, inserts a node on behalf of the supplied
     * key.  Whether originally present or freshly inserted,
     * ensureNodeExists returns the address of the relevant node.
     */
    node *ensureNodeExists(const Key& key);

    /**
     * Private Method: findNode
     * ------------------------
     * Returns the address of the node housing the supplied
     * key, or NULL if no such node exists.
     */
    const node *findNode(const Key& key) const;
};

#include "hash-map-impl.h"

