// DotFrame.java
/**
 Hosts a DotPanel.
 Implements the UI logic for save/saveAs/open/quit
*/

import java.awt.*;

import javax.swing.*;

import java.awt.event.*;
import java.io.*;

public class DotFrame extends JFrame {

	public static void main(String[] args) {
		try {
			UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
		} catch (Exception ignored) { }
		
		File fileToOpen = null;
		
		// Notice command line arg filename
		if (args.length != 0) {
			fileToOpen = new File(args[0]);
		}
		
		new DotFrame(fileToOpen);
        
	}
	
	private DotPanel dotPanel;
	private File saveFile; // the last place we saved, or null
	
	// Creates a new DotFrame
	// If passed a non-null file, opens that file
	public DotFrame(File file) {
		setTitle("DotFrame");
		
		dotPanel = new DotPanel(300, 300);
		if (file != null) dotPanel.open(file);
		add(dotPanel, BorderLayout.CENTER);
		
		
		JPanel box = new JPanel();
		box.setLayout(new BoxLayout(box, BoxLayout.Y_AXIS));
		add(box, BorderLayout.WEST);
        
		/*
		 Create the checkboxes and wire them to setters
		 on the DotPanel for each boolean feature.
		*/
		final JCheckBox print = new JCheckBox("Print");
		box.add(print);
		print.addActionListener( new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					dotPanel.setPrint(print.isSelected());
				}
		});
		
		final JCheckBox smart = new JCheckBox("Smart repaint", true);
		box.add(smart);
		smart.addActionListener( new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					dotPanel.setSmart(smart.isSelected());
				}
		});
		

		final JCheckBox old = new JCheckBox("Old repaint", true);
		box.add(old);
		old.addActionListener( new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					dotPanel.setOld(old.isSelected());
				}
		});
		
		final JCheckBox red = new JCheckBox("Red paint region");
		box.add(red);
		red.addActionListener( new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					dotPanel.setRed(red.isSelected());
				}
		});

		// Buttons for file save/open
		
		JButton button;
		button = new JButton("Save...");
		box.add(button);
		button.addActionListener( new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				doSave(false);
			}
		});
		
		button = new JButton("Save As...");
		box.add(button);
		button.addActionListener( new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				doSave(true);
			}
		});
		
		button = new JButton("Open...");
		box.add(button);
		button.addActionListener( new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				doOpen();
			}
		});
		

 		// Make a Save Image button
		JButton imageButton = new JButton("Save PNG Image");
		box.add(imageButton);
		imageButton.addActionListener( new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				JFileChooser chooser = createFileChooser();
				int status = chooser.showSaveDialog(DotFrame.this);
				// depending on the os, it may help if the user
				// gives the file a name ending in .png
				if (status == JFileChooser.APPROVE_OPTION) {
					File dest = chooser.getSelectedFile();
					dotPanel.saveImage(dest);
				}
			}
		});
		
		setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
		// Listener for the case where the user closes the frame
		// In that case, check the dirty bit
        addWindowListener(new WindowAdapter() {
        		public void windowClosing(WindowEvent event) {
        			boolean ok = doCheckQuit();
        			if (!ok) return;
        			System.exit(0);
        		}
        });

        
		pack();
		setVisible(true);
	}
	
	// Save (or saveAs) the panel
	public void doSave(boolean saveAs) {
		// prompt where to save if needed
		if (saveFile == null || saveAs) {
			JFileChooser chooser = createFileChooser();
			int status = chooser.showSaveDialog(this);
			if (status == JFileChooser.APPROVE_OPTION) {
				saveFile = chooser.getSelectedFile();
			}
			else return;  // i.e. cancel the whole operation
		}
		
		// Do the save, set the frame title
		dotPanel.save(saveFile);
		setTitle(saveFile.getName());
	}
	
	// Asks if they want to save, and if so does it.
	// Returns false if they cancel.
	public boolean saveOk() {
		int result = JOptionPane.showConfirmDialog(
				this, "Save changes first?", "Save?",
				JOptionPane.YES_NO_CANCEL_OPTION);
		
		if (result == JOptionPane.YES_OPTION) {
			doSave(false);
		}
		
		return (result != JOptionPane.CANCEL_OPTION);
	}
	
	// Checks if quiting is ok, and saves if needed.
	// Returns false if they cancel.
	public boolean doCheckQuit() {
		if (dotPanel.getDirty()) {
			boolean ok = saveOk();
			if (!ok) return false;
		}
		return true;
	}
	
	// Prompts for a file and reads it in
	// Asks to write out old contents first if dirty.
	public void doOpen() {
		if (dotPanel.getDirty()) {
			boolean ok = saveOk();
			if (!ok) return;
		}
		
		JFileChooser chooser = createFileChooser();
		int status = chooser.showOpenDialog(this);
		if (status == JFileChooser.APPROVE_OPTION) {
			dotPanel.open(chooser.getSelectedFile());
			// now that file is the "current" file for the window
			saveFile = chooser.getSelectedFile();
			setTitle(saveFile.getName());
		}
	}

	// Creates a new JFileChooser, doing the boilerplate
	// to start it in the current directory.
	private JFileChooser createFileChooser() {
		JFileChooser chooser = new JFileChooser();
		try {
			// The "." stuff attempts to open in the "current"
			// directory.
			File dir = new File(new File(".").getCanonicalPath());
			chooser.setCurrentDirectory(dir);
		}
		catch (Exception ex) {
			ex.printStackTrace();
		}
		return chooser;
	}
}

