// CensorTableModel.java
import javax.swing.*;
import javax.swing.table.*;
import javax.swing.event.*;


/*
 Presents a table model that censors out table cells that include
 the "censor" string.
 
 This is a "layered" or "delegate" solution.
 We do not store any cell data at all --
 we store a pointer to a delegate "real" table model that has the real data.
 In typical delegation fashion, when we receive a message to do something,
 we often pass the request back to the delegate for processing.
 We pass the data forward, but do the censoring on it.
 We listen to the delegate to hear about table model changes from it.
 We get setCensor() when the censor string changes, and in that case
 we have to re-think the censoring for every cell.
 This is a pretty complex use of MVC model-listener logic.
 
 Note: another solution would be to subclass off TableModel to create a censoring
 variant. The delegate strategy shown here is probably easier to get right.
*/
class CensorTableModel extends AbstractTableModel implements TableModelListener {
	private TableModel model;		// the "real" model
	private String censor;		// current censor string
	
	/*
	 Sets up the censoring model based on the given model.
	*/
	public CensorTableModel(TableModel model) {
		this.model = model;
		censor = "";
		
		// we listen to the real model for its changes
		model.addTableModelListener(this);
	}
	
	
	/*
	 Standard table model messages -- we just pass them back
	 to our delegate, the real table model.
	 Except, getData, which does the censoring.
	*/
	public String getColumnName(int col) {
		return model.getColumnName(col);
	}
	
	public int getColumnCount() { return model.getColumnCount(); }
	public int getRowCount() { return model.getRowCount(); }
	
	// This one is interesting: calls the delegate, then adjusts
	// the data.
	public Object getValueAt(int row, int col) {
		String elem = (String) model.getValueAt(row, col);
		if (isCensored(elem, censor)) return "(BEEP)";
		else return elem;
	}
		
	// We don't allow a censored cell to be edited
	public boolean isCellEditable(int row, int col) { return false; }
	// So we don't have to respond to the following...
	//public void setValueAt(Object value, int row, int col) {
	//	model.setValueAt(value, row, col);
	//}	
	
	
 	/*
	 We get this notification when the real table model
	 changes: column added, cell updated, etc.
	 We just pass it forward to our table, and it
	 will generate getXXX() requests as it sees fit.
	*/
	public void tableChanged(TableModelEvent e) {
		fireTableChanged(e);
	}
	
	
	/*
	 Changes the censor string.
	 
	 Implementation: In this case, we need to
	 check if any of the cells are now censored differently,
	 and do fireTableCellUpdated() to notify
	 the table of any cells that are changed.
	 Compares what each cell looked like with the 
	 old censor string vs. the new censor string to
	 decide if we need to tell our table to change or not.
	*/
	public void setCensor(String newCensor) {
		String oldCensor = censor;
		censor = newCensor.toLowerCase();
		
		// if the strings are the same, just forget it
		if (censor.equals(oldCensor)) return;
		
		// look at every cell, compare old censoring to new
		for (int row = 0; row<model.getRowCount(); row++) {
			for (int col = 0; col<model.getColumnCount(); col++) {
				String elem = (String) model.getValueAt(row, col);
				if (elem != null) {
					boolean old = isCensored(elem, oldCensor);
					boolean now = isCensored(elem, censor);
					
					// tell the table the cell changed if the
					// new state is different from the old state.
					if (old != now) fireTableCellUpdated(row, col);
				}
			}
		}
	}


	/*
	 Private utility.
	 Tests a cell string against a censor string to
	 see if it should be censored.
	 For censoring to happen, the censor string must
	 not be the empty string.
	 
	 note: example of "bottleneck" strategy -- in this way,
	 getValueAt() and setCensor() have *exactly the same*
	 notion of censoring. Two methods like that getting
	 out of sync for some edge case is a classic source of bugs.
	*/
	private boolean isCensored(String cell, String censorString) {
		return(cell!=null && !"".equals(censorString) &&
			cell.toLowerCase().indexOf(censorString) != -1);
	}
}


/*
 Same goal as above, but uses SUBCLASSING off the real model
 instead of keeping a pointer to a separate model object.
 This version also listens to the text field directly.
 Since we ARE a BasicTableModel, we have a copy of the data.
 We just adjust the data on the way out in getDataAt().
 
 This just shows off how you can do something with sublcassing
 or delegating.
*/

class CensorTableModel2 extends BasicTableModel {
	private String censor;
	private JTextField field;
	
	
	public CensorTableModel2() {
		censor = "";
	}

	
	
	private boolean isCensored(String cell, String censorString) {
		return(cell!=null && !"".equals(censorString) &&
			cell.toLowerCase().indexOf(censorString) != -1);
	}
	
	public void setCensor(String newCensor) {
		String oldCensor = censor;
		censor = newCensor.toLowerCase();
		
		// if the strings are the same, just forget it
		if (censor.equals(oldCensor)) return;
		
		// look at every cell, compare old censoring to new
		for (int row = 0; row<getRowCount(); row++) {
			for (int col = 0; col<getColumnCount(); col++) {
				String elem = (String) getValueAt(row, col);
				if (elem != null) {
					boolean old = isCensored(elem, oldCensor);
					boolean now = isCensored(elem, censor);
					
					// tell the table the cell changed if the
					// new state is different from the old state.
					if (old != now) fireTableCellUpdated(row, col);
				}
			}
		}
	}
	
	
	// We change this one method to do the censoring
	public Object getValueAt(int row, int col) {
		String elem = (String) super.getValueAt(row, col);
		if (isCensored(elem, censor)) return("(BEEP)");
		else return(elem);
	}
	
	// If editing happens -- well too bad. The string "(censor)"
	// will get put into the model.
}

