import numpy as np
import matplotlib.pyplot as plt
from scipy.stats import norm

# ----- Model parameters -----
mu_prior = 0.8     # prior mean of Y
sigma_prior = 1.0  # prior sd of Y

# Thresholds: equal bins under standard normal (k/5 quantiles)
taus = [norm.ppf(k/5.0) for k in range(1, 5)]
tau1, tau2, tau3, tau4 = taus

# ----- Define components of the posterior (up to a constant) -----

def prior_density(y):
    """Prior N(0.8, 1)."""
    return norm.pdf(y, loc=mu_prior, scale=sigma_prior)

def p1(y):
    """P(R = 2 | Y = y)."""
    return norm.cdf(tau1 - y) 

def p2(y):
    """P(R = 2 | Y = y)."""
    return norm.cdf(tau2 - y) - norm.cdf(tau1 - y)

def p3(y):
    """P(R = 3 | Y = y)."""
    return norm.cdf(tau3 - y) - norm.cdf(tau2 - y)

def p4(y):
    """P(R = 4 | Y = y)."""
    return norm.cdf(tau4 - y) - norm.cdf(tau3 - y)

def p5(y):
    """P(R = 5 | Y = y)."""
    return 1.0 - norm.cdf(tau4 - y)

def unnormalized_posterior(y, ratings):
    """
    Posterior density for Y up to a normalizing constant:
        p(Y=y | data) ∝ prior(y) * P4(y)^n4 * P5(y)^n5
    """
    total = prior_density(y) 
    total *= p1(y) ** ratings['1★'] 
    total *= p2(y) ** ratings['2★']
    total *= p3(y) ** ratings['3★']
    total *= p4(y) ** ratings['4★']
    total *= p5(y) ** ratings['5★']
    return total

def get_probability_better(book_2, book_1,
                           y_min=-3, y_max=3, num_points=2000):

    y = np.linspace(y_min, y_max, num_points)

    # Evaluate posteriors
    post1 = unnormalized_posterior(y, book_1)
    post2 = unnormalized_posterior(y, book_2)

    # Normalize
    post1 /= np.trapz(post1, y)
    post2 /= np.trapz(post2, y)

    # Compute probability Y2 > Y1 by double integral:
    # Integral post2(y2) * (Integral post1(y1) where y1 < y2)
    cdf1 = np.cumsum(post1) * (y[1] - y[0])  # numerical CDF of book 1

    prob = np.trapz(post2 * cdf1, y)

    return prob

# ----- Evaluate on a grid and plot -----

book_1 = {
    '1★':0, 
    '2★':0, 
    '3★':0, 
    '4★':1, 
    '5★':4
}
book_2 = {
    '1★':0, 
    '2★':1, 
    '3★':2, 
    '4★':20, 
    '5★':200
}

p_better = get_probability_better(book_2, book_1)
print("P(Book 2 > Book 1) =", p_better)