import numpy as np

def estimate_prob_book2_better(n_samples=200_000, seed=0):
    rng = np.random.default_rng(seed)
    
    # Rating counts in order [1, 2, 3, 4, 5]
    counts_book1 = np.array([0, 0, 0, 1, 4])
    counts_book2 = np.array([0, 1, 2, 20, 200])
    
    # Dirichlet(1,...,1) prior => posterior alpha = counts + 1
    alpha1 = counts_book1 + 1
    alpha2 = counts_book2 + 1
    
    # Sample rating probability vectors from the posterior for each book
    p1_samples = rng.dirichlet(alpha1, size=n_samples)  # shape (n_samples, 5)
    p2_samples = rng.dirichlet(alpha2, size=n_samples)  # shape (n_samples, 5)
    
    # For fixed p1, p2:
    #   P(R2 > R1) = sum_{i<j} p1[i] * p2[j]
    #
    # We'll compute this efficiently for all samples at once.
    
    # For each sample, cumulative sums of p2 from high rating to low rating
    # cum_p2_rev[:, k] = p2[:,k] + p2[:,k+1] + ... + p2[:,4]
    cum_p2_rev = np.flip(np.cumsum(np.flip(p2_samples, axis=1), axis=1), axis=1)
    
    probs = np.zeros(n_samples)
    # For each possible rating i for Book 1, multiply p1[i]
    # by probability Book 2 gets a strictly higher rating.
    for i in range(5):
        if i + 1 < 5:
            probs += p1_samples[:, i] * cum_p2_rev[:, i + 1]

    # Monte Carlo estimate of P(Book 2 rating > Book 1 rating)
    return probs.mean()

if __name__ == "__main__":
    est = estimate_prob_book2_better()
    print(f"Estimated P(Book 2 gets a higher rating than Book 1): {est:.4f}")