import numpy as np
import matplotlib.pyplot as plt
from scipy.stats import norm

# ----- Model parameters -----
mu_prior = 0.8     # prior mean of Y
sigma_prior = 1.0  # prior sd of Y

# Thresholds: equal bins under standard normal (k/5 quantiles)
taus = [norm.ppf(k/5.0) for k in range(1, 5)]
tau1, tau2, tau3, tau4 = taus

# Data: counts of ratings
n4 = 1
n5 = 4

# ----- Define components of the posterior (up to a constant) -----

def prior_density(y):
    """Prior N(0.8, 1)."""
    return norm.pdf(y, loc=mu_prior, scale=sigma_prior)

def p4(y):
    """P(R = 4 | Y = y)."""
    return norm.cdf(tau4 - y) - norm.cdf(tau3 - y)

def p5(y):
    """P(R = 5 | Y = y)."""
    return 1.0 - norm.cdf(tau4 - y)

def unnormalized_posterior(y):
    """
    Posterior density for Y up to a normalizing constant:
        p(Y=y | data) ∝ prior(y) * P4(y)^n4 * P5(y)^n5
    """
    return prior_density(y) * (p4(y) ** n4) * (p5(y) ** n5)

# ----- Evaluate on a grid and plot -----

ys = np.linspace(-1.0, 4.0, 1000)
post_vals = unnormalized_posterior(ys)

plt.figure(figsize=(8, 4))
plt.plot(ys, post_vals)
plt.xlabel("Y (book true quality)")
plt.ylabel("Unnormalized posterior density")
plt.title("Unnormalized posterior of Y given ratings (1×4★, 4×5★)")
plt.tight_layout()
plt.show()