'''
/* =======================================================================
   (c) 2015, Kre8 Technology, Inc.

   PROPRIETARY and CONFIDENTIAL
   ========================================================================*/
'''

import HS2_Rx #Sensors byte definition
import HS2_Tx #Effectors byte definition
import util #two's complement and its inverse functions

#code snipset from a file: HS2_Rx.py
#version_id = 0 #version/topology
#network_id = 1
#command_id = 2 #command/security
#signal_strength = 3
#proximity_left = 4
#...

#code snipset from a file: HS2_Tx.py
#version_id = 0 #version/topology
#network_id = 1
#command_id = 2 #command/security
#wheel_left = 3
#wheel_right = 4
#...

class RobotAPI(object):
    def __init__(self, peripheral):
        #private members
        self._peripheral = peripheral
        self._tx_packet = bytearray.fromhex(u'0000000000000000000000000000000000000000')
        self._rx_packet = bytearray.fromhex(u'0000000000000000000000000000000000000000')
        self._tx_packet[HS2_Tx.command_id] = 0x10
        self._temperature = 24.0
        self._battery = 3.6
        self._light = 0

    #public methods
    def set_sensors(self, buf, from_hex):
        if from_hex:
            self._rx_packet = bytearray.fromhex(buf)
        else:
            self._rx_packet = bytearray(buf)

    def get_effectors(self):
        return self._tx_packet

    def print_sensors(self, print_hex):
        if print_hex:
            for i in range(0, HS2_Rx.packet_size):
                print ("%2.2x"% int(self._rx_packet[i])),
            print ''
        else:
            print ("%#3.1x"% self.get_version()),
            print ("%#3.1x"% self.get_topology()),
            print ("%3d"% self.get_network_id()),
            print ("%#3.1x"% self.get_command()),
            print ("%#3.1x"% self.get_security()),
            print ("%4d"% self.get_signal()),
            print ("%3d"% self.get_proximity(0)),
            print ("%3d"% self.get_proximity(1)),
            print ("%3d"% self.get_floor(0)),
            print ("%3d"% self.get_floor(1)),
            print ("%6d"% self.get_acceleration(0)),
            print ("%6d"% self.get_acceleration(1)),
            print ("%6d"% self.get_acceleration(2)),
            print ("%3d"% self.get_flag()),
            print ("%5d"% self.get_light()),
            print ("%5.1f"% self.get_temperature()),
            print ("%4.2f"% self.get_battery()),
            print ("%#3.1x"% self.get_io_mode(0)),
            print ("%#3.1x"% self.get_io_mode(1)),
            print ("%3d"% self.get_port(0)),
            print ("%3d"% self.get_port(1)),
            print ("%3d"% self.get_linetracer_state())

    def reset(self):
        self.set_version(0)
        self.set_topology(0)
        self.set_network_id(0)
        self.set_command(1)
        self.set_security(0)
        self.set_wheel(0, 0)
        self.set_wheel(1, 0)
        self.set_led(0, 0)
        self.set_led(1, 0)
        self.set_buzzer(0)
        self.set_musical_note(0)
        self.set_line_tracer_mode_speed(0, 0)
        self.set_proximity_IR_current(0)
        self.set_gravity_range(0)
        self.set_gravity_bandwidth(0)
        self.set_io_mode(0, 0)
        self.set_io_mode(1, 0)
        self.set_port(0, 0)
        self.set_port(1, 0)
        self.set_wheel_balance(0)
        self.set_input_pull(0, 0)
        self.set_input_pull(1, 0)
        self.set_reserved(0)

    def get_version(self):
        return int(self._rx_packet[HS2_Rx.version_id] >> 4 & 0x0F)

    def get_topology(self):
        return int(self._rx_packet[HS2_Rx.version_id] & 0x0F)

    def get_acceleration(self, axis): #0:x, 1:y, 2:z
        if (axis < 0):
            axis = 0
        elif (axis > 2):
            axis = 2
        high = int(self._rx_packet[HS2_Rx.acc_x_high + 2 * axis])
        low = int(self._rx_packet[HS2_Rx.acc_x_low + 2 * axis])
        return util.twos_complement(high * 256 + low, 16)

    def get_battery(self):
        if (self.get_flag() == 1):
            self._battery = (2.0 + (int(self._rx_packet[HS2_Rx.battery]) * 0.01))
        return self._battery
        
    def get_io_mode(self, side):  #0: portA, 1: portB
        if (side == 1):
            return self._tx_packet[HS2_Tx.io_mode] & 0x0F
        else:
            return (self._tx_packet[HS2_Tx.io_mode] >> 4) & 0x0F

    def get_port(self, side): #0: portA, 1: portB
        mode = self.get_io_mode(side)
        if (mode == 0x00): # ADC
            #print self._rx_packet[HS2_Rx.port_a + side]
            value = (3.3 * int(self._rx_packet[HS2_Rx.port_a + side]) / 255)
            #value = int(self._rx_packet[HS2_Rx.port_a + side])
        elif (mode == 0x01): # DI
            value = int(self._rx_packet[HS2_Rx.port_a + side])
        else:
            value = int(self._rx_packet[HS2_Rx.port_a + side])
        return value

    def get_linetracer_state(self):
        return int(self._rx_packet[HS2_Rx.line_state])

    def set_version(self, value):
        if (value < 0x00):
            value = 0x00
        elif (value > 0x0F):
            value = 0x0F
        value = (value << 4) | (self._tx_packet[HS2_Tx.version_id] & 0x0F)
        self._tx_packet[HS2_Tx.version_id] = value

    def set_topology(self, value):
        if (value < 0x00):
            value = 0x00
        elif (value > 0x0F):
            value = 0x0F
        value = (self._tx_packet[HS2_Tx.version_id] & 0xF0) | value
        self._tx_packet[HS2_Tx.version_id] = value

    def set_line_tracer_mode_speed(self, mode, speed): # 0x11 ~ 0x6A (0: off), speed 7(100%)
        if (mode < 0):
            mode = 0
        elif (mode > 14):
            mode = 14
        if (speed < 0):
            speed = 0
        elif (speed > 7):
            speed = 7
        new_cmd = (self._tx_packet[HS2_Tx.line_tracer_mode_speed] >> 7) ^ 0x01;
        value = new_cmd << 7 | mode << 3 | speed        
        self._tx_packet[HS2_Tx.line_tracer_mode_speed] = value

    def set_io_mode(self, side, value): #0 : portA, 1: portB, #0 ~ 127 #0x00(ADC), 0x01(DI), 0x08(ASC), 0x09(PWM), 0x0A(DO)
        if side < 0:
            side = 0 
        elif side > 1:
            side = 1
        if (value == 0x00 
            or value == 0x01
            or value == 0x08
            or value == 0x09
            or value == 0x0A):
            if (side == 1):
                value = (self.get_io_mode(0) << 4) | value
            else:
                value = (value << 4) | self.get_io_mode(1)
            self._tx_packet[HS2_Tx.io_mode] = value
        else:
            print "unknown IO mode:", value

    def set_port(self, side, value): # 0: portA, 1: portB, # value: 0 ~ 255
        if side < 0:
            side = 0 
        elif side > 1:
            side = 1
        if (value < 0x00):
            value = 0x00
        elif (value > 0xFF):
            value = 0xFF
        self._tx_packet[HS2_Tx.port_a + side] = value

    def set_wheel_balance(self, value): # -128 ~ 127 (0: off)
        if (value < -128):
            value = -128
        elif (value > 127):
            value = 127
        value = util.twos_complement_inv(value, 8)
        self._tx_packet[HS2_Tx.wheel_balance] = value
 
    def set_input_pull(self, side, value): # side 0: portA, side 1: portB, value: 0~3
        if side < 0:
            side = 0 
        elif side > 1:
            side = 1
        if (value < 0):
            value = 0
        elif (value > 3):
            value = 3
        if side == 1:
            value = (value << 2) | (self._tx_packet[HS2_Tx.input_pull] & 0x03)
        else:
            value = (self._tx_packet[HS2_Tx.input_pull] & 0x0C) | value 
        value = (self._tx_packet[HS2_Tx.input_pull] & 0xF0) | value
        self._tx_packet[HS2_Tx.input_pull] = value 

    def set_reserved(self, value):
        if (value < 0x00):
            value = 0x00
        elif (value > 0xFF):
            value = 0xFF
        self._tx_packet[HS2_Tx.reserved] = value 

