%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Modularity_Eig
%  Given an adjacency matrix for a network
%  return its partition into modules, and the modularity 
%  measure for this partition. 
%
%  Input variables:
%  A:        the adjacency matrix
%  directed: binary whether the network is directed or not
%  min_comp: minimal size of module below which to stop partitioning
%  max_iter: the amount of maximum iterations of the algorithm
%            (useful to limit running time if A is large, in which 
%            case the network is divided into a maximum of
%            2^max_iter groups)
%
%  Output variables:
%  modules:    a vector with the group id for each node
%  Q:          the modularity measure
%  centrality: vector with module centrality scores per node
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [modules Q centrality clustertree] = Modularity_Eig(A,directed,min_comp,max_iter)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Initialize variables                   
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
modules = [];
Q = 0;
centrality = zeros(1,length(A));
clustertree = [];

global mod_eigenValues;
global mod_eigenVectors;

mod_eigenValues = [];
mod_eigenVectors = [];

if nargin < 2
    directed = 0;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Test that the matrix is symmetric                      
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if sum(sum(abs(A-A')))~=0 && not(directed)
    disp('Matrix must be symmetric')
    return;
end

if nargin < 3
    min_comp = 0;
end
if nargin < 4
    max_iter = 0;
end

if isempty(max_iter)
    max_iter = Inf;
end

if max_iter==0
    max_iter = Inf;
end



[comp comp_num] = Get_Components(A,directed,directed); % work on connected components to reduce complexity

modules = zeros(1,length(A));
centrality = zeros(1,length(A));

if comp_num>1
    clustertree = cell(comp_num,1);
end

for i=1:comp_num
%     cur_A = Reduce_Matrix(A,comp(:,i)');
    cur_A = A(comp(:,i)>0,comp(:,i)>0);
    [cur_modules tree]= Modulate_EV(cur_A,ones(1,length(cur_A)),0,directed,min_comp,max_iter);
    index = find(comp(:,i));
    group_id = max(modules);
    cur_cent = ComputeCentrality(mod_eigenValues, mod_eigenVectors);
    for j=1:length(cur_A)
        modules(index(j)) = cur_modules(j) + group_id;
        centrality(index(j)) = cur_cent(j);
    end
    mod_eigenValues = [];
    mod_eigenVectors = [];
    if comp_num>1
        clustertree{i} = tree;
    else
        clustertree = tree;
    end
        
end
    

% [A_nonzero index] = Compact_Matrix(A,0); % run algorithm only on nodes with degree>0
% [modules_nonzero] = Modulate_EV(A_nonzero,ones(1,length(A_nonzero)),0,min_comp,max_iter);
% 
% 
% modules = ones(1,length(A));
% 
% if length(modules_nonzero) < length(modules)
%     b = 1;
% else
%     b = 0;
% end
% 
% for i=1:length(modules_nonzero)
%     modules(index(i)) = modules_nonzero(i)+b;
% end

Q = ComputeModularity(A,modules,directed);
% centrality = ComputeCentrality(mod_eigenValues, mod_eigenVectors);
% 
clear global mod_eigenValues;
clear global mod_eigenVectors;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Modulate_EV
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [m ct] = Modulate_EV(A,g,iter,directed,min_comp,max_iter)

m = g;

if iter>=max_iter
    return;
end

if length(find(g)) <= min_comp
    return;
end

B = Modularity_Matrix(A,g);

if directed
    B = B + B';
end

s = Split(B);

if abs(sum(s))~=length(s) % in case the split produced nothing
    q = s*B*s';
    q = q/(2*sum(sum(abs((A(g==1,g==1))))));
else
    q = 0;
end

if q>0
    [g1,g2] = Reconstruct(s,g);
    [m1 ct1] = Modulate_EV(A,g1,iter+1,directed,min_comp,max_iter);
    [m2 ct2] = Modulate_EV(A,g2,iter+1,directed,min_comp,max_iter);
    [m ct] = Combine(m1,m2,ct1,ct2,q);
else
    ct = [];
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Modularity_Matrix
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function B = Modularity_Matrix(A,g)

if min(min(A)) < 0 % if there are negative weights, treat them as different types of ties
    [Aplus Aminus] = DivideWeights(A);
    Bplus = Modularity_Matrix(Aplus,g);
    Bminus = Modularity_Matrix(Aminus,g);
    B = Bplus - Bminus;
else
    m = sum(sum(A)); % number of edges in the graph

    Kin = sum(A);
    Kout = sum(A');
    K = Kin'*Kout; % matrix of ki*kj
    if m>0
        K = K./m;
    end 

    B = A'-K; % Bij = Aji - Kin(i)Kout(j)/m;

    B = Reduce_Matrix(B,g);
    B = B - diag(sum(B'));
end





% B = B.*abs(1-eye(length(B)));

% B = Reduce_Matrix(B,g); % B = Aij - kikj/2m
% 
% G = Reduce_Matrix(A,g);
% Kg = sum(G);
% Ki = sum(A);
% Dg = sum(Ki(g>0));
% if m>0
%     Ki = Ki.*(Dg/(2*m));
% end
% Ki = Ki(g>0);
% Kg = (Kg-Ki);
% 
% B = B-diag(Kg);




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Split
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function s = Split(B)

s = ones(1,length(B));
[v,d] = GetEigenValues(B);


if(max(d)<=0) % if the leading eigenvalue is non-positive (fixed for epsilon), then we cannot split
    return
end;

[m,max_eig] = max(diag(d));

max_v = v(:,max_eig);
max_v(abs(max_v)<(10^4*eps))=0;
s(max_v<=0)=-1;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% ComputeModularity
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function Q = ComputeModularity(A,m,directed)

B = Modularity_Matrix(A,ones(1,length(A)));

if directed
    B = B + B';
end
S = zeros(length(A),length(unique(m)));

for i=1:length(m)
    S(i,m(i))=1;
end

if sum(sum(abs((A))))>0
    Q = trace(S'*B*S)/sum(sum(abs((A))));
else
    Q = 0;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% ComputeCentrality
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function centrality = ComputeCentrality(d, v)

if length(d)<1
    centrality = 0;
    return;
end

[d_sort,ix] = sort(diag(d),'descend');

i = 1;
centrality = zeros(1,length(d_sort));
while d_sort(i)>0
    centrality = centrality + d_sort(i)*(v(:,ix(i)).^2)';
    i=i+1;
end

centrality = sqrt(centrality);



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Reconstruct
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [v1 v2] = Reconstruct(s,g)

v = zeros(1,length(g));
v1 = zeros(1,length(g));
v2 = zeros(1,length(g));

f = find(g);

for i=1:length(s)
    v(f(i))=s(i);
end

v1(v>0)=1;
v2(v<0)=1;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Combine
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [m ct] = Combine(m1,m2,ct1,ct2,dist)

m = m2 + max(m1);
m(m2==0)=0;
m = m + m1;

nnum1 = size(ct1,1)+1;
nnum2 = size(ct2,1)+1;
nnum = nnum1+nnum2;

if ~isempty(ct1)
    ct1nodes = ct1(:,[1,2]);
    ct1weights = ct1(:,3);
    ct1nodes(ct1nodes>nnum1) = ct1nodes(ct1nodes>nnum1)+nnum2;
    mxnodes1 = nnum + size(ct1,1);
%     mxnodes1 = max(max(ct1nodes));
else
    mxnodes1 = 1;
    ct1nodes = [];
    ct1weights = [];
end

if ~isempty(ct2)
    ct2nodes = ct2(:,[1,2]);
    ct2weights = ct2(:,3);
    ct2nodes(ct2nodes<=nnum2) = ct2nodes(ct2nodes<=nnum2)+nnum1;
    ct2nodes(ct2(:,[1,2])>nnum2) = ct2nodes(ct2(:,[1,2])>nnum2)+nnum1 + size(ct1,1);
    mxnodes2 = nnum + size(ct1,1) + size(ct2,1);
%     mxnodes2 = mxnodes1 + size(ct2,1);
%     mxnodes2 = mxnodes1 + max(max(ct2nodes));
else
    ct2nodes = [];
    mxnodes2 = nnum1+1;
    ct2weights = [];
end

ct = [ct1nodes,ct1weights;ct2nodes,ct2weights];
ct = [ct;mxnodes1,mxnodes2,dist];
% ct = [ct;nnum1+nnum2+size(ct1nodes,1),nnum1+nnum2+size(ct1nodes,1)+size(ct2nodes,1),dist];


% 
% 
% 
% 
% if nnum1>1
%     cnum1 = max(max(ct1(:,[1,2])))-nnum1;
% else
%     cnum1 = 0;
% end
% if nnum2>1
%     cnum2 = max(max(ct2(:,[1,2])))-nnum2;
% else
%     cnum2 = 0;
% end
% 
% n1 = ClusterTreeCutOff(ct1);
% n2 = ClusterTreeCutOff(ct2);
% 
% if ~isempty(ct1)
%     nodes1 = ct1(1:n1-1,:);
% else
%     nodes1 = [];
% end
% if ~isempty(ct2)
%     nodes2 = ct2(1:n2-1,:);
%     nodes2(:,[1,2]) = nodes2(:,[1,2]) + nnum1;
% else
%     nodes2 = [];
% end
% 
% if ~isempty(ct1)
%     clust1 = ct1(n1:end,:);
%     if ~isempty(clust1)
%         clust1(:,[1,2]) = clust1(:,[1,2])+nnum2;
%     end
% else
%     clust1 = [];
% end
% 
% if ~isempty(ct2)
%     clust2 = ct2(n2:end,:);
%     if ~isempty(clust2)
%         clust2(:,[1,2]) = clust2(:,[1,2])+nnum1+cnum1;
%     end
% else
%     clust2 = [];
% end
% 
% 
% ct = [nodes1;nodes2;clust1;clust2];
% 
% if ~isempty(ct1)
%     mx1 = max(max(ct1(:,[1,2])));
%     mx1b = max(max(ct1(:,3)));
% else
%     mx1 = 0;
%     mx1b = 0;
% end
% if ~isempty(ct2)
%     mx2 = max(max(ct2(:,[1,2])));
%     mx2b = max(max(ct2(:,3)));
% else
%     mx2 = 0;
%     mx2b = 0;
% end
% 
% if isempty(ct1)
%     mxclust1 = 1;
% else
%     mxclust1 = nnum1+nnum2+size(ct1,1);
% end
% 
% if isempty(ct2)
%     mxclust2 = 1+nnum1;
% else
%     mxclust2 = nnum1+nnum2+size(ct1,1)+size(ct2,1);
% end
% 
% ct = [ct;...
%     mxclust1,mxclust2,max(mx1b,mx2b)+1];

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% ClusterTreeCutOff
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function n = ClusterTreeCutOff(ct)

n = 2;
if ~isempty(ct)
    nnum = size(ct,1)+1;
    for i=1:nnum-1
        if ~isempty(find(ct(i,:)>nnum))
            n = i;
            break;
        end
    end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% GetEigenValues
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [v d] = GetEigenValues(B)

global mod_eigenValues;
global mod_eigenVectors;

[v,d] = eig(B);
d(abs(d)<(10^4*eps))=0; % correct numeric errors
v(abs(v)<(10^4*eps))=0;
if isempty(mod_eigenValues)
    mod_eigenValues = d;
end

if isempty(mod_eigenVectors)
    mod_eigenVectors = v;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% DivideWeights
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [Aplus Aminus] = DivideWeights(A)

Aplus = zeros(size(A));
Aminus = zeros(size(A));

Aplus(A>0) = A(A>0);
Aminus(A<0) = -A(A<0);

