% [n,m,q,c,R,mu,sigma] = RCA(data,maxs,option,bootsN)
% 
% Run an RCA analyis
% 
% data: NxM dataset
% maxs: 1xM vector of maximal values for each variable
% option: for calculating the distance. Defulat is relationality, but other
%         pdist options are also applicable
% bootsN: number of bootstrap samples. if =0, use mean and std of
%         relationality matrix
% 
% n: number of clusters
% m: cluster assignments
% q: modularity value
% c: centrality values
% R: relationality matrix
% mu: mean relationality value
% sigma: standard deviation of relationality values

function [n,m,q,c,R,mu,sigma,tree] = RCA(data,maxs,option,bootsN,alpha)

disp('Generating relationality matrix...');

if nargin<2
    maxs = max(data);
end

if nargin<3
    option = [];
end

if nargin<4
    bootsN = 1000;
end

if nargin<5
    alpha = 0.05;
end

if isempty(option) || strcmp(option,'relationality')
    R = Relationality(data,maxs);
else
    R = squareform(pdist(data,option));
end


disp('Bootstrapping...');

if ~isempty(option) && ~strcmp(option,'relationality')    
    switch option
        case {'euclidean'}
            R = 1-R./sqrt(sum(maxs.^2));
        case {'jaccard','cosine'}
            R = 1-R;
    end
    R = R.*(1-eye(size(R,1)));
end

if bootsN
    bootstrap = RCA_bootstrap(R,bootsN);
    mu = mean(bootstrap(:,1));
    sigma = mean(bootstrap(:,2));
else
    vec = R(logical(tril(ones(size(R,1)),-1)));
    mu = mean(vec);
    sigma = std(vec);
    clear vec;
end

if isempty(option) || strcmp(option,'relationality')
    R05 = abs(GetSignificantRelationality(R-mu,0,sigma,alpha));
else
    R05 = GetSignificantRelationality(R-mu,0,sigma,alpha);
    R05(R05<0) = 0;
end

disp('Finding clusters...');

[m q c tree] = Modularity_Eig(R05);

n = length(unique(m));





