###
# Computes the average value loss for instances generated using the taxi data
# and geographical constraints.
###
import numpy as np
from datetime import *
import os
import matplotlib as mpl
if os.environ.get('DISPLAY','') == '':
    print('no display found. Using non-interactive Agg backend')

mpl.use('Agg')
import matplotlib.pyplot as plt
import pandas as pd
from joblib import Parallel, delayed
import dateutil.parser
from gurobipy import *
import time
import networkx as nx
from geopy import distance

# Number of threads allowed for gurobi. Default = 0 (maximum threads available)
############################
__gurobi_threads = 1
############################


def read_data_by_time(taxiData, start_time, time_delta, random=False, num_jobs=100, seed=7):
    rides_in_time_interval = (taxiData["tpep_pickup_datetime"].map(lambda x: dateutil.parser.parse(x)) >= start_time) & (taxiData["tpep_pickup_datetime"].map(lambda x: dateutil.parser.parse(x)) < start_time+time_delta)
    taxiData = taxiData[rides_in_time_interval]
    if random:
        taxiData = taxiData.sample(num_jobs, random_state=seed)

    start = list(taxiData["tpep_pickup_datetime"])
    end = list(taxiData["tpep_dropoff_datetime"])
    revenue = list(taxiData["total_amount"])
    pickup_long = list(taxiData["pickup_longitude"])
    pickup_lat = list(taxiData["pickup_latitude"])
    dropoff_long = list(taxiData["dropoff_longitude"])
    dropoff_lat = list(taxiData["dropoff_latitude"])
    return start, end, revenue, pickup_lat, pickup_long, dropoff_lat, dropoff_long


## Efficient solution
## Given the time in which it starts, ends and revenue of each trip,
## it computes the efficient solution, that maximizes the sum of the
## revenues, for the number of provideres given by num_providers.
## It returns the total revenue of this solution.

def not_in_clique(job_1, job_2, list_of_cliques):
    for clique in list_of_cliques:
        if (job_1 in clique) and (job_2 in clique):
            return False
    return True


def efficient_sol(start, end, revenue, pickup_lat, pickup_long, dropoff_lat, dropoff_long, avgSpeed, num_providers,interval_start_time, interval_end_time):
    num_jobs = len(start)
    m = Model("Efficient")
    jobs = range(num_jobs)
    providers = range(num_providers)

    x = m.addVars(num_providers, num_jobs,  vtype=GRB.BINARY)

    m.setObjective(quicksum(revenue[job]*x[prov,job]
                   for job in jobs for prov in providers), GRB.MAXIMIZE)

    onlyOneProvPerJobCts = m.addConstrs((x.sum('*', job) <= 1
                      for job in jobs), "c1")

    clique_cts = []
    time_delta = (interval_end_time - interval_start_time)/20

    num_cliques = 20
    intersects_time_point = [[] for i in range(num_cliques)]
    for i in range(num_cliques):
        time_point = interval_start_time + i * time_delta

        for job in jobs:
            if dateutil.parser.parse(start[job])< time_point and dateutil.parser.parse(end[job]) > time_point:
                intersects_time_point[i].append(job)
        for prov in providers:
            clique_cts.append(m.addConstr(quicksum(x[prov, job] for job in intersects_time_point[i]) <= 1))


    noOverlappingJobsCts = []

    for prov in providers:
        for job_1 in jobs:
            for job_2 in jobs:
                if((job_2 is not job_1)
                    and
                    (not_in_clique(job_1, job_2, intersects_time_point))
                    and
                    ((start[job_1]<=start[job_2] and
                        start[job_2] <= end[job_1])
                        or
                    (start[job_2] <= start[job_1] and
                        start[job_1] <= end[job_2]))):
                    noOverlappingJobsCts.append(m.addConstr(x[prov,job_1] + x[prov,job_2] <= 1, "c2"))

    for prov in providers:
        for job_1 in jobs:
            for job_2 in jobs:
                    if ((job_2 is not job_1)
                        and
                        (start[job_1]<=start[job_2] and
                        start[job_2] >= end[job_1])):

                        if distance.distance((dropoff_lat[job_1], dropoff_long[job_1]),
                                      (pickup_lat[job_2], pickup_long[job_2])).miles / avgSpeed > (dateutil.parser.parse(start[job_2])-dateutil.parser.parse(end[job_1])).total_seconds()/360:
                            noOverlappingJobsCts.append(m.addConstr(x[prov,job_1] + x[prov,job_2] <= 1, "c2"))

    for prov in providers:
        for job_1 in jobs:
            for job_2 in jobs:
                    if ((job_2 is not job_1)
                        and
                        (start[job_2] <= start[job_1] and
                        start[job_1] >= end[job_2])):

                        if distance.distance((dropoff_lat[job_2], dropoff_long[job_2]),
                                      (pickup_lat[job_1], pickup_long[job_1])).miles / avgSpeed > (dateutil.parser.parse(start[job_1])-dateutil.parser.parse(end[job_2])).total_seconds()/360:
                            noOverlappingJobsCts.append(m.addConstr(x[prov,job_1] + x[prov,job_2] <= 1, "c2"))
    m.setParam('LogFile', " ")
    m.setParam( 'OutputFlag', False )

    ### Set the number of threads for GUROBI:
    ##########################################
    m.Params.threads = __gurobi_threads
    ##########################################

    m.optimize()
    status = m.status

    x_eff = {}
    for p in providers:
        for j in jobs:
            x_eff[(p,j)] = x[p,j].X

    return m.objVal, x_eff




## Max_min solution
## Given the time in which it starts, ends and revenue of each trip,
## it computes the max_min solution for the number of provideres
## given by num_providers. Note that right now it only maximizes total
## revenue conditional on the solution maximizing the minimum allocation.
## It returns the total revenue of this solution.

def max_min_sol(start, end, revenue, pickup_lat, pickup_long, dropoff_lat, dropoff_long, avgSpeed, num_providers, interval_start_time, interval_end_time):
    num_jobs = len(start)
    m = Model("Max_Min")
    jobs = range(num_jobs)
    providers = range(num_providers)

    x = m.addVars(num_providers, num_jobs,  vtype=GRB.BINARY)
    t = m.addVar()

    m.setObjective(t, GRB.MAXIMIZE)

    minCts = []

    for prov in providers:
        minCts.append(m.addConstr(quicksum(x[prov,job] * revenue[job] for job in jobs) >= t))

    onlyOneProvPerJobCts = m.addConstrs((x.sum('*', job) <= 1
                      for job in jobs), "c1")


    clique_cts = []
    time_delta = (interval_end_time - interval_start_time)/20

    num_cliques = 20
    intersects_time_point = [[] for i in range(num_cliques)]
    for i in range(num_cliques):
        time_point = interval_start_time + i * time_delta

        for job in jobs:
            if dateutil.parser.parse(start[job])< time_point and dateutil.parser.parse(end[job]) > time_point:
                intersects_time_point[i].append(job)
        for prov in providers:
            clique_cts.append(m.addConstr(quicksum(x[prov, job] for job in intersects_time_point[i]) <= 1))


    noOverlappingJobsCts = []

    for prov in providers:
        for job_1 in jobs:
            for job_2 in jobs:
                if((job_2 is not job_1)
                    and
                    (not_in_clique(job_1, job_2, intersects_time_point))
                    and
                    ((start[job_1]<=start[job_2] and
                        start[job_2] <= end[job_1])
                        or
                    (start[job_2] <= start[job_1] and
                        start[job_1] <= end[job_2]))):
                    noOverlappingJobsCts.append(m.addConstr(x[prov,job_1] + x[prov,job_2] <= 1, "c2"))



    for prov in providers:
        for job_1 in jobs:
            for job_2 in jobs:
                    if ((job_2 is not job_1)
                        and
                        (start[job_1]<=start[job_2] and
                        start[job_2] >= end[job_1])):

                        if distance.distance((dropoff_lat[job_1], dropoff_long[job_1]),
                                      (pickup_lat[job_2], pickup_long[job_2])).miles / avgSpeed > (dateutil.parser.parse(start[job_2])-dateutil.parser.parse(end[job_1])).total_seconds()/360:
                            noOverlappingJobsCts.append(m.addConstr(x[prov,job_1] + x[prov,job_2] <= 1, "c2"))


    for prov in providers:
        for job_1 in jobs:
            for job_2 in jobs:
                    if ((job_2 is not job_1)
                        and
                        (start[job_2] <= start[job_1] and
                        start[job_1] >= end[job_2])):

                        if distance.distance((dropoff_lat[job_2], dropoff_long[job_2]),
                                      (pickup_lat[job_1], pickup_long[job_1])).miles / avgSpeed > (dateutil.parser.parse(start[job_1])-dateutil.parser.parse(end[job_2])).total_seconds()/360:

                            noOverlappingJobsCts.append(m.addConstr(x[prov,job_1] + x[prov,job_2] <= 1, "c2"))

    m.setParam('LogFile', " ")
    m.setParam('TimeLimit', 2*60*60)
    m.setParam( 'OutputFlag', False )

    ### Set the number of threads for GUROBI:
    ##########################################
    m.Params.threads = __gurobi_threads
    ##########################################
    m.optimize()

    tempVarValues = [[0 for job in jobs] for prov in providers]
    tempBasisVal = [[0 for job in jobs] for prov in providers]
    for job in jobs:
        for prov in providers:
            tempVarValues[prov][job] = x[prov,job].getAttr("x")
 

    m.setParam('MIPGap', 0)
 

    #Second stage, maximize sum, subject to the all of the allocations being larger than the max Min
    maxMin = m.objVal
    m.setObjective(quicksum(revenue[job]*x[prov,job]
                   for job in jobs for prov in providers), GRB.MAXIMIZE)
    m.addConstr(t == maxMin)
    m.setParam('LogFile', " ")
    m.setParam( 'OutputFlag', False )

    ### Set the number of threads for GUROBI:
    ##########################################
    m.Params.threads = __gurobi_threads
    ##########################################
    m.update()

    for job in jobs:
        for prov in providers:
            x[prov,job].Start = tempVarValues[prov][job]

    m.optimize()
    status = m.status

    x_maxMin = {}
    for p in providers:
        for j in jobs:
            x_maxMin[(p,j)] = x[p,j].X


    return m.objVal, x_maxMin

def generate_graph(start, end):
    num_jobs = len(start)
    jobs = range(num_jobs)

    G = nx.Graph()
    G.add_nodes_from(jobs)


    for job_1 in jobs:
            for job_2 in jobs:
                if((job_2 is not job_1)
                    and
                    ((start[job_1]<=start[job_2] and
                        start[job_2] <= end[job_1])
                        or
                    (start[job_2] <= start[job_1] and
                        start[job_1] <= end[job_2]))):
                    G.add_edge(job_1,job_2)

    return G

def value_loss(num_prov, num_jobs, data, trial, window, directory, avgSpeed):
    eff = 0
    max_min = 0

    for d in data:
        start = d[0]
        end = d[1]
        revenue = d[2]
        pickup_lat = d[3]
        pickup_long = d[4]
        dropoff_lat = d[5]
        dropoff_long = d[6]
        s_time = d[7]
        e_time = d[8]



        partial_eff, x_eff = efficient_sol(start, end, revenue, pickup_lat, pickup_long, dropoff_lat, dropoff_long, avgSpeed, num_prov, s_time, e_time)
        partial_max_min, x_maxMin = max_min_sol(start, end, revenue, pickup_lat, pickup_long, dropoff_lat, dropoff_long, avgSpeed, num_prov, s_time, e_time)


        # G = generate_graph(start, end)
        # pos = nx.fruchterman_reingold_layout(G) #nx.spring_layout(G)
        # nx.draw(G,pos=pos, labels=dict(enumerate(revenue)), font_size=8, node_size=400)

        # # dens = nx.density(G)
        # # maxIS = nx.algorithms.clique.graph_clique_number(nx.algorithms.operators.unary.complement(G))
        # # maxClique = nx.algorithms.clique.graph_clique_number(G)

        # cm = plt.get_cmap('gnuplot')

        # colors = [cm(1. * i / num_prov) for i in range(num_prov)]
        # #colors = ['m','r','c']
        # shapes = ['8','p','h','s']
        # print("COLORS:  {}".format(colors))

        # if (partial_eff - partial_max_min) < 10e-6 * partial_eff:
        #     if not os.path.exists(directory+"/zeroLoss"):
        #                 os.makedirs(directory+"/zeroLoss")
        #     plt.savefig(directory+"/zeroLoss/Graph-NonDyn-trial-{}-NumProv-{}-stime-{}-etime-{}.pdf".format(trial, num_prov,s_time,e_time))
        #     plt.clf()
        #     nx.draw(G, pos, labels=dict(enumerate(revenue)), font_size=7, node_size=450, node_color='w', node_shape='o')
        #     ax = plt.gca() # to get the current axis
        #     ax.collections[0].set_edgecolor("#000000")
        #     for p in range(num_prov):
        #         nlist = [j for j in range(num_jobs) if x_eff[(p,j)] == 1]
        #         n_col = colors[p]
        #         n_shape = 'o'
        #         nx.draw_networkx_nodes(G, pos, nodelist=nlist, node_shape=n_shape, node_color=n_col,font_size=7, node_size = 250, labels=dict(enumerate(revenue)))
        #     nx.draw_networkx_labels(G, pos, font_size=7, labels=dict(enumerate(revenue)))
        #     plt.savefig(directory+"/zeroLoss/Graph-NonDyn-trial-{}-NumProv-{}-stime-{}-etime-{}-Eff-DiffShapes.pdf".format(trial, num_prov,s_time,e_time))
        #     plt.clf()

        #     nx.draw(G, pos, labels=dict(enumerate(revenue)), font_size=7, node_size=400, node_color='w')
        #     ax = plt.gca() # to get the current axis
        #     ax.collections[0].set_edgecolor("#000000")
        #     for p in range(num_prov):
        #         nlist = [j for j in range(num_jobs) if x_maxMin[(p,j)] == 1]
        #         n_col = colors[p]
        #         n_shape = 'o'
        #         nx.draw_networkx_nodes(G, pos, nodelist=nlist, node_shape=n_shape, node_color=n_col,font_size=7, node_size = 250, labels=dict(enumerate(revenue)))

        #     nx.draw_networkx_labels(G, pos, font_size=7, labels=dict(enumerate(revenue)))
        #     plt.savefig(directory+"/zeroLoss/Graph-NonDyn-trial-{}-NumProv-{}-stime-{}-etime-{}-Max-Min-DiffShapes.pdf".format(trial, num_prov,s_time,e_time))
        #     plt.clf()


        #     # densities_noLoss.append(dens)
        #     # max_clique_No_Loss.append(maxClique)
        #     # max_IS_NoLoss.append(maxIS)

        # else:


        #     if not os.path.exists(directory+"/posLoss"):
        #                 os.makedirs(directory+"/posLoss")
        #     plt.savefig(directory+"/posLoss/Graph-NonDyn-trial-{}-NumProv-{}-stime-{}-etime-{}.pdf".format(trial, num_prov,s_time,e_time))
        #     plt.clf()

        #     nx.draw(G, pos, labels=dict(enumerate(revenue)), font_size=7, node_size=400, node_color='w')
        #     ax = plt.gca() # to get the current axis
        #     ax.collections[0].set_edgecolor("#000000")
        #     for p in range(num_prov):
        #         nlist = [j for j in range(num_jobs) if x_eff[(p,j)] == 1]
        #         n_col = colors[p]
        #         n_shape = shapes[p]
        #         nx.draw_networkx_nodes(G, pos, nodelist=nlist, node_shape=n_shape, node_color=n_col,font_size=7, node_size = 250, labels=dict(enumerate(revenue)))

        #     nx.draw_networkx_labels(G, pos, font_size=7, labels=dict(enumerate(revenue)))
        #     plt.savefig(directory+"/posLoss/Graph-NonDyn-trial-{}-NumProv-{}-stime-{}-etime-{}-Eff-DiffShapes.pdf".format(trial, num_prov,s_time,e_time))
        #     plt.clf()

        #     nx.draw(G, pos, labels=dict(enumerate(revenue)), font_size=7, node_size=400, node_color='w')
        #     ax = plt.gca() # to get the current axis
        #     ax.collections[0].set_edgecolor("#000000")
        #     for p in range(num_prov):
        #         nlist = [j for j in range(num_jobs) if x_maxMin[(p,j)] == 1]
        #         n_col = colors[p]
        #         n_shape = shapes[p]
        #         nx.draw_networkx_nodes(G, pos, nodelist=nlist, node_shape=n_shape, node_color=n_col,font_size=7, node_size = 250, labels=dict(enumerate(revenue)))

        #     nx.draw_networkx_labels(G, pos, font_size=7, labels=dict(enumerate(revenue)))
        #     plt.savefig(directory+"/posLoss/Graph-NonDyn-trial-{}-NumProv-{}-stime-{}-etime-{}-Max-Min-DiffShapes.pdf".format(trial, num_prov,s_time,e_time))
        #     plt.clf()

        #     # densities_Loss.append(dens)
        #     # max_clique_Loss.append(maxClique)
        #     # max_IS_Loss.append(maxIS)

        # G.clear()

        eff += partial_eff
        max_min += partial_max_min
    price_of_rest = (eff - max_min)/eff


    f = open(directory+"/OutputTrial"+str(trial)+".txt", "a+")
    f.write(str(num_prov)+", "+str(price_of_rest)+"\n")
    f.close()

    print()
    print("#######################################")
    print("#######################################")
    print("#######################################")
    print("Finished solving for "+str(num_prov)+"providers")
    print("Efficient for "+str(num_prov)+" agents is: "+str(eff))
    print("Max_min for "+str(num_prov)+" agents is: "+str(max_min))
    print("#######################################")
    print("#######################################")
    print("#######################################")
    print()



def datetime_range(start, end, delta):
    current = start
    dts = []
    while current < end:
        dts.append(current)
        current += delta
    return dts


def multiple_value_loss(num_jobs, providers_list,
                         taxiData, dts,time_delta, trial, window, directory, avgSpeed):
    data = []
    for st in dts:
        start, end, revenue, pickup_lat, pickup_long, dropoff_lat, dropoff_long = read_data_by_time(taxiData, st, time_delta, True, num_jobs, trial)
        end_t = st + time_delta
        data.append((start, end, revenue, pickup_lat, pickup_long, dropoff_lat, dropoff_long, st, end_t))

    for num_prov in providers_list:
        value_loss(num_prov, num_jobs, data, trial, window, directory, avgSpeed)

def main():
    main_started = time.time()
    for window in [60]:
        for day in [8]:
            for ns_jobs in [120]:
                for region in ["All-Manhattan"]:
                    for avgSpeed in [7.44, 10.7]:

                        date = datetime(2016, 1, day).date()
                        start_time = datetime(2016, 1, day, 9)
                        time_delta = timedelta(minutes=window)
                        end_time = datetime(2016, 1, day, 17)
                        dts = datetime_range(start_time, end_time, time_delta)
                        num_jobs = ns_jobs
                        taxiData = pd.read_csv('../Data/CleanedData_'+str(date)+'.csv')
                        first_prov = 2
                        last_prov = 30
                        prov_interval = 1
                        providers_list = range(first_prov, last_prov, prov_interval)

                        # Number of times we sample from each time interval num_jobs jobs.
                        #######################################
                        number_of_trials = 50          ########
                        #######################################

                        directory = "../Output/"+str(window)+"_Min_Window_"+str(num_jobs)+"_jobs_NineToFiveData_nonDynamic-GeographicFeasMultipleTests_"+region+"_"+str(date)+"avgSpeed_"+str(avgSpeed)+"/"
                        if not os.path.exists(directory):
                            os.makedirs(directory)
                        main_started = time.time()
                        Parallel(n_jobs = 10)(delayed(multiple_value_loss)(num_jobs, providers_list, taxiData, dts, time_delta, trial, window, directory, avgSpeed) for trial in range(number_of_trials))

    main_finished = time.time()
    print("Total time elapsed = "+str(main_finished - main_started))

if __name__ == '__main__':
    main()
