###
# Computes the value loss for synthetic instances with fixed length and saves the results in
# the folder ../Output. It also generates graphs of the instances and saves them.
###
import numpy as np
import os
import matplotlib as mpl
if os.environ.get('DISPLAY','') == '':
    print('no display found. Using non-interactive Agg backend')
    mpl.use('Agg')
import matplotlib.pyplot as plt
import pandas as pd
from joblib import Parallel, delayed
from gurobipy import *
import time
from scipy.stats import truncnorm, uniform
import networkx as nx

# Number of threads allowed for gurobi. Default = 0 (maximum threads available)
############################
__gurobi_threads = 1
############################



## Efficient solution
## Given the time in which it starts, ends and revenue of each trip, 
## it computes the efficient solution, that maximizes the sum of the 
## revenues, for the number of provideres given by num_providers. 
## It returns the total revenue of this solution. 

def not_in_clique(job_1, job_2, list_of_cliques):
    for clique in list_of_cliques:
        if (job_1 in clique) and (job_2 in clique):
            return False
    return True


def efficient_sol(start, end, revenue, num_providers,interval_start_time, interval_end_time, previous_jobs = None):
    print("Started Eff Sol")
    num_jobs = len(start)
    m = Model("Efficient")
    jobs = range(num_jobs)
    providers = range(num_providers)

    x = m.addVars(num_providers, num_jobs,  vtype=GRB.BINARY)

    m.setObjective(quicksum(revenue[job]*x[prov,job] 
                   for job in jobs for prov in providers), GRB.MAXIMIZE)

    onlyOneProvPerJobCts = m.addConstrs((x.sum('*', job) <= 1
                      for job in jobs), "c1")

    clique_cts = []
    time_delta = (interval_end_time - interval_start_time)/20

    num_cliques = 20
    intersects_time_point = [[] for i in range(num_cliques)]
    for i in range(num_cliques):
        time_point = interval_start_time + i * time_delta
        
        for job in jobs:
            if start[job]< time_point and end[job] > time_point:
                intersects_time_point[i].append(job)
        for prov in providers:
            clique_cts.append(m.addConstr(quicksum(x[prov, job] for job in intersects_time_point[i]) <= 1))


    noOverlappingJobsCts = []

    for prov in providers:
        for job_1 in jobs:
            for job_2 in jobs:
                if((job_2 is not job_1) 
                    and
                    (not_in_clique(job_1, job_2, intersects_time_point))
                    and 
                    ((start[job_1]<=start[job_2] and 
                        start[job_2] <= end[job_1])
                        or 
                    (start[job_2] <= start[job_1] and 
                        start[job_1] <= end[job_2]))):
                    noOverlappingJobsCts.append(m.addConstr(x[prov,job_1] + x[prov,job_2] <= 1, "c2"))

    ### Set the number of threads for GUROBI:
    ##########################################
    m.Params.threads = __gurobi_threads
    ##########################################


    m.optimize()
    status = m.status
    print(status)
    print('Finished Efficient')
    print('The optimal objective is %g' % m.objVal)
    print('#######################################')
    
    if status == GRB.Status.OPTIMAL:
        print('Finished Efficient')
        print('The optimal objective is %g' % m.objVal)
        print('#######################################')
    return m.objVal




## Max_min solution
## Given the time in which it starts, ends and revenue of each trip, 
## it computes the max_min solution for the number of provideres 
## given by num_providers. Note that right now it only maximizes total
## revenue conditional on the solution maximizing the minimum allocation. 
## It returns the total revenue of this solution. 

def max_min_sol(start, end, revenue, num_providers, interval_start_time, interval_end_time):
    print("Started Max Min")
    num_jobs = len(start)
    m = Model("Max_Min")
    jobs = range(num_jobs)
    providers = range(num_providers)

    x = m.addVars(num_providers, num_jobs,  vtype=GRB.BINARY)
    t = m.addVar()

    m.setObjective(t, GRB.MAXIMIZE)

    minCts = []

    for prov in providers:
        minCts.append(m.addConstr(quicksum(x[prov,job] * revenue[job] for job in jobs) >= t))

    onlyOneProvPerJobCts = m.addConstrs((x.sum('*', job) <= 1
                      for job in jobs), "c1")


    clique_cts = []
    time_delta = (interval_end_time - interval_start_time)/20

    num_cliques = 20
    intersects_time_point = [[] for i in range(num_cliques)]
    for i in range(num_cliques):
        time_point = interval_start_time + i * time_delta
        
        for job in jobs:
            if start[job]< time_point and end[job] > time_point:
                intersects_time_point[i].append(job)
        for prov in providers:
            clique_cts.append(m.addConstr(quicksum(x[prov, job] for job in intersects_time_point[i]) <= 1))


    noOverlappingJobsCts = []

    for prov in providers:
        for job_1 in jobs:
            for job_2 in jobs:
                if((job_2 is not job_1) 
                    and
                    (not_in_clique(job_1, job_2, intersects_time_point))
                    and 
                    ((start[job_1]<=start[job_2] and 
                        start[job_2] <= end[job_1])
                        or 
                    (start[job_2] <= start[job_1] and 
                        start[job_1] <= end[job_2]))):
                    noOverlappingJobsCts.append(m.addConstr(x[prov,job_1] + x[prov,job_2] <= 1, "c2"))

    ### Set the number of threads for GUROBI:
    ##########################################
    m.Params.threads = __gurobi_threads
    ##########################################
    m.optimize()
    
    tempVarValues = [[0 for job in jobs] for prov in providers]
    tempBasisVal = [[0 for job in jobs] for prov in providers]
    for job in jobs:
        for prov in providers:
            tempVarValues[prov][job] = x[prov,job].getAttr("x")
    
    
    status = m.status
    print(status)
    print('Finished First phase of Max Min')
    print('The optimal objective is %g' % m.objVal)
    print('#######################################')
    
    print('The Minimum value any provider gets is %g' % m.objVal)
    print("Value of t: "+str(t.getAttr("X")))
    print(m.objVal)
    maxMin = m.objVal
    m.setObjective(quicksum(revenue[job]*x[prov,job] 
                   for job in jobs for prov in providers), GRB.MAXIMIZE)
    m.addConstr(t == t.getAttr("X"))


    for job in jobs:
        for prov in providers:
            x[prov,job].Start = tempVarValues[prov][job]
            # x[prov,job].VBasis = tempBasisVal[prov][job]

    
    ### Set the number of threads for GUROBI:
    ##########################################
    m.Params.threads = __gurobi_threads
    ##########################################

    m.optimize()

    status = m.status
    print(status)
    print('Finished Second phase of Max Min')
    if status != 2:    
        print('The model is infeasible; computing IIS')
        m.computeIIS()
        print('\nThe following constraint(s) cannot be satisfied:')
        for c in m.getConstrs():
            if c.IISConstr:
                print('%s' % c.constrName)
    print('The optimal objective is %g' % m.objVal)
    print('#######################################')
    if status == GRB.Status.OPTIMAL:
        print('The Max Min solution is %g' % m.objVal)
    #The solution is the Max_min (version where we only max the min revenue...)
    
    
    return m.objVal
    

def line_prepender(filename, line):
    with open(filename, 'r+') as f:
        content = f.read()
        f.seek(0, 0)
        f.write(line.rstrip('\r\n') + '\n' + content)

def generate_graph(start, end):
    num_jobs = len(start)
    jobs = range(num_jobs)

    G = nx.Graph()
    G.add_nodes_from(jobs)


    for job_1 in jobs:
            for job_2 in jobs:
                if((job_2 is not job_1) 
                    and 
                    ((start[job_1]<=start[job_2] and 
                        start[job_2] <= end[job_1])
                        or 
                    (start[job_2] <= start[job_1] and 
                        start[job_1] <= end[job_2]))):
                    G.add_edge(job_1,job_2)

    return G

def value_loss(num_prov, num_jobs, data, trial, length_var, directory):
    eff = 0
    max_min = 0
    d = data[0]
    start = d[0]
    end = d[1]
    revenue = d[2]
    s_time = d[3]
    e_time = d[4]

    eff = efficient_sol(start, end, revenue, num_prov, s_time, e_time)
    max_min = max_min_sol(start, end, revenue, num_prov,s_time, e_time)
    value_loss = (eff - max_min)/eff
    
    G = generate_graph(start, end)
    nx.draw(G, labels=dict(enumerate(revenue)), font_size=10, node_size=350)
    if value_loss < 0.000001:
        if not os.path.exists(directory+"/zeroLoss"):
                    os.makedirs(directory+"/zeroLoss")
        plt.savefig(directory+"/zeroLoss/Graph-NonDyn-trial-{}-NumProv-{}.pdf".format(trial, num_prov))
        plt.clf()
    else:
        if not os.path.exists(directory+"/posLoss"):
                    os.makedirs(directory+"/posLoss")
        plt.savefig(directory+"/posLoss/Graph-NonDyn-trial-{}-NumProv-{}.pdf".format(trial, num_prov))
        plt.clf()
    G.clear()

    f = open(directory+"/OutputTrial"+str(trial)+".txt", "a+")
    f.write(str(num_prov)+", "+str(value_loss)+"\n")
    f.close()

    print()
    print("#######################################")
    print("#######################################")
    print("#######################################")
    print("Finished solving for "+str(num_prov)+"providers")
    print("Efficient for "+str(num_prov)+" agents is: "+str(eff))
    print("Max_min for "+str(num_prov)+" agents is: "+str(max_min))
    print("#######################################")
    print("#######################################")
    print("#######################################")
    print()


def syntheticDataIsoLengthUnif(num_jobs, trial, s_t, e_t, length_var):
    ran = np.random.RandomState(int(trial^20))
    var = truncnorm.rvs(-1, np.inf, size=num_jobs, scale=length_var, loc=0, random_state=ran)
    revenue = [1+var[i] for i in range(num_jobs)]
    length = [1 for i in range(num_jobs)]
    start = uniform.rvs(s_t, e_t, size=num_jobs, random_state=ran)
    end = [start[i] + length[i] for i in range(num_jobs)]
    return start, end, revenue


def multiple_value_loss(num_jobs, providers_list, 
                         s_t, e_t, trial, length_var, directory):
    data = []
    start, end, revenue = syntheticDataIsoLengthUnif(num_jobs, trial, s_t, e_t, length_var)
    
    data.append((start, end, revenue, s_t, e_t))
    
    for num_prov in providers_list:
        value_loss(num_prov, num_jobs, data, trial, length_var, directory) 

def main(): 
    main_started = time.time()
    for var in [0.5, 0.45, 0.4, 0.35, 0.3, 0.25, 0.2, 0.15, 0.1, 0.05, 0.001]:
        for e_t in [1.3,1.5,1.7]:
            s_t = 0
            num_jobs = 30
            first_prov = 2
            last_prov = 32
            prov_interval = 2
            providers_list = range(first_prov, last_prov, prov_interval)
            length_var = var
            number_of_trials = 100
            directory = "../Output/Synthetic-fixedLen-Unif0_"+str(e_t)+"_length=rev=1+eps~TN(0,"+str(length_var)+")_"+str(num_jobs)+"Jobs+Graphs"
            if not os.path.exists(directory):
                os.makedirs(directory)
            Parallel(n_jobs = 8)(delayed(multiple_value_loss)(num_jobs, providers_list, 
                                                               s_t,e_t, trial, length_var, directory) 
                                                    for trial in range(number_of_trials))

        

    main_finished = time.time()
    print("Total time elapsed = "+str(main_finished - main_started))

if __name__ == '__main__':
    main()

