#!/usr/bin/env python3

import sys
import time
import ctypes

"""
Stanford CS106A Movie Grid Example (vertical)
Chris Gregg
"""
ESC = chr(27)
CSI = f'{ESC}['
DELAY = 0.1


def main():
    # for fancy colors and movement, Windows needs to be set up
    setup_windows_terminal()

    # open our own code to use as the characters
    with open(sys.argv[0]) as f:
        code = f.readlines()

    num_rows, num_cols = screen_dimensions()
    clear_screen()

    # set foreground color green, background color black
    # see here for ANSI codes:
    # https://en.wikipedia.org/wiki/ANSI_escape_code
    sys.stdout.write(f'{CSI}32;40m')
    sys.stdout.flush()

    run_matrix(code, num_rows, num_cols)


def get_line(code, y, num_cols):
    horiz_line = ''
    for x in range(num_cols):
        line = code[x % len(code)]
        ch = line[(len(line) - y) % len(line)]
        if ch == '\n':
            ch = ' '  # convert newline to space
        horiz_line += ch
    return horiz_line

def run_matrix(code, num_rows, num_cols):
    # create a horizontal line, one char from each line of code
    # for each line of code, grab a character to display vertically
    y = 0
    lines = []
    while True:
        next_line = get_line(code, y, num_cols)
        lines.insert(0, next_line)
        if len(lines) > num_rows:
            lines.pop()  # remove last line
        set_screen_pos(1, 1)
        for line in lines:
            print(line)
        # slow things down
        time.sleep(DELAY)
        y += 1


# --------------- UTILITY FUNCTIONS ----------------
def clear_screen():
    sys.stdout.write(f'{CSI}2J')
    sys.stdout.flush()


def set_screen_pos(row, col):
    # this is 1-based
    sys.stdout.write(f'{CSI}{row};{col}H')
    sys.stdout.flush()


def screen_dimensions():
    # save current pos
    sys.stdout.write(f'{ESC}7')
    sys.stdout.flush()
    # send cursor to 999, 999
    sys.stdout.write(f'{CSI}999;999H')
    sys.stdout.flush()
    # device status report
    sys.stdout.write(f'{CSI}6n')
    sys.stdout.flush()
    # should return CSIn;mR
    # get first two characters
    if getch() == ESC and getch() == '[':
        # get characters until we get a semicolon
        s = ''
        while True:
            ch = getch()
            if ch == ';':
                rows = int(s)
                s = ''
                while True:
                    ch = getch()
                    if ch == 'R':
                        cols = int(s)
                        # restore pos
                        sys.stdout.write(f'{ESC}8')
                        sys.stdout.flush()
                        return rows, cols
                    else:
                        s += ch
            else:
                s += ch
    else:
        sys.stdout.write(f'{ESC}8')
        sys.stdout.flush()
        return -1, -1  # didn't get a response


# read on character from terminal
# this is platform-specific (Windows/Unix)
# found at
# https://stackoverflow.com/a/510364/561677
class _Getch:
    """Gets a single character from standard input.  Does not echo to the
screen."""
    def __init__(self):
        try:
            self.impl = _GetchWindows()
        except ImportError:
            self.impl = _GetchUnix()

    def __call__(self): return self.impl()


class _GetchUnix:
    def __init__(self):
        import tty, sys

    def __call__(self):
        import sys, tty, termios
        fd = sys.stdin.fileno()
        old_settings = termios.tcgetattr(fd)
        try:
            tty.setraw(sys.stdin.fileno())
            ch = sys.stdin.read(1)
        finally:
            termios.tcsetattr(fd, termios.TCSADRAIN, old_settings)
        return ch


class _GetchWindows:
    def __init__(self):
        import msvcrt

    def __call__(self):
        import msvcrt
        return msvcrt.getch().decode('utf-8')


getch = _Getch()


def setup_windows_terminal():
    if sys.platform == 'win32':
        kernel32 = ctypes.windll.kernel32
        kernel32.SetConsoleMode(kernel32.GetStdHandle(-11), 7)

if __name__ == "__main__":
    main()
