from tkinter import Tk, Canvas
import random
import math
import time
import sys

CANVAS_WIDTH = 600
CANVAS_HEIGHT = 600
WINDOW_TOP_X = 300
WINDOW_TOP_Y = 300


class Ball:
    """
    The Ball class defines a "ball" that can bounce around the screen
    """
    def __init__(self, canvas, x=-1, y=-1, random_loc=False, fill=None):
        self.canvas = canvas
        self.canvas_object = None
        if random_loc:
            self.x = random.randint(0, canvas.winfo_width() - 1)
            self.y = random.randint(0, canvas.winfo_height() - 1)
        else:
            if x == -1:
                self.x = canvas.winfo_width() / 2
            else:
                self.x = x
            if y == -1:
                self.y = canvas.winfo_height() / 2
            else:
                self.y = y
        self.width = 30
        self.height = 30
        self.direction = random.random() * 360 / (2 * math.pi)
        if fill:
            self.fill = fill
        else:
            self.fill = f'#{random.randint(0, 16777215):06x}'

        self.outline = self.fill
        self.draw()

    def __str__(self):
        s = (f'x:{self.x}, y:{self.y}, '
             f'dir:{self.direction}, color:{self.fill}')
        return s

    def draw(self):
        self.canvas_object = self.canvas.create_oval(self.x, self.y, self.x +
                                           self.width, self.y + self.height,
                                           outline=self.outline,
                                           fill=self.fill)

    def update(self):
        x_dir = math.cos(self.direction)
        y_dir = math.sin(self.direction)

        self.x += x_dir * 10
        self.y += y_dir * 10

        if self.x < 0:
            self.x = 0
            x_dir *= -1

        if self.x > self.canvas.winfo_width() - self.width:
            self.x = self.canvas.winfo_width() - self.width
            x_dir *= -1

        if self.y < 0:
            self.y = 0
            y_dir *= -1

        if self.y > self.canvas.winfo_height() - self.height:
            self.y = self.canvas.winfo_height() - self.height
            y_dir *= -1

        self.direction = math.atan2(y_dir, x_dir)
        self.canvas.delete(self.canvas_object)

        self.draw()

    def collision(self, other_ball):
        return(math.sqrt((other_ball.x - self.x)**2 +
                         (other_ball.y - self.y)**2)) < self.width

    def handle_collision(self, other_ball):
        if self.collision(other_ball):
            tmp = other_ball.direction
            other_ball.direction = self.direction  # elastic collision?
            other_ball.update()
            # self.direction = self.direction + math.pi
            # if self.direction > 2 * math.pi:
            #     self.direction = 2 * math.pi - self.direction
            self.direction = tmp
            self.update()


def animate(canvas, ball_list, collide):
    if len(ball_list) <= 100:
        time.sleep(0.01 - 0.0001 * len(ball_list))
    for ball in ball_list:
        ball.update()
    if collide:
        for ball in ball_list:
            for other in ball_list:
                if ball == other:
                    continue
                ball.handle_collision(other)
    canvas.after(1, lambda: animate(canvas, ball_list, collide))


def make_canvas(width, height, title):
    tk = Tk()
    tk.minsize(width=width, height=height)
    tk.title(title)
    canvas = Canvas(tk, width=width + 1, height=height + 1)
    canvas.pack()
    canvas.update()
    return canvas, tk


def main():
    num_balls = 1
    random_loc = True
    collide = False

    if len(sys.argv) > 1:
        num_balls = int(sys.argv[1])

    if '--center' in sys.argv:
        random_loc = False

    if '--collide' in sys.argv:
        collide = True

    (canvas, tk) = make_canvas(CANVAS_WIDTH, CANVAS_HEIGHT, "Bouncing Ball")

    ball_list = [Ball(canvas, random_loc=random_loc, fill='black')]
    ball_list = ball_list + [Ball(canvas, random_loc=random_loc, fill='red')
                            for b in
                 range(num_balls - 1)]

    animate(canvas, ball_list, collide)
    tk.mainloop()


if __name__ == "__main__":
    main()

