#!/usr/bin/env python3
"""
Forest change detection using ghost.py utilities (revised).

Usage:
    python3 is_forest.py <new_dir> <old_dir>
"""

import sys
import os
import ghost
from simpleimage import SimpleImage


# ──────────────────────────────────────────────────────────────
# 1. Forest classifier
# ──────────────────────────────────────────────────────────────
def is_forest(p,
              min_brightness=10,
              max_brightness=160,
              max_red_blue_diff=20):
    """Heuristic forest classifier for RGB pixels.

    p: (r, g, b)

    Rules:
    - Forest is moderately dark (not too bright).
    - Green is the strongest channel.
    - Red cannot be much higher than blue (avoids yellowish agricultural fields).
    """
    r, g, b = p
    brightness = (r + g + b) / 3

    # 1) Brightness constraint
    if not (min_brightness <= brightness <= max_brightness):
        return False

    # 2) Green must dominate
    if not (g > r and g > b):
        return False

    # 3) Avoid yellow-green agriculture
    if (r - b) > max_red_blue_diff:
        return False

    return True


# ──────────────────────────────────────────────────────────────
# 2. Change-map with transparent overlay + stacked comparison
# ──────────────────────────────────────────────────────────────
def diff_image(img_new, img_old, alpha=0.5):
    """
    Create a change map on top of img_new (2024), with transparency.

    Returns a tall image with three stacked panels:
      1. overlay (2024 + red loss / green gain)
      2. plain 2024
      3. plain 2000
    """
    # Ensure identical dimensions
    w = min(img_old.width, img_new.width)
    h = min(img_old.height, img_new.height)

    overlay_img = SimpleImage.blank(w, h)

    loss = gain = 0

    for y in range(h):
        for x in range(w):
            old_p = img_old.get_color_tuple(x, y)
            new_p = img_new.get_color_tuple(x, y)

            old_forest = is_forest(old_p)
            new_forest = is_forest(new_p)

            base_r, base_g, base_b = new_p  # 2024 pixel

            if old_forest and not new_forest:
                # Forest loss: tint toward red
                overlay_r, overlay_g, overlay_b = 255, 0, 0
                loss += 1

            elif not old_forest and new_forest:
                # Forest gain: tint toward green
                overlay_r, overlay_g, overlay_b = 0, 255, 0
                gain += 1

            else:
                # No change
                overlay_img.set_rgb(x, y, base_r, base_g, base_b)
                continue

            # Alpha-blend the overlay onto the base 2024 pixel
            r = int((1 - alpha) * base_r + alpha * overlay_r)
            g = int((1 - alpha) * base_g + alpha * overlay_g)
            b = int((1 - alpha) * base_b + alpha * overlay_b)

            overlay_img.set_rgb(x, y, r, g, b)

    total = w * h
    print(f"Forest-loss : {100*loss/total:.2f}%  ({loss} pixels)")
    print(f"Forest-gain : {100*gain/total:.2f}%  ({gain} pixels)")

    # Build a tall image with 3 panels:
    #  top: overlay on 2024
    #  mid: plain 2000
    #  bot: plain 2024
    combined = SimpleImage.blank(w, h * 3)

    for y in range(h):
        for x in range(w):
            # Top panel: overlay image
            r, g, b = overlay_img.get_color_tuple(x, y)
            combined.set_rgb(x, y, r, g, b)

            # Middle panel: 2024
            r2, g2, b2 = img_new.get_color_tuple(x, y)
            combined.set_rgb(x, y + h, r2, g2, b2)

            # Bottom panel: 2000
            r3, g3, b3 = img_old.get_color_tuple(x, y)
            combined.set_rgb(x, y + 2*h, r3, g3, b3)

    return combined


# ──────────────────────────────────────────────────────────────
# 3. Main script logic
# ──────────────────────────────────────────────────────────────
def main():
    args = sys.argv[1:]
    if len(args) != 2:
        print("Usage: python3 demo_forest_change.py <new_dir> <old_dir>")
        sys.exit(1)

    dir_new, dir_old = args

    # Load images using ghost helper
    imgs_old = ghost.load_images(dir_old)
    imgs_new = ghost.load_images(dir_new)

    img_old = ghost.solve(imgs_old, '-good')
    img_new = ghost.solve(imgs_new, '-good')

    # Generate change map
    combined_view = diff_image(img_new, img_old)
    combined_view.show()


if __name__ == "__main__":
    main()
