#!/usr/bin/env python3

"""
Stanford Bluescreen Code - Nick Parlante
Provides -front and -back strategies.
Only the key "if" line needs to be written
in the do_front() and do_back() functions.

To run -front:
  python3 bluescreen.py -front monkey-500.jpg moon-600.jpg
To run -back
  python3 bluescreen.py -back monkey-500.jpg moon-600.jpg 100 50

Please see the handout for an explanation of the
two strategies. For the front strategy, the back image
must be at least as large as the front image,
or there may be an out-of-bounds error.
"""

import sys
from simpleimage import SimpleImage


def do_front(front_filename, back_filename):
    """
    Front strategy: loop over front image,
    detect blue (or whatever) pixels there,
    overwrite blue pixels with pixels from back.
    Return changed front image.
    """
    image = SimpleImage(front_filename)
    back = SimpleImage(back_filename)
    for y in range(image.height):
        for x in range(image.width):
            pixel = image.get_pixel(x, y)
            average = (pixel.red + pixel.green + pixel.blue) / 3

            # Your code - replace False with expression
            # detect pixels in front image (e.g. blue) that should
            # be replaced by pixels from back image.
            if False:
                back_pixel = back.get_pixel(x, y)
                pixel.red = back_pixel.red
                pixel.green = back_pixel.green
                pixel.blue = back_pixel.blue
    return image


def do_back(front_filename, back_filename, shift_x, shift_y):
    """
    Back strategy: loop over image,
    detect *non-blue* pixels.
    Copy those pixels to back, shifted by shift_x, shift_y.
    Pixels which fall outside of the background are ignored.
    Return changed back image.
    """
    image = SimpleImage(front_filename)
    back = SimpleImage(back_filename)
    # Loop over front image - copy non-blue pixels
    # to backkground
    for y in range(image.height):
        for x in range(image.width):
            pixel = image.get_pixel(x, y)
            average = (pixel.red + pixel.green + pixel.blue) / 3

            # Your code - replace False with expression
            # detect pixels in front image (e.g. not-blue) that should
            # be copied over to back image.
            if False:
                dest_x = x + shift_x
                dest_y = y + shift_y
                # Only copy pixels to back if they will be in-bounds
                if back.in_bounds(dest_x, dest_y):
                    back_pixel = back.get_pixel(dest_x, dest_y)
                    back_pixel.red = pixel.red
                    back_pixel.green = pixel.green
                    back_pixel.blue = pixel.blue
    return back



def main():
    args = sys.argv[1:]

    # 1/2025: change args to take -front or -back flags at start
    # to spell out what we're doing

    if len(args) == 3 and args[0] == '-front':
        image = do_front(args[1], args[2])
        image.show()
        return

    if len(args) == 5 and args[0] == '-back':
        image = do_back(args[1], args[2], int(args[3]), int(args[4]))
        image.show()
        return

    # If we get here, did not recognize command
    print('Command line args not recognized. 2 ways to call:')
    print('-front frontimg.jpg backimg.jpg')
    print('-back frontimg.jpg backimg.jpg shift-x shift-y')


if __name__ == '__main__':
    main()
