/*
 * File: main.cpp
 * --------------
 * Blank C++ project configured to use Stanford cslib and Qt
 */

#include "console.h"
#include "map.h"
#include "set.h"
#include "vector.h"
#include "stack.h"
#include "queue.h"
#include "SimpleTest.h" // IWYU pragma: keep (needed to quiet spurious warning)
using namespace std;
Vector<string> wordWrap(string paragraph, int maxLineLength);
long numberOfJumpingSolutions(int numberOfPads);
long solutionsHelper(int numberOfPads, Map<int, long> &memo);

/* Uncomment the Big O test you want to run
 *  (they each take a fair amount of time)
 */
// #define TEST_BIGO_A
// #define TEST_BIGO_B
#define TEST_BIGO_C
// #define TEST_BIGO_D
// #define TEST_BIGO_E
// #define TEST_BIGO_F

int main()
{
    if (runSimpleTests(SELECTED_TESTS)) {
        return 0;
    }
    return 0;
}

// problem 1: word wrap
Vector<string> wordWrap(string paragraph, int maxLineLength) {
    Vector<string> lines;
    string line;
    string word;
    for (int i = 0; i < paragraph.length(); i++) {
        char ch = paragraph[i];
        word += ch;
        if (ch == ' ' || i == paragraph.length() - 1) {
            // we have a word or end of paragraph
            if (line.length() + word.length() <= maxLineLength) {
                line += word;
            } else {
                lines.add(line);
                line = word;
            }
            word = "";
        }
    }
    // don't forget the last line
    if (line != "") {
        lines.add(line);
    }
    return lines;
}

PROVIDED_TEST("Test paragraph #1") {
    string paragraph = "One fish. Two fish. Red fish, blue fish.";
    Vector<string> wrapped = {"One ",
                              "fish. ",
                              "Two ",
                              "fish. ",
                              "Red ",
                              "fish, ",
                              "blue ",
                              "fish."};
    // cout << wordWrap(paragraph, 6);
    EXPECT_EQUAL(wordWrap(paragraph, 6), wrapped);
}

PROVIDED_TEST("Test paragraph #2") {
    string paragraph = "One fish. Two fish. Red fish, blue fish.";
    Vector<string> wrapped = {"One fish. ",
                              "Two fish. ",
                              "Red fish, ",
                              "blue fish."};
    // cout << wordWrap(paragraph, 10);
    EXPECT_EQUAL(wordWrap(paragraph, 10), wrapped);
}

PROVIDED_TEST("Test paragraph #3") {
    string paragraph = "One fish. Two fishes. Red fish, green fish.";
    Vector<string> wrapped = {"One fish. ",
                              "Two ",
                              "fishes. ",
                              "Red fish, ",
                              "green ",
                              "fish."};
    cout << wordWrap(paragraph, 10);
    EXPECT_EQUAL(wordWrap(paragraph, 10), wrapped);
}

PROVIDED_TEST("Test paragraph #4, multiple lines need wrapping") {
    string paragraph = "One fish. Two fish. Red fish, green fish.";
    Vector<string> wrapped = {"One fish. Two ",
                              "fish. Red ",
                              "fish, green ",
                              "fish."};
    // cout << wordWrap(paragraph, 14);
    EXPECT_EQUAL(wordWrap(paragraph, 14), wrapped);
}

PROVIDED_TEST("Test paragraph #5, long line") {
    string paragraph = "One fish. Two fish. Red fish, blue fish.";
    Vector<string> wrapped = {"One fish. Two fish. Red fish, blue fish."};
    // cout << wordWrap(paragraph, 6);
    EXPECT_EQUAL(wordWrap(paragraph, 60), wrapped);
}

PROVIDED_TEST("Test paragraph #6") {
    string paragraph = "This is a paragraph. Here is the second sentence.";
    Vector<string> wrapped = {"This is a ",
                             "paragraph. ",
                             "Here is the ",
                             "second ",
                             "sentence."};
    // cout << wordWrap(paragraph, 15);
    EXPECT_EQUAL(wordWrap(paragraph, 15), wrapped);
}

PROVIDED_TEST("Test paragraph #7, Declaration of Independence") {
    string paragraph = "We hold these truths to be self-evident, that all men are created equal, that they are endowed by their Creator with certain unalienable Rights, that among these are Life, Liberty and the pursuit of Happiness.--That to secure these rights, Governments are instituted among Men, deriving their just powers from the consent of the governed, --That whenever any Form of Government becomes destructive of these ends, it is the Right of the People to alter or to abolish it, and to institute new Government, laying its foundation on such principles and organizing its powers in such form, as to them shall seem most likely to effect their Safety and Happiness. Prudence, indeed, will dictate that Governments long established should not be changed for light and transient causes; and accordingly all experience hath shewn, that mankind are more disposed to suffer, while evils are sufferable, than to right themselves by abolishing the forms to which they are accustomed. But when a long train of abuses and usurpations, pursuing invariably the same Object evinces a design to reduce them under absolute Despotism, it is their right, it is their duty, to throw off such Government, and to provide new Guards for their future security.--Such has been the patient sufferance of these Colonies; and such is now the necessity which constrains them to alter their former Systems of Government. The history of the present King of Great Britain is a history of repeated injuries and usurpations, all having in direct object the establishment of an absolute Tyranny over these States. To prove this, let Facts be submitted to a candid world.";
    Vector<string> wrapped = {"We hold these truths to be self-evident, that all men are ",
                              "created equal, that they are endowed by their Creator with ",
                              "certain unalienable Rights, that among these are Life, ",
                              "Liberty and the pursuit of Happiness.--That to secure these ",
                              "rights, Governments are instituted among Men, deriving ",
                              "their just powers from the consent of the governed, --That ",
                              "whenever any Form of Government becomes destructive of ",
                              "these ends, it is the Right of the People to alter or to ",
                              "abolish it, and to institute new Government, laying its ",
                              "foundation on such principles and organizing its powers in ",
                              "such form, as to them shall seem most likely to effect ",
                              "their Safety and Happiness. Prudence, indeed, will dictate ",
                              "that Governments long established should not be changed for ",
                              "light and transient causes; and accordingly all experience ",
                              "hath shewn, that mankind are more disposed to suffer, while ",
                              "evils are sufferable, than to right themselves by ",
                              "abolishing the forms to which they are accustomed. But when ",
                              "a long train of abuses and usurpations, pursuing invariably ",
                              "the same Object evinces a design to reduce them under ",
                              "absolute Despotism, it is their right, it is their duty, to ",
                              "throw off such Government, and to provide new Guards for ",
                              "their future security.--Such has been the patient ",
                              "sufferance of these Colonies; and such is now the necessity ",
                              "which constrains them to alter their former Systems of ",
                              "Government. The history of the present King of Great ",
                              "Britain is a history of repeated injuries and usurpations, ",
                              "all having in direct object the establishment of an ",
                              "absolute Tyranny over these States. To prove this, let ",
                              "Facts be submitted to a candid world."};
    // cout << wordWrap(paragraph, 60);
    EXPECT_EQUAL(wordWrap(paragraph, 60), wrapped);
}

PROVIDED_TEST("Test paragraph #8, empty string") {
    string paragraph = "";
    Vector<string> wrapped = {};
    // cout << wordWrap(paragraph, 5);
    EXPECT_EQUAL(wordWrap(paragraph, 5), wrapped);
}

PROVIDED_TEST("Test paragraph #9, 1-char string") {
    string paragraph = "a";
    Vector<string> wrapped = {"a"};
    // cout << wordWrap(paragraph, 1);
    EXPECT_EQUAL(wordWrap(paragraph, 1), wrapped);
}

PROVIDED_TEST("Test paragraph #10, 3-char string") {
    string paragraph = "a b";
    Vector<string> wrapped = {"a ", "b"};
    // cout << wordWrap(paragraph, 2);
    EXPECT_EQUAL(wordWrap(paragraph, 2), wrapped);
}

// problem 2: unionStackAndQueue
Set<int> unionStackAndQueue(Stack<int>&stack, Queue<int>&queue) {
    Set<int> setUnion;
    Stack<int> extraStack;
    // step 1: dequeue entire queue to set and back into queue
    int origQueueSize = queue.size();
    for (int i = 0; i < origQueueSize; i++) {
        int n = queue.dequeue();
        setUnion.add(n);
        queue.enqueue(n);
    }

    // step 2: pop all of stack and put in set and temp stack
    while (!stack.isEmpty()) {
        int n = stack.pop();
        setUnion.add(n);
        extraStack.push(n);
    }

    // step 3: push all of extraStack back to stack
    while (!extraStack.isEmpty()) {
        int n = extraStack.pop();
        stack.push(n);
    }

    return setUnion;
}

PROVIDED_TEST("Basic stack and queue") {
    Stack<int> stack = {1, 2, 3, 4};
    Queue<int> queue = {3, 4, 5, 6, 7};

    Stack<int>sCopy = stack;
    Queue<int>qCopy = queue;

    Set<int> soln = {1, 2, 3, 4, 5, 6, 7};

    Set<int>studentSoln = unionStackAndQueue(stack, queue);

    EXPECT_EQUAL(studentSoln, soln);
    EXPECT_EQUAL(stack, sCopy);
    EXPECT_EQUAL(queue, qCopy);
}

PROVIDED_TEST("Empty stack and queue") {
    Stack<int> stack = {};
    Queue<int> queue = {};

    Stack<int>sCopy = stack;
    Queue<int>qCopy = queue;

    Set<int> soln = {};

    Set<int>studentSoln = unionStackAndQueue(stack, queue);

    EXPECT_EQUAL(studentSoln, soln);
    EXPECT_EQUAL(stack, sCopy);
    EXPECT_EQUAL(queue, qCopy);
}

PROVIDED_TEST("Duplicates in stack and queue") {
    Stack<int> stack = {1, 2, 3, 2, 1, 3};
    Queue<int> queue = {2, 3, 2, 4, 2, 3};

    Stack<int>sCopy = stack;
    Queue<int>qCopy = queue;

    Set<int> soln = {1, 2, 3, 4};

    Set<int>studentSoln = unionStackAndQueue(stack, queue);

    EXPECT_EQUAL(studentSoln, soln);
    EXPECT_EQUAL(stack, sCopy);
    EXPECT_EQUAL(queue, qCopy);
}

PROVIDED_TEST("Empty stack, full queue") {
    Stack<int> stack = {};
    Queue<int> queue = {1, 2, 3, 2, 1, 3};

    Stack<int>sCopy = stack;
    Queue<int>qCopy = queue;

    Set<int> soln = {1, 2, 3};

    Set<int>studentSoln = unionStackAndQueue(stack, queue);

    EXPECT_EQUAL(studentSoln, soln);
    EXPECT_EQUAL(stack, sCopy);
    EXPECT_EQUAL(queue, qCopy);
}

PROVIDED_TEST("Full stack, empty queue") {
    Stack<int> stack = {1, 2, 3, 2, 1, 3};
    Queue<int> queue = {};

    Stack<int>sCopy = stack;
    Queue<int>qCopy = queue;

    Set<int> soln = {1, 2, 3};

    Set<int>studentSoln = unionStackAndQueue(stack, queue);

    EXPECT_EQUAL(studentSoln, soln);
    EXPECT_EQUAL(stack, sCopy);
    EXPECT_EQUAL(queue, qCopy);
}

// problem 3: Big O

void bigOa(int N) {
    int total = 0;
    for (int i = 0; i < N; i++) {
        for (int j = 0; j < i; j++) {
            total++;
        }
    }
    cout << total << endl;
}

#ifdef TEST_BIGO_A
PROVIDED_TEST("BigOa Runtime test") {
    // O(N^2)
    const int MIN_N = 1024;
    const int MAX_N = MIN_N << 8; // increase for longer test
    for (int n = MIN_N; n < MAX_N; n *= 2) {
        cout << "Testing for N==" << n << endl;
        TIME_OPERATION(n, bigOa(n));
    }
}
#endif

void bigOb(int N, Map<int, int> &map, Set<int> &set) {
    // Assume the map and set already contain N elements
    for (int i = 0; i < 1000; i++) {
        if (set.contains((N + i) / 3)) {
            map[(N + i) / 3] = (N - 1) * 2;
        } else {
            map[(N + i) / 3] = N * 2;
        }
    }
}

#ifdef TEST_BIGO_B
PROVIDED_TEST("BigOb Runtime test") {
    // Should be O(log n)
    /* Note: this is particularly hard to test,
     * because log(n) is so fast, even for very large
     * values of N. Although this test may look like
     * it is demonstrating constant time, that is likely
     * because the processor is too fast to test
     * correctly
     */
    const int MIN_N = 32768;
    const int MAX_N = MIN_N << 8; // increase for longer test
    Map<int, int> map;
    Set<int> set;
    for (int n = MIN_N; n < MAX_N; n *= 2) {
        for (int i = 0; i < n; i++) {
            set.add(i);
            map[i] = i;
        }
        cout << "Testing for N==" << n << endl;
        TIME_OPERATION(n, bigOb(n, map, set));
    }
}
#endif

void bigOc(int N) {
    Queue<int> queue;
    for (int i = 0; i <= N * 3; i++) {
        queue.enqueue(i);
    }
    Stack<int> stack;
    while (!queue.isEmpty()) {
        int num = queue.dequeue();
        stack.push(num);
    }
}

#ifdef TEST_BIGO_C
PROVIDED_TEST("BigOc Runtime test") {
    // O(N)
    const int MIN_N = 1024;
    const int MAX_N = MIN_N << 13; // increase for longer test
    for (int n = MIN_N; n < MAX_N; n *= 2) {
        cout << "Testing for N==" << n << endl;
        TIME_OPERATION(n, bigOc(n));
    }
}
#endif

void bigOd(int N) {
    Vector<char> vec;
    for (int i = 0; i < N; i++) {
        vec.add(i % 128);
    }

    for (int i = 0; i < N; i++) {
        vec.add(i);
        vec.remove(0);
    }
}

#ifdef TEST_BIGO_D
PROVIDED_TEST("BigOd Runtime test") {
    // O(N^2)
    const int MIN_N = 1024;
    const int MAX_N = MIN_N << 10; // increase for longer test
    for (int n = MIN_N; n < MAX_N; n *= 2) {
        cout << "Testing for N==" << n << endl;
        TIME_OPERATION(n, bigOd(n));
    }
}
#endif

int bigOe(int N) {
    if (N < 0) {
        return 1;
    }
    int sum = 0;
    for (int i = 1; i <= 10; i++) {
        sum += bigOe(N - i);
    }
    return sum;
}

#ifdef TEST_BIGO_E
PROVIDED_TEST("BigOe Runtime test") {
    // O(a^N)
    const int MIN_N = 16;
    const int MAX_N = 31; // increase for longer test
    for (int n = MIN_N; n < MAX_N; n++) {
        cout << "Testing for N==" << n << endl;
        TIME_OPERATION(n, bigOe(n));
    }
}
#endif

int bigOf(int N, int top) {
    // assume that in original call, top = 0
    if (N < top) {
        return top;
    }
    cout << top << endl;
    return top + bigOf(N, top + 1);
}

#ifdef TEST_BIGO_F
PROVIDED_TEST("BigOf Runtime test") {
    // O(N)
    const int MIN_N = 128;
    const int MAX_N = MIN_N << 5; // increase for longer test
    for (int n = MIN_N; n < MAX_N; n *= 2) {
        cout << "Testing for N==" << n << endl;
        TIME_OPERATION(n, bigOf(n, 0));
    }
}
#endif

// problem 4: procedural recursion
long numberOfJumpingSolutions(int numberOfPads) {
    Map<int, long>memo = {{0, 0}, {1, 1}, {2, 2}, {3, 4}};
    return solutionsHelper(numberOfPads, memo);
}

long solutionsHelper(int numberOfPads, Map<int, long> &memo) {
    if (memo.containsKey(numberOfPads)) {
        return memo[numberOfPads];
    }
    long currentSolution = solutionsHelper(numberOfPads - 1, memo) +
                           solutionsHelper(numberOfPads - 2, memo) +
                           solutionsHelper(numberOfPads - 3, memo);
    memo[numberOfPads] = currentSolution;
    return currentSolution;
}

PROVIDED_TEST("Test lily pad jumps with 1 lily pad") {
    EXPECT_EQUAL(numberOfJumpingSolutions(1), 1);
}

PROVIDED_TEST("Test lily pad jumps with 2 lily pads") {
    EXPECT_EQUAL(numberOfJumpingSolutions(2), 2);
}

PROVIDED_TEST("Test lily pad jumps with 3 lily pads") {
    EXPECT_EQUAL(numberOfJumpingSolutions(3), 4);
}

PROVIDED_TEST("Test lily pad jumps with 4 lily pads") {
    EXPECT_EQUAL(numberOfJumpingSolutions(4), 7);
}

PROVIDED_TEST("Test lily pad jumps with 100 lily pads") {
    EXPECT_EQUAL(numberOfJumpingSolutions(100), 7367864567128947527);
}

PROVIDED_TEST("Test lily pad for sequence up to 11") {
    Vector<long> results = {0, 1, 2, 4, 7, 13, 24, 44, 81, 149, 274 };
    for (int i = 0; i < 11; i++) {
        EXPECT_EQUAL(numberOfJumpingSolutions(i), results[i]);
    }
}


