/*
 * File: treeTraversal.cpp
 * -------------
 */

#include <iostream>
#include "console.h"
#include "simpio.h"
#include "queue.h"
using namespace std;

struct TreeNode {
    string data;
    TreeNode *left;
    TreeNode *right;
};

int treeHeight(TreeNode* tree);
void prettyPrint(TreeNode* tree);
void printChars(int n, char c);
int longestData(TreeNode* tree);
string padString(string s, int len);

TreeNode* makeTree(string data) {
    TreeNode* t = new TreeNode;
    t->data = data;
    t->left = nullptr;
    t->right = nullptr;
    return t;
}

// Makes a sentence tree
TreeNode* initSentenceTree() {
    TreeNode* root = new TreeNode;
    root->data = "this";

    root->left = makeTree("is");
    root->left->left = makeTree("a");
    root->left->right = makeTree("correctly");

    root->right = makeTree("written");
    root->right->right = makeTree("sentence.");

    return root;
}

TreeNode* initNumberTree() {
    TreeNode* root = new TreeNode;
    root->data = "4";

    root->left = makeTree("2");
    root->left->left = makeTree("1");
    root->left->right = makeTree("3");

    root->right = makeTree("5");
    root->right->right = makeTree("6");

    return root;
}

void preOrder(TreeNode* tree) {
    if (tree != nullptr) {
        cout << tree->data << " ";
        preOrder(tree->left);
        preOrder(tree->right);
    }
}

void inOrder(TreeNode* tree) {
    if (tree != nullptr) {
        inOrder(tree->left);
        cout << tree->data << " ";
        inOrder(tree->right);
    }
}

void postOrder(TreeNode* tree) {
    if (tree != nullptr) {
        postOrder(tree->left);
        postOrder(tree->right);
        cout << tree->data << " ";
    }
}

void levelOrder(TreeNode*tree) {
    Queue<TreeNode*> queue;
    queue.enqueue(tree);
    while (!queue.isEmpty()) {
        TreeNode* element = queue.dequeue();
        if (element != nullptr) {
            cout << element->data << " ";
            queue.enqueue(element->left);
            queue.enqueue(element->right);
        }
    }
}

int main() {
    setConsoleFont("courier-32");

    int choice = -1;

    TreeNode *tree;

    cout << "1) Sentence Tree" << endl;
    cout << "2) Number Tree" << endl;
    choice = getInteger("Please choose 1-2: ");
    if (choice == 1) {
        tree = initSentenceTree();
    } else {
        tree = initNumberTree();
    }

    prettyPrint(tree);
    cout << endl;

    cout << "1) Pre-order" << endl;
    cout << "2) In-order" << endl;
    cout << "3) Post-order (Yoda?)" << endl;
    cout << "4) Level-order" << endl << endl;
    choice = -1;
    while (choice != 0) {
        choice = getInteger("Please choose 1-4 (0 to quit): ");
        switch (choice) {
        case 1: preOrder(tree); break;
        case 2: inOrder(tree); break;
        case 3: postOrder(tree); break;
        case 4: levelOrder(tree); break;
        }
        cout << endl << endl;
    }
    cout << "Goodbye!" << endl;
    return 0;
}

int treeHeight(TreeNode* tree) {
    if (tree == nullptr) {
        return 0;
    }
    int leftHeight = treeHeight(tree->left);
    int rightHeight = treeHeight(tree->right);
    if (leftHeight > rightHeight) {
        return leftHeight + 1;
    } else {
        return rightHeight + 1;
    }
}

int longestData(TreeNode* tree) {
    if (tree == nullptr) return 0;
    int longest = max(longestData(tree->left),
                      longestData(tree->right));
    return (max((int)tree->data.length(), longest));
}

void prettyPrint(TreeNode* tree)
{
    // calculate levels
    int numLevels = treeHeight(tree);
    int maxLength = longestData(tree);

    // go through each level and mark locations of numbers
    Queue<TreeNode*> q;
    q.enqueue(tree);
    int level = 0;
    int numberCount = 1 << level;
    int nextNumberOfCount = 0;
    int spacesSoFar = 0;
    while (!q.isEmpty()) {
        TreeNode* curr = q.dequeue();
        //             xxx
        //     xxx             xxx
        // xxx     xxx     xxx     xxx
        int first = (1 << (numLevels - level - 1)) - 1;
        int elementMult = 1 << (numLevels - level);
        int nextPos = (first + (nextNumberOfCount * elementMult)) * maxLength;

        printChars(nextPos - spacesSoFar, ' ');
        spacesSoFar = nextPos;
        if (curr != nullptr) {
            cout << padString(curr->data, maxLength);
        } else {
            printChars(maxLength, '-');
        }
        spacesSoFar += maxLength;
        if (curr != nullptr) {
            q.enqueue(curr->left);
            q.enqueue(curr->right);
        } else {
            q.enqueue(nullptr);
            q.enqueue(nullptr);
        }
        nextNumberOfCount++;
        if (nextNumberOfCount == numberCount) {
            nextNumberOfCount = 0;
            level++;
            numberCount = 1 << level;
            spacesSoFar = 0;
            if (level == numLevels) break; // don't print last level, which will be empty
            cout << endl;
        }
    }
    cout << endl;
}

void printChars(int n, char c)
{
    for (int i = 0; i < n; i++) {
        cout << c;
    }
}

string padString(string s, int len) {
    // make the string len long
    int sLen = s.length();
    if ((len - sLen) % 2 == 1) {
        s += " ";
        sLen++;
    }
    for (int i = 0; i < (len - sLen) / 2; i++) {
        s = string(" ") + s + " ";
    }
    return s;
}
