/*
 * calculator.cpp
 *
 * @author Cynthia Bailey
 * @version 2026-Win
 *
 * Demonstrates use of Stack ADT to implement a very simple Reverse Polish
 * Notation (RPN) calculator.
 *
 */
#include <iostream>   // for cout
#include "console.h"  // for Stanford libs
#include "simpio.h"   // for getting input from console
#include "stack.h"    // for parsing RPN notation
#include "strlib.h"   // for converting digit character to int
#include "SimpleTest.h" // for tests
using namespace std;

void printWelcome();
int  calculate(string expression);
bool isSupportedOperator(char op);
int  applyOperator(int lhs, char op, int rhs);

/*
 * This program asks user for an input file, in which each line is an
 * arithmetic expression in RPN format, and performs the calculations. After
 * each calculation, the user is asked if they would like to continue.
 */
int main()
{
    if (runSimpleTests(SELECTED_TESTS)) {
        return 0;
    }

    printWelcome();
    bool calcAgain = true;
    while (calcAgain) {
        string expression = getLine("Enter an expression: ");
        cout << "Answer = " << calculate(expression) << endl;
        calcAgain = getYesOrNo("Do you want another calculation (Y/N)? ");
    }
    return 0;
}

/*
 * Prints out an initial greeting message for the program.
 */
void printWelcome()
{
    cout << "Welcome to the CS106B RPN Calculator!" << endl;
    cout << endl;
    cout << "This program takes arithmetic expressions in RPN " << endl;
    cout << "format, and performs the calculations." << endl;
    cout << endl;
}

/*
 * Demonstrates use of a Stack to calculate an RPN arithmetic expression.
 * @param expression is an arithmetic expression in RPN form
 * @param result "returns" the result of the calculation
 * @return true if expression could be parsed and result was set
 *        successfully, else false.
 */
int calculate(string expression)
{
    Stack<int> memory;

    // Examine each character of input, left to right
    for (char c : expression) {
        // if digit, store it
        if (isdigit(c)) {
            memory.push(charToInteger(c));
        }

        // if operator, perform operation
        else if (isSupportedOperator(c) && memory.size() >= 2) {
            int op2 = memory.pop(); // op2 is the newer (right-hand side) one, so first pop
            int op1 = memory.pop(); // op1 is the older (left-hand side) one, so second pop
            memory.push(applyOperator(op1, c, op2));
        }

        // expression has syntax error
        else {
            error("Bad syntax in your expression.");
        }
    }

    // should be single number in memory, that's our answer
    if (memory.size() != 1) {
        error("Cannot parse expression.");
    }
    return memory.pop();
}

/*
 * Performs a check to make sure we support the arithmetic operator.
 * @param op is a character representation of the operator
 * @return true if op is one of * / + -, otherwise false
 */
bool isSupportedOperator(char op) {
    return op == '*' || op == '/' || op == '+' || op == '-';
}

/*
 * Performs a single arithmetic operator computation.
 * @pre op must be a valid operator (else throws exception)
 * @param lhs is the first or left-hand-side operand
 * @param op is a character representation of the operator
 * @param rhs is the second or right-hand-side operand
 * @return the result of the calculation
 */
int applyOperator(int lhs, char op, int rhs)
{
    switch (op) {
    case '*' :
        return lhs * rhs;
    case '/':
        return lhs / rhs;
    case '+':
        return lhs + rhs;
    case '-':
        return lhs - rhs;
    default:
        error("Invalid operator"); // we should have already checked for this
    }
}


/* * * * * * Test Cases * * * * * */

PROVIDED_TEST("applyOperator") {
    EXPECT_EQUAL(applyOperator(5, '+', 1), 6);
    EXPECT_EQUAL(applyOperator(5, '-', 1), 4);
    EXPECT_EQUAL(applyOperator(5, '/', 2), 2);
    EXPECT_EQUAL(applyOperator(5, '*', 2), 10);
}

PROVIDED_TEST("isSupportedOperator") {
    EXPECT(isSupportedOperator('+'));
    EXPECT(isSupportedOperator('-'));
    EXPECT(isSupportedOperator('/'));
    EXPECT(isSupportedOperator('*'));
    EXPECT(!isSupportedOperator('^'));
    EXPECT(!isSupportedOperator('('));
    EXPECT(!isSupportedOperator('5'));
}
4

PROVIDED_TEST("calculate 11+") {
    EXPECT_EQUAL(calculate("11+"), 2);
}

PROVIDED_TEST("calculate 99+7*73+/") {
    EXPECT_EQUAL(calculate("99+7*73+/"), 12);
}

PROVIDED_TEST("calculate 11++ (should fail to parse)") {
    EXPECT_ERROR(calculate("11++"));
}

STUDENT_TEST("test additional cases not covered by the provided tests") {
}
