/*
 * wordcount.cpp
 *
 * @author Cynthia Bailey
 * @version Winter 2026
 *
 * A program to demonstrate the use of Stanford library Map
 * to count the number of times each word occurs in a book
 * and make reports of the counts.
 */

#include <fstream>
#include <iostream>
#include "console.h"
#include "filelib.h"
#include "map.h"
#include "simpio.h"
#include "SimpleTest.h"
using namespace std;

void printCommonWords();
Vector<string> readWordsFromBook();
void countUniqueWords(Vector<string>& words, Map<string, int>& wordCounts);
int getOccurrencesCountForWord(Map<string, int>& wordCounts, string word);

static const int    FREQUENCY_THRESHOLD = 100;
static const string BOOK_TITLE          = "books/hurston.txt";


int main()
{
    if (runSimpleTests(SELECTED_TESTS)) {
        return 0;
    }

    printCommonWords();

    return 0;
}

/*
 * Takes vector containing full text (tokenized), and populates a map
 * of words -> word counts.
 * @param words is a Vector of tokens (words) to be counted.
 * @param wordCounts should be initially empty, and will be filled with
 *   associations of words with the count of that word's occurrences in
 *   the file.
 */
void countUniqueWords(Vector<string>& words, Map<string, int>& wordCounts)
{
    for (string word : words) {
        wordCounts[word]++;
    }
}

/*
 * Read a book. Print all words that appeared in the book
 * at least FREQUENCY_THRESHOLD times, in alphabetical order.
 */
void printCommonWords()
{
    Vector<string> allWords = readWordsFromBook();
    Map<string, int> wordCounts;
    countUniqueWords(allWords, wordCounts);
    for (string word : wordCounts) {
        if (wordCounts[word] >= FREQUENCY_THRESHOLD) {
            cout << word << " appeared " << wordCounts[word] << " times." << endl;
        }
    }
}

/*
 * Give us a word and we report how many times that word
 *   appeared in the book.
 * @param wordCounts associates words with the count of that word's
 *   occurrences in the file.
 * @param word is a word we want to query
 */
int getOccurrencesCountForWord(Map<string, int>& wordCounts, string word)
{
    if (wordCounts.containsKey(word)) {
        return wordCounts[word];
    }
    return 0;
}


/*
 * Read contents of filename, tokenized by whitespace, and store
 * each token as an element of the provided Vector.
 * @param words: the Vector to add all the file's words to.
 */
Vector<string> readWordsFromBook()
{
    Vector<string> words;
    ifstream infile;
    infile.open(BOOK_TITLE);
    string word;
    while (infile >> word) {
        words.add(word);
    }
    infile.close();

    return words;
}


/* * * * * * Test Cases * * * * * */

PROVIDED_TEST("getOccurrencesCountForWord")
{
    Vector<string> allWords = readWordsFromBook();
    Map<string, int> wordCounts;
    countUniqueWords(allWords, wordCounts);

    EXPECT_EQUAL(getOccurrencesCountForWord(wordCounts, "the"), 1636);
    EXPECT_EQUAL(getOccurrencesCountForWord(wordCounts, "her"), 656);
    EXPECT_EQUAL(getOccurrencesCountForWord(wordCounts, "rhinocerous"), 0);
}


STUDENT_TEST("test additional cases not covered by the provided tests")
{
    Vector<string> allWords = readWordsFromBook();
    Map<string, int> wordCounts;
    countUniqueWords(allWords, wordCounts);
    int nWords = wordCounts.size();
    string s = "aaa";
    for (int i = 0; i < 100; i++) {
        // count should be zero
        EXPECT_EQUAL(getOccurrencesCountForWord(wordCounts, s), 0);
        // querying these additional words should not add them to the structure
        EXPECT_EQUAL(wordCounts.size(), nWords);
        s += "a";
    }
}
