/*
 * target.cpp
 *
 * @author Cynthia Bailey
 * @version Winter 2026
 *
 * Demonstrates backtracking recursion to generate all possible
 * subsets. This program considers a list of items available for purchase,
 * and attempts to see if some subset of them will sum to exactly the
 * amount that is available on a gift card.
 */

#include <iostream>
#include "console.h"
#include "strlib.h"
#include "set.h"
#include "SimpleTest.h"
using namespace std;

class Item
{
public:
    Item(string n, int p)
    {
        name = n;
        price = p;
    }
    friend bool operator==(const Item& l, const Item& r)
    {
        return l.name == r.name && l.price == r.price;
    }
    string name;
    int price;
};

std::ostream& operator<<(std::ostream& os, const Item& item)
{
    os << item.name << " " << item.price;
    return os;
}


/* Function Prototypes */
bool canUseFullGiftCard(int giftCardAmt, Vector<Item>& itemsForSale, Vector<Item>& itemsToBuy);
bool canUseFullGiftCard(int giftCardAmt, Vector<Item>& itemsForSale, Vector<Item>& itemsToBuy, int index);


int main()
{
    if (runSimpleTests(SELECTED_TESTS)) {
        return 0;
    }
    cout << "All done, exiting" << endl;
    return 0;
}

/* canUseFullGiftCard()
 * (wrapper function)
 * The function seeks to find a subset of items to buy whose prices
 * will sum to EXACTLY the amount of funds available on the gift card.
 * @param giftCardAmt is the amount of funds available to purchase items
 * @param itemsForSale items that are available for purchase (a Vector not a
 *        Set, because some items could have more than one copy in stock, so
 *        they will appear more than once)
 * @param itemsToBuy is an initially-empty set that--if the function returns
 *        true--will contain the list of items that should be purchased
 * @returns true if a successful set of items was found, else false
 */
bool canUseFullGiftCard(int giftCardAmt, Vector<Item>& itemsForSale, Vector<Item>& itemsToBuy)
{
    return canUseFullGiftCard(giftCardAmt, itemsForSale, itemsToBuy, 0);
}

/* canUseFullGiftCard()
 * (recursive helper function)
 * See comment for wrapper function. This helper uses a similar pattern as
 * binary search, which took start/end indices to mark the region effectively in
 * use at this time. Our `index` parameter marks the place in the vector where
 * we should start considering elements for purchase.
 */
bool canUseFullGiftCard(int giftCardAmt, Vector<Item>& itemsForSale, Vector<Item>& itemsToBuy, int index)
{
    // base case success: card amount is spent down to 0 exactly
    if (giftCardAmt == 0) {
        return true;
    }

    // base case failure: we either overspent, or we need to spend more but there are
    //                    no more items for to consider, so we can't succeed
    if (giftCardAmt < 0 || index == itemsForSale.size()) {
        return false;
    }

    // recursive case: consider 1 next item (at `index`)
    Item item = itemsForSale[index];

    // ...or NOT BUY THE ITEM and go on to conider other additional purchases with
    //    the same amount to spend.
    if (canUseFullGiftCard(giftCardAmt, itemsForSale, itemsToBuy, index + 1)) {
        return true;
    }

    // Our two choices are that we can either BUY THE ITEM and go on to consider
    //   other additional purchases with less money to spend...
    itemsToBuy.add(item);
    if (canUseFullGiftCard(giftCardAmt - item.price, itemsForSale, itemsToBuy, index + 1)) {
        return true;
    }
    itemsToBuy.remove(itemsToBuy.size() - 1);

    return false; // if neither of the two options can work, we have exhausted all options
}




/* * * * * * Test Cases * * * * * */

PROVIDED_TEST("Simple successful example")
{
    Vector<Item> itemsForSale = {Item("soap", 2), Item("socks", 3), Item("shoes", 4), Item("shirt", 6)};
    Vector<Item> itemsToBuy;
    EXPECT(canUseFullGiftCard(7, itemsForSale, itemsToBuy));
    EXPECT_EQUAL(itemsToBuy, {Item("socks", 3), Item("shoes", 4)});
}

PROVIDED_TEST("Simple fail example")
{
    Vector<Item> itemsForSale = {Item("soap", 2), Item("socks", 3), Item("shoes", 4), Item("shirt", 6)};
    Vector<Item> itemsToBuy;
    EXPECT(!canUseFullGiftCard(100, itemsForSale, itemsToBuy));
}

PROVIDED_TEST("Edge case: no items for sale, but also $0 gift card = success")
{
    Vector<Item> itemsForSale = {};
    Vector<Item> itemsToBuy;
    EXPECT(canUseFullGiftCard(0, itemsForSale, itemsToBuy));
    EXPECT_EQUAL(itemsToBuy, {});
}
