/**
 * File: visuals.cpp
 * -----------------
 * @author Julie Zelenski, Fall 2021, CS106M
 *
 * Don't look too closely, you might go blind :-)
 */

#include "gbutton.h"
#include "gcolor.h"
#include "gevent.h"
#include "gobjects.h"
#include "random.h"
#include "strlib.h"
#include "pagerank.h"
#include "testing/SimpleTest.h"
#include "testing/Graphics.h"
using namespace std;

struct GraphicalNode {
    GPoint center;
    int rgb;
};

static GCanvas *pieCanvas, *networkCanvas;

int sum(const Map<string, int>& counts) {
    int total = 0;
    for (auto& k: counts) total += counts[k];
    return total;
}

int RGBForHue(int huedegree, double v) {
    // https://en.wikipedia.org/wiki/HSL_and_HSV
    double s = 1;
    double h = huedegree/360.0;
    int i = h * 6;
    double f = h * 6 - i;
    double p = v * (1 - s);
    double q = v * (1 - f * s);
    double t = v * (1 - (1 - f) * s);
    double r,g,b;

    switch (i % 6) { // which sextant
      case 0: r = v, g = t, b = p; break;
      case 1: r = q, g = v, b = p; break;
      case 2: r = p, g = v, b = t; break;
      case 3: r = p, g = q, b = v; break;
      case 4: r = t, g = p, b = v; break;
      case 5: r = v, g = p, b = q; break;
    }
    return GColor::convertRGBToRGB(r * 255, g * 255, b * 255);
}

GPoint midpoint(GPoint start, GPoint end, double fraction) {
   return {start.x + (end.x-start.x)*fraction, start.y + (end.y-start.y)*fraction};
}

void drawArrow(GCanvas *gc, GPoint start, GPoint end, double fraction) {
    double theta = vectorAngle(start.x - end.x, start.y - end.y);
    GPoint mid = midpoint(start, end, fraction);
    double delta = 24, r = 12;
    double angleL = theta + delta, angleR = theta - delta;
    GPoint arrowL = {mid.x + r * cosDegrees(angleL), mid.y - r * sinDegrees(angleL)};
    GPoint arrowR = {mid.x + r * cosDegrees(angleR), mid.y - r * sinDegrees(angleR)};
    gc->fillPolygon( {mid, arrowL, arrowR } );
}

void drawLineWithArrow(GCanvas *gc, GPoint start, GPoint end, bool twoway, double fraction = 0.6) {
    gc->drawLine(start, end);
    if (fraction != 0)  drawArrow(gc, start, end, fraction);
    if (twoway) drawArrow(gc, end, start, fraction);
}

void drawLink(GCanvas *gc, GPoint start, GPoint end, bool twoway, bool isActive = false) {
    if (isActive) {
        gc->setLineWidth(2);
        gc->setColor("Dark Gray");
        drawLineWithArrow(gc, start, end, false, 0);
    } else {
        gc->setColor("Light Gray");
        gc->setLineWidth(0.5);
        gc->setLineStyle(GObject::LINE_DASH);
        drawLineWithArrow(gc, start, end, twoway);
        gc->setLineStyle(GObject::LINE_SOLID);
    }
}

void drawLinks(GCanvas *gc, const Map<string, Set<string>>& links, const Map<string, GraphicalNode>& nodes, string surfer = "") {
    for (const auto& src: links) {
        for (const auto& dst: links[src]) {
            bool both = links[dst].contains(src); // if dst links back to src
            bool isActive = (src == surfer || (both && dst == surfer));
            if (!both || src < dst) { // don't draw it twice, only draw if src < dst
                if (isActive) {
                    auto other = surfer == src ? dst : src;
                    drawLink(gc, nodes[surfer].center, nodes[other].center, both, isActive);
                } else {
                    drawLink(gc, nodes[src].center, nodes[dst].center, both, isActive);

                }
            }
        }
    }
}

void fillWedge(GCanvas *gc, GPoint origin, double size, double start, double sweep) {
    GPoint ctr = {origin.x + size/2, origin.y + size/2}; // triangle of wedge
    GPoint corner1 = gc->drawPolarLine(ctr, size/2, start);
    GPoint corner2 = gc->drawPolarLine(ctr, size/2, start + sweep);
    gc->fillPolygon( {ctr, corner1, corner2 } );
    gc->fillArc(origin.x, origin.y, size, size, start, sweep);
}

void displayPieChart(const Map<string, Set<string>>& network, const Map<string, int>& counts) {
    double chartSize = 0.5 * min(pieCanvas->getWidth(), pieCanvas->getHeight());
    GPoint chartOrigin = {10, 10};
    Map<string,int> colors;

    double hue = 0, jump = 360/network.size();
    for (const auto&key: network) {
        colors[key] = RGBForHue(hue);
        hue += jump;
    }

    pieCanvas->clear();
    Vector<string> wedges = network.keys();
    // arrange in decreasing order, biggest wedge first
    sort(wedges.begin(), wedges.end(), [counts](string a, string b) {return counts[a] > counts[b];});
    int total = sum(counts);

    double angle = 0;
    for (const auto& w : wedges) {
        if (counts[w] > 0 && total) {
            double sweep = (360.0*counts[w])/total; // in degrees
            pieCanvas->setColor(colors[w]);
            fillWedge(pieCanvas, chartOrigin, chartSize, angle, sweep);
            angle += sweep;
        }
    }
    pieCanvas->setColor("black");
    pieCanvas->drawOval(chartOrigin.x, chartOrigin.y, chartSize, chartSize);
    pieCanvas->repaint();
}

void displayNode(GCanvas *gc, GraphicalNode p, int percent, bool highlight) {
    double size = 10;
    string label = "";
    int fontPx = 10;

    if (percent > 0) {
        size = min(100, max(20, percent*3));
        label = integerToString(percent) + "%";
        fontPx = percent < 10 ? 10 : 2 + size/3;
    }
    gc->setLineWidth(3);
    gc->setColor(highlight?  0: p.rgb); // border color matches fill = no visible stroke
    gc->setFillColor(p.rgb);
    gc->fillRect(p.center.x - size/2, p.center.y - size/2, size, size);
    if (!label.empty()) {
        gc->setColor("Black");
        gc->setFont("Helvetica-Bold-" + integerToString(fontPx));
        gc->drawString(label, p.center.x - fontPx + 1, p.center.y + fontPx);
        if (highlight) gc->drawString("😎", p.center.x - fontPx/2, p.center.y);
    }
}

void displayNetwork(const Map<string, Set<string>>& network, const Map<string, int>& counts, string surferLocation) {
    double canvasSize = min(networkCanvas->getWidth(), networkCanvas->getHeight());
    GPoint center = {canvasSize/2, canvasSize/2};
    double radius = canvasSize*.4;
    double angle = 0, sweep = 360/network.size();
    Map<string,GraphicalNode> nodes;

    for (const auto& key: network) {
        double x = center.x + radius * cosDegrees(angle);
        double y = center.y - radius * sinDegrees(angle);
        nodes[key].center = {x, y};
        nodes[key].rgb = RGBForHue(angle);
        angle += sweep;
    }
    networkCanvas->clear();
    drawLinks(networkCanvas, network, nodes, surferLocation);
    int total = sum(counts);
    for (const auto& key: network) {
        int percent = total == 0 ? 0 : (100*counts[key])/total;
        displayNode(networkCanvas, nodes[key], percent, surferLocation == key);
    }
    networkCanvas->repaint();
}

void displayState(const Map<string, Set<string>>& network, const Map<string, int>& counts, string surferLocation) {
    displayNetwork(network, counts, surferLocation);
    displayPieChart(network, counts);
}


static bool freeRunning = false, userStopped = false;
static int stepCount = 0;

void configGraphics(GWindow *win) {
    GDimension pieSize = {250, 250};
    GDimension networkSize = {win->getWidth() - pieSize.width, win->getHeight()};
    pieCanvas = new GCanvas(pieSize.width, pieSize.height);
    pieCanvas->setAutoRepaint(false);   // control refresh manually
    networkCanvas = new GCanvas(networkSize.width, networkSize.height);
    networkCanvas->setAutoRepaint(false); // control refresh manually
    win->addToRegion(networkCanvas, GWindow::REGION_EAST);
    win->addToRegion(pieCanvas, GWindow::REGION_WEST);

    freeRunning = userStopped = false;
    GButton *stepButton = new GButton("Step");
    stepButton->setActionListener([]() { stepCount++; });
    GButton *runButton = new GButton("Run");
    runButton->setActionListener([stepButton, runButton]()
        { freeRunning = true; stepButton->setEnabled(false); runButton->setEnabled(false);});
    GButton *stopButton = new GButton("Stop");
    stopButton->setActionListener([stepButton, runButton, stopButton]()
        { userStopped = true; stepButton->setEnabled(false); runButton->setEnabled(false); stopButton->setEnabled(false);});
    win->addToRegion(stepButton, GWindow::REGION_SOUTH);
    win->addToRegion(runButton, GWindow::REGION_SOUTH);
    win->addToRegion(stopButton, GWindow::REGION_SOUTH);
}

bool userContinues() {
    int before = stepCount;
    while (!freeRunning && stepCount == before && !userStopped) pause(1);
    return !userStopped;
}


/* * * * * * Test Cases * * * * * */

PROVIDED_TEST("Colors")
{
    SimpleTestGraphics g("HSV Color",100, 360, false);
    GWindow *win = g.getWindow();
    win->setAutoRepaint(false);

    for (int hue = 0; hue < 360; hue ++) { // hue by rows
        for (int percent = 0; percent < 100; percent++) {
            int rgb = RGBForHue(hue, percent/100.0);
            win->drawPixel(percent, hue, rgb);
        }
       if (hue % 10 == 0) win->repaint();
    }
}

PROVIDED_TEST("Arrows")
{
    double size = 500;
    SimpleTestGraphics g("Arrows", size, size, false);
    GWindow *win = g.getWindow();

    for (int i = 0; i < 100; i ++) {
        GPoint start = {randomReal(0, size), randomReal(0, size)};
        GPoint end = {randomReal(0, size), randomReal(0, size)};
        drawLink(win->getCanvas(), start, end, randomChance(0.5));
    }
}

