/**************************************************
 * QueenSafety Example
 *
 * (c) Jerry Cain, with updates by Cynthia Lee
 *
 * This program randomly places queens on a chess
 * board, and then checks each space on the board
 * to see if it would be safe for an opponent's
 * piece to be in that space.
 *
 * This example was selected to demonstrate many
 * of the key C++ and Stanford C++ library features
 * you will need to implement Assignment 1, The
 * Game of Life. In particular, you'll need a Grid
 * object that you should plan to pass by reference
 * at all times (and by const reference in cases
 * where you won't need to modify it). You'll also
 * need to handle mouse events and do some console
 * I/O. Happy coding!
 **************************************************/

/* *************** INCLUDES *************** */

#include <iostream>
#include <string>
using namespace std;

#include "grid.h"           // for the board state (model)
#include "random.h"         // for randomInteger() to randomly place queens
#include "simpio.h"         // for getLine() to pause play until ENTER
#include "gwindow.h"        // for the window (view)
#include "gobjects.h"       // for window graphics components (view)

/* *************** CONSTANTS *************** */

static const int    NUM_QUEENS	= 5;
static const int    BOARD_DIMENSION = 8;
static const int    NUM_ROWS = BOARD_DIMENSION;
static const int    NUM_COLS = BOARD_DIMENSION;
static const string LINE_COLOR = "Blue";
static const string QUEEN_MARK = "Q";
static const string QUEEN_COLOR = "Black";
static const string MARK_FONT = "Helvetica-Bold-24";
static const double SQUARE_SIZE = 32;
static const double MILLISECOND_DELAY = 250;
static const string SAFE_MARK = "S";
static const string SAFE_COLOR = "Green";
static const string DANGER_MARK = "X";
static const string DANGER_COLOR = "Red";


/* *************** FUNCTION DECLARATIONS *************** */

static void drawSquare(GWindow& window, int row, int col, string color);
static void clearBoard(Grid<bool>& board, GWindow& window);
static void markLocation(GWindow& window, string label, int row, int col, string color);
static void placeRandomQueens(Grid<bool>& board, GWindow& window);
static bool isDirectionSafe(const Grid<bool>& board, int row, int col, int drow, int dcol);
static void identifySafeLocations(const Grid<bool>& board, GWindow& window);
static bool isSafe(const Grid<bool>& board, int row, int col);


/* *************** FUNCTION DEFINITIONS *************** */


/* main()
 *
 * Main manages the overall sequence of events for the program.
 * The Grid<bool> board is created here--is TRUE in spaces where
 * a queen is present and FALSE in spaces that are empty. There
 * are no other pieces, nor is the safety/danger of a given space
 * represented in the Grid. Instead, we will display the safety/
 * danger of a given space in the view of the board in the GWindow
 * object.
 */
int main() {
    Grid<bool> board(NUM_ROWS, NUM_COLS);
    GWindow window;
    window.setWindowTitle("Queen Safety");
    window.setVisible(true);
    clearBoard(board, window);
    placeRandomQueens(board, window);
    cout << "Press ENTER to see which locations are safe." << endl;
    getLine();
    identifySafeLocations(board, window);
    return 0;
}

/* clearBoard()
 *
 * This function clears both the model (the Grid<bool>) and the view (the
 * GWindow) of the chess board.
 *
 * board: the chess board state
 * window: the view where we should draw the board grid lines
 */
static void clearBoard(Grid<bool>& board, GWindow& window) {
    for (int row = 0; row < board.numRows(); row++) {
        for (int col = 0; col < board.numCols(); col++) {
            board[row][col] = false;
            drawSquare(window, row, col, "Blue");
        }
    }
}

/* placeRandomQueens()
 *
 * This function places NUM_QUEENS queens on the board in randomly selected
 * locations.
 *
 * window: the view of the board where we will draw the queens
 * board: the board state where we will record the presence of the queens
 */
static void placeRandomQueens(Grid<bool>& board, GWindow& window) {
    int numplaced = 0;
    while (numplaced < NUM_QUEENS) {
        int row = randomInteger(0,board.numRows() - 1);
        int col = randomInteger(0,board.numCols() - 1);
        if (!board[row][col]) {
            // update model
            board[row][col] = true;
            // update view
            markLocation(window, QUEEN_MARK, row, col, QUEEN_COLOR);
            numplaced++;
        }
    }
}

/* identifySafeLocations()
 *
 * This marks all empty spaces on the board as either safe for placement
 * of a piece, or unsafe for placement of a piece (relative to the
 * locations of the queens, which are all of the opposing player).
 *
 * board: the current board state (locations of queens)
 * window: the view of the board state (where we will draw the marks)
 */
static void identifySafeLocations(const Grid<bool>& board, GWindow& window) {
    for (int row = 0; row < board.numRows(); row++) {
        for (int col = 0; col < board.numCols(); col++) {
            if (isSafe(board, row, col)) {
                markLocation(window, SAFE_MARK, row, col, SAFE_COLOR);
            }
            else {
                markLocation(window, DANGER_MARK, row, col, DANGER_COLOR);
            }
        }
    }
}

/* isSafe()
 *
 * This function returns true if the indicated location on the board
 * is a safe from the opponent's queens. Otherwise returns false.
 *
 * board: the current board state (locations of queens)
 * row, col: the chess board space in question
 */
static bool isSafe(const Grid<bool>& board, int row, int col) {
    for (int drow = -1; drow <= 1; drow++) {
        for (int dcol = -1; dcol <= 1; dcol++) {
            if (!isDirectionSafe(board, row, col, drow, dcol)) {
                return false;
            }
        }
    }
    return true;
}


/* isDirectionSafe()
 *
 * This function returns true if one can start at the given row/col and continue
 * in the indicated direction all the way to the edge of the board without
 * encountering a queen. Otherwise returns false.
 *
 * board: the current board state (locations of queens)
 * row, col: the "current" place where we will begin our check
 * drow, dcol: the direction of search (-1,-1 is NW, 1,1 is SE, 0,0 is stay in place, etc.)
 */
static bool isDirectionSafe(const Grid<bool>& board, int row, int col, int drow, int dcol) {
    if (drow == 0 && dcol == 0) {
        return !board[row][col];  // "stay in place" spot is safe iff no queen here
    }

    row += drow;
    col += dcol;
    while (board.inBounds(row, col) && !board[row][col]) {
        row += drow;
        col += dcol;
    }

    return !board.inBounds(row, col);
}


/* drawSquare()
 *
 * This function draws one square of the chess board, at the given
 * row/col. The color indicates the line color for the square.
 *
 * window: the view where we should draw the board grid lines
 * row, col: the location on the board where we should draw the grid lines
 * color: grid line color
 *
 */
static void drawSquare(GWindow& window, int row, int col, string color) {
    int ulx = (window.getWidth() - BOARD_DIMENSION * SQUARE_SIZE)/2 + col * SQUARE_SIZE;
    int uly = (window.getHeight() - BOARD_DIMENSION * SQUARE_SIZE)/2 + row * SQUARE_SIZE;
    window.setColor(color);
    window.drawRect(ulx, uly, SQUARE_SIZE, SQUARE_SIZE);
}

/* markLocation()
 *
 * This function is used to write a label to a given space on the chess board.
 *
 * window: the view of the board where we will make the mark
 * label: the text to write
 * row, col: the chess board space to mark
 * color: the color of the text to write
 */
static void markLocation(GWindow& window, string label, int row, int col, string color) {
    int cx = (window.getWidth() - BOARD_DIMENSION * SQUARE_SIZE)/2 + col * SQUARE_SIZE + SQUARE_SIZE/2;
    int cy = (window.getHeight() - BOARD_DIMENSION * SQUARE_SIZE)/2 + row * SQUARE_SIZE + SQUARE_SIZE/2;
    window.setColor(color);
    GLabel* mark = new GLabel(label);
    mark->setFont(MARK_FONT);
    mark->setColor(color);
    window.add(mark, cx - mark->getWidth()/2, cy + mark->getFontAscent()/2);
    pause(MILLISECOND_DELAY);
}

