/**
 * Facebook.js
 * -----------
 * Present the implementation of the facebookRefund function and
 * a simple test framework.
 */

"use strict";

/* Constants */
// The value of is supplied in a parallel FacebookConstants.js file, which
// needs to be included if the code below is to work as expected.

/**
 * Function: facebookRefund
 * ------------------------
 * Decides whether it was less expensive to purchase
 * Facebook stock at the time an order was placed or
 * the time the trade was executed and returns the
 * price difference between the two if the latter was
 * less expensive (and 0 otherwise).
 */
function facebookRefund(nShares, date, timeOrdered, timeExecuted) {
	let priceOrdered = findSharePrice(date, timeOrdered);
	let priceExecuted = findSharePrice(date, timeExecuted);
	let refund = nShares * (priceOrdered - priceExecuted);
	if (refund < 0) refund = 0;
	return refund;
}

/**
 * Function: findSharePrice
 * ------------------------
 * Returns the price of Facebook stock at the specified
 * time on the specified date.  If no price information is
 * available, an alert notifies the user and 0.0 is returned.
 */
function findSharePrice(date, time) {
	for (let i = 0; i < FB_SHARE_PRICE_DATA.length; i++) {
		let entry = FB_SHARE_PRICE_DATA[i];
		if (entry.date === date && entry.time === time) 
			return entry.price;
	}

	alert("No record for " + date + " " + time + ".");
	return 0.0;
}

/**
 * Function: TestFacebookRefund
 * ----------------------------
 * Semi-elaborate test framework to call facebookRefund in
 * a variety of ways and confirm that its return value is
 * always nonegative.
 */
function TestFacebookRefund() {
	let timestamps = extractAllTimestamps();
	for (let lh = 0; lh < timestamps.length; lh++) {
		for (let rh = lh + 1; rh < timestamps.length; rh++) {
			let timeOrdered = timestamps[lh];
			let timeExecuted = timestamps[rh];
			logExpectedRefund(timeOrdered, timeExecuted);
		}
	}
}

/**
 * Function: extractAllTimestamps
 * ------------------------------
 * Returns an array of all the times for which we have Facebook
 * stock price data on "5/18/2012".  It's assumed that all times
 * for any given date are stored in chronological order.
 */
function extractAllTimestamps() {
	let timestamps = [];
	for (let i = 0; i < FB_SHARE_PRICE_DATA.length; i++) {
		if (FB_SHARE_PRICE_DATA[i].date === "5/18/2012") {
			timestamps.push(FB_SHARE_PRICE_DATA[i].time);
		}
	}
	return timestamps;
}

/**
 * Function: logExpectedRefund
 * ---------------------------
 * Prints information about the execution of facebookRefund and what
 * it returns when the first two arguments are 1000 and "5/18/2012", and
 * the last two arguments are set to the two arguments supplied.
 */
function logExpectedRefund(timeOrdered, timeExecuted) {
	let header = "facebookRefund(1000, \"5/18/2012\", \"" + timeOrdered + ", " + timeExecuted + "\") --> $";
	console.log(header + facebookRefund(1000, "5/18/2012", timeOrdered, timeExecuted).toFixed(2));
}
