/*
 * File: YarnPattern.js
 * --------------------
 * This program uses the GLine class to simulate the process of winding a
 * piece of colored yarn around a set of pegs equally spaced along the
 * perimeter of a rectangle.  At each step, the yarn is stretched from
 * its current peg to the one DELTA pegs further on.
 */

/* Constants */

const GWINDOW_WIDTH = 1000;
const GWINDOW_HEIGHT = 625;
const N_ACROSS = 80;
const N_DOWN = 50;
const DELTA = 113;

/*
 * Creates a pattern that simulates the process of winding a piece of
 * yarn around an array of pegs along the edges of the graphics window.
 */

function YarnPattern() {
   let gw = GWindow(GWINDOW_WIDTH, GWINDOW_HEIGHT);
   let pegs = createPegArray(GWINDOW_WIDTH, GWINDOW_HEIGHT, N_ACROSS, N_DOWN);
   let thisPeg = 0;
   let nextPeg = -1;
   while (thisPeg !== 0 || nextPeg === -1) {
      nextPeg = (thisPeg + DELTA) % pegs.length;
      let p0 = pegs[thisPeg];
      let p1 = pegs[nextPeg];
      let line = GLine(p0.x, p0.y, p1.x, p1.y);
      line.setColor("Magenta");
      gw.add(line);
      thisPeg = nextPeg;
   }
}

/**
 * Creates an array of pegs around the perimeter of a rectangle with the
 * specified width and height.  The number of pegs in each dimension is
 * given by the values nAcross and nDown.
 */
function createPegArray(width, height, nAcross, nDown) {
   let dx = width/nAcross;
   let dy = height/nDown;
   let pegs = [];
   for (let i = 0; i < nAcross; i++) {
      pegs.push(Point(i * dx, 0));
   }
   for (let i = 0; i < nDown; i++) {
      pegs.push(Point(nAcross * dx, i * dy));
   }
   for (let i = nAcross; i > 0; i--) {
      pegs.push(Point(i * dx, nDown * dy));
   }
   for (let i = nDown; i > 0; i--) {
      pegs.push(Point(0, i * dy));
   }
   return pegs;
}

/**
 * Creates a new Point object.  If this function is called with no
 * arguments, it creates a Point object at the origin.
 */
function Point(x, y) {
   if (x === undefined) {
      x = 0;
      y = 0;
   }
   return { x: x, y: y };
}