# File: tokenscanner.py

"""
This file implements a simple version of a token scanner class.
"""

# A token scanner is an abstract data type that divides a string into
# individual tokens, which are strings of consecutive characters that
# form logical units.  This simplified version recognizes two token types:
#
#   1. A string of consecutive letters and digits
#   2. A single character string
#
# To use this class, you must first create a TokenScanner instance by
# calling its constructor:
#
#     scanner = TokenScanner()
#
# The next step is to call the setInput method to specify the string
# from which tokens are read, as follows:
#
#     scanner.setInput(s)
#
# Once you have initialized the scanner, you can retrieve the next token
# by calling
#
#    token = scanner.nextToken()
#
# To determine whether any tokens remain to be read, you can either
# call the predicate method scanner.hasMoreTokens() or check to see
# whether nextToken returns the empty string.
#
# The following code fragment serves as a pattern for processing each
# token in the string stored in the variable source:
#
#     scanner = TokenScanner(source)
#     while scanner.hasMoreTokens():
#         token = scanner.nextToken()
#         . . . code to process the token . . .
#
# By default, the TokenScanner class treats whitespace characters
# as operators and returns them as single-character tokens.  You
# can set the token scanner to ignore whitespace characters by
# making the following call:
#
#     scanner.ignoreWhitespace()

class TokenScanner:

    """This class implements a simple token scanner."""

# Constructor

    def __init__(self, source=""):
        """
        Creates a new TokenScanner object that scans the specified string.
        """
        self._source = source
        self._nch = len(source)
        self._cp = 0
        self._ignoreWhitespaceFlag = False

# Public methods

    def setInput(self, source):
        """
        Resets the input so that it comes from source.
        """
        self._source = source
        self._nch = len(source)
        self._cp = 0

    def nextToken(self):
        """
        Returns the next token from this scanner.  If nextToken is called
        when no tokens are available, it returns the empty string.
        """
        if self._ignoreWhitespaceFlag:
            self._skipWhitespace()
        if self._cp == self._nch:
            return ""
        token = self._source[self._cp]
        self._cp += 1
        if token.isalnum():
            while self._cp < self._nch and self._source[self._cp].isalnum():
                token += self._source[self._cp]
                self._cp += 1
        return token

    def hasMoreTokens(self):
        """
        Returns True if there are more tokens for this scanner to read.
        """
        if self._ignoreWhitespaceFlag:
            self._skipWhitespace()
        return self._cp < self._nch

    def ignoreWhitespace(self):
        """
        Tells the scanner to ignore whitespace characters.
        """
        self._ignoreWhitespaceFlag = True

# Private methods

    def _skipWhitespace(self):
        """
        Skips over any whitespace characters before the next token.
        """
        while self._cp < self._nch and self._source[self._cp].isspace():
            self._cp += 1

# Interactive test program for the TokenScanner class 

def TestTokenScanner():
    while True:
        line = input("Enter a string: ")
        if line == "": break
        scanner = TokenScanner(line)
        if line.startswith(" "):
            scanner.ignoreWhitespace()
        while scanner.hasMoreTokens():
            print('"' + scanner.nextToken() + '"')

# Startup code

if __name__ == "__main__":
    TestTokenScanner()
