/**
 * File: todo.js
 * -------------
 * Provides the necessary Javascript needed to wire up the
 * event handling necessary for my ToDo list to properly operate.
 */
"use strict";

/**
 * Function: onAddClick
 * --------------------
 * Invoked whenever the user clicks the Add button.
 * Whatever text resides in the one text input field
 * is extracted, and if its nonempty, a new todo list
 * entry is appended to the existing list by creating
 * a new li element and appending it to the list
 * of children already hanging from the <ul> element.
 *
 * Note that we are careful to add a text element as the
 * lone child, and we also add the correct double click
 * listener to the li item, just as we did for those that
 * existed at the time of the page load.
 */
function onAddClick(e) { // e argument is ignored
   let input = document.getElementById("item-text");
   let text = input.value.trim();
   input.value = "";
   if (text.length === 0) return;
   let ul = document.getElementById("to-do-list");
   let li = document.createElement("li");
   li.addEventListener("dblclick", onItemDoubleClick);
   let item = document.createTextNode(text);
   li.appendChild(item);
   ul.appendChild(li);
}

/**
 * Function: onClearClick
 * ----------------------
 * Fired whenever the clear button is clicked.  The function
 * repeatedly removes the last child from the relevant <ul>
 * element until all of its children are gone.  Ironically,
 * there is no removeAllChildren method.
 */
function onClearClick(e) { // e argument is ignored
   let ul = document.getElementById("to-do-list");
   while (ul.childNodes.length > 0) {
      ul.removeChild(ul.lastChild);
   }
}

/*
 * Function: onItemDoubleClick
 * ---------------------------
 * Fired whenever a list item (i.e. an <li> element is
 * double-clicked). The event object carries a target
 * attribute whose value is the element that triggered
 * the event.  We only add dblclick listeners to <li> 
 * elements (the original ones in the HTML file, and the
 * ones that are programmatically added).
 *
 * Fortunately, every single DOM node tracks its
 * parent, so the element that triggered the event
 * can asked the parent to remove it from its list
 * of children.
 */
function onItemDoubleClick(e) {
   e.target.parentNode.removeChild(e.target);
}

/**
 * Function: onKeyDown
 * -------------------
 * Triggered every single time the user presses
 * a key on the keyboard while the text input has
 * the focus.  All but one of the keys are irrelevant, 
 * but if the user presses the enter/return key, then
 * that's identified as an effort to effectively click
 * the add button.  And as it turns out, we reframe 
 * that very situation as an add button click: we call
 * the click method on the add button, which simulates
 * a button click as if we'd clicked it ourselves.
 */
function onKeyDown(e) {
   if (e.which === undefined) e.which = e.keyCode;
   const ENTER_KEY = 13;
   if (e.which !== ENTER_KEY) return;
   let addButton = document.getElementById("add-item");
   addButton.click();
}

/*
 * The following code block is executed once the
 * entire HTML file has been parsed and the DOM
 * has been fully constructed to represent it.  
 * Then, and only then, are we permitted to
 * search for elements by their ID
 */
document.addEventListener("DOMContentLoaded", function() {
   let addButton = document.getElementById("add-item");
   addButton.addEventListener("click", onAddClick);
   let clearButton = document.getElementById("clear-all-items");
   clearButton.addEventListener("click", onClearClick);
   let input = document.getElementById("item-text");
   input.addEventListener("keydown", onKeyDown);
   let listItems = document.getElementsByTagName("li");
   for (let i = 0; i < listItems.length; i++) {
      listItems[i].addEventListener("dblclick", onItemDoubleClick);
   }
});