/**
 * File: numbers-trivia.js
 * -----------------------
 * Implements a slide show of random facts about arbitrary
 * numbers between 1 and 200.
 */
"use strict";

function BootstrapNumberTrivia() {

   /* all four variables are referenced by inner callback functions */
   let contentArea = document.getElementById("showcase");
   let number = document.getElementById("number");
   let decrementButton = document.getElementById("decrement-button");
   let incrementButton = document.getElementById("increment-button");
   decrementButton.addEventListener("click", decrementNumber);
   incrementButton.addEventListener("click", incrementNumber);
   
   /**
    * Function: showcaseTrivia
    * ------------------------
    * Accepts the AsyncResponse object, rehydates the
    * payload into a bonafide object, and then extracts the
    * text field (which stores the trivial piece of information
    * about a particular number) and updates the main div
    * to show it.
    */
   function showcaseTrivia(response) {
      while (contentArea.childNodes.length > 0) {
         contentArea.removeChild(contentArea.lastChild);
      }
      let info = JSON.parse(response.getPayload());
      contentArea.appendChild(document.createTextNode(info.text));
   }

   /**
    * Function: initiateNumberTriviaFetch
    * -----------------------------------
    * Initiates a request for the JSON resource that comes with:
    *
    *   http://numbersapi.com/<n>?json=true
    *
    * where n is any well formed number.  Note that we
    * install the showcaseTrivia function to handle the
    * server response once it arrives.  We ignore any
    * server errors.
    */
   function initiateNumberTriviaFetch() {
      let req = AsyncRequest("http://number-trivia.com/" + number.value);
      req.addParams({json: true});
      req.setSuccessHandler(showcaseTrivia);
      req.send();
   }
   
   /**
    * Function: decrementNumber
    * -------------------------
    * Self-explanatory, save for the need to convert back and forth
    * between the numbers and their string forms by using the parseInt
    * function and the .toString() method.  Yes, even integers are
    * objects in JavaScript.
    */
   function decrementNumber(e) {
      incrementButton.disabled = false;
      let current = parseInt(number.value) - 1;
      number.value = current.toString();
      if (current === 1) decrementButton.disabled = true;
      initiateNumberTriviaFetch();
   }
   
   /**
    * Function: incrementNumber
    * -------------------------
    * Self-explanatory, save for the details highlighted in
    * documentation for decrementNumber.
    */
   function incrementNumber(e) {
      decrementButton.disabled = false;
      let current = parseInt(number.value) + 1;
      number.value = current.toString();
      if (current === 200) incrementButton.disabled = true;
      initiateNumberTriviaFetch();
   }
   
   /* seed initial presentation with a random number from [1, 200] */
   number.value = (Math.floor(Math.random() * 200) + 1).toString();
   initiateNumberTriviaFetch();
}

document.addEventListener("DOMContentLoaded", BootstrapNumberTrivia);
