--------------------------------------------------------------------------------
--
--  Murphi Model of the CHAP protocol, between Authenticator A end Peer P
--
--------------------------------------------------------------------------------
--
--  steps of the protocol modeled:
--
--   1. A->P: session + nonce
--   2. P->A: session + {session + password + nonce}hash
--   3. A->P: success || failure
--
--   A: authenticator, P: peer
--  We will also use the term "server" for Authenticator, and "client" for peer.
--
--------------------------------------------------------------------------------
-- SIMPLIFICATIONS:
--
-- This model only considers peers' passwords. It might be better to consider
-- "all" passwords (intruders and peers) but we don't see the point, given that
-- the server will only accept passwords from honest peers anyways.
--------------------------------------------------------------------------------



--------------------------------------------------------------------------------
-- Client Authentication:
--   Make sure that the server enters "SUCCESS" only with people who are
-- actually clients.
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
-- Server Authentication:
--   Make sure that the client enters "SUCCESS" only with people who are
-- actually servers.
--------------------------------------------------------------------------------

-- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
const

  NumAuthenticators:   1;   -- number of authenticators
  NumPeers:   1;            -- number of peers
  NumIntruders:    1;       -- number of intruders
  NetworkSize:     1;       -- max. number of outstanding messages in network
  MaxKnowledge:   15;       -- max. number of messages intruder can remember
  
  -- Protocol modifiers
  CheckServerAuth:       true;  -- true, if we want to verify that the 
  AttackerFakesSuccess:  false; -- set to true if the intruder will send a fake success message
  AttackerCanChangeUser: true;



-- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
-- Types defined for this MurPHI model.
type
	-- Set of authenticators
	AuthenticatorId:  scalarset (NumAuthenticators);
	-- Set of peers
	PeerId:           scalarset (NumPeers);
	-- Set of intruders
	IntruderId:       scalarset (NumIntruders);
	-- Set of all possible peers
	PeerIntruder: union {PeerId, IntruderId};
	-- Set of all possible authenticators
	AuthenticatorIntruder: union {AuthenticatorId, IntruderId};
  	-- Set of all agents
	AgentId:      union {AuthenticatorId, PeerId, IntruderId};

	MessageType : enum {      -- different types of messages
		M_Connect,             -- Peer says hello to authenticator (link begin)
		M_Challenge,           --  {Challenge,Chall_Id, Chall_Val}
		M_Response,            --  {Resp_Id, Resp_Val}
		M_Success,             --  Success, Peer has been authenticated
		M_Failure,             --  Failure, Peer has failed to be authenticated
		M_Failure_Expired,     --  Failure, expired password (only attacker sends this.)
		M_ChangePassword       --  Change password, sent by client
	};

	Session: record
		server:   AuthenticatorIntruder;
		client:   PeerIntruder;
	end;

	Message : record
		fromIntruder   : boolean;           -- Whether the intruder sent this message (so it doesn't learn it again)
		mType          : MessageType;       -- type of message
		session        : Session;           -- the id of peer + authenticator
		nonce				: AgentId;				-- server or peer nonce
		response			: NtResponse -- The NT Response or the whole authenticator response
	end;

	AuthenticatorStates : enum {
		A_SLEEP,                     -- waiting for link
		A_LINKED,                    -- state after connexion setup or at any given time during connection
		A_WAIT_RESPONSE,             -- waiting for response from peer
		A_FAILURE,                   -- authenticator refuses to authenticate peer
		A_SUCCESS                    -- authenticator accepts to authenticate peer
	};                                  --  (thinks responder is authenticated)

	Authenticator: record
		state:     AuthenticatorStates;
		peer: PeerIntruder;               -- agent who links to the authenticator
	end;
	
	PeerStates : enum {
		P_LINK,                      -- peer is going to make a connection
		P_WAIT_CHALLENGE,            -- peer is waiting for the server's challenge
		P_WAIT_OK,                   -- peer has sent response, waiting for server's OK
		P_SUCCESS,                   -- peer received server OK, and server is authenticated
		P_FAILURE,                   -- peer was told to buzz off
		P_BADAUTH                    -- authenticator failed to provide correct credentials
	};

	Peer : record
		state:     PeerStates;
		authenticator: AuthenticatorIntruder;
	end;
	
	PeerArray : array[PeerIntruder] of boolean;
	AuthenticatorArray : array[PeerIntruder] of boolean;
	SessionArray: array [AuthenticatorIntruder] of PeerArray;

   Intruder : record
      passwords      : PeerArray;                  -- known passwords
      serverNonces   : AuthenticatorArray;         -- known client nonces
      peerNonces     : PeerArray;                  -- known passwords
      NTResponses    : SessionArray;               -- known NTResponses
      AuthResponses  : SessionArray;               -- known Authenticator responses
      sessions			: SessionArray;               -- known sessions
   end;
    

-- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
var                                               -- state variables for:

  net: multiset[NetworkSize] of Message;          --  network
  auth: array[AuthenticatorId] of Authenticator;  --  authenticators
  peer: array[PeerId] of Peer;                    --  peers
  int: array[IntruderId] of Intruder;             --  intruders  

  

------------------------------------------------------------
------------------------------------------------------------
------------------------------------------------------------
------    ___  ______ _____ _   __________  ___  ____ ------
------   / _ )/ __/ // / _ | | / /  _/ __ \/ _ \/ __/ ------
------  / _  / _// _  / __ | |/ // // /_/ / , _/\ \   ------
------ /____/___/_//_/_/ |_|___/___/\____/_/|_/___/   ------
------                                                ------
------------------------------------------------------------
------------------------------------------------------------
------------------------------------------------------------



---------------------------
--   ___                 --
--  / _ \___ ___ _______ --
-- / ___/ -_) -_) __(_-< --
--/_/   \__/\__/_/ /___/ --
--                       --
---------------------------


-- peer i starts protocol with server or intruder j (step 1)
ruleset i: PeerId do
	ruleset j: AgentId do
		rule 20 "peer sends Hello (step 0)"

			peer[i].state = P_LINK &
			!ismember(j, PeerId) &               -- only servers and intruders
			multisetcount (l:net, true) < NetworkSize  -- only send a message if network is not full

		==>
    
		var
			outM: Message;   -- outgoing message

		begin
			undefine outM;
			outM.source  := i;
			outM.username:= i;
			outM.dest    := j;
			outM.mType   := M_Connect;
			multisetadd (outM,net);

			peer[i].state         := P_WAIT_CHALLENGE;
			peer[i].authenticator := j;
		end;
	end;
end;

-- peer i reacts to challenge (step 3)
ruleset i: PeerId do
	choose j: net do
		rule 20 "peer reacts to challenge (step 3)"

			peer[i].state = P_WAIT_CHALLENGE &
			net[j].dest = i &
			ismember(net[j].source,IntruderId) -- because everything goes through intruder
		==>

		var
			outM: Message;   -- outgoing message
			inM:  Message;   -- incoming message

		begin
			inM := net[j];
			multisetremove (j,net);

			if inM.username = peer[i].authenticator then   -- message comes from the right server

				if inM.mType = M_Challenge then   -- correct message type
					undefine outM;
					outM.source   := i;
					outM.dest     := peer[i].authenticator;
					outM.mType    := M_Response;
					outM.session  := inM.session;
					outM.nonce    := inM.nonce;
					outM.password := i;
					outM.username := i;

					multisetadd (outM,net);

					peer[i].state := P_WAIT_OK;
				end;

			end;
		end;
	end;
end;

ruleset i: PeerId do
	choose j: net do
		rule 20 "peer receives success/failure (step 5)"
		
			peer[i].state = P_WAIT_OK &
			net[j].dest = i &
			ismember(net[j].source,IntruderId) -- because everything goes through intruder
		==>
		
		var
			inM: Message; -- incoming
		
		begin
			inM := net[j];
			multisetremove(j, net);
			
			if inM.username = peer[i].authenticator then
				if inM.session.server = peer[i].authenticator &
				inM.session.client = i then
					if inM.mType = M_Success then
						peer[i].state := P_SUCCESS;
					elsif inM.mType = M_Failure then
						error "peer was refused";
						peer[i].state := P_FAILURE;
					end
				end;
			end;
		end;
	end;
end;


------------------------------------------------------------------
--   ___       __  __            __  _          __              --
--  / _ |__ __/ /_/ /  ___ ___  / /_(_)______ _/ /____  _______ --
-- / __ / // / __/ _ \/ -_) _ \/ __/ / __/ _ `/ __/ _ \/ __(_-< --
--/_/ |_\_,_/\__/_//_/\__/_//_/\__/_/\__/\_,_/\__/\___/_/ /___/ --
--                                                              --
------------------------------------------------------------------

-- authenticator i reacts to peer's connect (step 2)
ruleset i: AuthenticatorId do
	choose j: net do
		rule 20 "server reacts to peer's connect (step 2)"


			auth[i].state = A_SLEEP &
			net[j].dest = i &
			ismember(net[j].source,IntruderId)

		==>

		var
			outM: Message;   -- outgoing message
			inM:  Message;   -- incoming message

		begin
			inM := net[j];
			multisetremove (j,net);

			if inM.mType = M_Connect then   -- correct message type
				undefine outM;
				outM.source   := i;
				outM.dest     := inM.username;   -- identifier of peer
				outM.mType    := M_Challenge;
				outM.session.server  := i;
				outM.session.client  := inM.username;
				outM.nonce    := inM.username;
				outM.username := i;
				--outM.password := undefined;

				multisetadd (outM,net);

				auth[i].state     := A_WAIT_RESPONSE;
				auth[i].peer := inM.username;
			end;

		end;
	end;
end;

-- authenticator i reacts to challenge response (step 4)
ruleset i: AuthenticatorId do
	choose j: net do
		rule 20 "authenticator i reacts to challenge response (step 4)"

			auth[i].state = A_WAIT_RESPONSE &
			net[j].dest = i &
			ismember(net[j].source,IntruderId)

		==>
		
		var
			inM: Message;
			outM: Message;

		begin
			inM := net[j];
			multisetremove (j,net);
			
			if inM.session.server = i & inM.session.client = inM.username then   -- message has the right session ID
				if inM.mType = M_Response then   -- correct message type
					if inM.nonce = inM.username then    -- correct challenge id (nonce) received
						undefine outM;
						outM.source := i;
						outM.dest := inM.username;
						outM.session := inM.session;
						outM.nonce := inM.username;
						outM.username := i;
						
						if inM.password = inM.username then   -- correct password received
							auth[i].state := A_SUCCESS;
							outM.mType := M_Success;
						else
							auth[i].state := A_FAILURE;
							outM.mType := M_Failure;
						end;
						
						multisetadd(outM, net);
					end;
				end;
			end;

		end;
	end;
end;


---------------------------------------------
--   ____     __              __           --
--  /  _/__  / /_______ _____/ /__ _______ --
-- _/ // _ \/ __/ __/ // / _  / -_) __(_-< --
--/___/_//_/\__/_/  \_,_/\_,_/\__/_/ /___/ --
--                                         --
---------------------------------------------

-- intruder i intercepts messages
ruleset i: IntruderId do
	choose j: net do
		rule 10 "intruder intercepts"

			!ismember (net[j].source, IntruderId)    -- not for intruders' messages

		==>

		var
			temp: Message;

		begin
			alias msg: net[j] do   -- message to intercept

				alias messages: int[i].messages do
					temp := msg;
					undefine temp.source;   -- delete useless information
					undefine temp.dest;
					if multisetcount (l:messages,   -- add only if not there
					messages[l].mType = temp.mType &
					messages[l].nonce = temp.nonce &
					messages[l].session.server = temp.session.server &
					messages[l].session.client = temp.session.client &
					messages[l].username = temp.username &
					messages[l].password = temp.password ) = 0
						then
							multisetadd (temp, int[i].messages);
					end;
				end;

				multisetremove (j,net);
			end;
		end;
	end;
end;

-- intruder i sends recorded message
ruleset i: IntruderId do         -- arbitrary choice of
	choose j: int[i].messages do     --  recorded message
		ruleset k: AgentId do    --  destination
			rule 90 "intruder sends recorded message"

				!ismember(k, IntruderId) &                 -- not to intruders
				multisetcount (l:net, true) < NetworkSize

			==>

			var
				outM: Message;

			begin
				outM        := int[i].messages[j];
				outM.source := i;
				outM.dest   := k;

				multisetadd (outM,net);
			end;
		end;
	end;
end;

ruleset i: IntruderId do         -- arbitrary choice of
	choose j: int[i].messages do     --  recorded message
		ruleset u: AgentId do        --  claimed source
			ruleset k: AgentId do    --  destination
				rule 90 "intruder sends recorded message with new username"
				
					AttackerCanChangeUser &

					!ismember(k, IntruderId) &                 -- not to intruders
					!ismember(u, AuthenticatorId) &            -- not from servers
					int[i].messages[j].username != u &         -- not same username
					multisetcount (l:net, true) < NetworkSize

				==>

				var
					outM: Message;

				begin
					outM        := int[i].messages[j];
					outM.source := i;
					outM.dest   := k;

					outM.username := u;

					multisetadd (outM,net);
				end;
			end;
		end;
	end;
end;


-- intruder i fakes response to Peer Response
ruleset i: IntruderId do
	choose m: net do
		rule 10 "intruder fakes response to Peer Response"

			AttackerFakesSuccess &

			ismember(net[m].source, PeerId) & -- from a peer...
			ismember(net[m].dest, AuthenticatorId) & -- to an authenticator

			net[m].mType = M_Response -- only for a response message
		==>

		var
			outM: Message;
			inM: Message;
		begin
			inM := net[m];
			
			undefine outM;
			outM.source := i;
			outM.dest := inM.username;
			outM.session := inM.session;
			outM.nonce := inM.username;
			outM.username := inM.dest; -- make the message appear to come from server

			outM.mType := M_Success;
			multisetremove(m, net);
			multisetadd(outM, net);
		end;
	end;
end;

         


--------------------------------------------------------------------------------
-- startstate
--------------------------------------------------------------------------------

startstate
	-- initialize peers
	undefine peer;
	for i: PeerId do
		peer[i].state         := P_LINK;
		peer[i].authenticator := i;
	end;

	-- initialize servers
	undefine auth;
	for i: AuthenticatorId do
		auth[i].state    := A_SLEEP;
		auth[i].peer     := i;
	end;

	-- initialize intruders
	undefine int;
	for i: IntruderId do   -- the only nonce known is the own one
		for j: AgentId do  
			int[i].passwords[j] := false;
		end;
		int[i].passwords[i] := true;
	end;

	-- initialize network 
	undefine net;
end;



------------------------------------------------
--   ____                  _           __     --
--  /  _/__ _  _____ _____(_)__ ____  / /____ --
-- _/ // _ \ |/ / _ `/ __/ / _ `/ _ \/ __(_-< --
--/___/_//_/___/\_,_/_/ /_/\_,_/_//_/\__/___/ --
                                              --
------------------------------------------------

invariant "server correctly authenticated"
	forall i: PeerId do
		peer[i].state = P_SUCCESS &
		ismember(peer[i].authenticator, AuthenticatorId) &
		true
		
		->
		
		auth[peer[i].authenticator].peer = i &
		auth[peer[i].authenticator].state = A_SUCCESS
	end
;

-- "server talks to peer who is talking to same server"
invariant "server authenticates only peers connecting to it"
	forall i: AuthenticatorId do
		auth[i].state = A_SUCCESS &
		ismember(auth[i].peer, PeerId)
		
		->
		
		!CheckPeerAuth |
		peer[auth[i].peer].authenticator = i
	end
;
	
invariant "no peer passwords are known by intruders"
	forall i: IntruderId do
		forall j: AgentId do
			ismember(j, PeerId)
			->
			int[i].passwords[j] = false
		end
	end
;

