/*! \file uart.h \brief UART driver with buffer support. */
//*****************************************************************************
//
// File Name	: 'uart.h'
// Title		: UART driver with buffer support
// Author		: Pascal Stang - Copyright (C) 2000-2002
// Created		: 11/22/2000
// Revised		: 11/01/2001
// Version		: 1.3
// Target MCU	: ATMEL AVR Series
// Editor Tabs	: 4
//
// This code is distributed under the GNU Public License
//		which can be found at http://www.gnu.org/licenses/gpl.txt
//
//*****************************************************************************

#ifndef UART_H
#define UART_H

#include "global.h"
#include "buffer.h"

// default baud rate
// can be changed by using uartSetBaudRate()
#define UART_BAUD_RATE			9600

// buffer memory allocation defines
// buffer sizes
#define UART_TX_BUFFER_SIZE		0x0020	// number of bytes for uart transmit buffer
#define UART_RX_BUFFER_SIZE		0x0020	// number of bytes for uart receive buffer

// define this key if you wish to use
// external RAM for the	UART buffers
//#define UART_BUFFER_EXTERNAL_RAM
#ifdef UART_BUFFER_EXTERNAL_RAM
	// absolute address of uart buffers
	#define UART_TX_BUFFER_ADDR	0x1000
	#define UART_RX_BUFFER_ADDR	0x1100
#endif

// compatibility with dual-uart processors
// (if you need to use both uarts, please use the uart2 library)
#ifdef UDR0
	#define UDR					UDR0
	#define UBRR				UBRR0L
	#define SIG_UART_TRANS		SIG_UART0_TRANS
	#define SIG_UART_RECV		SIG_UART0_RECV
	#define SIG_UART_DATA		SIG_UART0_DATA
#endif
// compatibility for the mega161
#ifndef RXCIE
	#define RXCIE	RXCIE0
	#define TXCIE	TXCIE0
	#define UDRIE	UDRIE0
	#define RXEN	RXEN0
	#define TXEN	TXEN0
	#define CHR9	CHR90
	#define RXB8	RXB80
	#define TXB8	TXB80
#endif
#ifndef UBRR0L
	#define UBRR0L	UBRR0
	#define UBRR1L	UBRR1
#endif

// functions

//! initializes transmit and receive buffers
// called from uartInit()
void uartInitBuffers(void);

//! initializes uart
void uartInit(void);

//! sets the uart baud rate
void uartSetBaudRate(u32 baudrate);

//! returns pointer to the receive buffer structure
cBuffer* uartGetRxBuffer(void);

//! returns pointer to the transmit buffer structure
cBuffer* uartGetTxBuffer(void);

//! sends a single byte over the uart
void uartSendByte(u08 data);

//! gets a single byte from the uart receive buffer
// function returns TRUE if data was available, FALSE if not
// actual data is returned in variable pointed to by "data"
// example usage:
// char myReceivedByte;
// uartReceiveByte( &myReceivedByte );
u08 uartReceiveByte(u08* data);

//! returns TRUE/FALSE if receive buffer is empty/not-empty
u08 uartReceiveBufferIsEmpty(void);

//! flushes (deletes) all data from receive buffer
void uartFlushReceiveBuffer(void);

//! begins transmission of the transmit buffer under interrupt control
void uartSendTxBuffer(void);

//! sends a buffer of length nBytes via the uart using interrupt control
u08  uartSendBuffer(char *buffer, u16 nBytes);

#endif

