//*****************************************************************************
// File Name	: hello.c
// 
// Title		: Example "hello world" application
// Revision		: 1.0
// Notes		:	
// Target MCU	: Atmel AVR series
// Editor Tabs	: 4
// 
// Revision History:
// When			Who			Description of change
// -----------	-----------	-----------------------
// 14-Oct-2002	pstang		Created the program
//*****************************************************************************

 
//----- Include Files ---------------------------------------------------------
#include <io.h>			// include I/O definitions (port names, pin names, etc)
#include <sig-avr.h>	// include "signal" names (interrupt names)
#include <interrupt.h>	// include interrupt support

#include "global.h"		// include our global settings
#include "uart.h"		// include uart function library
#include "rprintf.h"	// include printf function library
#include "timer.h"		// include timer function library

//----- Defines ---------------------------------------------------------------
#define COUNT_UP	1		// Note: all CAPS is commonly used for #defines
#define COUNT_DOWN	2

//----- Functions -------------------------------------------------------------
void doCount(void);

//----- Global Variables ------------------------------------------------------
static unsigned char state;
static unsigned short count;

//----- Begin Code ------------------------------------------------------------
int main(void)
{
	// initialize the AVRlib libraries
	timerInit(); 				// initialize the timer system
	uartInit(); 				// initialize the UART (serial port)
	uartSetBaudRate(9600);		// set the baud rate to 9600
	rprintfInit(uartSendByte);	// make all rprintf statements use uart for output

	// pause for 100ms
	timerPause(100);

	// print a little intro message so we know things are working
	rprintf("\r\nHello World\r\n");

	// NOTE: inb and outb replaces the AVR assembly commands "in" and "out"
	// set port C pins to all output
	outb(DDRC, 0xFF);
	// set port C pins to a value of all 0 (logic low, or 0 volts)
	outb(PORTC, 0x00);

	// NOTE: The AVR assembly commands "SBI" and "CBI" are still usable in C
	// set interrupt 0 to trigger on falling edge
	cbi(MCUCR, ISC00);
	sbi(MCUCR, ISC01);
	// set interrupt 1 to trigger on falling edge
	cbi(MCUCR, ISC10);
	sbi(MCUCR, ISC11);

	// enable interrupt 0
	sbi(GIMSK, INT0);
	// enable interrupt 1
	sbi(GIMSK, INT1);
	// enable global interrupts
	sei();

	// AVRlib timer function library handles all timer interrupts
	// use timerAttach function to make doCount run on each overflow of Timer2
	timerAttach(TIMER2OVERFLOW_INT, doCount);
	// set the timer2 prescaler so the count rate is visible
	timer2SetPrescaler(TIMER_CLK_DIV1024);

	// set the state variable to default state (COUNT_UP)
	state = COUNT_UP;
	// reset the counter
	count = 0;
	
	// enter endless while loop
	while(1)
	{
		// print out the current count value
		rprintf("Count value: %d\r\n", count);
		// reflect count on the LEDs
		outb(PORTC, ~count);
		// pause for a moment
		timerPause(100);
	}
	
}

// perform count operation
void doCount(void)
{
	// check current state
	if(state == COUNT_UP)
		count++;
	if(state == COUNT_DOWN)
		count--;
}

// Interrupt 0 handler (service routine)
INTERRUPT(SIG_INTERRUPT0)
{
	// For debugging, print a '0' so you know the interrupt handler is working
	putchar('0');

	// modify the state
	state = COUNT_UP;
	
	// For debugging, print a 'c' so you know the handler has finished
	putchar('c');
}


// Interrupt 1 handler (service routine)
INTERRUPT(SIG_INTERRUPT1)
{
	// For debugging, print a '1' so you know the interrupt handler is working
	putchar('1');

	// modify the state
	state = COUNT_UP;

	// For debugging, print a 'd' so you know the handler has finished
	putchar('d');
}
 
