/******************
 * YING-ZONG HUANG
 * December, 2002
 * Wireless Freeform Input Device
 * Transmitter Code
 ******************/


//----- Include Files ---------------------------------------------------------
#include <io.h>			// include I/O definitions (port names, pin names, etc)
#include <sig-avr.h>	// include "signal" names (interrupt names)
#include <interrupt.h>	// include interrupt support
#include <math.h>

#include "global.h"		// include our global settings

//#define __DEBUG_MODE__
//----- Defines ---------------------------------------------------------------
#ifdef __DEBUG_MODE__
	#define BAUD_RATE	115200
	#define BAUD_DIV	((F_CPU/8/BAUD_RATE-1)+1)
#else
	#define BAUD_RATE	1200
	#define BAUD_DIV	(F_CPU/16/BAUD_RATE-1)	// non-double mode
#endif

#define M_DDR	DDRD
#define M_PIN	PIND
#define	M_BL	1<<5
#define	M_BR	1<<4
#define M_IDENT	(M_BL | M_BR)
#define M_MASK	(M_BL | M_BR)

#define	A_DDR	DDRD
#define A_PIN	PIND
#define A_X		1<<2
#define A_Y		1<<3
#define A_ICP	1<<6
#define A_MASK	(A_X | A_Y | A_ICP)

#define L_DDR	DDRD
#define	L_PORT	PORTD
#define	L_MASK	1<<7

#define U_DDR	DDRD	// usart
#define	U_MASK	1<<2

#define	MEM_BASE	0x0060	// base of the RAM
#define BUF_ICP_CHUNKS	40
#define BUF_ICP_SIZE	3
#define BUF_DIFF_CHUNKS	20	// how many elements of struct in buffer

#define FILT_ORD 1

#define sleep() ({ asm volatile ("sleep" "\n\t");})

typedef struct {
	u16 CurTime;
	u08 PinState;
} ICPT;

typedef struct {
	u16 onTime;
	u16 offTime;
} DIFFT;

typedef struct {
	DIFFT* BUF_DIFF;
	u08 bufDiffHead;
	u08 bufDiffTail;
	s16 pBufHistOut[FILT_ORD];
	s16 pBufHistIn[FILT_ORD];	// for filtering
} BDT;

typedef struct {
	u08 b;
	s08 x;
	s08 y;
	u08 btn_old;
	s16 XCenter;
	s16 YCenter;	
} MOUSET;

typedef s16 (*fnFiltT)(s16 In, s16* pLastIn, s16* pLastOut);

//----- Functions -------------------------------------------------------------
void InitAll(void);
inline void MemInit(void);
inline void USARTInit(void);
inline void ICPInit(void);
void WakeInit(void);
void USARTSend(u08 txData);
//u08 CheckButtons(void);
void EnQ(u08 Byte, u08* pQBuf, u08* pQHead, u08 Size);
u08 DeQ(u08* pQBuf, u08* pQTail, u08 Size);
void EnQW(u16 Word, u16* pQBuf, u08* pQHead, u08 Elems);
s16 LPF(s16 In, s16* pLastIn, s16* pLastOut);
s16 HPF(s16 In, s16* pLastIn, s16* pLastOut);
s16 fmul(s16 x, u08 y);
void MousePack(MOUSET* pm);
void GetFromICPQueue(void);
u08 ProcDiffQueue(BDT* bdt, s16* pOut);
s08 ShiftClip(s16 In);

//----- Global Variables ------------------------------------------------------
// shared buffers
ICPT BUF_ICP[BUF_ICP_CHUNKS];
DIFFT BUF_DIFFX[BUF_DIFF_CHUNKS];
DIFFT BUF_DIFFY[BUF_DIFF_CHUNKS];

// and buffer pointers
volatile register u08 bufICPHead asm("r18");
volatile register u08 bufICPTail asm("r19");
BDT bdtX = {BUF_DIFFX, 0, 0, 0, 0};
BDT bdtY = {BUF_DIFFY, 0, 0, 0, 0};
MOUSET coord = {0, 0, 0, 0, 0, 0};
volatile u08 MouseXmitStatus = 0;
enum {WaitInit=0, WaitDown, WaitUp};
u08 XState=WaitInit, YState=WaitInit;

//----- Begin Code ------------------------------------------------------------

SIGNAL(SIG_INPUT_CAPTURE1)
{	// all other interrupts disabled while in here, so we should hurry.
	register u08* ptr = (u08*)BUF_ICP + bufICPHead;
	*(ptr++) = inb(ICR1L);
	*(ptr++) = inb(ICR1H);
	*ptr = inb(A_PIN);
	bufICPHead += BUF_ICP_SIZE;
	if( bufICPHead == BUF_ICP_CHUNKS * BUF_ICP_SIZE )
		bufICPHead = 0;
	// work in clobber mode...
}

INTERRUPT(SIG_UART_TRANS)
{
	if(MouseXmitStatus == 1) {
		MouseXmitStatus++;
		USARTSend(coord.x);
	} else if(MouseXmitStatus == 2) {
		MouseXmitStatus++;
		USARTSend(coord.y);
	} else if(MouseXmitStatus == 4) {
		MouseXmitStatus++;
		USARTSend('2');
	} else	// done
		MouseXmitStatus = 0;
}

int main(void)
{
	s16 VX, VY;
	InitAll();
	u08 btns;
	u08 status;
	s08 x, y, old_x=0, old_y=0;
	u16	cnt=0; //s

	while(1) {
		GetFromICPQueue();
		status = ProcDiffQueue(&bdtX, &VX) && ProcDiffQueue(&bdtY, &VY);
		btns = (~inb(M_PIN) & M_MASK);	// switch state
		if( btns == M_IDENT ) {
#ifndef __DEBUG_MODE__
			if(MouseXmitStatus == 0) {	// if ready to send
				MouseXmitStatus = 4;	// transmit ident string
				USARTSend('M');
			}
#endif
			// calibrate level position
			coord.XCenter = VX;
			coord.YCenter = VY;
		} else if( status ) {	// has x y data
			VX -= coord.XCenter;
			VY -= coord.YCenter;
			x = -ShiftClip(VX);
			y = -ShiftClip(VY);
			if(x == old_x && y == old_y)	//s
				cnt++;	// s: count for sleeping
			else {	//s
				cnt = 0;	//s
				outb(L_PORT, inb(L_PORT) ^ L_MASK);	// toggle LED whenever there is movement
			}
			old_x = x; old_y = y;
			// if we are ready to send, then send, otherwise, throw data away
			if(cnt >= 1000) {	// about two seconds of no movement
				outb(L_PORT, inb(L_PORT) & ~L_MASK);	//turn off LED
				//outb(TIMSK, 1<<TOIE0);	// turn on wake interrupt, turn off icp interrupt
				//sleep();
				//outb(TIMSK, 1<<TICIE1);	// turn on icp interrupt, turn off timer interrupt
				//XState = YState = WaitInit;	// do icp edges from the first state
			}
			else {
#ifdef	__DEBUG_MODE__
				USARTSend(x);	// debug
#else
				if( (btns != coord.btn_old || x != 0 || y != 0)
					&& MouseXmitStatus == 0 ) {

					coord.b = btns;
					coord.x = x;
					coord.y = y;
					coord.btn_old = coord.b;
					MousePack(&coord);
					MouseXmitStatus=1;
					USARTSend(coord.b);	// get started by sending the first byte
				}
#endif
			}
		}
	}
}

s08 ShiftClip(s16 In)
{
	u08 sgn = (In >= 0);
	In = In >= 0 ? In>>5 : (-In)>>5;
	In -= 4;	// clip (dead zone)
	if(In < 0)
		In = 0;
	if(sgn)
		return In;
	else
		return -In;
}

u08 ProcDiffQueue(BDT* bdt, s16* pOut)
{
	s16 In;

	if(bdt->bufDiffHead != bdt->bufDiffTail) {	// have one more input point
		// acquire data
		In = (s16)(bdt->BUF_DIFF[bdt->bufDiffTail].onTime);	// grab on time interval
		if(++(bdt->bufDiffTail) == BUF_DIFF_CHUNKS)
			bdt->bufDiffTail = 0;
		*pOut = LPF(In, bdt->pBufHistIn, bdt->pBufHistOut);
		/* high pass filter here to get rid of tilt-acceleration, then integrate once for v.
			//OutX = -HPF(&LastInX, InX, &LastOutX);
			// now integrate by simply summing
			//VX += OutX >= 0 ? OutX>>7 : -((-OutX)>>7);
			//VX += (OutX >> 3);	// 3 bits of noise as observed.
			if(cnt == 10)
				VX = cnt = 0;	// reset velocity component
			else if((OutX >> 3) == 0)	// within noise, we are actually at rest frame
				cnt++;	// count how many points are like this
			*/
			//outb(PORTC, ~(((OutX>6) * 0xF0) | ((OutX<-6) * 0x0F )));
		return 1;
	}
	return 0;
}

void GetFromICPQueue(void)
{
//	enum {WaitInit=0, WaitDown, WaitUp};
//	static u08 XState=WaitInit, YState=WaitInit;
	static u16 XFirst=0, YFirst=0;
	static u08 bXY_old=0, bChg_old=0;
	u08 bXY, bChg;
	u16 CurTime;

	if(bufICPHead != bufICPTail) {	// there is data in the ICP queue
	// process it
		CurTime = *((u08*)BUF_ICP+(bufICPTail++));
		CurTime += *((u08*)BUF_ICP+(bufICPTail++)) << 8;
		bXY = *((u08*)BUF_ICP+bufICPTail);
		if(++bufICPTail == BUF_ICP_CHUNKS*BUF_ICP_SIZE)
			bufICPTail = 0;
		bChg = (bXY_old ^ bXY) & (A_X | A_Y);
		if(bChg) {	// changed mask
			if(bChg & A_X) {	// x changed
				switch(XState) {
				case WaitInit:
					if(bXY & A_X) {	// if x (changed to) high
						XState=WaitDown;
						XFirst = CurTime;
					}	// if x changed to low, ignore this datapoint
					break;
				case WaitDown:	// x changed to low
					bdtX.BUF_DIFF[bdtX.bufDiffHead].onTime = CurTime-XFirst;
					XFirst = CurTime;
					XState=WaitUp;	// now wait for up edge
					break;
				case WaitUp:	// x changed to up
					bdtX.BUF_DIFF[bdtX.bufDiffHead].offTime = CurTime-XFirst;
					XFirst = CurTime;
					XState=WaitDown;	// now wait for down edge
					if( ++(bdtX.bufDiffHead) == BUF_DIFF_CHUNKS )
						bdtX.bufDiffHead = 0;
					break;
				}
			}
			if(bChg & A_Y) {	// y changed
				switch(YState) {
				case WaitInit:
					if(bXY & A_Y) {	// if y (changed to) high
						YState=WaitDown;
						YFirst = CurTime;
					}	// if y changed to low, ignore this datapoint
					break;
				case WaitDown:	// y changed to low
					bdtY.BUF_DIFF[bdtY.bufDiffHead].onTime = CurTime-YFirst;
					YFirst = CurTime;
					YState=WaitUp;	// now wait for up edge
					break;
				case WaitUp:	// y changed to up
					bdtY.BUF_DIFF[bdtY.bufDiffHead].offTime = CurTime-YFirst;
					YFirst = CurTime;
					YState=WaitDown;	// now wait for down edge
					if( ++(bdtY.bufDiffHead) == BUF_DIFF_CHUNKS )
						bdtY.bufDiffHead = 0;
					break;
				}
			}
			//USARTSend('.');
		} else {	// nothing changed... missed an edge
			// recover that edge by looking at the last event:
			if(XState != WaitInit && YState != WaitInit) {
				if(bChg_old == A_X) {	// x last changed alone.. most recent is y change
					if(YState == WaitDown) {
						bdtY.BUF_DIFF[bdtY.bufDiffHead].onTime = XFirst-YFirst;
						bdtY.BUF_DIFF[bdtY.bufDiffHead].offTime = CurTime-XFirst;
						if( ++(bdtY.bufDiffHead) == BUF_DIFF_CHUNKS )
							bdtY.bufDiffHead = 0;
					} else {
						bdtY.BUF_DIFF[bdtY.bufDiffHead].offTime = XFirst-YFirst;
						if( ++(bdtY.bufDiffHead) == BUF_DIFF_CHUNKS )
							bdtY.bufDiffHead = 0;
						bdtY.BUF_DIFF[bdtY.bufDiffHead].onTime = CurTime-XFirst;
					}
					YFirst = CurTime;
				} else if(bChg_old == A_Y) {	// y last changed alone.. assume most recent is x change
					if(XState == WaitDown) {
						bdtX.BUF_DIFF[bdtX.bufDiffHead].onTime = YFirst-XFirst;
						bdtX.BUF_DIFF[bdtX.bufDiffHead].offTime = CurTime-YFirst;
						if( ++(bdtX.bufDiffHead) == BUF_DIFF_CHUNKS )
							bdtX.bufDiffHead = 0;
					} else {
						bdtX.BUF_DIFF[bdtX.bufDiffHead].offTime = YFirst-XFirst;
						if( ++(bdtX.bufDiffHead) == BUF_DIFF_CHUNKS )
							bdtX.bufDiffHead = 0;
						bdtX.BUF_DIFF[bdtX.bufDiffHead].onTime = CurTime-YFirst;
					}
					XFirst = CurTime;
				}
				//USARTSend('o');
			}
		}
		bXY_old = bXY;	// save current level state
		bChg_old = bChg;	// save changed state also
	}
}

s16 LPF(s16 In, s16* pLastIn, s16* pLastOut)
{	// length two IIR filter (first-order butterworth LPF: wc = 0.008pi ~ 2Hz)
	s16 ret = fmul(*pLastOut, 252) + fmul((In + *pLastIn), 2);
	*pLastIn = In;
	*pLastOut = ret;
	return ret;	// substract out low pass value.
}

s16 HPF(s16 In, s16* pLastIn, s16* pLastOut)
{	// high pass first-order Butterworth filter with cutoff around 2Hz.
	//return In;
	//s16 ret = fmul(*pLastOut, 250) + fmul((In - *pLastIn), 253);
	s16 ret = fmul(*pLastOut, 254) + In - *pLastIn;
	*pLastIn = In;
	*pLastOut = ret;
	return ret;
}

s16 fmul(s16 x, u08 y)
{	// gives the 16 bit signed integer equal to x * (y/256)
	u16 lsb, msb;
	s16 ret;
	if(x & 0x8000) {
		lsb = -x & 0x00FF;
		msb = (-x & 0xFF00) >> 8;
		ret = -(s16)((((lsb * y) >> 8) & 0x00FF) + (msb * y));
	} else {
		lsb = x & 0x00FF;
		msb = (x & 0xFF00) >> 8;
		ret = (((lsb * y) >> 8) & 0x00FF) + (msb * y);
	}
	return ret;
}

void MousePack(MOUSET* pm)
{
	u08 b = ((( pm->y & 0xC0 ) >> 4) & 0x0C);
	b |= ((( pm->x & 0xC0 ) >> 6) & 0x03);
	pm->b |= (b | (1 << 6));
	pm->x &= 0x3F;
	pm->y &= 0x3F;
}

void InitAll(void)
{
	MemInit();
	USARTInit(); 			// initialize the UART (serial port)
	ICPInit();
	WakeInit();
	outb(U_DDR, inb(U_DDR) | U_MASK);	// usart is output
	outb(L_DDR, inb(L_DDR) | L_MASK);	// led is output
	// mouse and accelerometers are inputs
	outb(M_DDR, ~(~inb(M_DDR) | M_MASK));
	outb(A_DDR, ~(~inb(A_DDR) | A_MASK));
	outb(L_PORT, inb(L_PORT) | L_MASK);	// turn on LED for debug
	outb(MCUCR, (1<<SE) | (1<<SM0) );	//idle mode (timers enabled)
	sei();	// enable all interrupts
	// oiii iioi
}

void MemInit(void)
{
	bufICPHead=0;
	bufICPTail=0;
}

void ICPInit(void)
{
	outb(TCCR1A, 0);
	outb(TCCR1B, (1<<ICES1 | 1<<CS10));	// rising edge for ICP, full speed
	outb(TIMSK, 1<<TICIE1);	// ICP interrupt enable
}

void WakeInit(void)
{
	outb(TCCR0, 5<<CS00);	// overflows at ~30.5 Hz.
}

void USARTInit(void)
{
	// set baud rate
	outb(UBRRL, (u08)BAUD_DIV);	// lower bits
	outb(UBRRH, (u08)(BAUD_DIV >> 8));	// upper bits
#ifdef __DEBUG_MODE__
	outb(UCSRA, 1 << U2X);	// double speed
	outb(UCSRB, (1<<RXEN)|(1<<TXEN));	// no transfer interrupt
	outb(UCSRC, (1<<URSEL)|(0<<USBS)|(3<<UCSZ0));	// 8N1
#else
	// disable double baud rate (less switching)
	outb(UCSRA, 0 << U2X);
	// set up transmit and receive
	outb(UCSRB, (1<<RXEN)|(1<<TXEN)|(1<<TXCIE));
	// frame format: 7N1
	outb(UCSRC, (1<<URSEL)|(0<<USBS)|(2<<UCSZ0));
#endif
}

void USARTSend(u08 txData)
{
	// wait for the transmitter to be ready
	//while(!bit_is_set(UCSRA, UDRE));		// loop until USR:UDRE is 1
	outb( UDR, txData );
}

/* efficient circular queue */

void EnQ(u08 Byte, u08* pQBuf, u08* pQHead, u08 Size)
{
	*(pQBuf+*pQHead) = Byte;
	*pQHead = (*pQHead+1>=Size) ? 0 : (*pQHead+1);
}

u08 DeQ(u08* pQBuf, u08* pQTail, u08 Size)
{
	u08 temp = *(pQBuf+*pQTail);
	*pQTail = (*pQTail+1>=Size) ? 0 : (*pQTail+1);
	return temp;
}

void EnQW(u16 Word, u16* pQBuf, u08* pQHead, u08 Elems)
{	// Elems is the number of 2-byte elements
	*(pQBuf+*pQHead) = Word;
	*pQHead = (*pQHead+1>=Elems) ? 0 : (*pQHead+1);
	/*if(pQBuf == (u16*)BUF_DIFFX)
		USARTSend('X');
	else
		USARTSend('Y');*/
	//USARTSend((u08)(Word>>8));
}
