/*****************************************************************************

Copyright (c) 2004 SensAble Technologies, Inc. All rights reserved.

OpenHaptics(TM) toolkit. The material embodied in this software and use of
this software is subject to the terms and conditions of the clickthrough
Development License Agreement.

For questions, comments or bug reports, go to forums at: 
    http://dsc.sensable.com

Module:
  ME328_Ass4.cpp
  This is the backbone code for ME 328, Assignment 4: Medical Image Analysis, Image Registration, and Needle Insertion
  
Authors/Version:
  Allison Okamura and Troy Adebar (May 2013)

*******************************************************************************/
#ifdef  _WIN64
#pragma warning (disable:4996)
#endif

#include <cstdio>
#include <cassert>
#include <windows.h>
#include <time.h>

#if defined(WIN32)
# include <conio.h>
#else
# include "conio.h"
#endif

#include <HD/hd.h>
#include <HDU/hduVector.h>
#include <HDU/hduError.h>
#include <HDU/hduMath.h>
#include <HDU/hduMatrix.h>

// Prototypes
hduVector3Dd GetNeedlePosition(double l,hduMatrix transformMatrix);

// Specify name of the haptic device
// You can find device name in "PHANTOM configuration"
#define DEVICE_NAME_1 "Phantom1"
#define MAXPTS 10000 

// Dual Phantom devices
HHD phantomId_1;

// Variables for keeping track of parts
char part;
#define PART_A 'A'
#define PART_B 'B'

/////////////////////////////////////////////////
// DATA RECORDING EDIT HERE
// Variables for data recording
LARGE_INTEGER	ticksPerSecond, thisTime, lastTime, deltaTime;
char			clockResult[100];
double		deltaTimeS = 0.0;
FILE		*output_file;
int			num_pts = 0;
int			recording = 0;
double		Position[10000];
double		Velocity[10000];
double		Force[10000];
double		timer = 0;
double      Time[10000];
clock_t		start_time


/*******************************************************************************
 Haptic feedback callback.  Controller for single robot.
*******************************************************************************/
HDCallbackCode HDCALLBACK Control(void *data)
{
    // position/velocity of device
    hduVector3Dd pos, vel;
    // force for device
    hduVector3Dd forceVec;

    // get current position and velocity of device
    hdBeginFrame(phantomId_1);
    hdGetDoublev(HD_CURRENT_POSITION, pos);
	hdGetDoublev(HD_CURRENT_VELOCITY, vel);
   
	// get current position of the stylus tip in the Omni workspace given its length 
	double stylusTipLength = 275; // [mm] <---- EDIT THIS IF YOUR NEEDLE IS A DIFFERENT LENGTH
	hduVector3Dd stylusTipPosition(0,0,0); 
	hduMatrix transformMatrix;
	hdGetDoublev(HD_CURRENT_TRANSFORM,transformMatrix);		
	stylusTipPosition = GetNeedlePosition(stylusTipLength,transformMatrix);
 
	/////////////////////////////////////////////////
	//************* START EDITING HERE *************//
	/////////////////////////////////////////////////

	// variables 
		
	switch (part) {
		case PART_A:// Prints position of needle tip to screen. 
			// You should not have to modify this case at all.
			// Change variable stylusTipLength above if you think your needle is a different length.
			printf("\t%.1f, %.3f, %.3f\r",stylusTipPosition[0],stylusTipPosition[1],stylusTipPosition[2]);
            break;
			
		case PART_B:// Needle insertion
			//Add your code here to control the needle to the target
			break;

		// when no problem part is selected, output zero force to device
		default:
			forceVec.set(0,0,0);
	}

	/////////////////////////////////////////////////
	//************* STOP EDITING HERE *************//
	/////////////////////////////////////////////////

	// Output forces to robot
	hdMakeCurrentDevice(phantomId_1);
    hdSetDoublev(HD_CURRENT_FORCE, forceVec);
    hdEndFrame(phantomId_1);

	// Get time stamp
	lastTime = thisTime; // Cache the time of the previous haptic function call
	QueryPerformanceCounter(&thisTime); // Find out what time it is now
	// Calculate time since last call in clock cycles and then convert to seconds.
	deltaTime.QuadPart = (thisTime.QuadPart - lastTime.QuadPart);
	deltaTimeS = (float) deltaTime.LowPart / (float) ticksPerSecond.QuadPart;

	

	/////////////////////////////////////////////////
	// DATA RECORDING EDIT HERE
	// Store x-direction data for MAXPTS (currently about 10 seconds worth of data)
	if (recording) {
		if (num_pts < MAXPTS) {
			Position[num_pts] = pos[0];
			Velocity[num_pts] = vel[0];
			Force[num_pts] = forceVec[0];
			timer += deltaTimeS;
			Time[num_pts] = timer;
			num_pts++;
		}
	}

    HDErrorInfo error;
    if (HD_DEVICE_ERROR(error = hdGetError()))
    {
        hduPrintError(stderr, &error, "Error during scheduler callback");
        if (hduIsSchedulerError(&error))
        {
            return HD_CALLBACK_DONE;
        }
    }

    return HD_CALLBACK_CONTINUE;
}


/*******************************************************************************
 * main function
   Initializes the devices, creates a callback to handles Control function, 
   terminates upon key press.
 ******************************************************************************/
int main(int argc, char* argv[])
{
    HDErrorInfo error;
	int quit = 0;
	int key;

    printf("Starting application\n");
    
    // Initialize both devices.  These needs to be called before any actions on the devices.
    
    // First device
	phantomId_1 = hdInitDevice(DEVICE_NAME_1);
	if (HD_DEVICE_ERROR(error = hdGetError()))
    {
        hduPrintError(stderr, &error, "Failed to initialize first haptic device");
        fprintf(stderr, "Make sure the configuration \"%s\" exists\n", DEVICE_NAME_1);
        fprintf(stderr, "\nPress any key to quit.\n");
        getchar();
        exit(-1);
    }

    printf("1. Found device %s\n",hdGetString(HD_DEVICE_MODEL_TYPE));
    hdEnable(HD_FORCE_OUTPUT);
    hdEnable(HD_FORCE_RAMPING);

	// Set the update rate at 1,000 Hz
	hdSetSchedulerRate(1000);
    hdStartScheduler();
    if (HD_DEVICE_ERROR(error = hdGetError()))
    {
        hduPrintError(stderr, &error, "Failed to start scheduler");
        fprintf(stderr, "\nPress any key to quit.\n");
        getchar();
        exit(-1);
    }

    // Schedule the control callback, which will then run at 
    // servoloop rates and command forces if the user penetrates the plane.
    HDCallbackCode hControlCallback = hdScheduleAsynchronous(
        Control, 0, HD_DEFAULT_SCHEDULER_PRIORITY);

    printf("\nPress A to read the needle tip position.\n");
    printf("Press B to insert the needle.\n");
	printf("Press R to record data for the next ten seconds.\n");
    printf("Press Q to quit.\n\n");

	QueryPerformanceFrequency(&ticksPerSecond);
	printf("There are %I64d ticks per second.\n\n", ticksPerSecond.QuadPart);

	while (!quit) {

	   if (!hdWaitForCompletion(hControlCallback, HD_WAIT_CHECK_STATUS))
        {
            fprintf(stderr, "\nThe main scheduler callback has exited\n");
            fprintf(stderr, "\nPress any key to quit.\n");
            _getch();
            quit = 1;
        }

	if (_kbhit()) {
		key = _getch();
		switch (key) {
			case 'a':
			case 'A':
				printf("Printing needle tip position.\n");
				part = PART_A;
				break;
			case 'b':
			case 'B':
				printf("Inserting needle.\n");
                                start_time = clock();
				part = PART_B;                         
				break;
			case 'r':
			case 'R':
				printf("Data will be in file data_output_%c.txt.\n", part);
				num_pts = 0;
				recording = 1;
				break;
			case 'q':
			case 'Q':
				quit = 1;
				break;
			default:
				printf("Rendering nothing.\n");
				part = 0;
		}
	}
	}

    // Cleanup and shutdown the haptic devices, cleanup all callbacks.
    hdStopScheduler();
    hdUnschedule(hControlCallback);

    if (phantomId_1 != HD_INVALID_HANDLE) {
        hdDisableDevice(phantomId_1);
        phantomId_1 = HD_INVALID_HANDLE;
    }

	/////////////////////////////////////////////////
	// DATA RECORDING EDIT HERE
	// Save data to file
	char buffer [20];
	sprintf(buffer, "data_output_%c.txt", part);
	output_file = fopen(buffer,"w");
	for (int i=0; i<num_pts; i++) {
		// for Matlab:
		//fprintf(output_file, "t(%d)=%5f;\t p(%d)=%5f;\t v(%d)=%5f;\t f(%d)=%5f; \n", i+1, Time[i], i+1, Position_1[i], i+1, Position_2[i], i+1, Velocity_1[i], i+1, Velocity_2[i], i+1, Force_1[i], i+1, Force_2[i]);
		// for input to Excel:
		fprintf(output_file, "%5f\t %5f\t %5f\t %5f\n", Time[i], Position[i], Velocity[i], Force[i]);
	}
	fclose(output_file);

    return 0;
}

// Needle Point Position Function
/*
PURPOSE----------------------------------------------------------------------------------
Return the position of a "needle" which is attached to the omni in the omni workspace
INPUTS------------------------------------------------------------------------------------
double l                    : length of the needle (measured from omni op point)  (mm)
hduMatrix transformMatrix   : the 4x4 homogeneous tranform for the omni op point  (-)
OUTPUTS-----------------------------------------------------------------------------------
hduVector3Dd needlePosition : the position of the needle tip in omni workspace    (mm)
*/

hduVector3Dd GetNeedlePosition(double l,hduMatrix transformMatrix)
{
        // (1) extract rotation matrix (from stylus to omni base)
        double rotationMatrix[3][3];
        transformMatrix.getRotationMatrix(rotationMatrix);

        // (2) find vector from operation point to needle tip
        hduVector3Dd needleVectorInOP(0,0,l); //length is in the z direction in the direction of stylus

        // (3) resolve needle vector in ground coordinates (couldn't find a built in 3x3 matrix multiply)
        hduVector3Dd needleVector(0,0,0);
        needleVector[0] = rotationMatrix[0][0]*needleVectorInOP[0] +
                          rotationMatrix[1][0]*needleVectorInOP[1] +
                          rotationMatrix[2][0]*needleVectorInOP[2];

        needleVector[1] = rotationMatrix[0][1]*needleVectorInOP[0] +
                          rotationMatrix[1][1]*needleVectorInOP[1] +
                          rotationMatrix[2][1]*needleVectorInOP[2];

        needleVector[2] = rotationMatrix[0][2]*needleVectorInOP[0] +
                          rotationMatrix[1][2]*needleVectorInOP[1] +
                          rotationMatrix[2][2]*needleVectorInOP[2];

        // (4) extract position of operational point
        hduVector3Dd omniPositionOP(0,0,0);
        omniPositionOP[0] = transformMatrix.get(3,0);
        omniPositionOP[1] = transformMatrix.get(3,1);
        omniPositionOP[2] = transformMatrix.get(3,2);

        // (5) add omni OP and needle vectors
        hduVector3Dd needlePosition;
        needlePosition = omniPositionOP + needleVector;

		// FOR TESTING
		//printf("\t%.3f, %.3f, %.3f\n",needlePosition[0],needlePosition[1],needlePosition[2]);

        // (6) return needle position
        return(needlePosition);
}



/*****************************************************************************/
