% File: checkData.m
% Authors: Jose Gutierrez and Anita Rogacs
% Usage: flags = checkData(tableData)
% ------------------------------------------------------------------
% This function generates flags that correspond to specific types of
% errors that may be encountered when running a calculation.
% Currently, the following flags are used:
%
% flag 0: the input table is not missing any data
% flag 1: array is missing one value (Cp or deltaH)
% flag 2: array is missing two values (Cp and deltaH)
% flag 3: a value is listed as 'NaN'
% flag 4: one or more concentration values are missing
% flag 5: table is empty
%flag 6: all concentrations are set to 0

function  [warningMsg] = displayMSGWarnings(dataTable, handles)

% warning: type = 1
% error: type = 2
% no warnings or errors: type = 0

try
    flags = checkData(dataTable);
    
    str = [];
    index1=0;
    index2=0;
    
    for i = 1:size(flags,1)
        flagNum = flags(i,1);
        flagRow = flags(i,2);
        
        if flagNum == 1
           errMsg = sprintf('Entry %i (%s): deltaCp not specified\n\n', flagRow, dataTable{flagRow,1});
            str = [str errMsg];
            index1 =1;
        elseif flagNum == 2
            errMsg = sprintf('Entry %i (%s): deltaH and deltaCp not specified \n\n', flagRow, dataTable{flagRow,1});
            str = [str errMsg];
            index2 =1;
        elseif flagNum == 3
            errMsg = sprintf('Entry %i (%s): electrolyte properties are missing\nCalculation could not proceed.\n\n', flagRow, dataTable{flagRow,1});
            str = [str errMsg];
        elseif flagNum == 4
            errMsg = sprintf('Entry %i (%s): concentration value missing\n\nCalculation could not proceed.\n\n', flagRow, dataTable{flagRow,1});
            str = [str errMsg];
        elseif flagNum == 5
            errMsg = sprintf('Electrolyte Solution Composition table is empty.\n\n');
            str = [str errMsg];
        elseif flagNum == 6
            errMsg = sprintf('Concentrations of all species are set to 0.');
            str = [str errMsg];
          
        end
    end
    
    if index1==1 || index2==1
         errMsg = sprintf('If popup disabled, NaN values are replaced with 0 for simulation\n\n');
         str = [str errMsg];
    end
    if index1==1
         errMsg = sprintf('Note: For most electrolytes, deltaCp have negligible contributions to pK (see README file for references)\n\n');
         str = [str errMsg];
    end
    if index2==1
       errMsg = sprintf('Note: Strong electrolytes (pKa<2 or pKa>12) fully dissociate in moderate pH (4-10) buffers.\nTherefore their deltaH and deltaCp have neglible or no contributions to the temperature dependence of\n(1) their effective mobility, and\n(2) the solution properties\n(see README file for references)\n\n');
       str = [str errMsg];
    end
    
    warningMsg = str;
    set(handles.message,'String',warningMsg);
    
catch err
% If an error occurs, it will occur when checkData is called in line 8.
% Therefore, we can be certain that the error involves the Electrolyte
% Solution Composition table.
    warningMsg = sprintf('errmsg'); % TODO: change this error message ('Nan', empty table)
    set(handles.message,'String',warningMsg);
end

end