#ifndef ROSENBERG_H
#define ROSENBERG_H

#include "tree.h"
#include "iomanip"
#include <random>
#include <time.h>

// Factorial (non-optimized)
long long int fact(int n){
    long long int value = 1;
    for(int i = n; i > 0; i--){
        value = value * i;
    }
    return value;
}

// Falling factorial
long long int fallingFact(int n, int k){
    long long int value = fact(n) / fact(n - k);
    return value;
}

// Rising factorial
long long int risingFact(int n, int k){
    long long int value = fact(n + k - 1) / fact(n - 1);
    //if(value == 0) cout << n << " " << k << endl;
    return value;
}

// W function from Rosenberg (2002)
long double wFunction(int m, int n, int x, int k, double T){
    long double num, den, summand, value;
    value = 0;
    den = 0;
    num = gFunction(m, x, T) * gFunction(n, k - x, T);
    for(int i = 1; i < k; i++){
        summand = gFunction(m, i, T) * gFunction(n, k - i, T);
        cout << summand << " " << i << " " << x << " " << k-x << endl;
        den += summand;
    }
    value = num / den;
    return value;
}

// I function
long double iFunction(int n, int k){
    long double num, den, value, a, b, c, d;
    if(k > n) return 0;
    else if(n == 0 && k != 0) return 0;
    else if(n == 0 && k == 0) return 1;
    else{
        a = fact(n);
        b = fact(n - 1);
        num = a * b;
        c = fact(k);
        d = fact(k - 1);
        den = pow(2, n - k)* d * c;
        value = num / den;
        return value;
    }
}

// W function (interweaving) for 2 coalescence sequences
long double wSub2(int x, int y){
    long double value, sum, b, c, d;
    long long int a;
    sum = x + y;
    a = fact(sum);
    b = fact(x);
    c = fact(y);
    d = b * c;
    value = a / d;
    return value;
}

// W function (interweaving) for 3 coalescence sequences
long double wSub3(int x, int y, int z){
    long double value, sum, b, c, d, e;
    long long int a;
    sum = x + y + z;
    a = fact(sum);
    b = fact(x);
    c = fact(y);
    d = fact(z);
    e = b * c * d;
    value = a / e;
    return value;
}

// W function (interweaving) for 4 coalescence sequences
long double wSub4(int w, int x, int y, int z){
    long double value, sum, b, c, d, e, f;
    long long int a;
    sum = w + x + y + z;
    a = fact(sum);
    f = fact(w);
    b = fact(x);
    c = fact(y);
    d = fact(z);
    e = f * b * c * d;
    value = a / e;
    return value;
}

// f function from Rosenberg (2002) as stated in Rosenberg (2003) with c=0
long double fSmallFunction(int k, int a, int b){
   long double value = 0;
    for(int x = max(1, k + 1 - b ); x <= a; x++){
        for(int y = max(1, k + 1 - x); y <= b; y++){
                long double term, term1n, term1d, term1, term2n, term2d, term2, term3n, term3d, term3, term4n, term4d, term4;
                term1n = biCo(a, x) * biCo(b, y);
                term1d = biCo(a+b, x+y);
                term2n = biCo(x+y, x);
                term2d = biCo(a+b, a);
                term3n = a + b;
                term3d = a * b;
                term4n = 2 * (x * y)*(x * y);
                term4d = (x  + y)*(x + y) * (x + y - 1);
                term1 = term1n / term1d;
                term2 = term2n / term2d;
                term3 = term3n / term3d;
                term4 = term4n / term4d;
                term = term1 * term2 * term3 * term4;
                value += term;
        }
    }
    return value;
}

// f function from Rosenberg (2002) as stated in Rosenberg (2003) with c > 0
long double fFunction(int k, int a, int b, int c){
   long double value = 0;
    for(int x = max(1, k + 1 - b - c); x <= a; x++){
        for(int y = max(1, k + 1 - x - c); y <= b; y++){
            for(int z = max(1, k + 1 - x - y); z <= c; z++){
                long double term, term1n, term1d, term1, term2n, term2d, term2, term3n, term3d, term3, term4n, term4d, term4;
                term1n =  biCo(a, x) * biCo(b, y) * biCo(c, z);
                term1d =  biCo(a+b+c, x+y+z);
                term2n = biCo(x+y+z, z) * biCo(x+y, x);
                term2d = biCo(a+b+c, c) * biCo(a+b, a);
                term3n = a + b + c;
                term3d = a * b * c;
                term4n = 2 * (x * y)*(x * y) * z;
                term4d = (x  + y + z)*(x + y + z) * (x + y + z - 1);
                term1 = term1n / term1d;
                term2 = term2n / term2d;
                term3 = term3n / term3d;
                term4 = term4n / term4d;
                term = term1 * term2 * term3 * term4;
                value += term;
            }
        }
    }
    return value;
}

// Result from Rosenberg (2003), not RM
long double r2003result(int r_A, int r_B, double T_A, double T_B){
    long double value = 0;
    for(int q_A = 0; q_A <= r_A; q_A++){
        for(int q_B = 0; q_B <= r_B; q_B++){
            if(q_A + q_B > 0){
                long double prod1, prod2, summand = 0;
                prod1 = gFunction(r_A, q_A, T_A) * gFunction(r_B, q_B, T_B);
                for(int k = 1; k <= q_B; k++){
                    summand += iFunction(q_A, 1) * iFunction(q_B, k) * wSub2(q_A - 1, q_B - k) * k * iFunction(k, 1);
                }
                if(iFunction(q_A + q_B, 1) > 10000 || iFunction(q_A + q_B, 1) < 0) prod2 = 0;
                else {prod2 = prod1 * summand / iFunction(q_A + q_B, 1);}
                value += prod2;
            }
        }
    }
    return value;
}

// Result from Rosenberg (2003), RM
long double r2003resultRM(int r_A, int r_B, double T_A, double T_B){
    long double value = 0;
    for(int q_A = 0; q_A <= r_A; q_A++){
        for(int q_B = 0; q_B <= r_B; q_B++){
            if(q_A + q_B > 1){
                long double prod1, prod2;
                long long int choose;
                prod1 = gFunction(r_A, q_A, T_A) * gFunction(r_B, q_B, T_B);
                choose = fact(q_A + q_B)/(fact(q_A) * fact(q_B)) * (q_A + q_B - 1);
                prod2 = prod1 * 2 / choose;
                value += prod2;
            }
        }
    }
    return value;
}

// Result from Rosenberg (2002); incorrect (but correctly implemented)
long double r2002result(int r, int s, int q, double T_3, double T_2){
    long double value = 0;
    if(T_2 > 0){
    for(int m = 1; m <= r; m++){
        for(int n = 1; n <= s; n++){
            for(int k = 1; k <= m + n; k++){
                for(int l = 1; l <= q; l++){
                        long double term1, term2, summand = 0;
                        term1 = gFunction(r, m, T_3) * gFunction(s, n, T_3) * gFunction(m + n, k, T_2) * gFunction(q, l, T_2 + T_3);
                        if(k == 1){
                            term2 = (1 - fSmallFunction(2, m, n)) * 2 / (l * (l + 1));
                        }
                        else {
                            for(int x = 1; x <= k - 1; x++){
                                cout << summand << endl;
                                summand += (1 - fFunction(3, x, k - x, l) - fFunction(3, x, l, k - x) - fFunction(3, k - x, l, x)) * wFunction(m, n, x, k, T_2);
                            }
                            term2 = (1 - fSmallFunction(k, m, n)) * summand;
                        }
                        cout << term1 * term2 << " " << m << " " << n << " " << k << " " << l << endl;
                        value += term1 * term2;
                    }
                }
        }
    }
    }
    else{
        for(int m = 1; m <= r; m++){
            for(int n = 1; n <= s; n++){
                    for(int l = 1; l <= q; l++){
                            long double term1, term2;
                            term1 = gFunction(r, m, T_3) * gFunction(s, n, T_3) * gFunction(q, l, T_3);
                            term2 = 1 - fFunction(3, m, n, l) - fFunction(3, m, l, n) - fFunction(3, n, l, m);
                            value += term1 * term2;
                            cout << term1 * term2 << " " << m << " " << n << " " << l << endl;
                            //if(term1 * term2 != 0 && k > 1 && l > 1) cout << m << " " << n << " " << k << " " << l << " " << setprecision(15) << term1 * term2 << endl;
                    }
            }
        }
    }
    return value;
}

// Reciprocal monophyly of three species.  Correct.
long double myresult(int r, int s, int q, double T_1, double T_2, double T_3, double T_4){
    long double value = 0;
    for(int m = 1; m <= r; m++){
     for(int n = 1; n <= s; n++){
      for(int z = 1; z <= q; z++){
       for(int x = 0; x <= m; x++){
        for(int y = 0; y <= n; y++){
         for(int m_I = 0; m_I <= 1; m_I++){
             long double gterm, k1, k2;
             gterm = gFunction(r, m, T_2) * gFunction(s, n, T_1) * gFunction(m + n, x + y + m_I, T_3) * gFunction(q, z, T_4);
             if(x > 0 && y > 0 && m_I == 0 && m >= x && n >= y){
                 //k2 = iFunction(m, x) * iFunction(n, y) * wSub2(m - x, n - y) / iFunction(m + n, x + y);  //standard form
                 k2 = biCo(x+y, x) * biCo(m-1, x-1) * biCo(n-1, y-1) / biCo(m+n-1, x+y-1) / biCo(m+n, m); //binomial coefficient form
             }
             else if(x == 0 && y == 0 && m_I == 1){
                 //k2 = iFunction(m, 1) * iFunction(n, 1) * wSub2(m - 1, n - 1) * iFunction(2, 1) / iFunction(m + n, 1); // standard form
                 k2 = 2 / (biCo(m+n, m) * (m + n - 1)); // binomial coefficient form
             }
             else{
                 k2 = 0;
             }
             if(x > 0 && y > 0 && m_I == 0){
                 long double num = 0;
//                 k1 = iFunction(x, 1) * iFunction(y, 1) * iFunction(z, 1) * wSub3(x-1, y-1, z-1) * iFunction(3, 1) / iFunction(x+y+z, 1); //standard form of simple version
                 //k1 = 12 / (biCo(x+y+z, z) * biCo(x+y, x) * (x + y + z - 1) * (x + y + z - 2)); //binomial coefficient form of simple version
                 //standard form of complicated version
                 for(int c = 1; c <= z; c++){
                    num += iFunction(x, 1) * iFunction(y, 1) * iFunction(z, c) * wSub3(x-1, y-1, z-c) * iFunction(c, 1);
                 }
                 for(int c = 1; c <= x; c++){
                    num += iFunction(y, 1) * iFunction(z, 1) * iFunction(x, c) * wSub3(y-1, z-1, x-c) * iFunction(c, 1);
                 }
                 for(int c = 1; c <= y; c++){
                    num += iFunction(x, 1) * iFunction(z, 1) * iFunction(y, c) * wSub3(x-1, z-1, y-c) * iFunction(c, 1);
                 }
                 k1 = num / iFunction(x+y+z, 1);
          }
                 //binomial coeffienct form of complicated version.  Note that these terms are the same only in the three-species case.  Currently wrong.
//                 k1 = 0;
//                 for(int c = 1; c <= z; c++){
//                    k1 += 4 / (biCo(x+y+z, z) * biCo(x+y, x) * (fallingFact(x+y+z-1, c+1)));
//                 }
//                 for(int c = 1; c <= x; c++){
//                    k1 += 4 / (biCo(x+y+z, z) * biCo(x+y, x) * (fallingFact(x+y+z-1, c+1)));
//                 }
//                 for(int c = 1; c <= y; c++){
//                    k1 += 4 / (biCo(x+y+z, z) * biCo(x+y, x) * (fallingFact(x+y+z-1, c+1)));
//                 }
//             }
             else if(x == 0 && y == 0 && m_I == 1){
                 k1 = 1.0 / biCo(z+1, 2);
                 //k1 = iFunction(z, 1) * iFunction(1, 1) * wSub2(z-1, 0) * iFunction(2, 1) / iFunction(z+1, 1);
             }
             else{
                 k1 = 0;
             }
             value += gterm * k1 * k2;
             if(gterm * k1 * k2 != 0 && (x+y+m_I > 1) && z > 1 && x == r && y == s && z == q) cout << m << " " << n << " " << x + y + m_I << " " << z << " "  << gterm << " " << k1 * iFunction(x+y+z, 1) << " " << endl;
         }
        }
       }
      }
     }
    }
    return value;
}

// Combinatorial term C(w, x, y, z)
long double biguglyfour(int w, int x, int y, int z){
    long double num = 0;
    for(int c_1 = 1; c_1 <= y; c_1++){
     for(int c_2 = 1; c_2 <= z; c_2++){
      for(int c_3 = 1; c_3 <= c_2; c_3++){
            num += iFunction(w, 1)*iFunction(x,1)*iFunction(y, c_1)*iFunction(z, c_2)*wSub4(w-1, x-1, y-c_1, z-c_2)*iFunction(c_1, 1)*iFunction(c_2, c_3)*wSub2(c_1-1, c_2-c_3)*iFunction(c_3, 1);
      }
     }
    }
    for(int c_1 = 1; c_1 <= z; c_1++){
     for(int c_2 = 1; c_2 <= y; c_2++){
      for(int c_3 = 1; c_3 <= c_2; c_3++){
            num += iFunction(w, 1)*iFunction(x,1)*iFunction(y, c_2)*iFunction(z, c_1)*wSub4(w-1, x-1, y-c_2, z-c_1)*iFunction(c_1, 1)*iFunction(c_2, c_3)*wSub2(c_1-1, c_2-c_3)*iFunction(c_3, 1);
      }
     }
    }
    for(int c_1 = 1; c_1 <= y; c_1++){
     for(int c_2 = 1; c_2 <= z; c_2++){
     num += iFunction(w, 1)*iFunction(x, 1)*iFunction(y, c_1)*iFunction(z, c_2)*wSub4(w-1, x-1, y-c_1, z-c_2)*iFunction(c_1, 1)*iFunction(c_2, 1)*wSub2(c_1-1, c_2-1);
     }
    }
    for(int c_1 = 1; c_1 <= x; c_1++){
     for(int c_2 = 1; c_2 <= z; c_2++){
      for(int c_3 = 1; c_3 <= c_2; c_3++){
            num += iFunction(w, 1)*iFunction(x,c_1)*iFunction(y, 1)*iFunction(z, c_2)*wSub4(w-1, x-c_1, y-1, z-c_2)*iFunction(c_1, 1)*iFunction(c_2, c_3)*wSub2(c_1-1, c_2-c_3)*iFunction(c_3, 1);
      }
     }
    }
     for(int c_1 = 1; c_1 <= z; c_1++){
      for(int c_2 = 1; c_2 <= x; c_2++){
       for(int c_3 = 1; c_3 <= c_2; c_3++){
             num += iFunction(w, 1)*iFunction(x, c_2)*iFunction(y, 1)*iFunction(z, c_1)*wSub4(w-1, x-c_2, y-1, z-c_1)*iFunction(c_1, 1)*iFunction(c_2, c_3)*wSub2(c_1-1, c_2-c_3)*iFunction(c_3, 1);
       }
      }
     }
    for(int c_1 = 1; c_1 <= x; c_1++){
     for(int c_2 = 1; c_2 <= z; c_2++){
     num += iFunction(w, 1)*iFunction(x, c_1)*iFunction(y, 1)*iFunction(z, c_2)*wSub4(w-1, x-c_1, y-1, z-c_2)*iFunction(c_1, 1)*iFunction(c_2, 1)*wSub2(c_1-1, c_2-1);
     }
    }
    for(int c_1 = 1; c_1 <= x; c_1++){
     for(int c_2 = 1; c_2 <= y; c_2++){
      for(int c_3 = 1; c_3 <= c_2; c_3++){
            num += iFunction(w, 1)*iFunction(x, c_1)*iFunction(y, c_2)*iFunction(z, 1)*wSub4(w-1, x-c_1, y-c_2, z-1)*iFunction(c_1, 1)*iFunction(c_2, c_3)*wSub2(c_1-1, c_2-c_3)*iFunction(c_3, 1);
      }
     }
    }
    for(int c_1 = 1; c_1 <= y; c_1++){
     for(int c_2 = 1; c_2 <= x; c_2++){
      for(int c_3 = 1; c_3 <= c_2; c_3++){
            num += iFunction(w, 1)*iFunction(x, c_2)*iFunction(y, c_1)*iFunction(z, 1)*wSub4(w-1, x-c_2, y-c_1, z-1)*iFunction(c_1, 1)*iFunction(c_2, c_3)*wSub2(c_1-1, c_2-c_3)*iFunction(c_3, 1);
      }
     }
    }
    for(int c_1 = 1; c_1 <= x; c_1++){
     for(int c_2 = 1; c_2 <= y; c_2++){
     num += iFunction(w, 1)*iFunction(x, c_1)*iFunction(y, c_2)*iFunction(z, 1)*wSub4(w-1, x-c_1, y-c_2, z-1)*iFunction(c_1, 1)*iFunction(c_2, 1)*wSub2(c_1-1, c_2-1);
     }
    }
    for(int c_1 = 1; c_1 <= w; c_1++){
     for(int c_2 = 1; c_2 <= z; c_2++){
      for(int c_3 = 1; c_3 <= c_2; c_3++){
            num += iFunction(w, c_1)*iFunction(x, 1)*iFunction(y, 1)*iFunction(z, c_2)*wSub4(w-c_1, x-1, y-1, z-c_2)*iFunction(c_1, 1)*iFunction(c_2, c_3)*wSub2(c_1-1, c_2-c_3)*iFunction(c_3, 1);
      }
     }
    }
    for(int c_1 = 1; c_1 <= z; c_1++){
     for(int c_2 = 1; c_2 <= w; c_2++){
      for(int c_3 = 1; c_3 <= c_2; c_3++){
            num += iFunction(w, c_2)*iFunction(x, 1)*iFunction(y, 1)*iFunction(z, c_1)*wSub4(w-c_2, x-1, y-1, z-c_1)*iFunction(c_1, 1)*iFunction(c_2, c_3)*wSub2(c_1-1, c_2-c_3)*iFunction(c_3, 1);
      }
     }
    }
    for(int c_1 = 1; c_1 <= w; c_1++){
     for(int c_2 = 1; c_2 <= z; c_2++){
     num += iFunction(w, c_1)*iFunction(x, 1)*iFunction(y, 1)*iFunction(z, c_2)*wSub4(w-c_1, x-1, y-1, z-c_2)*iFunction(c_1, 1)*iFunction(c_2, 1)*wSub2(c_1-1, c_2-1);
     }
    }
    for(int c_1 = 1; c_1 <= w; c_1++){
     for(int c_2 = 1; c_2 <= y; c_2++){
      for(int c_3 = 1; c_3 <= c_2; c_3++){
            num += iFunction(w, c_1)*iFunction(x, 1)*iFunction(y, c_2)*iFunction(z, 1)*wSub4(w-c_1, x-1, y-c_2, z-1)*iFunction(c_1, 1)*iFunction(c_2, c_3)*wSub2(c_1 - 1, c_2 - c_3)*iFunction(c_3, 1);
      }
     }
    }
    for(int c_1 = 1; c_1 <= y; c_1++){
     for(int c_2 = 1; c_2 <= w; c_2++){
      for(int c_3 = 1; c_3 <= c_2; c_3++){
            num += iFunction(w, c_2)*iFunction(x, 1)*iFunction(y, c_1)*iFunction(z, 1)*wSub4(w-c_2, x-1, y-c_1, z-1)*iFunction(c_1, 1)*iFunction(c_2, c_3)*wSub2(c_1 - 1, c_2 - c_3)*iFunction(c_3, 1);
      }
     }
    }
    for(int c_1 = 1; c_1 <= w; c_1++){
     for(int c_2 = 1; c_2 <= y; c_2++){
     num += iFunction(w, c_1)*iFunction(x, 1)*iFunction(y, c_2)*iFunction(z, 1)*wSub4(w-c_1, x-1, y-c_2, z-1)*iFunction(c_1, 1)*iFunction(c_2, 1)*wSub2(c_1-1, c_2-1);
     }
    }
    for(int c_1 = 1; c_1 <= w; c_1++){
     for(int c_2 = 1; c_2 <= x; c_2++){
      for(int c_3 = 1; c_3 <= c_2; c_3++){
            num += iFunction(w, c_1)*iFunction(x, c_2)*iFunction(y, 1)*iFunction(z, 1)*wSub4(w-c_1, x-c_2, y-1, z-1)*iFunction(c_1, 1)*iFunction(c_2, c_3)*wSub2(c_1-1, c_2-c_3)*iFunction(c_3, 1);
      }
     }
    }
    for(int c_1 = 1; c_1 <= x; c_1++){
     for(int c_2 = 1; c_2 <= w; c_2++){
      for(int c_3 = 1; c_3 <= c_2; c_3++){
            num += iFunction(w, c_2)*iFunction(x, c_1)*iFunction(y, 1)*iFunction(z, 1)*wSub4(w-c_2, x-c_1, y-1, z-1)*iFunction(c_1, 1)*iFunction(c_2, c_3)*wSub2(c_1-1, c_2-c_3)*iFunction(c_3, 1);
      }
     }
    }
    for(int c_1 = 1; c_1 <= w; c_1++){
     for(int c_2 = 1; c_2 <= x; c_2++){
     num += iFunction(w, c_1)*iFunction(x, c_2)*iFunction(y, 1)*iFunction(z, 1)*wSub4(w-c_1, x-c_2, y-1, z-1)*iFunction(c_1, 1)*iFunction(c_2, 1)*wSub2(c_1-1, c_2-1);
     }
    }
    long double k1 = num / (iFunction(w+x+y+z, 1));
    return k1;
}

// Combinatorial term C(0, x, y, z)
long double biguglythree(int x, int y, int z){
    long double num = 0;
    //k1 = iFunction(x, 1) * iFunction(y, 1) * iFunction(z, 1) * wSub3(x-1, y-1, z-1) * iFunction(3, 1) / iFunction(x+y+z, 1); //standard form of simple version
    //k1 = 12 / (biCo(x+y+z, z) * biCo(x+y, x) * (x + y + z - 1) * (x + y + z - 2)); //binomial coefficient form of simple version
    //standard form of complicated version
    for(int c = 1; c <= z; c++){
       num += iFunction(x, 1) * iFunction(y, 1) * iFunction(z, c) * wSub3(x-1, y-1, z-c) * iFunction(c, 1);
    }
    for(int c = 1; c <= x; c++){
       num += iFunction(y, 1) * iFunction(z, 1) * iFunction(x, c) * wSub3(y-1, z-1, x-c) * iFunction(c, 1);
    }
    for(int c = 1; c <= y; c++){
       num += iFunction(x, 1) * iFunction(z, 1) * iFunction(y, c) * wSub3(x-1, z-1, y-c) * iFunction(c, 1);
    }
    long double k1 = num / iFunction(x+y+z, 1);
    return k1;
}

// Draw from g function as a pdf
int gPdfDraw(int input, double T, Vector<double>& cointosses){
    //cout <<  "input " << input << endl;
    double cointoss = cointosses.get(0);
    cointosses.remove(0);
    //cout << cointoss << endl;
    long double currval = 0;
    long double nextval = 0;
    for(int i = input; i > 0; i--){
        long double interval = gFunction(input, i, T);
        nextval = currval + interval;
        if(cointoss <= nextval){
            if(i > input) error("I too big");
            return i;
        }
        else{
            currval = nextval;
        }
    }
}

// Coalescent simulation of single-node reciprocal monophyly for three species
bool threespcoalsim(int l, int m, int n, int m_1, int output, int& w, int& x, int& y, int& m_2, Vector<double>& cointosses){
    double wwcoal, xxcoal, yycoal, mmcoal, wxcoal, wycoal, xycoal, wmcoal, xmcoal, ymcoal;
    int wcurr, xcurr, ycurr, mcurr;
    wcurr = l;
    xcurr = m;
    ycurr = n;
    mcurr = m_1;
    int present = wcurr + xcurr + ycurr + mcurr;
    bool OK = true;
    while(present > output){
        double denom = biCo(present, 2);
        if(wcurr > 1){
            wwcoal = biCo(wcurr, 2) / denom;
        }
        else(wwcoal = 0);
        if(xcurr > 1){
            xxcoal = biCo(xcurr, 2) / denom;
        }
        else(xxcoal = 0);
        if(ycurr > 1){
            yycoal = biCo(ycurr, 2) / denom;
        }
        else(yycoal = 0);
        if(mcurr > 1){
            mmcoal = biCo(mcurr, 2) / denom;
        }
        else(mmcoal = 0);
        wxcoal = wcurr * xcurr / denom;
        wycoal = wcurr * ycurr / denom;
        xycoal = xcurr * ycurr / denom;
        wmcoal = wcurr * mcurr / denom;
        xmcoal = xcurr * mcurr / denom;
        ymcoal = ycurr * mcurr / denom;
        double stack = wwcoal + xxcoal + yycoal + mmcoal + wxcoal + wycoal + xycoal + wmcoal + xmcoal + ymcoal;
        //cout << wcurr << " " << xcurr << " " << ycurr << " " << mcurr << endl;
        if(stack != 1.0) error("3sp root stack != 1");
        else{
            double cointoss = cointosses.get(0);
            cointosses.remove(0);
            if(cointoss <= wwcoal){
                wcurr--;
            }
            else if(cointoss <= wwcoal + xxcoal){
                xcurr--;
            }
            else if(cointoss <= wwcoal + xxcoal + yycoal){
                ycurr--;
            }
            else if(cointoss <= wwcoal + xxcoal + yycoal + mmcoal){
                mcurr--;
            }
            else if(cointoss <= wwcoal + xxcoal + yycoal + mmcoal + wxcoal){
                if(wcurr == 1 && xcurr == 1){
                    wcurr--;
                    xcurr--;
                    mcurr++;
                }
                else{
                    OK = false;
                    return OK;
                }
            }
            else if(cointoss <= wwcoal + xxcoal + yycoal + mmcoal + wxcoal + wycoal){
                if(wcurr == 1 && ycurr == 1){
                    wcurr--;
                    ycurr--;
                    mcurr++;
                }
                else{
                    OK = false;
                    return OK;
                }
            }
            else if(cointoss <= wwcoal + xxcoal + yycoal + mmcoal + wxcoal + wycoal + xycoal){
                if(xcurr == 1 && ycurr == 1){
                    xcurr--;
                    ycurr--;
                    mcurr++;
                }
                else{
                    OK = false;
                    return OK;
                }
            }
            else if(cointoss <= wwcoal + xxcoal + yycoal + mmcoal + wxcoal + wycoal +  xycoal + wmcoal){
                if(wcurr == 1){
                    wcurr--;
                }
                else{
                    OK = false;
                    return OK;
                }
            }
            else if(cointoss <= wwcoal + xxcoal + yycoal + mmcoal + wxcoal + wycoal + xycoal + wmcoal + xmcoal){
                if(xcurr == 1){
                    xcurr--;
                }
                else{
                    OK = false;
                    return OK;
                }
            }
            else if(cointoss <= wwcoal + xxcoal + yycoal + mmcoal + wxcoal + wycoal + xycoal + wmcoal + xmcoal + ymcoal){
                if(ycurr == 1){
                    ycurr--;
                }
                else{
                    OK = false;
                    return OK;
                }
            }
            else{
                error("3sp cointoss out of bounds");
            }
        }
        //cout << wcurr << " " << xcurr << " " << ycurr << " " << mcurr << endl;
        present = wcurr + xcurr + ycurr + mcurr;
    }
    w = wcurr;
    x = xcurr;
    y = ycurr;
    m_2 = mcurr;
    return OK;
}


// Coalescent simulation of single-node four-species reciprocal monophyly.  Note: this function assumes you are at the root
bool fourspcoalsim(int w, int x, int y, int z, int m_L, int m_R, Vector<double>& cointosses){
    double wwcoal, xxcoal, yycoal, zzcoal, mmcoal, wxcoal, wycoal, wzcoal, xycoal, xzcoal, yzcoal, wmcoal, xmcoal, ymcoal, zmcoal;
    int wcurr, xcurr, ycurr, zcurr, mcurr;
    wcurr = w;
    xcurr = x;
    ycurr = y;
    zcurr = z;
    mcurr = m_L + m_R;
    int present = wcurr + xcurr + ycurr + zcurr + mcurr;
    bool OK = true;
    while(present > 1){
        double denom = biCo(present, 2);
        if(wcurr > 1){
            wwcoal = biCo(wcurr, 2) / denom;
        }
        else(wwcoal = 0);
        if(xcurr > 1){
            xxcoal = biCo(xcurr, 2) / denom;
        }
        else(xxcoal = 0);
        if(ycurr > 1){
            yycoal = biCo(ycurr, 2) / denom;
        }
        else(yycoal = 0);
        if(zcurr > 1){
            zzcoal = biCo(zcurr, 2) / denom;
        }
        else(zzcoal = 0);
        if(mcurr > 1){
            mmcoal = biCo(mcurr, 2) / denom;
        }
        else(mmcoal = 0);
        wxcoal = wcurr * xcurr / denom;
        wycoal = wcurr * ycurr / denom;
        wzcoal = wcurr * zcurr / denom;
        xycoal = xcurr * ycurr / denom;
        xzcoal = xcurr * zcurr / denom;
        yzcoal = ycurr * zcurr / denom;
        wmcoal = wcurr * mcurr / denom;
        xmcoal = xcurr * mcurr / denom;
        ymcoal = ycurr * mcurr / denom;
        zmcoal = zcurr * mcurr / denom;
        double stack = wwcoal + xxcoal + yycoal + zzcoal + mmcoal + wxcoal + wycoal + wzcoal + xycoal + xzcoal + yzcoal + wmcoal + xmcoal + ymcoal + zmcoal;
        //cout << wcurr << " " << xcurr << " " << ycurr << " " << zcurr << " " << mcurr << endl;
        //if(stack != 1.0) {
            //cout << "STACK " << stack << endl;
            //error("4sp root stack != 1");
        //}
        //else{
            double cointoss = cointosses.get(0);
            cointosses.remove(0);
            //cout << cointoss << endl;
            if(cointoss < wwcoal){
                wcurr--;
            }
            else if(cointoss < wwcoal + xxcoal){
                xcurr--;
            }
            else if(cointoss < wwcoal + xxcoal + yycoal){
                ycurr--;
            }
            else if(cointoss < wwcoal + xxcoal + yycoal + zzcoal){
                zcurr--;
            }
            else if(cointoss < wwcoal + xxcoal + yycoal + zzcoal + mmcoal){
                mcurr--;
            }
            else if(cointoss < wwcoal + xxcoal + yycoal + zzcoal + mmcoal + wxcoal){
                if(wcurr == 1 && xcurr == 1){
                    wcurr--;
                    xcurr--;
                    mcurr++;
                }
                else{
                    OK = false;
                    return OK;
                }
            }
            else if(cointoss < wwcoal + xxcoal + yycoal + zzcoal + mmcoal + wxcoal + wycoal){
                if(wcurr == 1 && ycurr == 1){
                    wcurr--;
                    ycurr--;
                    mcurr++;
                }
                else{
                    OK = false;
                    return OK;
                }
            }
            else if(cointoss < wwcoal + xxcoal + yycoal + zzcoal + mmcoal + wxcoal + wycoal + wzcoal){
                if(wcurr == 1 && zcurr == 1){
                    wcurr--;
                    zcurr--;
                    mcurr++;
                }
                else{
                    OK = false;
                    return OK;
                }
            }
            else if(cointoss < wwcoal + xxcoal + yycoal + zzcoal + mmcoal + wxcoal + wycoal + wzcoal + xycoal){
                if(xcurr == 1 && ycurr == 1){
                    xcurr--;
                    ycurr--;
                    mcurr++;
                }
                else{
                    OK = false;
                    return OK;
                }
            }
            else if(cointoss < wwcoal + xxcoal + yycoal + zzcoal + mmcoal + wxcoal + wycoal + wzcoal + xycoal + xzcoal){
                if(xcurr == 1 && zcurr == 1){
                    xcurr--;
                    zcurr--;
                    mcurr++;
                }
                else{
                    OK = false;
                    return OK;
                }
            }
            else if(cointoss < wwcoal + xxcoal + yycoal + zzcoal + mmcoal + wxcoal + wycoal + wzcoal + xycoal + xzcoal + yzcoal){
                if(ycurr == 1 && zcurr == 1){
                    ycurr--;
                    zcurr--;
                    mcurr++;
                }
                else{
                    OK = false;
                    return OK;
                }
            }
            else if(cointoss < wwcoal + xxcoal + yycoal + zzcoal + mmcoal + wxcoal + wycoal + wzcoal + xycoal + xzcoal + yzcoal + wmcoal){
                if(wcurr == 1){
                    wcurr--;
                }
                else{
                    OK = false;
                    break;
                }
            }
            else if(cointoss < wwcoal + xxcoal + yycoal + zzcoal + mmcoal + wxcoal + wycoal + wzcoal + xycoal + xzcoal + yzcoal + wmcoal + xmcoal){
                if(xcurr == 1){
                    xcurr--;
                }
                else{
                    OK = false;
                    return OK;
                }
            }
            else if(cointoss < wwcoal + xxcoal + yycoal + zzcoal + mmcoal + wxcoal + wycoal + wzcoal + xycoal + xzcoal + yzcoal + wmcoal + xmcoal + ymcoal){
                if(ycurr == 1){
                    ycurr--;
                }
                else{
                    OK = false;
                    return OK;
                }
            }
            else if(cointoss < wwcoal + xxcoal + yycoal + zzcoal + mmcoal + wxcoal + wycoal + wzcoal + xycoal + xzcoal + yzcoal + wmcoal + xmcoal + ymcoal + zmcoal){
                if(zcurr == 1){
                    zcurr--;
                }
                else{
                    OK = false;
                    return OK;
                }
            }
            else{
                error("4sp cointoss out of bounds");
            }
        //}
        present = wcurr + xcurr + ycurr + zcurr + mcurr;
    }
    return OK;
}

// Coalescence simulation of single-node two-species reciprocal monophyly
bool twospcoalsim(int k, int l, int output, int& w, int& x, int& m, Vector<double>& cointosses){
    double kkcoal, llcoal, klcoal;
    int kcurr, lcurr, mcurr, total, present;
    kcurr = k;
    lcurr = l;
    mcurr = 0;
    total = kcurr + lcurr + mcurr;
    present = total;
    bool OK = true;
    while(present > output){
        double denom = biCo(present, 2);
        if(kcurr > 1){
            kkcoal = biCo(kcurr, 2) / denom;
        }
        else kkcoal = 0;
        if(lcurr > 1){
            llcoal = biCo(lcurr, 2) / denom;
        }
        else llcoal = 0;
        klcoal = kcurr * lcurr / denom;
        double stack = kkcoal + llcoal + klcoal;
        //cout << kcurr << " " << lcurr << " " << mcurr << endl;
        if(stack != 1.0) error("2sp stack != 1");
        else{
            double cointoss = cointosses.get(0);
            cointosses.remove(0);
            if(cointoss <= kkcoal){
                kcurr--;
            }
            else if(cointoss <= kkcoal + llcoal){
                lcurr--;
            }
            else if(cointoss <= kkcoal + llcoal + klcoal){
                if(kcurr == 1 && lcurr == 1){
                    kcurr--;
                    lcurr--;
                    mcurr++;
                }
                else{
                    OK = false;
                    return OK;
                }
            }
            else(error("2sp cointoss out of bounds"));
            present = kcurr + lcurr + mcurr;
        }
    }
    w = kcurr;
    x = lcurr;
    m = mcurr;
    return OK;
}

// full simulation of four-species reciprocal monophyly
bool simulation(int p, int q, int r, int s,double T1, double T2, double T3, double T4, double T5, double T6, bool symmetric, Vector<double>& cointosses){
    // WLOG pq coalesces first in the species tree
    srand(time(NULL));
    if(symmetric){
        // First choose an outcome of the leaf branches: choose a k, l, m, n.  RM is never violated in this stage.
        int k, l, m, n;
        k = gPdfDraw(p, T4, cointosses);
        l = gPdfDraw(q, T3, cointosses);
        m = gPdfDraw(r, T2, cointosses);
        n = gPdfDraw(s, T1, cointosses);
        // Now choose outcomes of the left and right branches into the root (k+l --> w+x+m_L and m+n --> y+z+m_R)
        int wxm_L, yzm_R, w, x, m_L, y, z, m_R;
        wxm_L = gPdfDraw(k+l, T5, cointosses);
        yzm_R = gPdfDraw(m+n, T6, cointosses);
        // Now perform one instance of the coalescence, quitting if RM is violated
        bool leftbranch, rightbranch;
        //First, the left branch
        leftbranch = twospcoalsim(k, l, wxm_L, w, x, m_L, cointosses);
        if(!leftbranch) return false;
        else{
            // If that works, the right branch
            rightbranch = twospcoalsim(m, n, yzm_R, y, z, m_R, cointosses);
            if(!rightbranch) return false;
            else{
                // If that works, then we do the root
                bool rootbool = fourspcoalsim(w, x, y, z, m_L, m_R, cointosses);
                return rootbool;
            }
        }
    }
    // Now the asymmetric case
    else{
        // First choose an outcome of the leaf branches: choose a k1, k2, n, z.  RM is never violated in this stage.
        int k1, k2, n, z;
        k1 = gPdfDraw(p, T4, cointosses);
        k2 = gPdfDraw(q, T3, cointosses);
        n = gPdfDraw(r, T2, cointosses);
        z = gPdfDraw(s, T1, cointosses);
        //cout << "chosen k1, k2, n, z: " << k1 << " " << k2 << " " << n << " " << z << endl;
        // Now choose the outcome of the first internal branch (k1 + k2 --> lmm_1)
        int lmm_1, l, m, m_1;
        lmm_1 = gPdfDraw(k1+k2, T5, cointosses);
        //cout << "chosen lmm_1: " << lmm_1 << endl;
        //Now perform one instance of the coalescent for this branch, quitting if RM is violated
        bool firstbranch, secondbranch;
        firstbranch = twospcoalsim(k1, k2, lmm_1, l, m, m_1, cointosses);
        //cout << "firstbranch is: " << firstbranch << endl;
        if(!firstbranch) {
            //cout << k1 << " " << k2 << " " << l << " " << m << " " << n << " " << m_1 << endl;
            return false;
        }
        else{
            // If that works, do the next branch.
            // Choose the outcome of the next internal branch (lmm_1 + n --> wxym_2)
            int wxym_2, w, x, y, m_2;
            wxym_2 = gPdfDraw(lmm_1 + n, T6, cointosses);
            //cout << "chosen wxym_2: " << wxym_2 << endl;
            //Now perform one instance of the coalescent for this branch, quitting if RM is violated
            secondbranch = threespcoalsim(l, m, n, m_1, wxym_2, w, x, y, m_2, cointosses);
            //cout << "secondbranch is: " << secondbranch << endl;
            if(!secondbranch) {
                //cout << l << " " << m << " " << n << " " << m_1 << " " << wxym_2 << " " << w << " " << x << " " << y << " " << m_2 << endl;
                return false;
            }
            else{
                // If that works, do the root.  m_1 is always 0 in this case just because of how I defined the function
                bool rootbool = fourspcoalsim(w, x, y, z, 0, m_2, cointosses);
                //if(!rootbool) cout << k1 << " " << k2 << " " << l << " " << m << " " << n << " " << m_1 << " " << w << " " << x << " " << y << " "  << z << " " << m_2 << endl;
                return rootbool;
            }
        }

    }
}

bool simulation3(int p, int q, int r, double T1, double T2, double T3, double T4, Vector<double>& cointosses){
    // WLOG pq coalesces first in the species tree
    srand(time(NULL));
        // First choose an outcome of the leaf branches: choose an m, n, l.  RM is never violated in this stage.
        int m, n, z;
        m = gPdfDraw(p, T1, cointosses);
        n = gPdfDraw(q, T3, cointosses);
        z = gPdfDraw(r, T4, cointosses);
        //cout << "chosen k1, k2, n, z: " << k1 << " " << k2 << " " << n << " " << z << endl;
        // Now choose the outcome of the first internal branch (m + n --> klm_1)
        int klm_1, k, l, m_1;
        klm_1 = gPdfDraw(m+n, T2, cointosses);
        //cout << "chosen lmm_1: " << lmm_1 << endl;
        //Now perform one instance of the coalescent for this branch, quitting if RM is violated
        bool firstbranch, secondbranch;
        firstbranch = twospcoalsim(m, n, klm_1, k, l, m_1, cointosses);
        //cout << "firstbranch is: " << firstbranch << endl;
        if(!firstbranch) {
            //cout << k1 << " " << k2 << " " << l << " " << m << " " << n << " " << m_1 << endl;
            return false;
        }
            else{
                // If that works, do the root.  m_1 is always 0 in this case just because of how I defined the function
            int w, x, y, m_2;
            bool rootbool = threespcoalsim(k, l, q, m_1, 1, w, x, y, m_2, cointosses);
                //if(!rootbool) cout << k1 << " " << k2 << " " << l << " " << m << " " << n << " " << m_1 << " " << w << " " << x << " " << y << " "  << z << " " << m_2 << endl;
                return rootbool;
            }
}

long double myresult4(int p, int q, int r, int s, double T_1, double T_2, double T_3, double T_4, double T_5, double T_6, bool symmetric){
    long double value = 0;
    long double gterm, k1, k2, k3, num;
    // symmetric topology
    if(symmetric){
        for(int k = 1; k <= p; k++){
         for(int l = 1; l <= q; l++){
          for(int m = 1; m <= r; m++){
           for(int n = 1; n <= s; n++){
            for(int w = 0; w <= k; w++){
             for(int x = 0; x <= l; x++){
              for(int y = 0; y <= m; y++){
               for(int z = 0; z <= n; z++){
                for(int m_R = 0; m_R <= 1; m_R++){
                 for(int m_L = 0; m_L <= 1; m_L++){
                     gterm = gFunction(p, k, T_4) * gFunction(q, l, T_3) * gFunction(r, m, T_2) * gFunction(s, n, T_1) * gFunction(k+l, w+x+m_L, T_5) * gFunction(m+n, y+z+m_R, T_6);
                     if(w == 0 && x == 0 && m_L == 1 && l >= 1 && k >= 1){
                         k2 = iFunction(k, 1) * iFunction(l, 1) * wSub2(k-1, l-1) * iFunction(2, 1) / iFunction(k+l, 1);
                     }
                     else if(w > 0 && x > 0 && m_L == 0 && l >= x && k >= w){
                         k2 = iFunction(k, w) * iFunction(l, x) * wSub2(k-w, l-x) / iFunction(k+l, w+x);
                     }
                     else{
                         k2 = 0;
                     }
                     if(y == 0 && z == 0 && m_R == 1 && m >= 1 && n >= 1){
                         k3 = iFunction(m, 1) * iFunction(n, 1) * wSub2(m-1, n-1) * iFunction(2, 1) / iFunction(m+n, 1);
                     }
                     else if(y > 0 && z > 0 && m_R == 0 && m >= y && n >= z){
                         k3 = iFunction(m, y) * iFunction(n, z) * wSub2(m-y, n-z) / iFunction(m+n, y+z);
                     }
                     else{
                         k3 = 0;
                     }
                     if(m_L == 1 && w == 0 && x == 0 && y >= 1 && z >= 1){
                         k1 = iFunction(y, 1) * iFunction(z, 1) * wSub2(y-1, z-1) * iFunction(3, 1) / iFunction(y+z+1, 1);
                     }
                     else if(m_R == 1 && y == 0 && z == 0 && w >= 1 && x >= 1){
                         k1 = iFunction(w, 1) * iFunction(x, 1) * wSub2(w-1, x-1) * iFunction(3, 1) / iFunction(w+x+1, 1);
                     }
                     else if(w > 0 && x > 0 && y > 0 && z > 0 && m_R == 0 && m_L == 0){
                         k1 = biguglyfour(w, x, y, z);
                     }
                     else if(w == 0 && x == 0 && y == 0 && z == 0 && m_L == 1 && m_R == 1){
                         k1 = 1;
                     }
                     else{
                         k1 = 0;
                     }
                     //if(k1 != k1) cout << "k1 " << w << " " << x  << " " << y << " " << z << " " << m_L << " " << m_R << endl;
                     //if(k2 != k2) cout << "k2 " << k << " " << l  << " " << w << " " << x << " " << m_L << endl;
                     //if(k3 != k3) cout << "k3 " << m << " " << n  << " " << y << " " << z << " " << m_R << endl;
                     //if(gterm != gterm) cout << "gterm " << k << " " << l << " " << m << " " << n << " " << w << " " << x  << " " << y << " " << z << " " << m_L << " " << m_R << endl;
                     value += gterm * k1 * k2 * k3;
                     //if(gterm > 0) cout << k << " " << l << " " << m << " " << n << " " << w << " " << x  << " " << y << " " << z << " " << m_L << " " << m_R << " " << gterm << " "  << k1 << " " << k2 << " " << k3 << endl;
                    }
                }
               }
              }
             }
            }
           }
          }
         }
        }
    }
    // Asymmetric topology
    else{
        for(int k_2 = 1; k_2 <= q; k_2++){
         for(int k_1 = 1; k_1 <= p; k_1++){
          for(int l = 0; l <= k_1; l++){
           for(int m = 0; m <= k_2; m++){
            for(int n = 1; n <= r; n++){
             for(int z = 1; z <= s; z++){
              for(int y = 0; y <= n; y++){
               for(int w = 0; w <= l; w++){
                for(int x = 0; x <= m; x++){
                 for(int m_1 = 0; m_1 <= 1; m_1++){
                  for(int m_2 = 0; m_2 <= 1; m_2++){
                     gterm = gFunction(p, k_1, T_4) * gFunction(q, k_2, T_3) * gFunction(r, n, T_2) * gFunction(s, z, T_1) * gFunction(k_1 + k_2, l + m + m_1, T_5) * gFunction(l + m + n + m_1, w + x + y + m_2, T_6);
                     if(l > 0 && m > 0 && m_1 == 0 && k_1 >= l && k_2 >= m){
                         k3 = iFunction(k_1, l) * iFunction(k_2, m) * wSub2(k_1 - l, k_2 - m) / iFunction(k_1 + k_2, l+m);
                     }
                     else if(l == 0 && m == 0 && m_1 == 1 && k_1 >= 1 && k_2 >= 1){
                         k3 = iFunction(k_1, 1) * iFunction(k_2, 1) * wSub2(k_1 - 1, k_2 - 1) / iFunction(k_1 + k_2, 1);
                     }
                     else{
                         k3 = 0;
                     }
                     //Now k2, three possible species
                     if(w == 0 && x == 0 && y == 0 && m_2 == 1 && m_1 == 0 && l > 0 && m > 0 && n > 0){
                         k2 = biguglythree(l, m, n);
                     }
                     else if(w > 0 && x == 0 && y == 0 && l > 0 && m > 0 && n > 0 && m_2 == 1 && m_1 == 0){
                         long double num = 0;
                         for(int c = 1; c <= l; c++){
                             num += iFunction(l, c) * iFunction(m, 1) * iFunction(n, 1) * wSub3(l-c, m-1, n-1) * iFunction(c, w);
                         }
                         k2 = num / iFunction(l+m+n, w+1);
                     }
                     else if(x > 0 && w == 0 && y == 0 && l > 0 && m > 0 && n > 0 && m_2 == 1 && m_1 == 0){
                         long double num = 0;
                         for(int c = 1; c <= m; c++){
                             num += iFunction(l, 1) * iFunction(m, c) * iFunction(n, 1) * wSub3(l-1, m-c, n-1) * iFunction(c, x);
                         }
                         k2 = num / iFunction(l+m+n, x+1);
                     }
                     else if(y > 0 && w == 0 && x == 0 && l > 0 && m > 0 && n > 0 && m_2 == 1 && m_1 == 0){
                         long double num = 0;
                         for(int c = 1; c <= n; c++){
                             num += iFunction(l, 1) * iFunction(m, 1) * iFunction(n, c) * wSub3(l-1, m-1, n-c) * iFunction(c, y);
                         }
                         k2 = num / iFunction(l+m+n, y+1);
                     }
                     else if(w > 0 && x > 0 && y > 0 && m_1 == 0 && m_2 == 0 && l > 0 && m > 0 && n > 0){
                         k2 = iFunction(l, w) * iFunction(m, x) * iFunction(n, y) * wSub3(l-w, m-x, n-y) / iFunction(l+m+n, w+x+y);
                     }
                     else if(w == 0 && x == 0 && l == 0 && m == 0 && y > 0 && m_1 == 1 && m_2 == 1 && n > 0){
                         k2 = iFunction(n, y) / iFunction(n+1, y+1);
                     }
                     else if(w == 0 && x == 0 && l == 0 && m == 0 && y == 0 && m_1 == 1 && m_2 == 1 && n > 0){
                         k2 = iFunction(n, 1) / iFunction(n+1, 1);
                     }
                     else{
                         k2 = 0;
                     }
                     //Now k1, the root (four possible species)
                     if(w > 0 && x > 0 && y > 0 && z > 0 && m_2 == 0){
                         k1 = biguglyfour(w, x, y, z);
                     }
                     else if(x == 0 && y == 0 && w > 0 && m_2 == 1 && z > 0){
                         k1 = biguglythree(w, z, m_2);
                     }
                     else if(w == 0 && y == 0 && x > 0 && m_2 == 1 && z > 0){
                         k1 = biguglythree(x, z, m_2);
                     }
                     else if(w == 0 && x == 0 && y > 0 && m_2 == 1 && z > 0){
                         k1 = biguglythree(y, z, m_2);
                     }
                     else if(w == 0 && x == 0 && y == 0 && z > 0 && m_2 == 1){
                         k1 = iFunction(z, 1) / iFunction(z+1, 1);
                     }
                     else{
                         k1 = 0;
                     }
                     value += gterm * k1 * k2 * k3;
                     //if(gterm > 0) cout << k_1 << " " << k_2 << " " << l << " " << m << " " << n << " " << w << " " << x  << " " << y << " " << z << " " << m_1 << " " << m_2 << " " << gterm << " "  << k1 << " " << k2 << " " << k3 << endl;
                    }
                 }
                }
               }
              }
             }
            }
           }
          }
         }
        }
    }
    return value;
}



#endif // ROSENBERG_H
