function [U,Sigma,V,numiter]  = SVT(n,Omega,b,tau,delta,maxiter,tol,EPS)
% [U,Sigma,V,numiter]  = SVT(n,Omega,b,tau,delta,maxiter,tol)
%
% Finds the minimum of tau ||X||_* + .5 || X ||_F^2 
%
% subject to P_Omega(X) = P_Omega(M)
%
% using linear Bregman iterations
%
% Usage:  [U,S,V,numiter]  = SVT(n,Omega,b,delta,maxiter,tol)
%
% Inputs:
%
%   n - size of the matrix X assumed n(1) by n(2). If n is a single integer, it
% is understood that n(1) = n(2). 
%
%   Omega - set of observed entries
%
%   b - data vector of the form M(Omega)
%
%   tau - parameter defining the objective functional 
%
%   delta - step size.  Choose delta less than 2 to be safe but
%       conservative; choose delta closer to n(1)*n(2)/length(Omega)
%       to be riskier (i.e. algorithm may diverge)
%
%   maxiter - maximum number of iterations
%
%   tol - stopping criteria (default: 1e-4)
%
% Outputs: matrix X stored in SVD format X = U*diag(S)*V'
% 
%   U - n1xr left singular vectors 
% 
%   S - rx1 singular values
%
%   V - n2xr right singular vectors 
%
%   numiter - number of iterations to achieve convergence

% Description: 
% Reference:
%
%    Cai, Candes and Shen
%    A singular value thresholding algorithm for matrix completion
%    Submitted for publication, October 2008.
%
% Written by: Emmanuel Candes
% Email: emmanuel@acm.caltech.edu
% Created: October 2008

% Modified: Stephen Becker, March 2009

% VERBOSE = false;
VERBOSE = 1;    % a little bit of output
% VERBOSE = 2;    % even more output

if nargin < 8 || isempty(EPS)
    EPS = false;
end
if nargin < 7 || isempty(tol)
    tol = 1e-4;
end
if nargin < 6 || isempty(maxiter)
    maxiter = 500;
end
    
if length(n) == 1,
    n1 = n(1); n2 = n1;
elseif length(n) == 2,
    n1 = n(1); n2 = n(2);
end
if n1*n2 < 100*100, SMALLSCALE = true; else SMALLSCALE = false; end

m = length(Omega); [temp,indx] = sort(Omega); 
% simpler: sort b also
incre = 5; 
normb = norm(b);

[i, j] = ind2sub([n1,n2], Omega);
if EPS
%     delta = delta/2; tau = 2*tau;
%     y1 = max(b-EPS,0); y2 = max(-b-EPS,0);
    y1 = max(b,0); y2 = max(-b,0);
    Y = sparse(i,j,y1-y2,n1,n2,m);
    normProjM = normest(Y);
    k0 = ceil(tau/(delta*normProjM));
    y1 = k0*delta*y1;
    y2 = k0*delta*y2;
    updateSparse(Y,y1-y2,indx);
else
    Y = sparse(i,j,b,n1,n2,m);
    normProjM = normest(Y);
    k0 = ceil(tau/(delta*normProjM));
    normb = norm(b);
    y = k0*delta*b; % kicking by k0 steps
    updateSparse(Y,y,indx);
end
r = 0;

if VERBOSE==1, fprintf('\nIteration:   '); end
for k = 1:maxiter,
    if VERBOSE==1, fprintf('\b\b\b\b%4d',k);  end
    s = r + 1;
    
    if SMALLSCALE
        [U,Sigma,V] = svd(full(Y));
    else
        OK = 0;
        while ~OK && ( s <= min(n1,n2) )
            [U,Sigma,V] = lansvd(Y,s,'L');
            OK = (Sigma(s,s) <= tau);
            s = s + incre;
        end
    end
   
    sigma = diag(Sigma); r = sum(sigma > tau);
    U = U(:,1:r); V = V(:,1:r); sigma = sigma(1:r) - tau; Sigma = diag(sigma);
    
    x = XonOmega(U*diag(sigma),V,Omega);
    if VERBOSE == 2
        fprintf('iteration %4d, rank is %2d, rel. residual is %.1e\n',k,r,norm(x-b)/normb);
    end
    if (norm(x-b)/normb < tol)
        break
    end
    if (norm(x-b)/normb > 1e5)
        disp('Divergence!');
        break
    end
    
    if EPS
        y1 = max( y1 + delta*( -(x-b) - EPS), 0 );
        y2 = max( y2 + delta*(  (x-b) - EPS), 0 );
        updateSparse(Y,y1-y2,indx);
    else
        y = y + delta*(b-x);
        updateSparse(Y,y,indx);
    end
end

if VERBOSE==1, fprintf('\n'); end
numiter = k;
