% PanelMuH_main     
%
%  Estimate mu, sigma, delta and eta in the IRS public use panel data using the
%  cross-section of growth rates of incomes.
%
%    Basic method: throw away the bottom growth rates (DeltaDrop) and use the 
%        main part of the distribution to recover mu and sigma
%
%  Evaluates a range of alternatives.
%   - PanelMuH-Final.log contains the Final Results
%   - PanelMuH-Details.log contains the details of each alternative.
%
% 09/23/16 -- MuH version
% 10/2/14 -- Updated to make this the main estimation in the Schumpeter paper.
% 5/28/14 -- First version

if exist('PanelMuH-Final.log'); delete('PanelMuH-Final.log'); end;
if exist('PanelMuH-Details.log'); delete('PanelMuH-Details.log'); end;

diary('PanelMuH-Details.log');
fprintf(['PanelMuH_main                 ' date]);
disp ' ';
disp ' ';
help PanelMuH_main

diary('PanelMuH-Final.log');
fprintf(['PanelMuH                 ' date]);
disp ' ';
disp ' ';
help PanelMuH_main

clear;
more off;
warning off;

% IncomeType: 0=Wage+Salaries, 1=Wages/Salaries+EntrepreneurialIncome,  2=Entrepreneurial only
%   pbar    : [0 eta1980] ==> choose to match eta1980. Otherwise if scalar, use that pbar

MainParameters=[
%IncPctile DeltaDrop IncomeType
    95         40       1
    95         40       0
    95         40       2
    90         40       1
    97         40       1
    95         50       1
    ];

IncomeTypeStr={'Income Type: Wages and Salaries',
               'Income Type: Wages and Salaries + Entrepreneurial Income',
               'Income Type: Entrepreneurial Income only'}


Years=(1979:1989)'; % This is first year, Results will display Years+1 = last year
T=length(Years);
K=4; % mutildeH sigmaH delta eta
deltaCD=.02      % Endogenous creative destruction (constant over time)
PermTrans = 1/3  % Fraction of var(growth) that is permanent shock - use for eta
pbar = 1
MuTildePctile = 95  % mutilde is the 95th percentile of the growth rate distribution
etaStart=0.33

NumberAlternatives=size(MainParameters,1);
Results=zeros(T,K+3,NumberAlternatives);
pbarAlt=zeros(NumberAlternatives,1);

for i=1:NumberAlternatives;
    IncPctileCutoff=MainParameters(i,1);
    DeltaDrop=MainParameters(i,2);
    IncomeType=MainParameters(i,3);

    diary('PanelMuH-Details.log');
    disp ' '; disp ' ';
    disp '-------------------------------------------------';
    fprintf(' CASE: IncPctileCutoff = %2.0f   DeltaDrop=%2.0f\n',[IncPctileCutoff DeltaDrop]);
    disp(['   ' IncomeTypeStr{IncomeType+1}]);
    disp '-------------------------------------------------';
    disp ' ';

    Params=zeros(T,K)*NaN; Nobs=zeros(T,1)*NaN;
    for t=1:T;
        [mutildeH sigmaH delta eta n]=panelmuH(Years(t),IncPctileCutoff,MuTildePctile,DeltaDrop,deltaCD,pbar,PermTrans,IncomeType);
        Params(t,:)=[mutildeH sigmaH delta eta]; Nobs(t)=n;
    end;
    mutildeH=Params(:,1);
    sigmaH=Params(:,2);
    delta=Params(:,3);
    eta=Params(:,4);

    diary('PanelMuH-Final.log');
    disp ' '; disp ' ';

    % Alternative: choose pbar s.t. eta in first year equals etaStart
    mm=mutildeH./sigmaH.^2;
    deltabar=delta-deltaCD;
    if deltabar<0.01; 
        disp 'PROBLEM: deltabar<0.01 in PanelMuH.m. Setting deltabar=.01...';
        deltabar=.01
    end;
    zeta0=1/etaStart;
    pbarAlt(i)=.5*sigmaH(1)^2*( (mm(1)+zeta0)^2-mm(1)^2) - deltabar(1);
    squigAlt=-mm+sqrt(mm.^2+2*(deltabar+pbarAlt(i))./sigmaH.^2);
    etaAlt=1./squigAlt;

    disp '-------------------------------------------------';
    disp '                   FINAL RESULTS '
    disp '-------------------------------------------------';
    disp ' ';
    fprintf(' IncPctileCutoff = %2.0f   DeltaDrop=%2.0f    deltaCD=%5.3f\n',[IncPctileCutoff DeltaDrop deltaCD]);
    disp(['     ' IncomeTypeStr{IncomeType+1}]);    
    fprintf('     pbarAlt=%8.4f so that eta(1)=%8.4f\n',[pbarAlt(i) etaStart]);
    cshow(' ',[Years+1 Params etaAlt Nobs],'%7.0f %10.4f %10.4f %10.4f %10.4f %10.4f %10.0f','Year mutildeH sigmaP dltBoth eta etaAlt Nobs');
    Results(:,:,i)=[Years+1 Params etaAlt Nobs];
    
end;
save -v7 PanelMuHResults.mat;


diary off;
