% TransitionDynamicsGuvenen.m
%
%   -- Called from TransitionDynamicsMaster with "CaseName", 
%   -- Considers full set of shocks from GuvenenEta.m, rather than just one shock at a time.
%
%  Simulate the transition dynamics of the inequality model
%  assuming entrepreneurial effort is fixed at its steady state value.
%  That is, we shut off the transition dynamics associated with entreprenuerial effort 
%  choice in the HJB equation (because solving that as well is very hard, while this is easy).
%
%  For the same reason, assume that s (research allocation) jumps to its steady state value.
%  So the only source of transition dynamics is in the evolution of the distribution of x.
%
%  Discretize: 200,000 entrepreneurs, with initial x drawn from initial SS distribution.
%   - Shock the economy (e.g. increase phiH) ==> new SS effort choice e* and new s* (==> delta* and g*)
%   - Let the discretized GBM evolve, so that the distribution of {x} evolves.
%   - Keep track of eta and y:=Y/Lbar and show how they evolve over time.
%
%  This is a rough guide to transition dynamics in the model. Goal is to illustrate two things:
%   1. Transition dynamics are fast for our estimated pbar.
%   2. Observed growth over 10 or 20 years looks different from SS growth bc of the level effect.
%
%  Graphs: show the initial steady-state for 20 years, then the shock hits in 1975.
%     How quickly does eta rise to the new steady state?
%     What does GDP look like?

diarychad(['TransitionDynamicsMaster_' CaseName]);

disp ' '; disp ' ';
disp '==========================================================================';
disp(['                     CaseName: ' CaseName]);
fprintf('                NumGridPoints: %6.0f\n',NumGridPoints);
fprintf('                PL_percentile: %8.5f\n',PL_percentile);
fprintf('                           dt:    1 /%3.0f\n',1/dt);
disp '==========================================================================';
disp ' '; disp ' ';

% Create etaUS -- See FractalInequality.m
load WorldTopInequalityDatabase2016;
T=2015;
yrs_data=(1970:T)';
yy=yrs_data-yr0;
s3=is001(:,USA)./is01(:,USA);
etaUS=1+log10(s3(yy));


% Code from GuvenenEta.m for key case:
load Guvenen;
yr0=1980;
deltaCD=0.06 % 0.02
PermTransVar=1/4; % Fraction of variance that is due to the permanent shock
EndingYear=2006   % Stop there so that the financial crisis does not overly influence smoothing
MyYears=(1981:EndingYear)';
YearsToSmooth=(1981:EndingYear)-yr0;
disp ' ';
disp ' -------- Top 1%/2.5%/5% ---------';
disp ' ';
%Eta01=guveneneta(ProbStayHighGrowth(:,2),GrowthTop1Earners.p99,GrowthTop1Earners.sdH,Delta_e(:,1),deltaCD,MyYears,yr0,PermTransVar,HPsmoother);
[Eta125,SpeedOfConvergence,mutildeH_G,sigmaH_G,pBar_G,delta_e_G,smootheddata]=guveneneta(ProbStayHighGrowth(:,2),GrowthTop1Earners.p97_5,GrowthTop1Earners.sdH,Delta_e(:,1),deltaCD,MyYears,yr0,PermTransVar,HPsmoother,[],YearsToSmooth);

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Time path of inputs: We cannot "solve" for implied SS values of s and g from the SSA or IRS
%   data -- we do not know how to decompose mutildeH into phiH, tau, pbar, etc.
%   Since we've already shown in the TransitionDynamics.m results that these s/g effects are
%   very small, we will shut those off for the SSA/IRS exercises.
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


% Setup parameters
oo=ones(length(MyYears),1);
sigmaL=.10 %2/3*sigmaH_G(1)*oo;
deltabar = delta_e_G-deltaCD;
figname='TransitionDynamicsMaster';


% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Time path of inputs
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Initial conditions and exogenous s
eta0=[]; %0.39; % Initial value to fit...
s_old=.1;
s_new=.1 ;
g_old=.02*100;
g_new=.02*100;
muL_old=0.004;
mutildeL=muL_old*oo-.5*sigmaL.^2;
Year0=1965;
YearT=2050;

% Now run the routine for transitiondynamics:
[eta,gdp,yrs,ytrend]=transitiondynamics(mutildeH_G,mutildeL,sigmaH_G,sigmaL,pBar_G,qbar,deltabar,NumGridPoints,alpha,s_old,s_new,g_old,yrs_data,etaUS,MyYears,Year0,YearT,figname,CaseName,dt,PL_percentile,eta0);
chadtimer

% Save the results for later analysis
eval(['data_' CaseName '=table(yrs,eta,gdp,ytrend);']);
diary off

