% TransitionDynamicsMaster.m
%
%  Master program to call various subprograms for solving for transition dynamics.
%  The fundamental program is transitiondynamics_1shock.m, which uses Ben Moll's
%  techniques. (That is a modification of his fig4b and fig5b.m programs for the
%  GLLM Econometrica 2016 paper.)
%
%  Simulate the transition dynamics of the inequality model
%  assuming entrepreneurial effort is fixed at its steady state value.
%  That is, we shut off the transition dynamics associated with entreprenuerial effort 
%  choice in the HJB equation (because solving that as well is very hard, while this is easy).
%
%  For the same reason, assume that s (research allocation) jumps to its steady state value.
%  So the only source of transition dynamics is in the evolution of the distribution of x.
%
%  Runtime: Around 12 minutes

diarychad('TransitionDynamicsMaster')
clear all;
definecolors;
tic

% Basic Parameters 
NumGridPoints=2000;
dt=1/25
PL_percentile=0.9999;




% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ONE Shock at a time
%   phiH, tau, pBar
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

YearsToShow=75;

CaseName='phiH'
ParamToShock='phiH';
ParamToShockValues=[0.955 1.415]; % These are the values in IdeaModelUSonly ==> phiH shock to match US data
SetParamsIdeaModel
TransitionDynamicsShocks

CaseName='tau'
ParamToShock='tau';
ParamToShockValues=[.455 0.192]; % These are the values in IdeaModelTaxes
SetParamsIdeaModel
phiH=1.4
Lbar=35
TransitionDynamicsShocks

CaseName='pBar'
ParamToShock='pBar';
ParamToShockValues=[1.5 0.93]; % These are the values in IdeaModelPBar
SetParamsIdeaModel
TransitionDynamicsShocks


% SHOCKS Now plot all the results on single graphs
yrs=data_phiH.yrs;
yy=find(yrs>=1970);
figure(1); figsetup; makefigwide
plot(yrs(yy),data_tau.eta(yy),'-','Color',mygreen,'LineWidth',LW);
plot(yrs(yy),data_pBar.eta(yy),'-','Color',mypurp,'LineWidth',LW);
plot(yrs(yy),data_phiH.eta(yy),'-','Color',myblue,'LineWidth',LW);
plot(yrs(yy),eta_new*ones(size(yy)),'--','Color',mygreen,'LineWidth',1);
plot(yrs_data,etaUS,'-','Color',myred,'LineWidth',2);
plot(yrs_data,etaUS,'o','Color',myred,'LineWidth',2);
chadfig2('Year','Pareto Inequality,\eta',1,0);
line([1983.5 1989.5],[.53 .523],'Color',myblack,'LineWidth',.5)
text(1990,.52,'\phi_H')
text(1977,.50,'\tau')
text(1983,.42,'$\bar{p}$','interpreter','latex');
print('-depsc','TransitionDynamicsMaster_eta');


tt=(1:length(yrs));
figure(2); figsetup; makefigwide
plot(yrs(tt),log(ytrend(tt)),'--','Color',myred,'LineWidth',1);
plot(yrs(tt),log(data_tau.gdp(tt)),'-','Color',mygreen,'LineWidth',LW);
plot(yrs(tt),log(data_pBar.gdp(tt)),'-','Color',mypurp,'LineWidth',LW);
plot(yrs(tt),log(data_phiH.gdp(tt)),'-','Color',myblue,'LineWidth',LW);
vstep=100;
v1=floor(min(gdp(tt))/vstep)*vstep;
v2=ceil(max(gdp(tt))/vstep)*vstep;
vals=vstep*2.^(0:ceil(log2(v2/vstep))); %v1:vstep:v2;
relabelaxis(log(vals),num2str(vals'),'y');
chadfig2('Year','GDP per person',1,0);
line([2014 2022],log([620 560]),'Color',myblack,'LineWidth',.5)
line([1996 1999.5],log([325 285]),'Color',myblack,'LineWidth',.5)
text(2010,log(650),'\phi_H') %text(2010,log(700),'\phi_H')
text(2030,log(850),'\tau')
text(2000,log(280),'$\bar{p}$','interpreter','latex');
print('-depsc','TransitionDynamicsMaster_gdp');


% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SEQUENCE of shocks from IRS / SSA data
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% GUVENEN transition dynamics
CaseName='Guvenen'
SetParamsIdeaModel
TransitionDynamicsGuvenen


% IRS Transition Dynamics
CaseName='IRSboth'
SetParamsIdeaModel
IRSCase=1;  % Wages + Entrepreneurial Income
HPsmoother=100;  % Pass HPsmoother=[] to not smooth
TransitionDynamicsIRS


CaseName='IRSentre'
SetParamsIdeaModel
IRSCase=3;  % Entrepreneurial Income
HPsmoother=100;  % Pass HPsmoother=[] to not smooth
TransitionDynamicsIRS

% CaseName='IRSraw'
% SetParamsIdeaModel
% IRSCase=1;  % Wages + Entrepreneurial Income
% HPsmoother=[];  % Pass HPsmoother=[] to not smooth
% TransitionDynamicsIRS



% IRS / Guvenen: Now plot all the results on single graphs
yrs=data_IRSboth.yrs;
yy=find(yrs>=1970);
figure(1); figsetup; makefigwide
plot(yrs(yy),data_IRSentre.eta(yy),'-','Color',mygreen,'LineWidth',LW);
plot(yrs(yy),data_Guvenen.eta(yy),'-','Color',mypurp,'LineWidth',LW);
plot(yrs(yy),data_IRSboth.eta(yy),'-','Color',myblue,'LineWidth',LW);
%plot(yrs(yy),eta_new*oo(yy),'--','Color',mygreen,'LineWidth',1);
plot(yrs_data,etaUS,'-','Color',myred,'LineWidth',2);
plot(yrs_data,etaUS,'o','Color',myred,'LineWidth',2);
chadfig2('Year','Pareto Inequality,\eta',1,0);
text(2020,.70,mlstring('Entrepreneurial income (IRS data)'));
text(2020,.495,'Wages and salaries (SSA data)');
text(2020,.608,mlstring('Wages, salaries, and entre-\\ preneurial income (IRS data)'));
print('-depsc','TransitionDynamicsMaster_etaIRS');

% gdp
tt=(1:length(yrs));
figure(2); figsetup; makefigwide
plot(yrs(tt),log(data_IRSboth.ytrend(tt)),'--','Color',myred,'LineWidth',1);
plot(yrs(tt),log(data_Guvenen.gdp(tt)),'-','Color',mypurp,'LineWidth',LW);
plot(yrs(tt),log(data_IRSentre.gdp(tt)),'-','Color',mygreen,'LineWidth',LW);
plot(yrs(tt),log(data_IRSboth.gdp(tt)),'-','Color',myblue,'LineWidth',LW);
vstep=100;
v1=floor(min(gdp(tt))/vstep)*vstep;
v2=ceil(max(gdp(tt))/vstep)*vstep;
vals=vstep*2.^(0:ceil(log2(v2/vstep))); %v1:vstep:v2;
relabelaxis(log(vals),num2str(vals'),'y');
chadfig2('Year','GDP per person',1,0);
text(2020,log(695),mlstring('Entrepreneurial income\\   (IRS data)'));
line([2029 2023],log([300 345]),'Color',myblack,'LineWidth',.5)
text(2030,log(300),mlstring('Wages and salaries\\  (SSA data)'));
text(1985,log(400),mlstring('Wages, salaries, and\\ entrepreneurial income\\  (IRS data)'));
line([2000 2006],log([330 270]),'Color',myblack,'LineWidth',.5)
print('-depsc','TransitionDynamicsMaster_gdpIRS');


save TransitionDynamicsMaster;

