!-----------------------------------------------------------------------
! PSCF - Polymer Self-Consistent Field Theory
! Copyright (2002-2016) Regents of the University of Minnesota
! contact: David Morse, morse012@umn.edu
!
! This program is free software; you can redistribute it and/or modify
! it under the terms of the GNU General Public License as published by
! the Free Software Foundation. A copy of this license is included in
! the LICENSE file in the top-level PSCF directory. 
!----------------------------------------------------------------------
!****m* scf/space_groups_mod
! MODULE
!   space_groups_mod
! PURPOSE
!   Module contains only the subroutine space_groups, which returns 
!   a space group if passed a standard Hermann-Mauguin symbol or 
!   space group number, or if passed the name of a file containing 
!   the group elements.
! SOURCE
!----------------------------------------------------------------------
module space_groups_mod
   USE const_mod
   USE group_mod
   implicit none
   !***
 
contains
 
   !-------------------------------------------------------------------
   !****p* space_groups_mod/space_groups
   ! SUBROUTINE
   !   space_groups(group_name,g,[found_arg])
   ! PURPOSE
   !   Routine contains lookup table for crystallographic space groups.
   !   This routine is essentially a hard-coded database of the standard
   !   crystallographic space groups.
   ! ARGUMENTS
   !   character(*)     :: group_name  
   !   type(group_type) :: g           ! group
   !   logical          :: found_arg   ! returns true if found (optional)
   ! COMMENT
   !   On input:
   !      group_name = an International Hermann-Mauguin symbol or number
   !                    for a 1, 2, or 3D space group, or the name of a 
   !                    file containing the group
   !
   !   On output:
   !      group_name = trimmed group name + any added default setting
   !      g          = the group
   !      found_arg  = .true. if group_name was found within routine
   !
   !   Algorithm:
   !      Routine first looks for space group with a Hermann-Mauguin 
   !      symbol or number that matches the string trim(group_name). 
   !      It then tries extensions of group_name with default settings.
   !      If this fails, it then tries to read the group from a file
   !      named trim(group_name), which must contain the symmetry
   !      operations for the group in the format defined in the
   !      input_group and output_group routines of module group_mod.
   !      Because the routine first tries to match a standard symbol
   !      or space group number, the file name must not be an integer
   !      or the ascii text representation of a group.
   !
   !   One-Dimensional Groups:
   !
   !      The only allowed one-dimensional groups are '1' (no symmetry)
   !      and '-1' (centrosymmetric). The only allowed crystal_system is
   !      'lamellar'.
   !
   !   Two-Dimensional Plane Groups:
   ! 
   !      Plane group names are those used in the International Tables
   !      for Crystallography, Volume A. To format these as ascii text
   !      strings, we introduce a space between logical elements. Each
   !      of the plane group names begins with a lower case 'p' or 'c',
   !      where 'c' indicates a centered group. For example:
   ! 
   !      plane group #  2, group_name = 'p 2'
   !                  # 12, group_name = 'p 4 g m'
   !                  # 17, group_name = 'p 6 m m'
   !
   !   Three-Dimensional Plane Groups:
   ! 
   !      Space group names are the Hermann-Mauguin symbols used in the 
   !      International Tables for Crystallography, Volume A. To format 
   !      these as ascii text strings, we use the following rules: 
   !      1) Add a single space between logical elements. 2) Inversion 
   !      bars are represented by a - sign prefix. 3) screw axis 
   !      subscripts are represented by integer suffixes, with no 
   !      preceding space. For example:
   ! 
   !      space group # 154  group_name = 'P 32 2 1'
   !                  # 176  group_name = 'P 63/m'
   !                  # 230, group_name = 'I a -3 d'
   !   
   !      Groups with more than one possible setting are represented by
   !      strings of the form
   !   
   !               group_name//' : '//setting
   ! 
   !      where setting = 1,2,H, or R is a single character that 
   !      indicates a choice of setting. Setting of such groups are
   !      labelled by either 1 and 2 or by H and R (where the latter 
   !      represents hexagonal and rhombohedral settings). For example
   !   
   !           group #  48, group_name =  'P n n n : 1' (setting 1)
   !                 #  48, group_name =  'P n n n : 2' (setting 2)
   !
   !           group # 146, group_name =  'R 3 : H' (hexag. setting)
   !           group # 146, group_name =  'R 3 : R' (Rhomb. setting)
   !   
   !      If the routine cannot match the input group_name as a group
   !      symbol or number, it next tries to match the extension of 
   !      the group_name by default settings '2' and 'H'.
   !   
   !      A list of valid names, with settings, is given in the file
   !      src/crystal/SGNAMES 
   !
   !   Sources:
   !
   !      The select case structure for 3D groups was automatically
   !      generated from an output of the C++ package cctbx.
   !      (http://cctbx.sourceforge.net)
   !     
   ! SOURCE
   !-------------------------------------------------------------------
   subroutine space_groups(group_name,g,found_arg)

   character(len=*), intent(INOUT)  :: group_name
   type(group_type), intent(OUT)    :: g           ! group
   logical, intent(OUT), optional   :: found_arg   ! true if found
   !***

   integer :: i        ! dummy loop index
   logical :: found    ! group_name has been found
   logical :: extend2  ! extension group_name//" : 2" has been tried
   logical :: extendH  ! extension group_name//" : H" has been tried
   character*20 :: group_name_in  ! input value of group_name

   ! Store trimmed input value of group_name for safekeeping
   group_name    = trim(group_name)
   group_name_in = group_name

   ! Extensions of group_name with setting 2 & H have not been tried
   extend2       = .false.
   extendH       = .false.

   ! Return to this point to try extending group_name with settings
10 found = .true.

   if (dim == 1) then
   
      select case(trim(group_name))
      case('1', 'P 1', 'p 1') 
       
         !Space group   #   1
         !Group setting #   1
       
         g%order   =   1
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%v(1)     =  0.00
       
       
      case('-1', 'P -1', 'p -1', '2') 
       
         !Space group   #   2
         !Group setting #   2
       
         g%order   =   2
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%v(1)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%v(1)     =  0.00
       
      case default
   
         found = .false. 
   
      end select

   else if (dim == 2) then

      select case(trim(group_name))
      case('p 1', 'P 1', '1')

         !Plane group   #   1
         !Crystal System: Oblique

         g%order   =   1
         do i=1, g%order
            g%s(i)%m = 0.0_long
            g%s(i)%v = 0.0_long
         enddo

         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00

      case('p 2', 'P 2', '2')

         !Plane group   #   2
         !Crystal System: Oblique

         g%order   =   2
         do i=1, g%order
            g%s(i)%m = 0.0_long
            g%s(i)%v = 0.0_long
         enddo

         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00

         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00

      case('p m', 'P m', '3')

         !Plane group   #  3
         !Crystal System: Rectangular
         !Description: Reflection x - > -x

         g%order   =   2
         do i=1, g%order
            g%s(i)%m = 0.0_long
            g%s(i)%v = 0.0_long
         enddo

         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00

         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   =  1.00

      case('p g', 'P g', '4')

         !Plane group   #  4
         !Crystal System: Rectangular
         !Description: Glide plane -x, y+1/2

         g%order   =   2
         do i=1, g%order
            g%s(i)%m = 0.0_long
            g%s(i)%v = 0.0_long
         enddo

         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00

         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   =  1.00
         g%s(2)%v(2)     =  0.50

      case('c m', 'C m', '5')

         !Plane group   #  5
         !Patterson:    c2mm
         !Crystal System: Rectangular
         !Description: Centered + mirror x -> -x (rhombus)

         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
            g%s(i)%v = 0.0_long
         enddo

         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00

         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   =  1.00

         g%s(3)%m(1,1)   =  1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50

         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50

      case('p 2 m m', 'P 2 m m', '6')

         !Plane group   #  6
         !Patterson:    p2mm
         !Crystal System: Rectangular

         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
            g%s(i)%v = 0.0_long
         enddo

         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00

         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00

         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00

         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00

      case('p 2 m g', 'P 2 m g', '7')

         !Plane group   #  7
         !Crystal System: Rectangular
         !Patterson:   p2mm

         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
            g%s(i)%v = 0.0_long
         enddo

         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00

         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00

         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%v(1)     =  0.50

         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%v(1)     =  0.50

      case('p 2 g g', 'P 2 g g', '8')

         !Plane group     #  8
         !Crystal System: Rectangular
         !Patterson:      p2mm

         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
            g%s(i)%v = 0.0_long
         enddo

         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00

         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00

         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%v(1)     =  0.50

         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%v(1)     =  0.50

      case('c 2 m m', 'C 2 m m', '9')

         !Plane group     #  9
         !Crystal System: Rectangular
         !Patterson:      c2mm

         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
            g%s(i)%v = 0.0_long
         enddo

         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00

         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00

         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00

         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50

         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   = -1.00

         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50

         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50

         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50

      case('p 4', '10')

         !Plane group     #  10
         !Crystal System: Square
         !Patterson:      p4mm

         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
            g%s(i)%v = 0.0_long
         enddo

         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00

         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00

         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00

         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00

      case('p 4 m m', 'P 4 m m', '11')

         !Plane group   #  11
         !Crystal System: Square
         !Patterson: p4mm

         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
            g%s(i)%v = 0.0_long
         enddo

         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00

         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00

         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00

         g%s(4)%m(1,2)   =  1.00
         g%s(4)%m(2,1)   = -1.00

         g%s(5)%m(1,2)   = -1.00
         g%s(5)%m(2,1)   =  1.00

         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00

         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   = -1.00

         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   =  1.00

      case('p 4 g m', 'P 4 g m', '12')

         !Plane group     #  12
         !Crystal System: Square
         !Patterson:      p4mm

         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
            g%s(i)%v = 0.0_long
         enddo

         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00

         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00

         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00

         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50

         g%s(5)%m(1,2)   = -1.00
         g%s(5)%m(2,1)   =  1.00

         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50

         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50

         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50

      case('p 3', 'P 3', '13')

         !Plane group     #  13
         !Crystal System: Hexagonal
         !Patterson:      p6

         g%order   =   3
         do i=1, g%order
            g%s(i)%m = 0.0_long
            g%s(i)%v = 0.0_long
         enddo

         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00

         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00

         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00

      case('p 3 m 1', 'P 3 m 1', '14')

         !Plane group     # 14
         !Crystal System: Hexagonal
         !Patterson:      p6mm

         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
            g%s(i)%v = 0.0_long
         enddo

         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00

         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00

         g%s(3)%m(1,1)   =  1.00
         g%s(3)%m(2,1)   =  1.00
         g%s(3)%m(2,2)   = -1.00

         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(1,2)   =  1.00
         g%s(4)%m(2,1)   = -1.00

         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,2)   =  1.00

         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   = -1.00

      case('p 3 1 m', 'P 3 1 m', '15')

         !Plane group     # 15
         !Crystal System: Hexagonal
         !Patterson:      p6mm

         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
            g%s(i)%v = 0.0_long
         enddo

         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00

         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00

         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   =  1.00

         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(1,2)   =  1.00
         g%s(4)%m(2,1)   = -1.00

         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00

         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,2)   = -1.00

      case('p 6', 'P 6', '16')

         !Plane group     # 16
         !Crystal System: Hexagonal
         !Patterson:      p6

         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
            g%s(i)%v = 0.0_long
         enddo

         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00

         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00

         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   = -1.00

         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(1,2)   =  1.00
         g%s(4)%m(2,1)   = -1.00

         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00

         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   =  1.00

      case('p 6 m m', 'P 6 m m', '17')

         !Plane group     # 17
         !Crystal System: Hexagonal
         !Patterson:      p6mm

         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
            g%s(i)%v = 0.0_long
         enddo

         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00

         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00

         g%s(3)%m(1,1)   =  1.00
         g%s(3)%m(2,1)   =  1.00
         g%s(3)%m(2,2)   = -1.00

         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00

         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00

         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,2)   =  1.00

         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00

         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00

         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,1)   = -1.00
         g%s(9)%m(2,2)   =  1.00

         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,1)  =  1.00

         g%s(11)%m(1,1)  =  1.00
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,2)  = -1.00

         g%s(12)%m(1,2)  =  1.00
         g%s(12)%m(2,1)  =  1.00

      case default
   
         found = .false. 
   
      end select

   else if (dim == 3) then

      select case(trim(group_name))
      case('P 1', '1') 
       
         !Space group   #   1
         !Group setting #   1
       
         g%order   =   1
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
       
      case('P -1', '2')                            
       
         !Space group   #   2
         !Group setting #   2
       
         g%order   =   2
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
       
      case('P 1 2 1', '3')                         
       
         !Space group   #   3
         !Group setting #   3
       
         g%order   =   2
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   =  1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
       
      case('P 1 21 1', '4')                        
       
         !Space group   #   4
         !Group setting #   4
       
         g%order   =   2
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   =  1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.00
       
       
      case('C 1 2 1', '5')                         
       
         !Space group   #   5
         !Group setting #   5
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   =  1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   =  1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
       
      case('P 1 m 1', '6')                         
       
         !Space group   #   6
         !Group setting #   6
       
         g%order   =   2
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
       
      case('P 1 c 1', '7')                         
       
         !Space group   #   7
         !Group setting #   7
       
         g%order   =   2
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
       
      case('C 1 m 1', '8')                         
       
         !Space group   #   8
         !Group setting #   8
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   =  1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
       
      case('C 1 c 1', '9')                         
       
         !Space group   #   9
         !Group setting #   9
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   =  1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.50
       
       
      case('P 1 2/m 1', '10')                       
       
         !Space group   #  10
         !Group setting #  10
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   =  1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   = -1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
       
      case('P 1 21/m 1', '11')                      
       
         !Space group   #  11
         !Group setting #  11
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   =  1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   = -1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     = -0.50
         g%s(4)%v(3)     =  0.00
       
       
      case('C 1 2/m 1', '12')                       
       
         !Space group   #  12
         !Group setting #  12
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   =  1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   = -1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.00
       
       
      case('P 1 2/c 1', '13')                       
       
         !Space group   #  13
         !Group setting #  13
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   =  1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   = -1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     = -0.50
       
       
      case('P 1 21/c 1', '14')                      
       
         !Space group   #  14
         !Group setting #  14
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   =  1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   = -1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     = -0.50
         g%s(4)%v(3)     = -0.50
       
       
      case('C 1 2/c 1', '15')                       
       
         !Space group   #  15
         !Group setting #  15
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   =  1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   = -1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     = -0.50
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     = -0.50
       
       
      case('P 2 2 2', '16')                         
       
         !Space group   #  16
         !Group setting #  16
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
       
      case('P 2 2 21', '17')                        
       
         !Space group   #  17
         !Group setting #  17
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
       
      case('P 21 21 2', '18')                       
       
         !Space group   #  18
         !Group setting #  18
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
       
      case('P 21 21 21', '19')                      
       
         !Space group   #  19
         !Group setting #  19
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
       
      case('C 2 2 21', '20')                        
       
         !Space group   #  20
         !Group setting #  20
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   = -1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
       
      case('C 2 2 2', '21')                         
       
         !Space group   #  21
         !Group setting #  21
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   = -1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.00
       
       
      case('F 2 2 2', '22')                         
       
         !Space group   #  22
         !Group setting #  22
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   = -1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  =  1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  = -1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,1)  = -1.00
         g%s(15)%m(2,2)  =  1.00
         g%s(15)%m(3,3)  = -1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  = -1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.00
       
       
      case('I 2 2 2', '23')                         
       
         !Space group   #  23
         !Group setting #  23
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   = -1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
       
      case('I 21 21 21', '24')                      
       
         !Space group   #  24
         !Group setting #  24
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  1.00000000
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  1.00000000
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   = -1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  1.00000000
         g%s(8)%v(3)     =  0.50
       
       
      case('P m m 2', '25')                         
       
         !Space group   #  25
         !Group setting #  25
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
       
      case('P m c 21', '26')                        
       
         !Space group   #  26
         !Group setting #  26
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
       
      case('P c c 2', '27')                         
       
         !Space group   #  27
         !Group setting #  27
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
       
      case('P m a 2', '28')                         
       
         !Space group   #  28
         !Group setting #  28
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
       
      case('P c a 21', '29')                        
       
         !Space group   #  29
         !Group setting #  29
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
       
      case('P n c 2', '30')                         
       
         !Space group   #  30
         !Group setting #  30
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.50
       
       
      case('P m n 21', '31')                        
       
         !Space group   #  31
         !Group setting #  31
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
       
      case('P b a 2', '32')                         
       
         !Space group   #  32
         !Group setting #  32
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
       
      case('P n a 21', '33')                        
       
         !Space group   #  33
         !Group setting #  33
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
       
      case('P n n 2', '34')                         
       
         !Space group   #  34
         !Group setting #  34
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.50
       
       
      case('C m m 2', '35')                         
       
         !Space group   #  35
         !Group setting #  35
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.00
       
       
      case('C m c 21', '36')                        
       
         !Space group   #  36
         !Group setting #  36
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
       
      case('C c c 2', '37')                         
       
         !Space group   #  37
         !Group setting #  37
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
       
      case('A m m 2', '38')                         
       
         !Space group   #  38
         !Group setting #  38
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
       
      case('A b m 2', '39')                         
       
         !Space group   #  39
         !Group setting #  39
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  1.00000000
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  1.00000000
         g%s(8)%v(3)     =  0.50
       
       
      case('A m a 2', '40')                         
       
         !Space group   #  40
         !Group setting #  40
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
       
      case('A b a 2', '41')                         
       
         !Space group   #  41
         !Group setting #  41
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  1.00000000
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  1.00000000
         g%s(8)%v(3)     =  0.50
       
       
      case('F m m 2', '42')                         
       
         !Space group   #  42
         !Group setting #  42
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,1)  = -1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  =  1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  =  1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(2,2)  = -1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,1)  = -1.00
         g%s(15)%m(2,2)  =  1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.00
       
       
      case('F d d 2', '43')                         
       
         !Space group   #  43
         !Group setting #  43
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.25
         g%s(3)%v(2)     =  0.25
         g%s(3)%v(3)     =  0.25
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.25
         g%s(4)%v(2)     =  0.25
         g%s(4)%v(3)     =  0.25
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.25
         g%s(7)%v(2)     =  0.75000000
         g%s(7)%v(3)     =  0.75000000
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.25
         g%s(8)%v(2)     =  0.75000000
         g%s(8)%v(3)     =  0.75000000
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,1)  = -1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  0.75000000
         g%s(11)%v(2)    =  0.25
         g%s(11)%v(3)    =  0.75000000
       
         g%s(12)%m(1,1)  =  1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.75000000
         g%s(12)%v(2)    =  0.25
         g%s(12)%v(3)    =  0.75000000
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  =  1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(2,2)  = -1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,1)  = -1.00
         g%s(15)%m(2,2)  =  1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.75000000
         g%s(15)%v(2)    =  0.75000000
         g%s(15)%v(3)    =  0.25
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.75000000
         g%s(16)%v(2)    =  0.75000000
         g%s(16)%v(3)    =  0.25
       
       
      case('I m m 2', '44')                         
       
         !Space group   #  44
         !Group setting #  44
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
       
      case('I b a 2', '45')                         
       
         !Space group   #  45
         !Group setting #  45
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  1.00000000
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  1.00000000
       
       
      case('I m a 2', '46')                         
       
         !Space group   #  46
         !Group setting #  46
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  1.00000000
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  1.00000000
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
       
      case('P m m m', '47')                         
       
         !Space group   #  47
         !Group setting #  47
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
       
      case('P n n n : 2', '48')                     
       
         !Space group   #  48
         !Group setting #  48
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     = -0.50
         g%s(6)%v(3)     = -0.50
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     = -0.50
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     = -0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     = -0.50
         g%s(8)%v(2)     = -0.50
         g%s(8)%v(3)     =  0.00
       
       
      case('P n n n : 1')                     
       
         !Space group   #  48
         !Group setting #  49
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
       
      case('P c c m', '49')                         
       
         !Space group   #  49
         !Group setting #  50
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     = -0.50
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     = -0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
       
      case('P b a n : 2', '50')                     
       
         !Space group   #  50
         !Group setting #  51
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     = -0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     = -0.50
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     = -0.50
         g%s(8)%v(2)     = -0.50
         g%s(8)%v(3)     =  0.00
       
       
      case('P b a n : 1')                     
       
         !Space group   #  50
         !Group setting #  52
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.00
       
       
      case('P m m a', '51')                         
       
         !Space group   #  51
         !Group setting #  53
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     = -0.50
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     = -0.50
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
       
      case('P n n a', '52')                         
       
         !Space group   #  52
         !Group setting #  54
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     = -0.50
         g%s(6)%v(3)     = -0.50
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     = -0.50
         g%s(7)%v(2)     = -0.50
         g%s(7)%v(3)     = -0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     = -0.50
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
       
      case('P m n a', '53')                         
       
         !Space group   #  53
         !Group setting #  55
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     = -0.50
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     = -0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     = -0.50
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     = -0.50
       
       
      case('P c c a', '54')                         
       
         !Space group   #  54
         !Group setting #  56
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     = -0.50
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     = -0.50
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     = -0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     = -0.50
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
       
      case('P b a m', '55')                         
       
         !Space group   #  55
         !Group setting #  57
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     = -0.50
         g%s(6)%v(2)     = -0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     = -0.50
         g%s(7)%v(2)     = -0.50
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
       
      case('P c c n', '56')                         
       
         !Space group   #  56
         !Group setting #  58
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     = -0.50
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     = -0.50
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     = -0.50
         g%s(7)%v(3)     = -0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     = -0.50
         g%s(8)%v(2)     = -0.50
         g%s(8)%v(3)     =  0.00
       
       
      case('P b c m', '57')                         
       
         !Space group   #  57
         !Group setting #  59
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     = -0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     = -0.50
         g%s(7)%v(3)     = -0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     = -0.50
       
       
      case('P n n m', '58')                         
       
         !Space group   #  58
         !Group setting #  60
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     = -0.50
         g%s(6)%v(2)     = -0.50
         g%s(6)%v(3)     = -0.50
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     = -0.50
         g%s(7)%v(2)     = -0.50
         g%s(7)%v(3)     = -0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
       
      case('P m m n : 2', '59')                     
       
         !Space group   #  59
         !Group setting #  61
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     = -0.50
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     = -0.50
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     = -0.50
         g%s(8)%v(2)     = -0.50
         g%s(8)%v(3)     =  0.00
       
       
      case('P m m n : 1')                     
       
         !Space group   #  59
         !Group setting #  62
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.00
       
       
      case('P b c n', '60')                         
       
         !Space group   #  60
         !Group setting #  63
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     = -0.50
         g%s(6)%v(2)     = -0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     = -0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     = -0.50
         g%s(8)%v(2)     = -0.50
         g%s(8)%v(3)     = -0.50
       
       
      case('P b c a', '61')                         
       
         !Space group   #  61
         !Group setting #  64
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     = -0.50
         g%s(6)%v(2)     = -0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     = -0.50
         g%s(7)%v(3)     = -0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     = -0.50
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     = -0.50
       
       
      case('P n m a', '62')                         
       
         !Space group   #  62
         !Group setting #  65
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     = -0.50
         g%s(6)%v(2)     = -0.50
         g%s(6)%v(3)     = -0.50
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     = -0.50
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     = -0.50
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     = -0.50
       
       
      case('C m c m', '63')                         
       
         !Space group   #  63
         !Group setting #  66
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     = -0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     = -0.50
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,1)  =  1.00
         g%s(15)%m(2,2)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    = -0.50
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    = -0.50
       
       
      case('C m c a', '64')                         
       
         !Space group   #  64
         !Group setting #  67
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     = -0.50
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     = -0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     = -0.50
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     = -0.50
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  1.00000000
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  1.00000000
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,1)  =  1.00
         g%s(15)%m(2,2)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    = -0.50
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    = -0.50
       
       
      case('C m m m', '65')                         
       
         !Space group   #  65
         !Group setting #  68
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,1)  =  1.00
         g%s(15)%m(2,2)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.00
       
       
      case('C c c m', '66')                         
       
         !Space group   #  66
         !Group setting #  69
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     = -0.50
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     = -0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    = -0.50
       
         g%s(15)%m(1,1)  =  1.00
         g%s(15)%m(2,2)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    = -0.50
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.00
       
       
      case('C m m a', '67')                         
       
         !Space group   #  67
         !Group setting #  70
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     = -0.50
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     = -0.50
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  1.00000000
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  1.00000000
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,1)  =  1.00
         g%s(15)%m(2,2)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.00
       
       
      case('C c c a : 2', '68')                     
       
         !Space group   #  68
         !Group setting #  71
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     = -0.50
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     = -0.50
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     = -0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     = -0.50
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  1.00000000
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  1.00000000
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    = -0.50
       
         g%s(15)%m(1,1)  =  1.00
         g%s(15)%m(2,2)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    = -0.50
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.00
       
       
      case('C c c a : 1')                     
       
         !Space group   #  68
         !Group setting #  72
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  1.00000000
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  1.00000000
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,1)  =  1.00
         g%s(15)%m(2,2)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  1.00000000
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  1.00000000
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.50
       
       
      case('F m m m', '69')                         
       
         !Space group   #  69
         !Group setting #  73
       
         g%order   =  32
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,1)  =  1.00
         g%s(15)%m(2,2)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.50
       
         g%s(17)%m(1,1)  =  1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  =  1.00
         g%s(17)%v(1)    =  0.50
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.50
       
         g%s(18)%m(1,1)  =  1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  = -1.00
         g%s(18)%v(1)    =  0.50
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.50
       
         g%s(19)%m(1,1)  = -1.00
         g%s(19)%m(2,2)  =  1.00
         g%s(19)%m(3,3)  = -1.00
         g%s(19)%v(1)    =  0.50
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    =  0.50
       
         g%s(20)%m(1,1)  = -1.00
         g%s(20)%m(2,2)  = -1.00
         g%s(20)%m(3,3)  =  1.00
         g%s(20)%v(1)    =  0.50
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    =  0.50
       
         g%s(21)%m(1,1)  = -1.00
         g%s(21)%m(2,2)  = -1.00
         g%s(21)%m(3,3)  = -1.00
         g%s(21)%v(1)    =  0.50
         g%s(21)%v(2)    =  0.00
         g%s(21)%v(3)    =  0.50
       
         g%s(22)%m(1,1)  = -1.00
         g%s(22)%m(2,2)  =  1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.50
         g%s(22)%v(2)    =  0.00
         g%s(22)%v(3)    =  0.50
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(2,2)  = -1.00
         g%s(23)%m(3,3)  =  1.00
         g%s(23)%v(1)    =  0.50
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    =  0.50
       
         g%s(24)%m(1,1)  =  1.00
         g%s(24)%m(2,2)  =  1.00
         g%s(24)%m(3,3)  = -1.00
         g%s(24)%v(1)    =  0.50
         g%s(24)%v(2)    =  0.00
         g%s(24)%v(3)    =  0.50
       
         g%s(25)%m(1,1)  =  1.00
         g%s(25)%m(2,2)  =  1.00
         g%s(25)%m(3,3)  =  1.00
         g%s(25)%v(1)    =  0.50
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  0.00
       
         g%s(26)%m(1,1)  =  1.00
         g%s(26)%m(2,2)  = -1.00
         g%s(26)%m(3,3)  = -1.00
         g%s(26)%v(1)    =  0.50
         g%s(26)%v(2)    =  0.50
         g%s(26)%v(3)    =  0.00
       
         g%s(27)%m(1,1)  = -1.00
         g%s(27)%m(2,2)  =  1.00
         g%s(27)%m(3,3)  = -1.00
         g%s(27)%v(1)    =  0.50
         g%s(27)%v(2)    =  0.50
         g%s(27)%v(3)    =  0.00
       
         g%s(28)%m(1,1)  = -1.00
         g%s(28)%m(2,2)  = -1.00
         g%s(28)%m(3,3)  =  1.00
         g%s(28)%v(1)    =  0.50
         g%s(28)%v(2)    =  0.50
         g%s(28)%v(3)    =  0.00
       
         g%s(29)%m(1,1)  = -1.00
         g%s(29)%m(2,2)  = -1.00
         g%s(29)%m(3,3)  = -1.00
         g%s(29)%v(1)    =  0.50
         g%s(29)%v(2)    =  0.50
         g%s(29)%v(3)    =  0.00
       
         g%s(30)%m(1,1)  = -1.00
         g%s(30)%m(2,2)  =  1.00
         g%s(30)%m(3,3)  =  1.00
         g%s(30)%v(1)    =  0.50
         g%s(30)%v(2)    =  0.50
         g%s(30)%v(3)    =  0.00
       
         g%s(31)%m(1,1)  =  1.00
         g%s(31)%m(2,2)  = -1.00
         g%s(31)%m(3,3)  =  1.00
         g%s(31)%v(1)    =  0.50
         g%s(31)%v(2)    =  0.50
         g%s(31)%v(3)    =  0.00
       
         g%s(32)%m(1,1)  =  1.00
         g%s(32)%m(2,2)  =  1.00
         g%s(32)%m(3,3)  = -1.00
         g%s(32)%v(1)    =  0.50
         g%s(32)%v(2)    =  0.50
         g%s(32)%v(3)    =  0.00
       
       
      case('F d d d : 2', '70')                     
       
         !Space group   #  70
         !Group setting #  74
       
         g%order   =  32
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.25
         g%s(2)%v(3)     =  0.25
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.25
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.25
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.25
         g%s(4)%v(2)     =  0.25
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     = -0.25
         g%s(6)%v(3)     = -0.25
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     = -0.25
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     = -0.25
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     = -0.25
         g%s(8)%v(2)     = -0.25
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.75000000
         g%s(10)%v(3)    =  0.75000000
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.25
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.75000000
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.25
         g%s(12)%v(2)    =  0.75000000
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.25
         g%s(14)%v(3)    =  0.25
       
         g%s(15)%m(1,1)  =  1.00
         g%s(15)%m(2,2)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    = -0.25
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.25
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    = -0.25
         g%s(16)%v(2)    =  0.25
         g%s(16)%v(3)    =  0.50
       
         g%s(17)%m(1,1)  =  1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  =  1.00
         g%s(17)%v(1)    =  0.50
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.50
       
         g%s(18)%m(1,1)  =  1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  = -1.00
         g%s(18)%v(1)    =  0.50
         g%s(18)%v(2)    =  0.25
         g%s(18)%v(3)    =  0.75000000
       
         g%s(19)%m(1,1)  = -1.00
         g%s(19)%m(2,2)  =  1.00
         g%s(19)%m(3,3)  = -1.00
         g%s(19)%v(1)    =  0.75000000
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    =  0.75000000
       
         g%s(20)%m(1,1)  = -1.00
         g%s(20)%m(2,2)  = -1.00
         g%s(20)%m(3,3)  =  1.00
         g%s(20)%v(1)    =  0.75000000
         g%s(20)%v(2)    =  0.25
         g%s(20)%v(3)    =  0.50
       
         g%s(21)%m(1,1)  = -1.00
         g%s(21)%m(2,2)  = -1.00
         g%s(21)%m(3,3)  = -1.00
         g%s(21)%v(1)    =  0.50
         g%s(21)%v(2)    =  0.00
         g%s(21)%v(3)    =  0.50
       
         g%s(22)%m(1,1)  = -1.00
         g%s(22)%m(2,2)  =  1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.50
         g%s(22)%v(2)    = -0.25
         g%s(22)%v(3)    =  0.25
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(2,2)  = -1.00
         g%s(23)%m(3,3)  =  1.00
         g%s(23)%v(1)    =  0.25
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    =  0.25
       
         g%s(24)%m(1,1)  =  1.00
         g%s(24)%m(2,2)  =  1.00
         g%s(24)%m(3,3)  = -1.00
         g%s(24)%v(1)    =  0.25
         g%s(24)%v(2)    = -0.25
         g%s(24)%v(3)    =  0.50
       
         g%s(25)%m(1,1)  =  1.00
         g%s(25)%m(2,2)  =  1.00
         g%s(25)%m(3,3)  =  1.00
         g%s(25)%v(1)    =  0.50
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  0.00
       
         g%s(26)%m(1,1)  =  1.00
         g%s(26)%m(2,2)  = -1.00
         g%s(26)%m(3,3)  = -1.00
         g%s(26)%v(1)    =  0.50
         g%s(26)%v(2)    =  0.75000000
         g%s(26)%v(3)    =  0.25
       
         g%s(27)%m(1,1)  = -1.00
         g%s(27)%m(2,2)  =  1.00
         g%s(27)%m(3,3)  = -1.00
         g%s(27)%v(1)    =  0.75000000
         g%s(27)%v(2)    =  0.50
         g%s(27)%v(3)    =  0.25
       
         g%s(28)%m(1,1)  = -1.00
         g%s(28)%m(2,2)  = -1.00
         g%s(28)%m(3,3)  =  1.00
         g%s(28)%v(1)    =  0.75000000
         g%s(28)%v(2)    =  0.75000000
         g%s(28)%v(3)    =  0.00
       
         g%s(29)%m(1,1)  = -1.00
         g%s(29)%m(2,2)  = -1.00
         g%s(29)%m(3,3)  = -1.00
         g%s(29)%v(1)    =  0.50
         g%s(29)%v(2)    =  0.50
         g%s(29)%v(3)    =  0.00
       
         g%s(30)%m(1,1)  = -1.00
         g%s(30)%m(2,2)  =  1.00
         g%s(30)%m(3,3)  =  1.00
         g%s(30)%v(1)    =  0.50
         g%s(30)%v(2)    =  0.25
         g%s(30)%v(3)    = -0.25
       
         g%s(31)%m(1,1)  =  1.00
         g%s(31)%m(2,2)  = -1.00
         g%s(31)%m(3,3)  =  1.00
         g%s(31)%v(1)    =  0.25
         g%s(31)%v(2)    =  0.50
         g%s(31)%v(3)    = -0.25
       
         g%s(32)%m(1,1)  =  1.00
         g%s(32)%m(2,2)  =  1.00
         g%s(32)%m(3,3)  = -1.00
         g%s(32)%v(1)    =  0.25
         g%s(32)%v(2)    =  0.25
         g%s(32)%v(3)    =  0.00
       
       
      case('F d d d : 1')                     
       
         !Space group   #  70
         !Group setting #  75
       
         g%order   =  32
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.25
         g%s(5)%v(2)     =  0.25
         g%s(5)%v(3)     =  0.25
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.25
         g%s(6)%v(2)     =  0.25
         g%s(6)%v(3)     =  0.25
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.25
         g%s(7)%v(2)     =  0.25
         g%s(7)%v(3)     =  0.25
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.25
         g%s(8)%v(2)     =  0.25
         g%s(8)%v(3)     =  0.25
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.25
         g%s(13)%v(2)    =  0.75000000
         g%s(13)%v(3)    =  0.75000000
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.25
         g%s(14)%v(2)    =  0.75000000
         g%s(14)%v(3)    =  0.75000000
       
         g%s(15)%m(1,1)  =  1.00
         g%s(15)%m(2,2)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.25
         g%s(15)%v(2)    =  0.75000000
         g%s(15)%v(3)    =  0.75000000
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.25
         g%s(16)%v(2)    =  0.75000000
         g%s(16)%v(3)    =  0.75000000
       
         g%s(17)%m(1,1)  =  1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  =  1.00
         g%s(17)%v(1)    =  0.50
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.50
       
         g%s(18)%m(1,1)  =  1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  = -1.00
         g%s(18)%v(1)    =  0.50
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.50
       
         g%s(19)%m(1,1)  = -1.00
         g%s(19)%m(2,2)  =  1.00
         g%s(19)%m(3,3)  = -1.00
         g%s(19)%v(1)    =  0.50
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    =  0.50
       
         g%s(20)%m(1,1)  = -1.00
         g%s(20)%m(2,2)  = -1.00
         g%s(20)%m(3,3)  =  1.00
         g%s(20)%v(1)    =  0.50
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    =  0.50
       
         g%s(21)%m(1,1)  = -1.00
         g%s(21)%m(2,2)  = -1.00
         g%s(21)%m(3,3)  = -1.00
         g%s(21)%v(1)    =  0.75000000
         g%s(21)%v(2)    =  0.25
         g%s(21)%v(3)    =  0.75000000
       
         g%s(22)%m(1,1)  = -1.00
         g%s(22)%m(2,2)  =  1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.75000000
         g%s(22)%v(2)    =  0.25
         g%s(22)%v(3)    =  0.75000000
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(2,2)  = -1.00
         g%s(23)%m(3,3)  =  1.00
         g%s(23)%v(1)    =  0.75000000
         g%s(23)%v(2)    =  0.25
         g%s(23)%v(3)    =  0.75000000
       
         g%s(24)%m(1,1)  =  1.00
         g%s(24)%m(2,2)  =  1.00
         g%s(24)%m(3,3)  = -1.00
         g%s(24)%v(1)    =  0.75000000
         g%s(24)%v(2)    =  0.25
         g%s(24)%v(3)    =  0.75000000
       
         g%s(25)%m(1,1)  =  1.00
         g%s(25)%m(2,2)  =  1.00
         g%s(25)%m(3,3)  =  1.00
         g%s(25)%v(1)    =  0.50
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  0.00
       
         g%s(26)%m(1,1)  =  1.00
         g%s(26)%m(2,2)  = -1.00
         g%s(26)%m(3,3)  = -1.00
         g%s(26)%v(1)    =  0.50
         g%s(26)%v(2)    =  0.50
         g%s(26)%v(3)    =  0.00
       
         g%s(27)%m(1,1)  = -1.00
         g%s(27)%m(2,2)  =  1.00
         g%s(27)%m(3,3)  = -1.00
         g%s(27)%v(1)    =  0.50
         g%s(27)%v(2)    =  0.50
         g%s(27)%v(3)    =  0.00
       
         g%s(28)%m(1,1)  = -1.00
         g%s(28)%m(2,2)  = -1.00
         g%s(28)%m(3,3)  =  1.00
         g%s(28)%v(1)    =  0.50
         g%s(28)%v(2)    =  0.50
         g%s(28)%v(3)    =  0.00
       
         g%s(29)%m(1,1)  = -1.00
         g%s(29)%m(2,2)  = -1.00
         g%s(29)%m(3,3)  = -1.00
         g%s(29)%v(1)    =  0.75000000
         g%s(29)%v(2)    =  0.75000000
         g%s(29)%v(3)    =  0.25
       
         g%s(30)%m(1,1)  = -1.00
         g%s(30)%m(2,2)  =  1.00
         g%s(30)%m(3,3)  =  1.00
         g%s(30)%v(1)    =  0.75000000
         g%s(30)%v(2)    =  0.75000000
         g%s(30)%v(3)    =  0.25
       
         g%s(31)%m(1,1)  =  1.00
         g%s(31)%m(2,2)  = -1.00
         g%s(31)%m(3,3)  =  1.00
         g%s(31)%v(1)    =  0.75000000
         g%s(31)%v(2)    =  0.75000000
         g%s(31)%v(3)    =  0.25
       
         g%s(32)%m(1,1)  =  1.00
         g%s(32)%m(2,2)  =  1.00
         g%s(32)%m(3,3)  = -1.00
         g%s(32)%v(1)    =  0.75000000
         g%s(32)%v(2)    =  0.75000000
         g%s(32)%v(3)    =  0.25
       
       
      case('I m m m', '71')                         
       
         !Space group   #  71
         !Group setting #  76
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,1)  =  1.00
         g%s(15)%m(2,2)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.50
       
       
      case('I b a m', '72')                         
       
         !Space group   #  72
         !Group setting #  77
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     = -0.50
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     = -0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  1.00000000
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  1.00000000
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,1)  =  1.00
         g%s(15)%m(2,2)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.50
       
       
      case('I b c a', '73')                         
       
         !Space group   #  73
         !Group setting #  78
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     = -0.50
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     = -0.50
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     = -0.50
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  1.00000000
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  1.00000000
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  1.00000000
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,1)  =  1.00
         g%s(15)%m(2,2)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.50
       
       
      case('I m m a', '74')                         
       
         !Space group   #  74
         !Group setting #  79
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     = -0.50
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     = -0.50
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  1.00000000
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  1.00000000
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,1)  =  1.00
         g%s(15)%m(2,2)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.50
       
       
      case('P 4', '75')                             
       
         !Space group   #  75
         !Group setting #  80
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
       
      case('P 41', '76')                            
       
         !Space group   #  76
         !Group setting #  81
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.25
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.75000000
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
       
      case('P 42', '77')                            
       
         !Space group   #  77
         !Group setting #  82
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
       
      case('P 43', '78')                            
       
         !Space group   #  78
         !Group setting #  83
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.75000000
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.25
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
       
      case('I 4', '79')                             
       
         !Space group   #  79
         !Group setting #  84
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   = -1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
       
      case('I 41', '80')                            
       
         !Space group   #  80
         !Group setting #  85
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.75000000
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.75000000
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  1.00000000
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  1.25000000
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  1.00000000
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  1.25000000
       
         g%s(8)%m(1,1)   = -1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
       
      case('P -4', '81')                            
       
         !Space group   #  81
         !Group setting #  86
       
         g%order   =   4
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   =  1.00
         g%s(2)%m(2,1)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   = -1.00
         g%s(3)%m(2,1)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
       
      case('I -4', '82')                            
       
         !Space group   #  82
         !Group setting #  87
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   =  1.00
         g%s(2)%m(2,1)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   = -1.00
         g%s(3)%m(2,1)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   = -1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
       
      case('P 4/m', '83')                           
       
         !Space group   #  83
         !Group setting #  88
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
       
      case('P 42/m', '84')                          
       
         !Space group   #  84
         !Group setting #  89
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     = -0.50
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     = -0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
       
      case('P 4/n : 2', '85')                       
       
         !Space group   #  85
         !Group setting #  90
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     = -0.50
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     = -0.50
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     = -0.50
         g%s(8)%v(2)     = -0.50
         g%s(8)%v(3)     =  0.00
       
       
      case('P 4/n : 1')                       
       
         !Space group   #  85
         !Group setting #  91
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.00
       
       
      case('P 42/n : 2', '86')                      
       
         !Space group   #  86
         !Group setting #  92
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     = -0.50
         g%s(6)%v(3)     = -0.50
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     = -0.50
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     = -0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     = -0.50
         g%s(8)%v(2)     = -0.50
         g%s(8)%v(3)     =  0.00
       
       
      case('P 42/n : 1')                      
       
         !Space group   #  86
         !Group setting #  93
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
       
      case('I 4/m', '87')                           
       
         !Space group   #  87
         !Group setting #  94
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,1)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,2)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,2)  =  1.00
         g%s(14)%m(2,1)  = -1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  =  1.00
         g%s(15)%m(3,3)  = -1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.50
       
       
      case('I 41/a : 2', '88')                      
       
         !Space group   #  88
         !Group setting #  95
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.25
         g%s(2)%v(2)     =  0.75000000
         g%s(2)%v(3)     =  0.75000000
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.25
         g%s(3)%v(2)     =  0.25
         g%s(3)%v(3)     =  0.25
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     = -0.25
         g%s(6)%v(2)     = -0.75000000
         g%s(6)%v(3)     = -0.75000000
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     = -0.25
         g%s(7)%v(2)     = -0.25
         g%s(7)%v(3)     = -0.25
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     = -0.50
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,1)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.75000000
         g%s(10)%v(2)    =  1.25000000
         g%s(10)%v(3)    =  1.25000000
       
         g%s(11)%m(1,2)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  0.75000000
         g%s(11)%v(2)    =  0.75000000
         g%s(11)%v(3)    =  0.75000000
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  1.00000000
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,2)  =  1.00
         g%s(14)%m(2,1)  = -1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.25
         g%s(14)%v(2)    = -0.25
         g%s(14)%v(3)    = -0.25
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  =  1.00
         g%s(15)%m(3,3)  = -1.00
         g%s(15)%v(1)    =  0.25
         g%s(15)%v(2)    =  0.25
         g%s(15)%v(3)    =  0.25
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.50
       
       
      case('I 41/a : 1')                      
       
         !Space group   #  88
         !Group setting #  96
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.75000000
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.75000000
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.75000000
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.75000000
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,1)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  1.00000000
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  1.25000000
       
         g%s(11)%m(1,2)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  1.00000000
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  1.25000000
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  1.00000000
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  1.25000000
       
         g%s(14)%m(1,2)  =  1.00
         g%s(14)%m(2,1)  = -1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  =  1.00
         g%s(15)%m(3,3)  = -1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  1.00000000
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  1.25000000
       
       
      case('P 4 2 2', '89')                         
       
         !Space group   #  89
         !Group setting #  97
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
       
      case('P 4 21 2', '90')                        
       
         !Space group   #  90
         !Group setting #  98
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
       
      case('P 41 2 2', '91')                        
       
         !Space group   #  91
         !Group setting #  99
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.25
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.75000000
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.75000000
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.25
       
       
      case('P 41 21 2', '92')                       
       
         !Space group   #  92
         !Group setting # 100
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.25
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.75000000
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.75000000
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.25
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
       
      case('P 42 2 2', '93')                        
       
         !Space group   #  93
         !Group setting # 101
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
       
      case('P 42 21 2', '94')                       
       
         !Space group   #  94
         !Group setting # 102
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
       
      case('P 43 2 2', '95')                        
       
         !Space group   #  95
         !Group setting # 103
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.75000000
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.25
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.25
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.75000000
       
       
      case('P 43 21 2', '96')                       
       
         !Space group   #  96
         !Group setting # 104
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.75000000
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.25
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.25
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.75000000
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
       
      case('I 4 2 2', '97')                         
       
         !Space group   #  97
         !Group setting # 105
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,1)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,2)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  =  1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  = -1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  =  1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(2,2)  = -1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,1)  =  1.00
         g%s(15)%m(3,3)  = -1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,2)  = -1.00
         g%s(16)%m(2,1)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.50
       
       
      case('I 41 2 2', '98')                        
       
         !Space group   #  98
         !Group setting # 106
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.75000000
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.75000000
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.75000000
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.75000000
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,1)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  1.00000000
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  1.25000000
       
         g%s(11)%m(1,2)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  1.00000000
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  1.25000000
       
         g%s(12)%m(1,1)  =  1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  = -1.00
         g%s(12)%v(1)    =  1.00000000
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  1.25000000
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  =  1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  1.00000000
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  1.25000000
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(2,2)  = -1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,1)  =  1.00
         g%s(15)%m(3,3)  = -1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,2)  = -1.00
         g%s(16)%m(2,1)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.50
       
       
      case('P 4 m m', '99')                         
       
         !Space group   #  99
         !Group setting # 107
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
       
      case('P 4 b m', '100')                         
       
         !Space group   # 100
         !Group setting # 108
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.00
       
       
      case('P 42 c m', '101')                        
       
         !Space group   # 101
         !Group setting # 109
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
       
      case('P 42 n m', '102')                        
       
         !Space group   # 102
         !Group setting # 110
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
       
      case('P 4 c c', '103')                         
       
         !Space group   # 103
         !Group setting # 111
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
       
      case('P 4 n c', '104')                         
       
         !Space group   # 104
         !Group setting # 112
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
       
      case('P 42 m c', '105')                        
       
         !Space group   # 105
         !Group setting # 113
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
       
      case('P 42 b c', '106')                        
       
         !Space group   # 106
         !Group setting # 114
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
       
      case('I 4 m m', '107')                         
       
         !Space group   # 107
         !Group setting # 115
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,1)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,2)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  =  1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  = -1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.50
       
       
      case('I 4 c m', '108')                         
       
         !Space group   # 108
         !Group setting # 116
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,1)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,2)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  =  1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  1.00000000
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  = -1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  1.00000000
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  1.00000000
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  1.00000000
       
       
      case('I 41 m d', '109')                        
       
         !Space group   # 109
         !Group setting # 117
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.75000000
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.75000000
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.75000000
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.75000000
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,1)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  1.00000000
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  1.25000000
       
         g%s(11)%m(1,2)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  1.00000000
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  1.25000000
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  =  1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  = -1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  1.00000000
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  1.25000000
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  1.00000000
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  1.25000000
       
       
      case('I 41 c d', '110')                        
       
         !Space group   # 110
         !Group setting # 118
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.75000000
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.75000000
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.25
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.25
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,1)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  1.00000000
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  1.25000000
       
         g%s(11)%m(1,2)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  1.00000000
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  1.25000000
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  =  1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  1.00000000
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  = -1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  1.00000000
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  1.00000000
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.75000000
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  1.00000000
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.75000000
       
       
      case('P -4 2 m', '111')                        
       
         !Space group   # 111
         !Group setting # 119
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   =  1.00
         g%s(2)%m(2,1)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   = -1.00
         g%s(3)%m(2,1)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
       
      case('P -4 2 c', '112')                        
       
         !Space group   # 112
         !Group setting # 120
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   =  1.00
         g%s(2)%m(2,1)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   = -1.00
         g%s(3)%m(2,1)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
       
      case('P -4 21 m', '113')                       
       
         !Space group   # 113
         !Group setting # 121
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   =  1.00
         g%s(2)%m(2,1)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   = -1.00
         g%s(3)%m(2,1)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.00
       
       
      case('P -4 21 c', '114')                       
       
         !Space group   # 114
         !Group setting # 122
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   =  1.00
         g%s(2)%m(2,1)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   = -1.00
         g%s(3)%m(2,1)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
       
      case('P -4 m 2', '115')                        
       
         !Space group   # 115
         !Group setting # 123
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   =  1.00
         g%s(2)%m(2,1)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   = -1.00
         g%s(3)%m(2,1)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
       
      case('P -4 c 2', '116')                        
       
         !Space group   # 116
         !Group setting # 124
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   =  1.00
         g%s(2)%m(2,1)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   = -1.00
         g%s(3)%m(2,1)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
       
      case('P -4 b 2', '117')                        
       
         !Space group   # 117
         !Group setting # 125
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   =  1.00
         g%s(2)%m(2,1)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   = -1.00
         g%s(3)%m(2,1)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.00
       
       
      case('P -4 n 2', '118')                        
       
         !Space group   # 118
         !Group setting # 126
       
         g%order   =   8
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   =  1.00
         g%s(2)%m(2,1)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   = -1.00
         g%s(3)%m(2,1)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
       
      case('I -4 m 2', '119')                        
       
         !Space group   # 119
         !Group setting # 127
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   =  1.00
         g%s(2)%m(2,1)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   = -1.00
         g%s(3)%m(2,1)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,2)  =  1.00
         g%s(13)%m(2,1)  =  1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,2)  = -1.00
         g%s(14)%m(2,1)  = -1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,1)  = -1.00
         g%s(15)%m(2,2)  =  1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.50
       
       
      case('I -4 c 2', '120')                        
       
         !Space group   # 120
         !Group setting # 128
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   =  1.00
         g%s(2)%m(2,1)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   = -1.00
         g%s(3)%m(2,1)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,2)  =  1.00
         g%s(13)%m(2,1)  =  1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  1.00000000
       
         g%s(14)%m(1,2)  = -1.00
         g%s(14)%m(2,1)  = -1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  1.00000000
       
         g%s(15)%m(1,1)  = -1.00
         g%s(15)%m(2,2)  =  1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  1.00000000
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  1.00000000
       
       
      case('I -4 2 m', '121')                        
       
         !Space group   # 121
         !Group setting # 129
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   =  1.00
         g%s(2)%m(2,1)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   = -1.00
         g%s(3)%m(2,1)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  =  1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  = -1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  =  1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(2,2)  = -1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.50
       
       
      case('I -4 2 d', '122')                        
       
         !Space group   # 122
         !Group setting # 130
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   =  1.00
         g%s(2)%m(2,1)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   = -1.00
         g%s(3)%m(2,1)   =  1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.75000000
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.75000000
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.75000000
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.75000000
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  =  1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  = -1.00
         g%s(12)%v(1)    =  1.00000000
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  1.25000000
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  =  1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  1.00000000
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  1.25000000
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(2,2)  = -1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  1.00000000
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  1.25000000
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  1.00000000
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  1.25000000
       
       
      case('P 4/m m m', '123')                       
       
         !Space group   # 123
         !Group setting # 131
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
       
      case('P 4/m c c', '124')                       
       
         !Space group   # 124
         !Group setting # 132
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    = -0.50
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    = -0.50
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    = -0.50
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    = -0.50
       
       
      case('P 4/n b m : 2', '125')                   
       
         !Space group   # 125
         !Group setting # 133
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    = -0.50
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    = -0.50
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    = -0.50
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    = -0.50
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    = -0.50
         g%s(14)%v(2)    = -0.50
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    = -0.50
         g%s(16)%v(2)    = -0.50
         g%s(16)%v(3)    =  0.00
       
       
      case('P 4/n b m : 1')                   
       
         !Space group   # 125
         !Group setting # 134
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.00
       
       
      case('P 4/n n c : 2', '126')                   
       
         !Space group   # 126
         !Group setting # 135
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    = -0.50
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    = -0.50
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    = -0.50
         g%s(12)%v(3)    = -0.50
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    = -0.50
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    = -0.50
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    = -0.50
         g%s(14)%v(2)    = -0.50
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    = -0.50
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    = -0.50
         g%s(16)%v(2)    = -0.50
         g%s(16)%v(3)    = -0.50
       
       
      case('P 4/n n c : 1')                   
       
         !Space group   # 126
         !Group setting # 136
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.50
       
       
      case('P 4/m b m', '127')                       
       
         !Space group   # 127
         !Group setting # 137
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    = -0.50
         g%s(12)%v(2)    = -0.50
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    = -0.50
         g%s(13)%v(2)    = -0.50
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    = -0.50
         g%s(15)%v(2)    = -0.50
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    = -0.50
         g%s(16)%v(2)    = -0.50
         g%s(16)%v(3)    =  0.00
       
       
      case('P 4/m n c', '128')                       
       
         !Space group   # 128
         !Group setting # 138
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    = -0.50
         g%s(12)%v(2)    = -0.50
         g%s(12)%v(3)    = -0.50
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    = -0.50
         g%s(13)%v(2)    = -0.50
         g%s(13)%v(3)    = -0.50
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    = -0.50
         g%s(15)%v(2)    = -0.50
         g%s(15)%v(3)    = -0.50
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    = -0.50
         g%s(16)%v(2)    = -0.50
         g%s(16)%v(3)    = -0.50
       
       
      case('P 4/n m m : 2', '129')                   
       
         !Space group   # 129
         !Group setting # 139
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    = -0.50
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    = -0.50
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    = -0.50
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    = -0.50
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    = -0.50
         g%s(14)%v(2)    = -0.50
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    = -0.50
         g%s(15)%v(2)    = -0.50
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
       
      case('P 4/n m m : 1')                   
       
         !Space group   # 129
         !Group setting # 140
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.00
       
       
      case('P 4/n c c : 2', '130')                   
       
         !Space group   # 130
         !Group setting # 141
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    = -0.50
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    = -0.50
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    = -0.50
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    = -0.50
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    = -0.50
         g%s(13)%v(3)    = -0.50
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    = -0.50
         g%s(14)%v(2)    = -0.50
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    = -0.50
         g%s(15)%v(2)    = -0.50
         g%s(15)%v(3)    = -0.50
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    = -0.50
       
       
      case('P 4/n c c : 1')                   
       
         !Space group   # 130
         !Group setting # 142
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    = -0.50
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    = -0.50
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    = -0.50
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    = -0.50
       
       
      case('P 42/m m c', '131')                      
       
         !Space group   # 131
         !Group setting # 143
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    = -0.50
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    = -0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    = -0.50
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    = -0.50
       
       
      case('P 42/m c m', '132')                      
       
         !Space group   # 132
         !Group setting # 144
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    = -0.50
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    = -0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    = -0.50
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    = -0.50
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
       
      case('P 42/n b c : 2', '133')                  
       
         !Space group   # 133
         !Group setting # 145
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    = -0.50
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    = -0.50
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    = -0.50
         g%s(11)%v(3)    = -0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    = -0.50
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    = -0.50
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    = -0.50
         g%s(14)%v(2)    = -0.50
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    = -0.50
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    = -0.50
         g%s(16)%v(2)    = -0.50
         g%s(16)%v(3)    = -0.50
       
       
      case('P 42/n b c : 1')                  
       
         !Space group   # 133
         !Group setting # 146
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.50
       
       
      case('P 42/n n m : 2', '134')                  
       
         !Space group   # 134
         !Group setting # 147
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    = -0.50
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    = -0.50
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    = -0.50
         g%s(11)%v(3)    = -0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    = -0.50
         g%s(12)%v(3)    = -0.50
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    = -0.50
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    = -0.50
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    = -0.50
         g%s(14)%v(2)    = -0.50
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    = -0.50
         g%s(16)%v(2)    = -0.50
         g%s(16)%v(3)    =  0.00
       
       
      case('P 42/n n m : 1')                  
       
         !Space group   # 134
         !Group setting # 148
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
       
      case('P 42/m b c', '135')                      
       
         !Space group   # 135
         !Group setting # 149
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    = -0.50
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    = -0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    = -0.50
         g%s(12)%v(2)    = -0.50
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    = -0.50
         g%s(13)%v(2)    = -0.50
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    = -0.50
         g%s(15)%v(2)    = -0.50
         g%s(15)%v(3)    = -0.50
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    = -0.50
         g%s(16)%v(2)    = -0.50
         g%s(16)%v(3)    = -0.50
       
       
      case('P 42/m n m', '136')                      
       
         !Space group   # 136
         !Group setting # 150
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    = -0.50
         g%s(10)%v(2)    = -0.50
         g%s(10)%v(3)    = -0.50
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    = -0.50
         g%s(11)%v(2)    = -0.50
         g%s(11)%v(3)    = -0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    = -0.50
         g%s(12)%v(2)    = -0.50
         g%s(12)%v(3)    = -0.50
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    = -0.50
         g%s(13)%v(2)    = -0.50
         g%s(13)%v(3)    = -0.50
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
       
      case('P 42/n m c : 2', '137')                  
       
         !Space group   # 137
         !Group setting # 151
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    = -0.50
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    = -0.50
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    = -0.50
         g%s(11)%v(3)    = -0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    = -0.50
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    = -0.50
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    = -0.50
         g%s(14)%v(2)    = -0.50
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    = -0.50
         g%s(15)%v(2)    = -0.50
         g%s(15)%v(3)    = -0.50
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    = -0.50
       
       
      case('P 42/n m c : 1')                  
       
         !Space group   # 137
         !Group setting # 152
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.50
       
       
      case('P 42/n c m : 2', '138')                  
       
         !Space group   # 138
         !Group setting # 153
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    = -0.50
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    = -0.50
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    = -0.50
         g%s(11)%v(3)    = -0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    = -0.50
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    = -0.50
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    = -0.50
         g%s(13)%v(3)    = -0.50
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    = -0.50
         g%s(14)%v(2)    = -0.50
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    = -0.50
         g%s(15)%v(2)    = -0.50
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
       
      case('P 42/n c m : 1')                  
       
         !Space group   # 138
         !Group setting # 154
       
         g%order   =  16
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.00
       
       
      case('I 4/m m m', '139')                       
       
         !Space group   # 139
         !Group setting # 155
       
         g%order   =  32
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  =  1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  =  1.00
         g%s(17)%v(1)    =  0.50
         g%s(17)%v(2)    =  0.50
         g%s(17)%v(3)    =  0.50
       
         g%s(18)%m(1,2)  = -1.00
         g%s(18)%m(2,1)  =  1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.50
         g%s(18)%v(2)    =  0.50
         g%s(18)%v(3)    =  0.50
       
         g%s(19)%m(1,2)  =  1.00
         g%s(19)%m(2,1)  = -1.00
         g%s(19)%m(3,3)  =  1.00
         g%s(19)%v(1)    =  0.50
         g%s(19)%v(2)    =  0.50
         g%s(19)%v(3)    =  0.50
       
         g%s(20)%m(1,1)  =  1.00
         g%s(20)%m(2,2)  = -1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.50
         g%s(20)%v(2)    =  0.50
         g%s(20)%v(3)    =  0.50
       
         g%s(21)%m(1,1)  = -1.00
         g%s(21)%m(2,2)  =  1.00
         g%s(21)%m(3,3)  = -1.00
         g%s(21)%v(1)    =  0.50
         g%s(21)%v(2)    =  0.50
         g%s(21)%v(3)    =  0.50
       
         g%s(22)%m(1,1)  = -1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.50
         g%s(22)%v(2)    =  0.50
         g%s(22)%v(3)    =  0.50
       
         g%s(23)%m(1,2)  =  1.00
         g%s(23)%m(2,1)  =  1.00
         g%s(23)%m(3,3)  = -1.00
         g%s(23)%v(1)    =  0.50
         g%s(23)%v(2)    =  0.50
         g%s(23)%v(3)    =  0.50
       
         g%s(24)%m(1,2)  = -1.00
         g%s(24)%m(2,1)  = -1.00
         g%s(24)%m(3,3)  = -1.00
         g%s(24)%v(1)    =  0.50
         g%s(24)%v(2)    =  0.50
         g%s(24)%v(3)    =  0.50
       
         g%s(25)%m(1,1)  = -1.00
         g%s(25)%m(2,2)  = -1.00
         g%s(25)%m(3,3)  = -1.00
         g%s(25)%v(1)    =  0.50
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  0.50
       
         g%s(26)%m(1,2)  =  1.00
         g%s(26)%m(2,1)  = -1.00
         g%s(26)%m(3,3)  = -1.00
         g%s(26)%v(1)    =  0.50
         g%s(26)%v(2)    =  0.50
         g%s(26)%v(3)    =  0.50
       
         g%s(27)%m(1,2)  = -1.00
         g%s(27)%m(2,1)  =  1.00
         g%s(27)%m(3,3)  = -1.00
         g%s(27)%v(1)    =  0.50
         g%s(27)%v(2)    =  0.50
         g%s(27)%v(3)    =  0.50
       
         g%s(28)%m(1,1)  = -1.00
         g%s(28)%m(2,2)  =  1.00
         g%s(28)%m(3,3)  =  1.00
         g%s(28)%v(1)    =  0.50
         g%s(28)%v(2)    =  0.50
         g%s(28)%v(3)    =  0.50
       
         g%s(29)%m(1,1)  =  1.00
         g%s(29)%m(2,2)  = -1.00
         g%s(29)%m(3,3)  =  1.00
         g%s(29)%v(1)    =  0.50
         g%s(29)%v(2)    =  0.50
         g%s(29)%v(3)    =  0.50
       
         g%s(30)%m(1,1)  =  1.00
         g%s(30)%m(2,2)  =  1.00
         g%s(30)%m(3,3)  = -1.00
         g%s(30)%v(1)    =  0.50
         g%s(30)%v(2)    =  0.50
         g%s(30)%v(3)    =  0.50
       
         g%s(31)%m(1,2)  = -1.00
         g%s(31)%m(2,1)  = -1.00
         g%s(31)%m(3,3)  =  1.00
         g%s(31)%v(1)    =  0.50
         g%s(31)%v(2)    =  0.50
         g%s(31)%v(3)    =  0.50
       
         g%s(32)%m(1,2)  =  1.00
         g%s(32)%m(2,1)  =  1.00
         g%s(32)%m(3,3)  =  1.00
         g%s(32)%v(1)    =  0.50
         g%s(32)%v(2)    =  0.50
         g%s(32)%v(3)    =  0.50
       
       
      case('I 4/m c m', '140')                       
       
         !Space group   # 140
         !Group setting # 156
       
         g%order   =  32
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    = -0.50
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    = -0.50
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    = -0.50
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    = -0.50
       
         g%s(17)%m(1,1)  =  1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  =  1.00
         g%s(17)%v(1)    =  0.50
         g%s(17)%v(2)    =  0.50
         g%s(17)%v(3)    =  0.50
       
         g%s(18)%m(1,2)  = -1.00
         g%s(18)%m(2,1)  =  1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.50
         g%s(18)%v(2)    =  0.50
         g%s(18)%v(3)    =  0.50
       
         g%s(19)%m(1,2)  =  1.00
         g%s(19)%m(2,1)  = -1.00
         g%s(19)%m(3,3)  =  1.00
         g%s(19)%v(1)    =  0.50
         g%s(19)%v(2)    =  0.50
         g%s(19)%v(3)    =  0.50
       
         g%s(20)%m(1,1)  =  1.00
         g%s(20)%m(2,2)  = -1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.50
         g%s(20)%v(2)    =  0.50
         g%s(20)%v(3)    =  1.00000000
       
         g%s(21)%m(1,1)  = -1.00
         g%s(21)%m(2,2)  =  1.00
         g%s(21)%m(3,3)  = -1.00
         g%s(21)%v(1)    =  0.50
         g%s(21)%v(2)    =  0.50
         g%s(21)%v(3)    =  1.00000000
       
         g%s(22)%m(1,1)  = -1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.50
         g%s(22)%v(2)    =  0.50
         g%s(22)%v(3)    =  0.50
       
         g%s(23)%m(1,2)  =  1.00
         g%s(23)%m(2,1)  =  1.00
         g%s(23)%m(3,3)  = -1.00
         g%s(23)%v(1)    =  0.50
         g%s(23)%v(2)    =  0.50
         g%s(23)%v(3)    =  1.00000000
       
         g%s(24)%m(1,2)  = -1.00
         g%s(24)%m(2,1)  = -1.00
         g%s(24)%m(3,3)  = -1.00
         g%s(24)%v(1)    =  0.50
         g%s(24)%v(2)    =  0.50
         g%s(24)%v(3)    =  1.00000000
       
         g%s(25)%m(1,1)  = -1.00
         g%s(25)%m(2,2)  = -1.00
         g%s(25)%m(3,3)  = -1.00
         g%s(25)%v(1)    =  0.50
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  0.50
       
         g%s(26)%m(1,2)  =  1.00
         g%s(26)%m(2,1)  = -1.00
         g%s(26)%m(3,3)  = -1.00
         g%s(26)%v(1)    =  0.50
         g%s(26)%v(2)    =  0.50
         g%s(26)%v(3)    =  0.50
       
         g%s(27)%m(1,2)  = -1.00
         g%s(27)%m(2,1)  =  1.00
         g%s(27)%m(3,3)  = -1.00
         g%s(27)%v(1)    =  0.50
         g%s(27)%v(2)    =  0.50
         g%s(27)%v(3)    =  0.50
       
         g%s(28)%m(1,1)  = -1.00
         g%s(28)%m(2,2)  =  1.00
         g%s(28)%m(3,3)  =  1.00
         g%s(28)%v(1)    =  0.50
         g%s(28)%v(2)    =  0.50
         g%s(28)%v(3)    =  0.00
       
         g%s(29)%m(1,1)  =  1.00
         g%s(29)%m(2,2)  = -1.00
         g%s(29)%m(3,3)  =  1.00
         g%s(29)%v(1)    =  0.50
         g%s(29)%v(2)    =  0.50
         g%s(29)%v(3)    =  0.00
       
         g%s(30)%m(1,1)  =  1.00
         g%s(30)%m(2,2)  =  1.00
         g%s(30)%m(3,3)  = -1.00
         g%s(30)%v(1)    =  0.50
         g%s(30)%v(2)    =  0.50
         g%s(30)%v(3)    =  0.50
       
         g%s(31)%m(1,2)  = -1.00
         g%s(31)%m(2,1)  = -1.00
         g%s(31)%m(3,3)  =  1.00
         g%s(31)%v(1)    =  0.50
         g%s(31)%v(2)    =  0.50
         g%s(31)%v(3)    =  0.00
       
         g%s(32)%m(1,2)  =  1.00
         g%s(32)%m(2,1)  =  1.00
         g%s(32)%m(3,3)  =  1.00
         g%s(32)%v(1)    =  0.50
         g%s(32)%v(2)    =  0.50
         g%s(32)%v(3)    =  0.00
       
       
      case('I 41/a m d : 2', '141')                  
       
         !Space group   # 141
         !Group setting # 157
       
         g%order   =  32
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.25
         g%s(2)%v(2)     =  0.75000000
         g%s(2)%v(3)     =  0.25
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.25
         g%s(3)%v(2)     =  0.25
         g%s(3)%v(3)     =  0.75000000
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.25
         g%s(7)%v(2)     =  0.75000000
         g%s(7)%v(3)     =  0.25
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.25
         g%s(8)%v(2)     =  0.25
         g%s(8)%v(3)     =  0.75000000
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    = -0.25
         g%s(10)%v(2)    = -0.75000000
         g%s(10)%v(3)    = -0.25
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    = -0.25
         g%s(11)%v(2)    = -0.25
         g%s(11)%v(3)    = -0.75000000
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    = -0.50
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    = -0.50
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    = -0.25
         g%s(15)%v(2)    = -0.75000000
         g%s(15)%v(3)    = -0.25
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    = -0.25
         g%s(16)%v(2)    = -0.25
         g%s(16)%v(3)    = -0.75000000
       
         g%s(17)%m(1,1)  =  1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  =  1.00
         g%s(17)%v(1)    =  0.50
         g%s(17)%v(2)    =  0.50
         g%s(17)%v(3)    =  0.50
       
         g%s(18)%m(1,2)  = -1.00
         g%s(18)%m(2,1)  =  1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.75000000
         g%s(18)%v(2)    =  1.25000000
         g%s(18)%v(3)    =  0.75000000
       
         g%s(19)%m(1,2)  =  1.00
         g%s(19)%m(2,1)  = -1.00
         g%s(19)%m(3,3)  =  1.00
         g%s(19)%v(1)    =  0.75000000
         g%s(19)%v(2)    =  0.75000000
         g%s(19)%v(3)    =  1.25000000
       
         g%s(20)%m(1,1)  =  1.00
         g%s(20)%m(2,2)  = -1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.50
         g%s(20)%v(2)    =  0.50
         g%s(20)%v(3)    =  0.50
       
         g%s(21)%m(1,1)  = -1.00
         g%s(21)%m(2,2)  =  1.00
         g%s(21)%m(3,3)  = -1.00
         g%s(21)%v(1)    =  0.50
         g%s(21)%v(2)    =  1.00000000
         g%s(21)%v(3)    =  0.50
       
         g%s(22)%m(1,1)  = -1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.50
         g%s(22)%v(2)    =  1.00000000
         g%s(22)%v(3)    =  0.50
       
         g%s(23)%m(1,2)  =  1.00
         g%s(23)%m(2,1)  =  1.00
         g%s(23)%m(3,3)  = -1.00
         g%s(23)%v(1)    =  0.75000000
         g%s(23)%v(2)    =  1.25000000
         g%s(23)%v(3)    =  0.75000000
       
         g%s(24)%m(1,2)  = -1.00
         g%s(24)%m(2,1)  = -1.00
         g%s(24)%m(3,3)  = -1.00
         g%s(24)%v(1)    =  0.75000000
         g%s(24)%v(2)    =  0.75000000
         g%s(24)%v(3)    =  1.25000000
       
         g%s(25)%m(1,1)  = -1.00
         g%s(25)%m(2,2)  = -1.00
         g%s(25)%m(3,3)  = -1.00
         g%s(25)%v(1)    =  0.50
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  0.50
       
         g%s(26)%m(1,2)  =  1.00
         g%s(26)%m(2,1)  = -1.00
         g%s(26)%m(3,3)  = -1.00
         g%s(26)%v(1)    =  0.25
         g%s(26)%v(2)    = -0.25
         g%s(26)%v(3)    =  0.25
       
         g%s(27)%m(1,2)  = -1.00
         g%s(27)%m(2,1)  =  1.00
         g%s(27)%m(3,3)  = -1.00
         g%s(27)%v(1)    =  0.25
         g%s(27)%v(2)    =  0.25
         g%s(27)%v(3)    = -0.25
       
         g%s(28)%m(1,1)  = -1.00
         g%s(28)%m(2,2)  =  1.00
         g%s(28)%m(3,3)  =  1.00
         g%s(28)%v(1)    =  0.50
         g%s(28)%v(2)    =  0.50
         g%s(28)%v(3)    =  0.50
       
         g%s(29)%m(1,1)  =  1.00
         g%s(29)%m(2,2)  = -1.00
         g%s(29)%m(3,3)  =  1.00
         g%s(29)%v(1)    =  0.50
         g%s(29)%v(2)    =  0.00
         g%s(29)%v(3)    =  0.50
       
         g%s(30)%m(1,1)  =  1.00
         g%s(30)%m(2,2)  =  1.00
         g%s(30)%m(3,3)  = -1.00
         g%s(30)%v(1)    =  0.50
         g%s(30)%v(2)    =  0.00
         g%s(30)%v(3)    =  0.50
       
         g%s(31)%m(1,2)  = -1.00
         g%s(31)%m(2,1)  = -1.00
         g%s(31)%m(3,3)  =  1.00
         g%s(31)%v(1)    =  0.25
         g%s(31)%v(2)    = -0.25
         g%s(31)%v(3)    =  0.25
       
         g%s(32)%m(1,2)  =  1.00
         g%s(32)%m(2,1)  =  1.00
         g%s(32)%m(3,3)  =  1.00
         g%s(32)%v(1)    =  0.25
         g%s(32)%v(2)    =  0.25
         g%s(32)%v(3)    = -0.25
       
       
      case('I 41/a m d : 1')                  
       
         !Space group   # 141
         !Group setting # 158
       
         g%order   =  32
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.75000000
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.75000000
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.75000000
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.75000000
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.75000000
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.75000000
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.75000000
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.75000000
       
         g%s(17)%m(1,1)  =  1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  =  1.00
         g%s(17)%v(1)    =  0.50
         g%s(17)%v(2)    =  0.50
         g%s(17)%v(3)    =  0.50
       
         g%s(18)%m(1,2)  = -1.00
         g%s(18)%m(2,1)  =  1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  1.00000000
         g%s(18)%v(2)    =  0.50
         g%s(18)%v(3)    =  1.25000000
       
         g%s(19)%m(1,2)  =  1.00
         g%s(19)%m(2,1)  = -1.00
         g%s(19)%m(3,3)  =  1.00
         g%s(19)%v(1)    =  1.00000000
         g%s(19)%v(2)    =  0.50
         g%s(19)%v(3)    =  1.25000000
       
         g%s(20)%m(1,1)  =  1.00
         g%s(20)%m(2,2)  = -1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  1.00000000
         g%s(20)%v(2)    =  0.50
         g%s(20)%v(3)    =  1.25000000
       
         g%s(21)%m(1,1)  = -1.00
         g%s(21)%m(2,2)  =  1.00
         g%s(21)%m(3,3)  = -1.00
         g%s(21)%v(1)    =  1.00000000
         g%s(21)%v(2)    =  0.50
         g%s(21)%v(3)    =  1.25000000
       
         g%s(22)%m(1,1)  = -1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.50
         g%s(22)%v(2)    =  0.50
         g%s(22)%v(3)    =  0.50
       
         g%s(23)%m(1,2)  =  1.00
         g%s(23)%m(2,1)  =  1.00
         g%s(23)%m(3,3)  = -1.00
         g%s(23)%v(1)    =  0.50
         g%s(23)%v(2)    =  0.50
         g%s(23)%v(3)    =  0.50
       
         g%s(24)%m(1,2)  = -1.00
         g%s(24)%m(2,1)  = -1.00
         g%s(24)%m(3,3)  = -1.00
         g%s(24)%v(1)    =  0.50
         g%s(24)%v(2)    =  0.50
         g%s(24)%v(3)    =  0.50
       
         g%s(25)%m(1,1)  = -1.00
         g%s(25)%m(2,2)  = -1.00
         g%s(25)%m(3,3)  = -1.00
         g%s(25)%v(1)    =  1.00000000
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  1.25000000
       
         g%s(26)%m(1,2)  =  1.00
         g%s(26)%m(2,1)  = -1.00
         g%s(26)%m(3,3)  = -1.00
         g%s(26)%v(1)    =  0.50
         g%s(26)%v(2)    =  0.50
         g%s(26)%v(3)    =  0.50
       
         g%s(27)%m(1,2)  = -1.00
         g%s(27)%m(2,1)  =  1.00
         g%s(27)%m(3,3)  = -1.00
         g%s(27)%v(1)    =  0.50
         g%s(27)%v(2)    =  0.50
         g%s(27)%v(3)    =  0.50
       
         g%s(28)%m(1,1)  = -1.00
         g%s(28)%m(2,2)  =  1.00
         g%s(28)%m(3,3)  =  1.00
         g%s(28)%v(1)    =  0.50
         g%s(28)%v(2)    =  0.50
         g%s(28)%v(3)    =  0.50
       
         g%s(29)%m(1,1)  =  1.00
         g%s(29)%m(2,2)  = -1.00
         g%s(29)%m(3,3)  =  1.00
         g%s(29)%v(1)    =  0.50
         g%s(29)%v(2)    =  0.50
         g%s(29)%v(3)    =  0.50
       
         g%s(30)%m(1,1)  =  1.00
         g%s(30)%m(2,2)  =  1.00
         g%s(30)%m(3,3)  = -1.00
         g%s(30)%v(1)    =  1.00000000
         g%s(30)%v(2)    =  0.50
         g%s(30)%v(3)    =  1.25000000
       
         g%s(31)%m(1,2)  = -1.00
         g%s(31)%m(2,1)  = -1.00
         g%s(31)%m(3,3)  =  1.00
         g%s(31)%v(1)    =  1.00000000
         g%s(31)%v(2)    =  0.50
         g%s(31)%v(3)    =  1.25000000
       
         g%s(32)%m(1,2)  =  1.00
         g%s(32)%m(2,1)  =  1.00
         g%s(32)%m(3,3)  =  1.00
         g%s(32)%v(1)    =  1.00000000
         g%s(32)%v(2)    =  0.50
         g%s(32)%v(3)    =  1.25000000
       
       
      case('I 41/a c d : 2', '142')                  
       
         !Space group   # 142
         !Group setting # 159
       
         g%order   =  32
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.25
         g%s(2)%v(2)     =  0.75000000
         g%s(2)%v(3)     =  0.25
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.25
         g%s(3)%v(2)     =  0.25
         g%s(3)%v(3)     =  0.75000000
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.25
         g%s(7)%v(2)     =  0.75000000
         g%s(7)%v(3)     =  0.75000000
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.25
         g%s(8)%v(2)     =  0.25
         g%s(8)%v(3)     =  0.25
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    = -0.25
         g%s(10)%v(2)    = -0.75000000
         g%s(10)%v(3)    = -0.25
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    = -0.25
         g%s(11)%v(2)    = -0.25
         g%s(11)%v(3)    = -0.75000000
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    = -0.50
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    = -0.50
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    = -0.50
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    = -0.25
         g%s(15)%v(2)    = -0.75000000
         g%s(15)%v(3)    = -0.75000000
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    = -0.25
         g%s(16)%v(2)    = -0.25
         g%s(16)%v(3)    = -0.25
       
         g%s(17)%m(1,1)  =  1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  =  1.00
         g%s(17)%v(1)    =  0.50
         g%s(17)%v(2)    =  0.50
         g%s(17)%v(3)    =  0.50
       
         g%s(18)%m(1,2)  = -1.00
         g%s(18)%m(2,1)  =  1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.75000000
         g%s(18)%v(2)    =  1.25000000
         g%s(18)%v(3)    =  0.75000000
       
         g%s(19)%m(1,2)  =  1.00
         g%s(19)%m(2,1)  = -1.00
         g%s(19)%m(3,3)  =  1.00
         g%s(19)%v(1)    =  0.75000000
         g%s(19)%v(2)    =  0.75000000
         g%s(19)%v(3)    =  1.25000000
       
         g%s(20)%m(1,1)  =  1.00
         g%s(20)%m(2,2)  = -1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.50
         g%s(20)%v(2)    =  0.50
         g%s(20)%v(3)    =  1.00000000
       
         g%s(21)%m(1,1)  = -1.00
         g%s(21)%m(2,2)  =  1.00
         g%s(21)%m(3,3)  = -1.00
         g%s(21)%v(1)    =  1.00000000
         g%s(21)%v(2)    =  0.50
         g%s(21)%v(3)    =  0.50
       
         g%s(22)%m(1,1)  = -1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.50
         g%s(22)%v(2)    =  1.00000000
         g%s(22)%v(3)    =  0.50
       
         g%s(23)%m(1,2)  =  1.00
         g%s(23)%m(2,1)  =  1.00
         g%s(23)%m(3,3)  = -1.00
         g%s(23)%v(1)    =  0.75000000
         g%s(23)%v(2)    =  1.25000000
         g%s(23)%v(3)    =  1.25000000
       
         g%s(24)%m(1,2)  = -1.00
         g%s(24)%m(2,1)  = -1.00
         g%s(24)%m(3,3)  = -1.00
         g%s(24)%v(1)    =  0.75000000
         g%s(24)%v(2)    =  0.75000000
         g%s(24)%v(3)    =  0.75000000
       
         g%s(25)%m(1,1)  = -1.00
         g%s(25)%m(2,2)  = -1.00
         g%s(25)%m(3,3)  = -1.00
         g%s(25)%v(1)    =  0.50
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  0.50
       
         g%s(26)%m(1,2)  =  1.00
         g%s(26)%m(2,1)  = -1.00
         g%s(26)%m(3,3)  = -1.00
         g%s(26)%v(1)    =  0.25
         g%s(26)%v(2)    = -0.25
         g%s(26)%v(3)    =  0.25
       
         g%s(27)%m(1,2)  = -1.00
         g%s(27)%m(2,1)  =  1.00
         g%s(27)%m(3,3)  = -1.00
         g%s(27)%v(1)    =  0.25
         g%s(27)%v(2)    =  0.25
         g%s(27)%v(3)    = -0.25
       
         g%s(28)%m(1,1)  = -1.00
         g%s(28)%m(2,2)  =  1.00
         g%s(28)%m(3,3)  =  1.00
         g%s(28)%v(1)    =  0.50
         g%s(28)%v(2)    =  0.50
         g%s(28)%v(3)    =  0.00
       
         g%s(29)%m(1,1)  =  1.00
         g%s(29)%m(2,2)  = -1.00
         g%s(29)%m(3,3)  =  1.00
         g%s(29)%v(1)    =  0.00
         g%s(29)%v(2)    =  0.50
         g%s(29)%v(3)    =  0.50
       
         g%s(30)%m(1,1)  =  1.00
         g%s(30)%m(2,2)  =  1.00
         g%s(30)%m(3,3)  = -1.00
         g%s(30)%v(1)    =  0.50
         g%s(30)%v(2)    =  0.00
         g%s(30)%v(3)    =  0.50
       
         g%s(31)%m(1,2)  = -1.00
         g%s(31)%m(2,1)  = -1.00
         g%s(31)%m(3,3)  =  1.00
         g%s(31)%v(1)    =  0.25
         g%s(31)%v(2)    = -0.25
         g%s(31)%v(3)    = -0.25
       
         g%s(32)%m(1,2)  =  1.00
         g%s(32)%m(2,1)  =  1.00
         g%s(32)%m(3,3)  =  1.00
         g%s(32)%v(1)    =  0.25
         g%s(32)%v(2)    =  0.25
         g%s(32)%v(3)    =  0.25
       
       
      case('I 41/a c d : 1')                  
       
         !Space group   # 142
         !Group setting # 160
       
         g%order   =  32
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.75000000
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.75000000
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.25
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.25
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.75000000
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,1)  =  1.00
         g%s(14)%m(2,2)  =  1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.75000000
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.25
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.25
       
         g%s(17)%m(1,1)  =  1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  =  1.00
         g%s(17)%v(1)    =  0.50
         g%s(17)%v(2)    =  0.50
         g%s(17)%v(3)    =  0.50
       
         g%s(18)%m(1,2)  = -1.00
         g%s(18)%m(2,1)  =  1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  1.00000000
         g%s(18)%v(2)    =  0.50
         g%s(18)%v(3)    =  1.25000000
       
         g%s(19)%m(1,2)  =  1.00
         g%s(19)%m(2,1)  = -1.00
         g%s(19)%m(3,3)  =  1.00
         g%s(19)%v(1)    =  1.00000000
         g%s(19)%v(2)    =  0.50
         g%s(19)%v(3)    =  1.25000000
       
         g%s(20)%m(1,1)  =  1.00
         g%s(20)%m(2,2)  = -1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  1.00000000
         g%s(20)%v(2)    =  0.50
         g%s(20)%v(3)    =  0.75000000
       
         g%s(21)%m(1,1)  = -1.00
         g%s(21)%m(2,2)  =  1.00
         g%s(21)%m(3,3)  = -1.00
         g%s(21)%v(1)    =  1.00000000
         g%s(21)%v(2)    =  0.50
         g%s(21)%v(3)    =  0.75000000
       
         g%s(22)%m(1,1)  = -1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.50
         g%s(22)%v(2)    =  0.50
         g%s(22)%v(3)    =  0.50
       
         g%s(23)%m(1,2)  =  1.00
         g%s(23)%m(2,1)  =  1.00
         g%s(23)%m(3,3)  = -1.00
         g%s(23)%v(1)    =  0.50
         g%s(23)%v(2)    =  0.50
         g%s(23)%v(3)    =  1.00000000
       
         g%s(24)%m(1,2)  = -1.00
         g%s(24)%m(2,1)  = -1.00
         g%s(24)%m(3,3)  = -1.00
         g%s(24)%v(1)    =  0.50
         g%s(24)%v(2)    =  0.50
         g%s(24)%v(3)    =  1.00000000
       
         g%s(25)%m(1,1)  = -1.00
         g%s(25)%m(2,2)  = -1.00
         g%s(25)%m(3,3)  = -1.00
         g%s(25)%v(1)    =  1.00000000
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  1.25000000
       
         g%s(26)%m(1,2)  =  1.00
         g%s(26)%m(2,1)  = -1.00
         g%s(26)%m(3,3)  = -1.00
         g%s(26)%v(1)    =  0.50
         g%s(26)%v(2)    =  0.50
         g%s(26)%v(3)    =  0.50
       
         g%s(27)%m(1,2)  = -1.00
         g%s(27)%m(2,1)  =  1.00
         g%s(27)%m(3,3)  = -1.00
         g%s(27)%v(1)    =  0.50
         g%s(27)%v(2)    =  0.50
         g%s(27)%v(3)    =  0.50
       
         g%s(28)%m(1,1)  = -1.00
         g%s(28)%m(2,2)  =  1.00
         g%s(28)%m(3,3)  =  1.00
         g%s(28)%v(1)    =  0.50
         g%s(28)%v(2)    =  0.50
         g%s(28)%v(3)    =  1.00000000
       
         g%s(29)%m(1,1)  =  1.00
         g%s(29)%m(2,2)  = -1.00
         g%s(29)%m(3,3)  =  1.00
         g%s(29)%v(1)    =  0.50
         g%s(29)%v(2)    =  0.50
         g%s(29)%v(3)    =  1.00000000
       
         g%s(30)%m(1,1)  =  1.00
         g%s(30)%m(2,2)  =  1.00
         g%s(30)%m(3,3)  = -1.00
         g%s(30)%v(1)    =  1.00000000
         g%s(30)%v(2)    =  0.50
         g%s(30)%v(3)    =  1.25000000
       
         g%s(31)%m(1,2)  = -1.00
         g%s(31)%m(2,1)  = -1.00
         g%s(31)%m(3,3)  =  1.00
         g%s(31)%v(1)    =  1.00000000
         g%s(31)%v(2)    =  0.50
         g%s(31)%v(3)    =  0.75000000
       
         g%s(32)%m(1,2)  =  1.00
         g%s(32)%m(2,1)  =  1.00
         g%s(32)%m(3,3)  =  1.00
         g%s(32)%v(1)    =  1.00000000
         g%s(32)%v(2)    =  0.50
         g%s(32)%v(3)    =  0.75000000
       
       
      case('P 3', '143')                             
       
         !Space group   # 143
         !Group setting # 161
       
         g%order   =   3
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
       
      case('P 31', '144')                            
       
         !Space group   # 144
         !Group setting # 162
       
         g%order   =   3
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.3333333333333
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.6666666666666
       
       
      case('P 32', '145')                            
       
         !Space group   # 145
         !Group setting # 163
       
         g%order   =   3
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.6666666666666
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.3333333333333
       
       
      case('R 3 : H')                         
       
         !Space group   # 146
         !Group setting # 164
       
         g%order   =   9
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.3333333333333
         g%s(4)%v(2)     =  0.6666666666666
         g%s(4)%v(3)     =  0.6666666666666
       
         g%s(5)%m(1,2)   = -1.00
         g%s(5)%m(2,1)   =  1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.3333333333333
         g%s(5)%v(2)     =  0.6666666666666
         g%s(5)%v(3)     =  0.6666666666666
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.3333333333333
         g%s(6)%v(2)     =  0.6666666666666
         g%s(6)%v(3)     =  0.6666666666666
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.6666666666666
         g%s(7)%v(2)     =  0.3333333333333
         g%s(7)%v(3)     =  0.3333333333333
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.6666666666666
         g%s(8)%v(2)     =  0.3333333333333
         g%s(8)%v(3)     =  0.3333333333333
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,1)   = -1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.6666666666666
         g%s(9)%v(2)     =  0.3333333333333
         g%s(9)%v(3)     =  0.3333333333333
       
       
      case('R 3 : R', '146')                         
       
         !Space group   # 146
         !Group setting # 165
       
         g%order   =   3
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,3)   =  1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,1)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
       
      case('P -3', '147')                            
       
         !Space group   # 147
         !Group setting # 166
       
         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
       
      case('R -3 : H')                        
       
         !Space group   # 148
         !Group setting # 167
       
         g%order   =  18
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.3333333333333
         g%s(7)%v(2)     =  0.6666666666666
         g%s(7)%v(3)     =  0.6666666666666
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.3333333333333
         g%s(8)%v(2)     =  0.6666666666666
         g%s(8)%v(3)     =  0.6666666666666
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,1)   = -1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.3333333333333
         g%s(9)%v(2)     =  0.6666666666666
         g%s(9)%v(3)     =  0.6666666666666
       
         g%s(10)%m(1,1)  = -1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.3333333333333
         g%s(10)%v(2)    =  0.6666666666666
         g%s(10)%v(3)    =  0.6666666666666
       
         g%s(11)%m(1,2)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.3333333333333
         g%s(11)%v(2)    =  0.6666666666666
         g%s(11)%v(3)    =  0.6666666666666
       
         g%s(12)%m(1,1)  =  1.00
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,1)  =  1.00
         g%s(12)%m(3,3)  = -1.00
         g%s(12)%v(1)    =  0.3333333333333
         g%s(12)%v(2)    =  0.6666666666666
         g%s(12)%v(3)    =  0.6666666666666
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  =  1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.6666666666666
         g%s(13)%v(2)    =  0.3333333333333
         g%s(13)%v(3)    =  0.3333333333333
       
         g%s(14)%m(1,2)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(2,2)  = -1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.6666666666666
         g%s(14)%v(2)    =  0.3333333333333
         g%s(14)%v(3)    =  0.3333333333333
       
         g%s(15)%m(1,1)  = -1.00
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.6666666666666
         g%s(15)%v(2)    =  0.3333333333333
         g%s(15)%v(3)    =  0.3333333333333
       
         g%s(16)%m(1,1)  = -1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.6666666666666
         g%s(16)%v(2)    =  0.3333333333333
         g%s(16)%v(3)    =  0.3333333333333
       
         g%s(17)%m(1,2)  =  1.00
         g%s(17)%m(2,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.6666666666666
         g%s(17)%v(2)    =  0.3333333333333
         g%s(17)%v(3)    =  0.3333333333333
       
         g%s(18)%m(1,1)  =  1.00
         g%s(18)%m(1,2)  = -1.00
         g%s(18)%m(2,1)  =  1.00
         g%s(18)%m(3,3)  = -1.00
         g%s(18)%v(1)    =  0.6666666666666
         g%s(18)%v(2)    =  0.3333333333333
         g%s(18)%v(3)    =  0.3333333333333
       
       
      case('R -3 : R', '148')                        
       
         !Space group   # 148
         !Group setting # 168
       
         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,3)   =  1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,1)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   = -1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,2)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,3)   = -1.00
         g%s(6)%m(3,1)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
       
      case('P 3 1 2', '149')                         
       
         !Space group   # 149
         !Group setting # 169
       
         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
       
      case('P 3 2 1', '150')                         
       
         !Space group   # 150
         !Group setting # 170
       
         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
       
      case('P 31 1 2', '151')                        
       
         !Space group   # 151
         !Group setting # 171
       
         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.3333333333333
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.6666666666666
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.6666666666666
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.3333333333333
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
       
      case('P 31 2 1', '152')                        
       
         !Space group   # 152
         !Group setting # 172
       
         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.3333333333333
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.6666666666666
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.6666666666666
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.3333333333333
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
       
      case('P 32 1 2', '153')                        
       
         !Space group   # 153
         !Group setting # 173
       
         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.6666666666666
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.3333333333333
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.3333333333333
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.6666666666666
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
       
      case('P 32 2 1', '154')                        
       
         !Space group   # 154
         !Group setting # 174
       
         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.6666666666666
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.3333333333333
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.3333333333333
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.6666666666666
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
       
      case('R 3 2 : H')                       
       
         !Space group   # 155
         !Group setting # 175
       
         g%order   =  18
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.3333333333333
         g%s(7)%v(2)     =  0.6666666666666
         g%s(7)%v(3)     =  0.6666666666666
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.3333333333333
         g%s(8)%v(2)     =  0.6666666666666
         g%s(8)%v(3)     =  0.6666666666666
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,1)   = -1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.3333333333333
         g%s(9)%v(2)     =  0.6666666666666
         g%s(9)%v(3)     =  0.6666666666666
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.3333333333333
         g%s(10)%v(2)    =  0.6666666666666
         g%s(10)%v(3)    =  0.6666666666666
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.3333333333333
         g%s(11)%v(2)    =  0.6666666666666
         g%s(11)%v(3)    =  0.6666666666666
       
         g%s(12)%m(1,2)  =  1.00
         g%s(12)%m(2,1)  =  1.00
         g%s(12)%m(3,3)  = -1.00
         g%s(12)%v(1)    =  0.3333333333333
         g%s(12)%v(2)    =  0.6666666666666
         g%s(12)%v(3)    =  0.6666666666666
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  =  1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.6666666666666
         g%s(13)%v(2)    =  0.3333333333333
         g%s(13)%v(3)    =  0.3333333333333
       
         g%s(14)%m(1,2)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(2,2)  = -1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.6666666666666
         g%s(14)%v(2)    =  0.3333333333333
         g%s(14)%v(3)    =  0.3333333333333
       
         g%s(15)%m(1,1)  = -1.00
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.6666666666666
         g%s(15)%v(2)    =  0.3333333333333
         g%s(15)%v(3)    =  0.3333333333333
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(1,2)  = -1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.6666666666666
         g%s(16)%v(2)    =  0.3333333333333
         g%s(16)%v(3)    =  0.3333333333333
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.6666666666666
         g%s(17)%v(2)    =  0.3333333333333
         g%s(17)%v(3)    =  0.3333333333333
       
         g%s(18)%m(1,2)  =  1.00
         g%s(18)%m(2,1)  =  1.00
         g%s(18)%m(3,3)  = -1.00
         g%s(18)%v(1)    =  0.6666666666666
         g%s(18)%v(2)    =  0.3333333333333
         g%s(18)%v(3)    =  0.3333333333333
       
       
      case('R 3 2 : R', '155')                       
       
         !Space group   # 155
         !Group setting # 176
       
         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,3)   =  1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,1)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,1)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,3)   = -1.00
         g%s(6)%m(3,2)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
       
      case('P 3 m 1', '156')                         
       
         !Space group   # 156
         !Group setting # 177
       
         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(1,2)   =  1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,1)   =  1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
       
      case('P 3 1 m', '157')                         
       
         !Space group   # 157
         !Group setting # 178
       
         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   =  1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(1,2)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
       
      case('P 3 c 1', '158')                         
       
         !Space group   # 158
         !Group setting # 179
       
         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(1,2)   =  1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,1)   =  1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
       
      case('P 3 1 c', '159')                         
       
         !Space group   # 159
         !Group setting # 180
       
         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   =  1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(1,2)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
       
      case('R 3 m : H')                       
       
         !Space group   # 160
         !Group setting # 181
       
         g%order   =  18
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(1,2)   =  1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,1)   =  1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.3333333333333
         g%s(7)%v(2)     =  0.6666666666666
         g%s(7)%v(3)     =  0.6666666666666
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.3333333333333
         g%s(8)%v(2)     =  0.6666666666666
         g%s(8)%v(3)     =  0.6666666666666
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,1)   = -1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.3333333333333
         g%s(9)%v(2)     =  0.6666666666666
         g%s(9)%v(3)     =  0.6666666666666
       
         g%s(10)%m(1,1)  = -1.00
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,2)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.3333333333333
         g%s(10)%v(2)    =  0.6666666666666
         g%s(10)%v(3)    =  0.6666666666666
       
         g%s(11)%m(1,1)  =  1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(2,2)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  0.3333333333333
         g%s(11)%v(2)    =  0.6666666666666
         g%s(11)%v(3)    =  0.6666666666666
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,1)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.3333333333333
         g%s(12)%v(2)    =  0.6666666666666
         g%s(12)%v(3)    =  0.6666666666666
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  =  1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.6666666666666
         g%s(13)%v(2)    =  0.3333333333333
         g%s(13)%v(3)    =  0.3333333333333
       
         g%s(14)%m(1,2)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(2,2)  = -1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.6666666666666
         g%s(14)%v(2)    =  0.3333333333333
         g%s(14)%v(3)    =  0.3333333333333
       
         g%s(15)%m(1,1)  = -1.00
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.6666666666666
         g%s(15)%v(2)    =  0.3333333333333
         g%s(15)%v(3)    =  0.3333333333333
       
         g%s(16)%m(1,1)  = -1.00
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.6666666666666
         g%s(16)%v(2)    =  0.3333333333333
         g%s(16)%v(3)    =  0.3333333333333
       
         g%s(17)%m(1,1)  =  1.00
         g%s(17)%m(2,1)  =  1.00
         g%s(17)%m(2,2)  = -1.00
         g%s(17)%m(3,3)  =  1.00
         g%s(17)%v(1)    =  0.6666666666666
         g%s(17)%v(2)    =  0.3333333333333
         g%s(17)%v(3)    =  0.3333333333333
       
         g%s(18)%m(1,2)  = -1.00
         g%s(18)%m(2,1)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.6666666666666
         g%s(18)%v(2)    =  0.3333333333333
         g%s(18)%v(3)    =  0.3333333333333
       
       
      case('R 3 m : R', '160')                       
       
         !Space group   # 160
         !Group setting # 182
       
         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,3)   =  1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,1)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   =  1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,1)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,3)   =  1.00
         g%s(6)%m(3,2)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
       
      case('R 3 c : H')                       
       
         !Space group   # 161
         !Group setting # 183
       
         g%order   =  18
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   = -1.00
         g%s(4)%m(1,2)   =  1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   =  1.00
         g%s(5)%m(2,1)   =  1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   =  1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.3333333333333
         g%s(7)%v(2)     =  0.6666666666666
         g%s(7)%v(3)     =  0.6666666666666
       
         g%s(8)%m(1,2)   = -1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.3333333333333
         g%s(8)%v(2)     =  0.6666666666666
         g%s(8)%v(3)     =  0.6666666666666
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,1)   = -1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.3333333333333
         g%s(9)%v(2)     =  0.6666666666666
         g%s(9)%v(3)     =  0.6666666666666
       
         g%s(10)%m(1,1)  = -1.00
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,2)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.3333333333333
         g%s(10)%v(2)    =  0.6666666666666
         g%s(10)%v(3)    =  1.16666996
       
         g%s(11)%m(1,1)  =  1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(2,2)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  0.3333333333333
         g%s(11)%v(2)    =  0.6666666666666
         g%s(11)%v(3)    =  1.16666996
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,1)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.3333333333333
         g%s(12)%v(2)    =  0.6666666666666
         g%s(12)%v(3)    =  1.16666996
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  =  1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.6666666666666
         g%s(13)%v(2)    =  0.3333333333333
         g%s(13)%v(3)    =  0.3333333333333
       
         g%s(14)%m(1,2)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(2,2)  = -1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.6666666666666
         g%s(14)%v(2)    =  0.3333333333333
         g%s(14)%v(3)    =  0.3333333333333
       
         g%s(15)%m(1,1)  = -1.00
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.6666666666666
         g%s(15)%v(2)    =  0.3333333333333
         g%s(15)%v(3)    =  0.3333333333333
       
         g%s(16)%m(1,1)  = -1.00
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  =  1.00
         g%s(16)%v(1)    =  0.6666666666666
         g%s(16)%v(2)    =  0.3333333333333
         g%s(16)%v(3)    =  0.83333302
       
         g%s(17)%m(1,1)  =  1.00
         g%s(17)%m(2,1)  =  1.00
         g%s(17)%m(2,2)  = -1.00
         g%s(17)%m(3,3)  =  1.00
         g%s(17)%v(1)    =  0.6666666666666
         g%s(17)%v(2)    =  0.3333333333333
         g%s(17)%v(3)    =  0.83333302
       
         g%s(18)%m(1,2)  = -1.00
         g%s(18)%m(2,1)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.6666666666666
         g%s(18)%v(2)    =  0.3333333333333
         g%s(18)%v(3)    =  0.83333302
       
       
      case('R 3 c : R', '161')                       
       
         !Space group   # 161
         !Group setting # 184
       
         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,3)   =  1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,1)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   =  1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,3)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,1)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,3)   =  1.00
         g%s(6)%m(3,2)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
       
      case('P -3 1 m', '162')                        
       
         !Space group   # 162
         !Group setting # 185
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(1,2)   = -1.00
         g%s(9)%m(2,1)   =  1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  =  1.00
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,2)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
       
      case('P -3 1 c', '163')                        
       
         !Space group   # 163
         !Group setting # 186
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(1,2)   = -1.00
         g%s(9)%m(2,1)   =  1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    = -0.50
       
         g%s(11)%m(1,1)  =  1.00
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,2)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    = -0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    = -0.50
       
       
      case('P -3 m 1', '164')                        
       
         !Space group   # 164
         !Group setting # 187
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(1,2)   = -1.00
         g%s(9)%m(2,1)   =  1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,1)  = -1.00
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,2)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  =  1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(2,2)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,1)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
       
      case('P -3 c 1', '165')                        
       
         !Space group   # 165
         !Group setting # 188
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(1,2)   = -1.00
         g%s(9)%m(2,1)   =  1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,1)  = -1.00
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,2)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    = -0.50
       
         g%s(11)%m(1,1)  =  1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(2,2)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    = -0.50
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,1)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    = -0.50
       
       
      case('R -3 m : H')                      
       
         !Space group   # 166
         !Group setting # 189
       
         g%order   =  36
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(1,2)   = -1.00
         g%s(9)%m(2,1)   =  1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,1)  = -1.00
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,2)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  =  1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(2,2)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,1)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  =  1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.3333333333333
         g%s(13)%v(2)    =  0.6666666666666
         g%s(13)%v(3)    =  0.6666666666666
       
         g%s(14)%m(1,2)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(2,2)  = -1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.3333333333333
         g%s(14)%v(2)    =  0.6666666666666
         g%s(14)%v(3)    =  0.6666666666666
       
         g%s(15)%m(1,1)  = -1.00
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.3333333333333
         g%s(15)%v(2)    =  0.6666666666666
         g%s(15)%v(3)    =  0.6666666666666
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(1,2)  = -1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.3333333333333
         g%s(16)%v(2)    =  0.6666666666666
         g%s(16)%v(3)    =  0.6666666666666
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.3333333333333
         g%s(17)%v(2)    =  0.6666666666666
         g%s(17)%v(3)    =  0.6666666666666
       
         g%s(18)%m(1,2)  =  1.00
         g%s(18)%m(2,1)  =  1.00
         g%s(18)%m(3,3)  = -1.00
         g%s(18)%v(1)    =  0.3333333333333
         g%s(18)%v(2)    =  0.6666666666666
         g%s(18)%v(3)    =  0.6666666666666
       
         g%s(19)%m(1,1)  = -1.00
         g%s(19)%m(2,2)  = -1.00
         g%s(19)%m(3,3)  = -1.00
         g%s(19)%v(1)    =  0.3333333333333
         g%s(19)%v(2)    =  0.6666666666666
         g%s(19)%v(3)    =  0.6666666666666
       
         g%s(20)%m(1,2)  =  1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(2,2)  =  1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.3333333333333
         g%s(20)%v(2)    =  0.6666666666666
         g%s(20)%v(3)    =  0.6666666666666
       
         g%s(21)%m(1,1)  =  1.00
         g%s(21)%m(1,2)  = -1.00
         g%s(21)%m(2,1)  =  1.00
         g%s(21)%m(3,3)  = -1.00
         g%s(21)%v(1)    =  0.3333333333333
         g%s(21)%v(2)    =  0.6666666666666
         g%s(21)%v(3)    =  0.6666666666666
       
         g%s(22)%m(1,1)  = -1.00
         g%s(22)%m(1,2)  =  1.00
         g%s(22)%m(2,2)  =  1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.3333333333333
         g%s(22)%v(2)    =  0.6666666666666
         g%s(22)%v(3)    =  0.6666666666666
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(2,1)  =  1.00
         g%s(23)%m(2,2)  = -1.00
         g%s(23)%m(3,3)  =  1.00
         g%s(23)%v(1)    =  0.3333333333333
         g%s(23)%v(2)    =  0.6666666666666
         g%s(23)%v(3)    =  0.6666666666666
       
         g%s(24)%m(1,2)  = -1.00
         g%s(24)%m(2,1)  = -1.00
         g%s(24)%m(3,3)  =  1.00
         g%s(24)%v(1)    =  0.3333333333333
         g%s(24)%v(2)    =  0.6666666666666
         g%s(24)%v(3)    =  0.6666666666666
       
         g%s(25)%m(1,1)  =  1.00
         g%s(25)%m(2,2)  =  1.00
         g%s(25)%m(3,3)  =  1.00
         g%s(25)%v(1)    =  0.6666666666666
         g%s(25)%v(2)    =  0.3333333333333
         g%s(25)%v(3)    =  0.3333333333333
       
         g%s(26)%m(1,2)  = -1.00
         g%s(26)%m(2,1)  =  1.00
         g%s(26)%m(2,2)  = -1.00
         g%s(26)%m(3,3)  =  1.00
         g%s(26)%v(1)    =  0.6666666666666
         g%s(26)%v(2)    =  0.3333333333333
         g%s(26)%v(3)    =  0.3333333333333
       
         g%s(27)%m(1,1)  = -1.00
         g%s(27)%m(1,2)  =  1.00
         g%s(27)%m(2,1)  = -1.00
         g%s(27)%m(3,3)  =  1.00
         g%s(27)%v(1)    =  0.6666666666666
         g%s(27)%v(2)    =  0.3333333333333
         g%s(27)%v(3)    =  0.3333333333333
       
         g%s(28)%m(1,1)  =  1.00
         g%s(28)%m(1,2)  = -1.00
         g%s(28)%m(2,2)  = -1.00
         g%s(28)%m(3,3)  = -1.00
         g%s(28)%v(1)    =  0.6666666666666
         g%s(28)%v(2)    =  0.3333333333333
         g%s(28)%v(3)    =  0.3333333333333
       
         g%s(29)%m(1,1)  = -1.00
         g%s(29)%m(2,1)  = -1.00
         g%s(29)%m(2,2)  =  1.00
         g%s(29)%m(3,3)  = -1.00
         g%s(29)%v(1)    =  0.6666666666666
         g%s(29)%v(2)    =  0.3333333333333
         g%s(29)%v(3)    =  0.3333333333333
       
         g%s(30)%m(1,2)  =  1.00
         g%s(30)%m(2,1)  =  1.00
         g%s(30)%m(3,3)  = -1.00
         g%s(30)%v(1)    =  0.6666666666666
         g%s(30)%v(2)    =  0.3333333333333
         g%s(30)%v(3)    =  0.3333333333333
       
         g%s(31)%m(1,1)  = -1.00
         g%s(31)%m(2,2)  = -1.00
         g%s(31)%m(3,3)  = -1.00
         g%s(31)%v(1)    =  0.6666666666666
         g%s(31)%v(2)    =  0.3333333333333
         g%s(31)%v(3)    =  0.3333333333333
       
         g%s(32)%m(1,2)  =  1.00
         g%s(32)%m(2,1)  = -1.00
         g%s(32)%m(2,2)  =  1.00
         g%s(32)%m(3,3)  = -1.00
         g%s(32)%v(1)    =  0.6666666666666
         g%s(32)%v(2)    =  0.3333333333333
         g%s(32)%v(3)    =  0.3333333333333
       
         g%s(33)%m(1,1)  =  1.00
         g%s(33)%m(1,2)  = -1.00
         g%s(33)%m(2,1)  =  1.00
         g%s(33)%m(3,3)  = -1.00
         g%s(33)%v(1)    =  0.6666666666666
         g%s(33)%v(2)    =  0.3333333333333
         g%s(33)%v(3)    =  0.3333333333333
       
         g%s(34)%m(1,1)  = -1.00
         g%s(34)%m(1,2)  =  1.00
         g%s(34)%m(2,2)  =  1.00
         g%s(34)%m(3,3)  =  1.00
         g%s(34)%v(1)    =  0.6666666666666
         g%s(34)%v(2)    =  0.3333333333333
         g%s(34)%v(3)    =  0.3333333333333
       
         g%s(35)%m(1,1)  =  1.00
         g%s(35)%m(2,1)  =  1.00
         g%s(35)%m(2,2)  = -1.00
         g%s(35)%m(3,3)  =  1.00
         g%s(35)%v(1)    =  0.6666666666666
         g%s(35)%v(2)    =  0.3333333333333
         g%s(35)%v(3)    =  0.3333333333333
       
         g%s(36)%m(1,2)  = -1.00
         g%s(36)%m(2,1)  = -1.00
         g%s(36)%m(3,3)  =  1.00
         g%s(36)%v(1)    =  0.6666666666666
         g%s(36)%v(2)    =  0.3333333333333
         g%s(36)%v(3)    =  0.3333333333333
       
       
      case('R -3 m : R', '166')                      
       
         !Space group   # 166
         !Group setting # 190
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,3)   =  1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,1)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,1)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,3)   = -1.00
         g%s(6)%m(3,2)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,2)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   = -1.00
         g%s(9)%m(2,3)   = -1.00
         g%s(9)%m(3,1)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,1)  =  1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  =  1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,2)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
       
      case('R -3 c : H')                      
       
         !Space group   # 167
         !Group setting # 191
       
         g%order   =  36
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(2,2)   = -1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,1)   =  1.00
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,3)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(2,2)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(1,2)   = -1.00
         g%s(9)%m(2,1)   =  1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,1)  = -1.00
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,2)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    = -0.50
       
         g%s(11)%m(1,1)  =  1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(2,2)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    = -0.50
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,1)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    = -0.50
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  =  1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.3333333333333
         g%s(13)%v(2)    =  0.6666666666666
         g%s(13)%v(3)    =  0.6666666666666
       
         g%s(14)%m(1,2)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(2,2)  = -1.00
         g%s(14)%m(3,3)  =  1.00
         g%s(14)%v(1)    =  0.3333333333333
         g%s(14)%v(2)    =  0.6666666666666
         g%s(14)%v(3)    =  0.6666666666666
       
         g%s(15)%m(1,1)  = -1.00
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,1)  = -1.00
         g%s(15)%m(3,3)  =  1.00
         g%s(15)%v(1)    =  0.3333333333333
         g%s(15)%v(2)    =  0.6666666666666
         g%s(15)%v(3)    =  0.6666666666666
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(1,2)  = -1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.3333333333333
         g%s(16)%v(2)    =  0.6666666666666
         g%s(16)%v(3)    =  1.16666996
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.3333333333333
         g%s(17)%v(2)    =  0.6666666666666
         g%s(17)%v(3)    =  1.16666996
       
         g%s(18)%m(1,2)  =  1.00
         g%s(18)%m(2,1)  =  1.00
         g%s(18)%m(3,3)  = -1.00
         g%s(18)%v(1)    =  0.3333333333333
         g%s(18)%v(2)    =  0.6666666666666
         g%s(18)%v(3)    =  1.16666996
       
         g%s(19)%m(1,1)  = -1.00
         g%s(19)%m(2,2)  = -1.00
         g%s(19)%m(3,3)  = -1.00
         g%s(19)%v(1)    =  0.3333333333333
         g%s(19)%v(2)    =  0.6666666666666
         g%s(19)%v(3)    =  0.6666666666666
       
         g%s(20)%m(1,2)  =  1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(2,2)  =  1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.3333333333333
         g%s(20)%v(2)    =  0.6666666666666
         g%s(20)%v(3)    =  0.6666666666666
       
         g%s(21)%m(1,1)  =  1.00
         g%s(21)%m(1,2)  = -1.00
         g%s(21)%m(2,1)  =  1.00
         g%s(21)%m(3,3)  = -1.00
         g%s(21)%v(1)    =  0.3333333333333
         g%s(21)%v(2)    =  0.6666666666666
         g%s(21)%v(3)    =  0.6666666666666
       
         g%s(22)%m(1,1)  = -1.00
         g%s(22)%m(1,2)  =  1.00
         g%s(22)%m(2,2)  =  1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.3333333333333
         g%s(22)%v(2)    =  0.6666666666666
         g%s(22)%v(3)    =  0.16666700
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(2,1)  =  1.00
         g%s(23)%m(2,2)  = -1.00
         g%s(23)%m(3,3)  =  1.00
         g%s(23)%v(1)    =  0.3333333333333
         g%s(23)%v(2)    =  0.6666666666666
         g%s(23)%v(3)    =  0.16666700
       
         g%s(24)%m(1,2)  = -1.00
         g%s(24)%m(2,1)  = -1.00
         g%s(24)%m(3,3)  =  1.00
         g%s(24)%v(1)    =  0.3333333333333
         g%s(24)%v(2)    =  0.6666666666666
         g%s(24)%v(3)    =  0.16666700
       
         g%s(25)%m(1,1)  =  1.00
         g%s(25)%m(2,2)  =  1.00
         g%s(25)%m(3,3)  =  1.00
         g%s(25)%v(1)    =  0.6666666666666
         g%s(25)%v(2)    =  0.3333333333333
         g%s(25)%v(3)    =  0.3333333333333
       
         g%s(26)%m(1,2)  = -1.00
         g%s(26)%m(2,1)  =  1.00
         g%s(26)%m(2,2)  = -1.00
         g%s(26)%m(3,3)  =  1.00
         g%s(26)%v(1)    =  0.6666666666666
         g%s(26)%v(2)    =  0.3333333333333
         g%s(26)%v(3)    =  0.3333333333333
       
         g%s(27)%m(1,1)  = -1.00
         g%s(27)%m(1,2)  =  1.00
         g%s(27)%m(2,1)  = -1.00
         g%s(27)%m(3,3)  =  1.00
         g%s(27)%v(1)    =  0.6666666666666
         g%s(27)%v(2)    =  0.3333333333333
         g%s(27)%v(3)    =  0.3333333333333
       
         g%s(28)%m(1,1)  =  1.00
         g%s(28)%m(1,2)  = -1.00
         g%s(28)%m(2,2)  = -1.00
         g%s(28)%m(3,3)  = -1.00
         g%s(28)%v(1)    =  0.6666666666666
         g%s(28)%v(2)    =  0.3333333333333
         g%s(28)%v(3)    =  0.83333302
       
         g%s(29)%m(1,1)  = -1.00
         g%s(29)%m(2,1)  = -1.00
         g%s(29)%m(2,2)  =  1.00
         g%s(29)%m(3,3)  = -1.00
         g%s(29)%v(1)    =  0.6666666666666
         g%s(29)%v(2)    =  0.3333333333333
         g%s(29)%v(3)    =  0.83333302
       
         g%s(30)%m(1,2)  =  1.00
         g%s(30)%m(2,1)  =  1.00
         g%s(30)%m(3,3)  = -1.00
         g%s(30)%v(1)    =  0.6666666666666
         g%s(30)%v(2)    =  0.3333333333333
         g%s(30)%v(3)    =  0.83333302
       
         g%s(31)%m(1,1)  = -1.00
         g%s(31)%m(2,2)  = -1.00
         g%s(31)%m(3,3)  = -1.00
         g%s(31)%v(1)    =  0.6666666666666
         g%s(31)%v(2)    =  0.3333333333333
         g%s(31)%v(3)    =  0.3333333333333
       
         g%s(32)%m(1,2)  =  1.00
         g%s(32)%m(2,1)  = -1.00
         g%s(32)%m(2,2)  =  1.00
         g%s(32)%m(3,3)  = -1.00
         g%s(32)%v(1)    =  0.6666666666666
         g%s(32)%v(2)    =  0.3333333333333
         g%s(32)%v(3)    =  0.3333333333333
       
         g%s(33)%m(1,1)  =  1.00
         g%s(33)%m(1,2)  = -1.00
         g%s(33)%m(2,1)  =  1.00
         g%s(33)%m(3,3)  = -1.00
         g%s(33)%v(1)    =  0.6666666666666
         g%s(33)%v(2)    =  0.3333333333333
         g%s(33)%v(3)    =  0.3333333333333
       
         g%s(34)%m(1,1)  = -1.00
         g%s(34)%m(1,2)  =  1.00
         g%s(34)%m(2,2)  =  1.00
         g%s(34)%m(3,3)  =  1.00
         g%s(34)%v(1)    =  0.6666666666666
         g%s(34)%v(2)    =  0.3333333333333
         g%s(34)%v(3)    = -0.16666700
       
         g%s(35)%m(1,1)  =  1.00
         g%s(35)%m(2,1)  =  1.00
         g%s(35)%m(2,2)  = -1.00
         g%s(35)%m(3,3)  =  1.00
         g%s(35)%v(1)    =  0.6666666666666
         g%s(35)%v(2)    =  0.3333333333333
         g%s(35)%v(3)    = -0.16666700
       
         g%s(36)%m(1,2)  = -1.00
         g%s(36)%m(2,1)  = -1.00
         g%s(36)%m(3,3)  =  1.00
         g%s(36)%v(1)    =  0.6666666666666
         g%s(36)%v(2)    =  0.3333333333333
         g%s(36)%v(3)    = -0.16666700
       
       
      case('R -3 c : R', '167')                      
       
         !Space group   # 167
         !Group setting # 192
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,3)   =  1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,1)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   = -1.00
         g%s(4)%m(3,3)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,3)   = -1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,1)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,3)   = -1.00
         g%s(6)%m(3,2)   = -1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   = -1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,2)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   = -1.00
         g%s(9)%m(2,3)   = -1.00
         g%s(9)%m(3,1)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    = -0.50
         g%s(10)%v(2)    = -0.50
         g%s(10)%v(3)    = -0.50
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,1)  =  1.00
         g%s(11)%v(1)    = -0.50
         g%s(11)%v(2)    = -0.50
         g%s(11)%v(3)    = -0.50
       
         g%s(12)%m(1,1)  =  1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,2)  =  1.00
         g%s(12)%v(1)    = -0.50
         g%s(12)%v(2)    = -0.50
         g%s(12)%v(3)    = -0.50
       
       
      case('P 6', '168')                             
       
         !Space group   # 168
         !Group setting # 193
       
         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
       
      case('P 61', '169')                            
       
         !Space group   # 169
         !Group setting # 194
       
         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.16666700
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.83333302
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.3333333333333
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.6666666666666
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
       
      case('P 65', '170')                            
       
         !Space group   # 170
         !Group setting # 195
       
         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.83333302
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.16666700
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.6666666666666
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.3333333333333
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
       
      case('P 62', '171')                            
       
         !Space group   # 171
         !Group setting # 196
       
         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.3333333333333
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.6666666666666
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.6666666666666
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.3333333333333
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
       
      case('P 64', '172')                            
       
         !Space group   # 172
         !Group setting # 197
       
         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.6666666666666
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.3333333333333
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.3333333333333
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.6666666666666
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
       
      case('P 63', '173')                            
       
         !Space group   # 173
         !Group setting # 198
       
         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
       
      case('P -6', '174')                            
       
         !Space group   # 174
         !Group setting # 199
       
         g%order   =   6
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(1,2)   =  1.00
         g%s(2)%m(2,1)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   = -1.00
         g%s(3)%m(2,1)   =  1.00
         g%s(3)%m(2,2)   = -1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(2,2)   =  1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
       
      case('P 6/m', '175')                           
       
         !Space group   # 175
         !Group setting # 200
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   = -1.00
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   = -1.00
         g%s(9)%m(2,1)   =  1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(2,2)  =  1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  =  1.00
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  =  1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
       
      case('P 63/m', '176')                          
       
         !Space group   # 176
         !Group setting # 201
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,2)   = -1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   = -1.00
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     = -0.50
       
         g%s(9)%m(1,2)   = -1.00
         g%s(9)%m(2,1)   =  1.00
         g%s(9)%m(2,2)   = -1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     = -0.50
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(2,2)  =  1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  =  1.00
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,1)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  =  1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    = -0.50
       
       
      case('P 6 2 2', '177')                         
       
         !Space group   # 177
         !Group setting # 202
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   = -1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,1)   =  1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(1,2)  =  1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  =  1.00
         g%s(12)%m(2,1)  =  1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
       
      case('P 61 2 2', '178')                        
       
         !Space group   # 178
         !Group setting # 203
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.16666700
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.83333302
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.3333333333333
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.6666666666666
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.6666666666666
       
         g%s(8)%m(1,1)   = -1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,1)   =  1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.3333333333333
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.83333302
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(1,2)  =  1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  =  1.00
         g%s(12)%m(2,1)  =  1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.16666700
       
       
      case('P 65 2 2', '179')                        
       
         !Space group   # 179
         !Group setting # 204
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.83333302
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.16666700
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.6666666666666
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.3333333333333
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.3333333333333
       
         g%s(8)%m(1,1)   = -1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,1)   =  1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.6666666666666
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.16666700
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(1,2)  =  1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  =  1.00
         g%s(12)%m(2,1)  =  1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.83333302
       
       
      case('P 62 2 2', '180')                        
       
         !Space group   # 180
         !Group setting # 205
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.3333333333333
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.6666666666666
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.6666666666666
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.3333333333333
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.3333333333333
       
         g%s(8)%m(1,1)   = -1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,1)   =  1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.6666666666666
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.6666666666666
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(1,2)  =  1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  =  1.00
         g%s(12)%m(2,1)  =  1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.3333333333333
       
       
      case('P 64 2 2', '181')                        
       
         !Space group   # 181
         !Group setting # 206
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.6666666666666
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.3333333333333
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.3333333333333
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.6666666666666
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.6666666666666
       
         g%s(8)%m(1,1)   = -1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,1)   =  1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.3333333333333
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.3333333333333
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(1,2)  =  1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  =  1.00
         g%s(12)%m(2,1)  =  1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.6666666666666
       
       
      case('P 63 2 2', '182')                        
       
         !Space group   # 182
         !Group setting # 207
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   = -1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,1)   =  1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(1,2)  =  1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  =  1.00
         g%s(12)%m(2,1)  =  1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.50
       
       
      case('P 6 m m', '183')                         
       
         !Space group   # 183
         !Group setting # 208
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   = -1.00
         g%s(9)%m(2,1)   = -1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  =  1.00
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,2)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
       
      case('P 6 c c', '184')                         
       
         !Space group   # 184
         !Group setting # 209
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,2)   = -1.00
         g%s(9)%m(2,1)   = -1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,1)  =  1.00
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,2)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.50
       
       
      case('P 63 c m', '185')                        
       
         !Space group   # 185
         !Group setting # 210
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,2)   = -1.00
         g%s(9)%m(2,1)   = -1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  =  1.00
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,2)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
       
      case('P 63 m c', '186')                        
       
         !Space group   # 186
         !Group setting # 211
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   = -1.00
         g%s(9)%m(2,1)   = -1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,1)  =  1.00
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,2)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.50
       
       
      case('P -6 m 2', '187')                        
       
         !Space group   # 187
         !Group setting # 212
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(1,2)   =  1.00
         g%s(2)%m(2,1)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   = -1.00
         g%s(3)%m(2,1)   =  1.00
         g%s(3)%m(2,2)   = -1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  =  1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,1)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
       
      case('P -6 c 2', '188')                        
       
         !Space group   # 188
         !Group setting # 213
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(1,2)   =  1.00
         g%s(2)%m(2,1)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   = -1.00
         g%s(3)%m(2,1)   =  1.00
         g%s(3)%m(2,2)   = -1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   = -1.00
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,1)  =  1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,1)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.50
       
       
      case('P -6 2 m', '189')                        
       
         !Space group   # 189
         !Group setting # 214
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(1,2)   =  1.00
         g%s(2)%m(2,1)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   = -1.00
         g%s(3)%m(2,1)   =  1.00
         g%s(3)%m(2,2)   = -1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  =  1.00
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,2)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
       
      case('P -6 2 c', '190')                        
       
         !Space group   # 190
         !Group setting # 215
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(1,2)   =  1.00
         g%s(2)%m(2,1)   = -1.00
         g%s(2)%m(3,3)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   = -1.00
         g%s(3)%m(2,1)   =  1.00
         g%s(3)%m(2,2)   = -1.00
         g%s(3)%m(3,3)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,2)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,3)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,1)   =  1.00
         g%s(9)%m(2,2)   =  1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  =  1.00
         g%s(10)%m(2,1)  =  1.00
         g%s(10)%m(3,3)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,1)  =  1.00
         g%s(11)%m(1,2)  = -1.00
         g%s(11)%m(2,2)  = -1.00
         g%s(11)%m(3,3)  =  1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,1)  = -1.00
         g%s(12)%m(2,2)  =  1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.50
       
       
      case('P 6/m m m', '191')                       
       
         !Space group   # 191
         !Group setting # 216
       
         g%order   =  24
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   = -1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,1)   =  1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(1,2)  =  1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  =  1.00
         g%s(12)%m(2,1)  =  1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(1,2)  =  1.00
         g%s(14)%m(2,1)  = -1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  =  1.00
         g%s(15)%m(2,2)  = -1.00
         g%s(15)%m(3,3)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  = -1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  =  1.00
         g%s(17)%m(1,2)  = -1.00
         g%s(17)%m(2,1)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(1,2)  =  1.00
         g%s(18)%m(2,2)  =  1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,1)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(2,2)  = -1.00
         g%s(19)%m(3,3)  =  1.00
         g%s(19)%v(1)    =  0.00
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    =  0.00
       
         g%s(20)%m(1,1)  =  1.00
         g%s(20)%m(2,2)  =  1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.00
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    =  0.00
       
         g%s(21)%m(1,2)  = -1.00
         g%s(21)%m(2,1)  = -1.00
         g%s(21)%m(3,3)  =  1.00
         g%s(21)%v(1)    =  0.00
         g%s(21)%v(2)    =  0.00
         g%s(21)%v(3)    =  0.00
       
         g%s(22)%m(1,2)  =  1.00
         g%s(22)%m(2,1)  =  1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.00
         g%s(22)%v(2)    =  0.00
         g%s(22)%v(3)    =  0.00
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(1,2)  = -1.00
         g%s(23)%m(2,2)  = -1.00
         g%s(23)%m(3,3)  =  1.00
         g%s(23)%v(1)    =  0.00
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    =  0.00
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,1)  = -1.00
         g%s(24)%m(2,2)  =  1.00
         g%s(24)%m(3,3)  =  1.00
         g%s(24)%v(1)    =  0.00
         g%s(24)%v(2)    =  0.00
         g%s(24)%v(3)    =  0.00
       
       
      case('P 6/m c c', '192')                       
       
         !Space group   # 192
         !Group setting # 217
       
         g%order   =  24
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   = -1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,1)   =  1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(1,2)  =  1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  =  1.00
         g%s(12)%m(2,1)  =  1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(1,2)  =  1.00
         g%s(14)%m(2,1)  = -1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  =  1.00
         g%s(15)%m(2,2)  = -1.00
         g%s(15)%m(3,3)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  = -1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  =  1.00
         g%s(17)%m(1,2)  = -1.00
         g%s(17)%m(2,1)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(1,2)  =  1.00
         g%s(18)%m(2,2)  =  1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    = -0.50
       
         g%s(19)%m(1,1)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(2,2)  = -1.00
         g%s(19)%m(3,3)  =  1.00
         g%s(19)%v(1)    =  0.00
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    = -0.50
       
         g%s(20)%m(1,1)  =  1.00
         g%s(20)%m(2,2)  =  1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.00
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    =  0.00
       
         g%s(21)%m(1,2)  = -1.00
         g%s(21)%m(2,1)  = -1.00
         g%s(21)%m(3,3)  =  1.00
         g%s(21)%v(1)    =  0.00
         g%s(21)%v(2)    =  0.00
         g%s(21)%v(3)    = -0.50
       
         g%s(22)%m(1,2)  =  1.00
         g%s(22)%m(2,1)  =  1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.00
         g%s(22)%v(2)    =  0.00
         g%s(22)%v(3)    = -0.50
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(1,2)  = -1.00
         g%s(23)%m(2,2)  = -1.00
         g%s(23)%m(3,3)  =  1.00
         g%s(23)%v(1)    =  0.00
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    = -0.50
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,1)  = -1.00
         g%s(24)%m(2,2)  =  1.00
         g%s(24)%m(3,3)  =  1.00
         g%s(24)%v(1)    =  0.00
         g%s(24)%v(2)    =  0.00
         g%s(24)%v(3)    = -0.50
       
       
      case('P 63/m c m', '193')                      
       
         !Space group   # 193
         !Group setting # 218
       
         g%order   =  24
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,1)   = -1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,1)   =  1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(1,2)  =  1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  =  1.00
         g%s(12)%m(2,1)  =  1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(1,2)  =  1.00
         g%s(14)%m(2,1)  = -1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    = -0.50
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  =  1.00
         g%s(15)%m(2,2)  = -1.00
         g%s(15)%m(3,3)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    = -0.50
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  = -1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  =  1.00
         g%s(17)%m(1,2)  = -1.00
         g%s(17)%m(2,1)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(1,2)  =  1.00
         g%s(18)%m(2,2)  =  1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    = -0.50
       
         g%s(19)%m(1,1)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(2,2)  = -1.00
         g%s(19)%m(3,3)  =  1.00
         g%s(19)%v(1)    =  0.00
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    = -0.50
       
         g%s(20)%m(1,1)  =  1.00
         g%s(20)%m(2,2)  =  1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.00
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    = -0.50
       
         g%s(21)%m(1,2)  = -1.00
         g%s(21)%m(2,1)  = -1.00
         g%s(21)%m(3,3)  =  1.00
         g%s(21)%v(1)    =  0.00
         g%s(21)%v(2)    =  0.00
         g%s(21)%v(3)    = -0.50
       
         g%s(22)%m(1,2)  =  1.00
         g%s(22)%m(2,1)  =  1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.00
         g%s(22)%v(2)    =  0.00
         g%s(22)%v(3)    =  0.00
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(1,2)  = -1.00
         g%s(23)%m(2,2)  = -1.00
         g%s(23)%m(3,3)  =  1.00
         g%s(23)%v(1)    =  0.00
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    =  0.00
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,1)  = -1.00
         g%s(24)%m(2,2)  =  1.00
         g%s(24)%m(3,3)  =  1.00
         g%s(24)%v(1)    =  0.00
         g%s(24)%v(2)    =  0.00
         g%s(24)%v(3)    =  0.00
       
       
      case('P 63/m m c', '194')                      
       
         !Space group   # 194
         !Group setting # 219
       
         g%order   =  24
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(1,2)   = -1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,3)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,1)   = -1.00
         g%s(3)%m(2,2)   =  1.00
         g%s(3)%m(3,3)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,1)   =  1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,3)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,1)   = -1.00
         g%s(5)%m(1,2)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,3)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,1)   =  1.00
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,2)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,1)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(2,2)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,1)   = -1.00
         g%s(8)%m(2,2)   = -1.00
         g%s(8)%m(3,3)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,1)   =  1.00
         g%s(9)%m(3,3)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,1)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(1,2)  =  1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  =  1.00
         g%s(12)%m(2,1)  =  1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,1)  = -1.00
         g%s(14)%m(1,2)  =  1.00
         g%s(14)%m(2,1)  = -1.00
         g%s(14)%m(3,3)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    = -0.50
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,1)  =  1.00
         g%s(15)%m(2,2)  = -1.00
         g%s(15)%m(3,3)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    = -0.50
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,1)  = -1.00
         g%s(16)%m(2,2)  =  1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  =  1.00
         g%s(17)%m(1,2)  = -1.00
         g%s(17)%m(2,1)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(1,2)  =  1.00
         g%s(18)%m(2,2)  =  1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,1)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(2,2)  = -1.00
         g%s(19)%m(3,3)  =  1.00
         g%s(19)%v(1)    =  0.00
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    =  0.00
       
         g%s(20)%m(1,1)  =  1.00
         g%s(20)%m(2,2)  =  1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.00
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    = -0.50
       
         g%s(21)%m(1,2)  = -1.00
         g%s(21)%m(2,1)  = -1.00
         g%s(21)%m(3,3)  =  1.00
         g%s(21)%v(1)    =  0.00
         g%s(21)%v(2)    =  0.00
         g%s(21)%v(3)    =  0.00
       
         g%s(22)%m(1,2)  =  1.00
         g%s(22)%m(2,1)  =  1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.00
         g%s(22)%v(2)    =  0.00
         g%s(22)%v(3)    = -0.50
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(1,2)  = -1.00
         g%s(23)%m(2,2)  = -1.00
         g%s(23)%m(3,3)  =  1.00
         g%s(23)%v(1)    =  0.00
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    = -0.50
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,1)  = -1.00
         g%s(24)%m(2,2)  =  1.00
         g%s(24)%m(3,3)  =  1.00
         g%s(24)%v(1)    =  0.00
         g%s(24)%v(2)    =  0.00
         g%s(24)%v(3)    = -0.50
       
       
      case('P 2 3', '195')                           
       
         !Space group   # 195
         !Group setting # 220
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,3)   =  1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,1)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,3)   = -1.00
         g%s(4)%m(3,1)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,2)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,3)   =  1.00
         g%s(6)%m(3,1)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,3)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,2)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   = -1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   = -1.00
         g%s(9)%m(3,1)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
       
      case('F 2 3', '196')                           
       
         !Space group   # 196
         !Group setting # 221
       
         g%order   =  48
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,3)   =  1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,1)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,3)   = -1.00
         g%s(4)%m(3,1)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,2)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,3)   =  1.00
         g%s(6)%m(3,1)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,3)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,2)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   = -1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   = -1.00
         g%s(9)%m(3,1)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  =  1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,3)  =  1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  =  1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  =  1.00
         g%s(15)%m(3,1)  =  1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,2)  = -1.00
         g%s(16)%m(2,3)  = -1.00
         g%s(16)%m(3,1)  =  1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.50
       
         g%s(17)%m(1,3)  =  1.00
         g%s(17)%m(2,1)  = -1.00
         g%s(17)%m(3,2)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.50
         g%s(17)%v(3)    =  0.50
       
         g%s(18)%m(1,2)  = -1.00
         g%s(18)%m(2,3)  =  1.00
         g%s(18)%m(3,1)  = -1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.50
         g%s(18)%v(3)    =  0.50
       
         g%s(19)%m(1,3)  = -1.00
         g%s(19)%m(2,1)  = -1.00
         g%s(19)%m(3,2)  =  1.00
         g%s(19)%v(1)    =  0.00
         g%s(19)%v(2)    =  0.50
         g%s(19)%v(3)    =  0.50
       
         g%s(20)%m(1,3)  = -1.00
         g%s(20)%m(2,1)  =  1.00
         g%s(20)%m(3,2)  = -1.00
         g%s(20)%v(1)    =  0.00
         g%s(20)%v(2)    =  0.50
         g%s(20)%v(3)    =  0.50
       
         g%s(21)%m(1,2)  =  1.00
         g%s(21)%m(2,3)  = -1.00
         g%s(21)%m(3,1)  = -1.00
         g%s(21)%v(1)    =  0.00
         g%s(21)%v(2)    =  0.50
         g%s(21)%v(3)    =  0.50
       
         g%s(22)%m(1,1)  =  1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,3)  = -1.00
         g%s(22)%v(1)    =  0.00
         g%s(22)%v(2)    =  0.50
         g%s(22)%v(3)    =  0.50
       
         g%s(23)%m(1,1)  = -1.00
         g%s(23)%m(2,2)  =  1.00
         g%s(23)%m(3,3)  = -1.00
         g%s(23)%v(1)    =  0.00
         g%s(23)%v(2)    =  0.50
         g%s(23)%v(3)    =  0.50
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,2)  = -1.00
         g%s(24)%m(3,3)  =  1.00
         g%s(24)%v(1)    =  0.00
         g%s(24)%v(2)    =  0.50
         g%s(24)%v(3)    =  0.50
       
         g%s(25)%m(1,1)  =  1.00
         g%s(25)%m(2,2)  =  1.00
         g%s(25)%m(3,3)  =  1.00
         g%s(25)%v(1)    =  0.50
         g%s(25)%v(2)    =  0.00
         g%s(25)%v(3)    =  0.50
       
         g%s(26)%m(1,3)  =  1.00
         g%s(26)%m(2,1)  =  1.00
         g%s(26)%m(3,2)  =  1.00
         g%s(26)%v(1)    =  0.50
         g%s(26)%v(2)    =  0.00
         g%s(26)%v(3)    =  0.50
       
         g%s(27)%m(1,2)  =  1.00
         g%s(27)%m(2,3)  =  1.00
         g%s(27)%m(3,1)  =  1.00
         g%s(27)%v(1)    =  0.50
         g%s(27)%v(2)    =  0.00
         g%s(27)%v(3)    =  0.50
       
         g%s(28)%m(1,2)  = -1.00
         g%s(28)%m(2,3)  = -1.00
         g%s(28)%m(3,1)  =  1.00
         g%s(28)%v(1)    =  0.50
         g%s(28)%v(2)    =  0.00
         g%s(28)%v(3)    =  0.50
       
         g%s(29)%m(1,3)  =  1.00
         g%s(29)%m(2,1)  = -1.00
         g%s(29)%m(3,2)  = -1.00
         g%s(29)%v(1)    =  0.50
         g%s(29)%v(2)    =  0.00
         g%s(29)%v(3)    =  0.50
       
         g%s(30)%m(1,2)  = -1.00
         g%s(30)%m(2,3)  =  1.00
         g%s(30)%m(3,1)  = -1.00
         g%s(30)%v(1)    =  0.50
         g%s(30)%v(2)    =  0.00
         g%s(30)%v(3)    =  0.50
       
         g%s(31)%m(1,3)  = -1.00
         g%s(31)%m(2,1)  = -1.00
         g%s(31)%m(3,2)  =  1.00
         g%s(31)%v(1)    =  0.50
         g%s(31)%v(2)    =  0.00
         g%s(31)%v(3)    =  0.50
       
         g%s(32)%m(1,3)  = -1.00
         g%s(32)%m(2,1)  =  1.00
         g%s(32)%m(3,2)  = -1.00
         g%s(32)%v(1)    =  0.50
         g%s(32)%v(2)    =  0.00
         g%s(32)%v(3)    =  0.50
       
         g%s(33)%m(1,2)  =  1.00
         g%s(33)%m(2,3)  = -1.00
         g%s(33)%m(3,1)  = -1.00
         g%s(33)%v(1)    =  0.50
         g%s(33)%v(2)    =  0.00
         g%s(33)%v(3)    =  0.50
       
         g%s(34)%m(1,1)  =  1.00
         g%s(34)%m(2,2)  = -1.00
         g%s(34)%m(3,3)  = -1.00
         g%s(34)%v(1)    =  0.50
         g%s(34)%v(2)    =  0.00
         g%s(34)%v(3)    =  0.50
       
         g%s(35)%m(1,1)  = -1.00
         g%s(35)%m(2,2)  =  1.00
         g%s(35)%m(3,3)  = -1.00
         g%s(35)%v(1)    =  0.50
         g%s(35)%v(2)    =  0.00
         g%s(35)%v(3)    =  0.50
       
         g%s(36)%m(1,1)  = -1.00
         g%s(36)%m(2,2)  = -1.00
         g%s(36)%m(3,3)  =  1.00
         g%s(36)%v(1)    =  0.50
         g%s(36)%v(2)    =  0.00
         g%s(36)%v(3)    =  0.50
       
         g%s(37)%m(1,1)  =  1.00
         g%s(37)%m(2,2)  =  1.00
         g%s(37)%m(3,3)  =  1.00
         g%s(37)%v(1)    =  0.50
         g%s(37)%v(2)    =  0.50
         g%s(37)%v(3)    =  0.00
       
         g%s(38)%m(1,3)  =  1.00
         g%s(38)%m(2,1)  =  1.00
         g%s(38)%m(3,2)  =  1.00
         g%s(38)%v(1)    =  0.50
         g%s(38)%v(2)    =  0.50
         g%s(38)%v(3)    =  0.00
       
         g%s(39)%m(1,2)  =  1.00
         g%s(39)%m(2,3)  =  1.00
         g%s(39)%m(3,1)  =  1.00
         g%s(39)%v(1)    =  0.50
         g%s(39)%v(2)    =  0.50
         g%s(39)%v(3)    =  0.00
       
         g%s(40)%m(1,2)  = -1.00
         g%s(40)%m(2,3)  = -1.00
         g%s(40)%m(3,1)  =  1.00
         g%s(40)%v(1)    =  0.50
         g%s(40)%v(2)    =  0.50
         g%s(40)%v(3)    =  0.00
       
         g%s(41)%m(1,3)  =  1.00
         g%s(41)%m(2,1)  = -1.00
         g%s(41)%m(3,2)  = -1.00
         g%s(41)%v(1)    =  0.50
         g%s(41)%v(2)    =  0.50
         g%s(41)%v(3)    =  0.00
       
         g%s(42)%m(1,2)  = -1.00
         g%s(42)%m(2,3)  =  1.00
         g%s(42)%m(3,1)  = -1.00
         g%s(42)%v(1)    =  0.50
         g%s(42)%v(2)    =  0.50
         g%s(42)%v(3)    =  0.00
       
         g%s(43)%m(1,3)  = -1.00
         g%s(43)%m(2,1)  = -1.00
         g%s(43)%m(3,2)  =  1.00
         g%s(43)%v(1)    =  0.50
         g%s(43)%v(2)    =  0.50
         g%s(43)%v(3)    =  0.00
       
         g%s(44)%m(1,3)  = -1.00
         g%s(44)%m(2,1)  =  1.00
         g%s(44)%m(3,2)  = -1.00
         g%s(44)%v(1)    =  0.50
         g%s(44)%v(2)    =  0.50
         g%s(44)%v(3)    =  0.00
       
         g%s(45)%m(1,2)  =  1.00
         g%s(45)%m(2,3)  = -1.00
         g%s(45)%m(3,1)  = -1.00
         g%s(45)%v(1)    =  0.50
         g%s(45)%v(2)    =  0.50
         g%s(45)%v(3)    =  0.00
       
         g%s(46)%m(1,1)  =  1.00
         g%s(46)%m(2,2)  = -1.00
         g%s(46)%m(3,3)  = -1.00
         g%s(46)%v(1)    =  0.50
         g%s(46)%v(2)    =  0.50
         g%s(46)%v(3)    =  0.00
       
         g%s(47)%m(1,1)  = -1.00
         g%s(47)%m(2,2)  =  1.00
         g%s(47)%m(3,3)  = -1.00
         g%s(47)%v(1)    =  0.50
         g%s(47)%v(2)    =  0.50
         g%s(47)%v(3)    =  0.00
       
         g%s(48)%m(1,1)  = -1.00
         g%s(48)%m(2,2)  = -1.00
         g%s(48)%m(3,3)  =  1.00
         g%s(48)%v(1)    =  0.50
         g%s(48)%v(2)    =  0.50
         g%s(48)%v(3)    =  0.00
       
       
      case('I 2 3', '197')                           
       
         !Space group   # 197
         !Group setting # 222
       
         g%order   =  24
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,3)   =  1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,1)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,3)   = -1.00
         g%s(4)%m(3,1)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,2)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,3)   =  1.00
         g%s(6)%m(3,1)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,3)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,2)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   = -1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   = -1.00
         g%s(9)%m(3,1)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  =  1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,3)  =  1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  =  1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  =  1.00
         g%s(15)%m(3,1)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,2)  = -1.00
         g%s(16)%m(2,3)  = -1.00
         g%s(16)%m(3,1)  =  1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.50
       
         g%s(17)%m(1,3)  =  1.00
         g%s(17)%m(2,1)  = -1.00
         g%s(17)%m(3,2)  = -1.00
         g%s(17)%v(1)    =  0.50
         g%s(17)%v(2)    =  0.50
         g%s(17)%v(3)    =  0.50
       
         g%s(18)%m(1,2)  = -1.00
         g%s(18)%m(2,3)  =  1.00
         g%s(18)%m(3,1)  = -1.00
         g%s(18)%v(1)    =  0.50
         g%s(18)%v(2)    =  0.50
         g%s(18)%v(3)    =  0.50
       
         g%s(19)%m(1,3)  = -1.00
         g%s(19)%m(2,1)  = -1.00
         g%s(19)%m(3,2)  =  1.00
         g%s(19)%v(1)    =  0.50
         g%s(19)%v(2)    =  0.50
         g%s(19)%v(3)    =  0.50
       
         g%s(20)%m(1,3)  = -1.00
         g%s(20)%m(2,1)  =  1.00
         g%s(20)%m(3,2)  = -1.00
         g%s(20)%v(1)    =  0.50
         g%s(20)%v(2)    =  0.50
         g%s(20)%v(3)    =  0.50
       
         g%s(21)%m(1,2)  =  1.00
         g%s(21)%m(2,3)  = -1.00
         g%s(21)%m(3,1)  = -1.00
         g%s(21)%v(1)    =  0.50
         g%s(21)%v(2)    =  0.50
         g%s(21)%v(3)    =  0.50
       
         g%s(22)%m(1,1)  =  1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,3)  = -1.00
         g%s(22)%v(1)    =  0.50
         g%s(22)%v(2)    =  0.50
         g%s(22)%v(3)    =  0.50
       
         g%s(23)%m(1,1)  = -1.00
         g%s(23)%m(2,2)  =  1.00
         g%s(23)%m(3,3)  = -1.00
         g%s(23)%v(1)    =  0.50
         g%s(23)%v(2)    =  0.50
         g%s(23)%v(3)    =  0.50
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,2)  = -1.00
         g%s(24)%m(3,3)  =  1.00
         g%s(24)%v(1)    =  0.50
         g%s(24)%v(2)    =  0.50
         g%s(24)%v(3)    =  0.50
       
       
      case('P 21 3', '198')                          
       
         !Space group   # 198
         !Group setting # 223
       
         g%order   =  12
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,3)   =  1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,1)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,3)   = -1.00
         g%s(4)%m(3,1)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,3)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,2)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,3)   =  1.00
         g%s(6)%m(3,1)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,3)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,2)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,3)   = -1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   = -1.00
         g%s(9)%m(3,1)   = -1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.50
       
       
      case('I 21 3', '199')                          
       
         !Space group   # 199
         !Group setting # 224
       
         g%order   =  24
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,3)   =  1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,1)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,3)   = -1.00
         g%s(4)%m(3,1)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,2)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,3)   =  1.00
         g%s(6)%m(3,1)   = -1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,3)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,2)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   = -1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   = -1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   = -1.00
         g%s(9)%m(3,1)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  =  1.00
         g%s(13)%m(2,2)  =  1.00
         g%s(13)%m(3,3)  =  1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,3)  =  1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  =  1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  =  1.00
         g%s(15)%m(3,1)  =  1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,2)  = -1.00
         g%s(16)%m(2,3)  = -1.00
         g%s(16)%m(3,1)  =  1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  1.00000000
         g%s(16)%v(3)    =  0.50
       
         g%s(17)%m(1,3)  =  1.00
         g%s(17)%m(2,1)  = -1.00
         g%s(17)%m(3,2)  = -1.00
         g%s(17)%v(1)    =  0.50
         g%s(17)%v(2)    =  0.50
         g%s(17)%v(3)    =  1.00000000
       
         g%s(18)%m(1,2)  = -1.00
         g%s(18)%m(2,3)  =  1.00
         g%s(18)%m(3,1)  = -1.00
         g%s(18)%v(1)    =  1.00000000
         g%s(18)%v(2)    =  0.50
         g%s(18)%v(3)    =  0.50
       
         g%s(19)%m(1,3)  = -1.00
         g%s(19)%m(2,1)  = -1.00
         g%s(19)%m(3,2)  =  1.00
         g%s(19)%v(1)    =  0.50
         g%s(19)%v(2)    =  1.00000000
         g%s(19)%v(3)    =  0.50
       
         g%s(20)%m(1,3)  = -1.00
         g%s(20)%m(2,1)  =  1.00
         g%s(20)%m(3,2)  = -1.00
         g%s(20)%v(1)    =  1.00000000
         g%s(20)%v(2)    =  0.50
         g%s(20)%v(3)    =  0.50
       
         g%s(21)%m(1,2)  =  1.00
         g%s(21)%m(2,3)  = -1.00
         g%s(21)%m(3,1)  = -1.00
         g%s(21)%v(1)    =  0.50
         g%s(21)%v(2)    =  0.50
         g%s(21)%v(3)    =  1.00000000
       
         g%s(22)%m(1,1)  =  1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,3)  = -1.00
         g%s(22)%v(1)    =  0.50
         g%s(22)%v(2)    =  0.50
         g%s(22)%v(3)    =  1.00000000
       
         g%s(23)%m(1,1)  = -1.00
         g%s(23)%m(2,2)  =  1.00
         g%s(23)%m(3,3)  = -1.00
         g%s(23)%v(1)    =  1.00000000
         g%s(23)%v(2)    =  0.50
         g%s(23)%v(3)    =  0.50
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,2)  = -1.00
         g%s(24)%m(3,3)  =  1.00
         g%s(24)%v(1)    =  0.50
         g%s(24)%v(2)    =  1.00000000
         g%s(24)%v(3)    =  0.50
       
       
      case('P m -3', '200')                          
       
         !Space group   # 200
         !Group setting # 225
       
         g%order   =  24
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,3)   =  1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,1)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,3)   = -1.00
         g%s(4)%m(3,1)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,2)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,3)   =  1.00
         g%s(6)%m(3,1)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,3)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,2)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   = -1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   = -1.00
         g%s(9)%m(3,1)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  = -1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,3)  =  1.00
         g%s(16)%m(3,1)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,3)  = -1.00
         g%s(17)%m(2,1)  =  1.00
         g%s(17)%m(3,2)  =  1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,2)  =  1.00
         g%s(18)%m(2,3)  = -1.00
         g%s(18)%m(3,1)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,3)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,2)  = -1.00
         g%s(19)%v(1)    =  0.00
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    =  0.00
       
         g%s(20)%m(1,3)  =  1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,2)  =  1.00
         g%s(20)%v(1)    =  0.00
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    =  0.00
       
         g%s(21)%m(1,2)  = -1.00
         g%s(21)%m(2,3)  =  1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.00
         g%s(21)%v(2)    =  0.00
         g%s(21)%v(3)    =  0.00
       
         g%s(22)%m(1,1)  = -1.00
         g%s(22)%m(2,2)  =  1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.00
         g%s(22)%v(2)    =  0.00
         g%s(22)%v(3)    =  0.00
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(2,2)  = -1.00
         g%s(23)%m(3,3)  =  1.00
         g%s(23)%v(1)    =  0.00
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    =  0.00
       
         g%s(24)%m(1,1)  =  1.00
         g%s(24)%m(2,2)  =  1.00
         g%s(24)%m(3,3)  = -1.00
         g%s(24)%v(1)    =  0.00
         g%s(24)%v(2)    =  0.00
         g%s(24)%v(3)    =  0.00
       
       
      case('P n -3 : 2', '201')                      
       
         !Space group   # 201
         !Group setting # 226
       
         g%order   =  24
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,3)   =  1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,1)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,3)   = -1.00
         g%s(4)%m(3,1)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,2)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,3)   =  1.00
         g%s(6)%m(3,1)   = -1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,3)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,2)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   = -1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   = -1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   = -1.00
         g%s(9)%m(3,1)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  = -1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,3)  =  1.00
         g%s(16)%m(3,1)  = -1.00
         g%s(16)%v(1)    = -0.50
         g%s(16)%v(2)    = -0.50
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,3)  = -1.00
         g%s(17)%m(2,1)  =  1.00
         g%s(17)%m(3,2)  =  1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    = -0.50
         g%s(17)%v(3)    = -0.50
       
         g%s(18)%m(1,2)  =  1.00
         g%s(18)%m(2,3)  = -1.00
         g%s(18)%m(3,1)  =  1.00
         g%s(18)%v(1)    = -0.50
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    = -0.50
       
         g%s(19)%m(1,3)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,2)  = -1.00
         g%s(19)%v(1)    = -0.50
         g%s(19)%v(2)    = -0.50
         g%s(19)%v(3)    =  0.00
       
         g%s(20)%m(1,3)  =  1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,2)  =  1.00
         g%s(20)%v(1)    = -0.50
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    = -0.50
       
         g%s(21)%m(1,2)  = -1.00
         g%s(21)%m(2,3)  =  1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.00
         g%s(21)%v(2)    = -0.50
         g%s(21)%v(3)    = -0.50
       
         g%s(22)%m(1,1)  = -1.00
         g%s(22)%m(2,2)  =  1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.00
         g%s(22)%v(2)    = -0.50
         g%s(22)%v(3)    = -0.50
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(2,2)  = -1.00
         g%s(23)%m(3,3)  =  1.00
         g%s(23)%v(1)    = -0.50
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    = -0.50
       
         g%s(24)%m(1,1)  =  1.00
         g%s(24)%m(2,2)  =  1.00
         g%s(24)%m(3,3)  = -1.00
         g%s(24)%v(1)    = -0.50
         g%s(24)%v(2)    = -0.50
         g%s(24)%v(3)    =  0.00
       
       
      case('P n -3 : 1')                      
       
         !Space group   # 201
         !Group setting # 227
       
         g%order   =  24
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,3)   =  1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,1)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,3)   = -1.00
         g%s(4)%m(3,1)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,2)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,3)   =  1.00
         g%s(6)%m(3,1)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,3)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,2)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   = -1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   = -1.00
         g%s(9)%m(3,1)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  = -1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,3)  =  1.00
         g%s(16)%m(3,1)  = -1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.50
       
         g%s(17)%m(1,3)  = -1.00
         g%s(17)%m(2,1)  =  1.00
         g%s(17)%m(3,2)  =  1.00
         g%s(17)%v(1)    =  0.50
         g%s(17)%v(2)    =  0.50
         g%s(17)%v(3)    =  0.50
       
         g%s(18)%m(1,2)  =  1.00
         g%s(18)%m(2,3)  = -1.00
         g%s(18)%m(3,1)  =  1.00
         g%s(18)%v(1)    =  0.50
         g%s(18)%v(2)    =  0.50
         g%s(18)%v(3)    =  0.50
       
         g%s(19)%m(1,3)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,2)  = -1.00
         g%s(19)%v(1)    =  0.50
         g%s(19)%v(2)    =  0.50
         g%s(19)%v(3)    =  0.50
       
         g%s(20)%m(1,3)  =  1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,2)  =  1.00
         g%s(20)%v(1)    =  0.50
         g%s(20)%v(2)    =  0.50
         g%s(20)%v(3)    =  0.50
       
         g%s(21)%m(1,2)  = -1.00
         g%s(21)%m(2,3)  =  1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.50
         g%s(21)%v(2)    =  0.50
         g%s(21)%v(3)    =  0.50
       
         g%s(22)%m(1,1)  = -1.00
         g%s(22)%m(2,2)  =  1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.50
         g%s(22)%v(2)    =  0.50
         g%s(22)%v(3)    =  0.50
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(2,2)  = -1.00
         g%s(23)%m(3,3)  =  1.00
         g%s(23)%v(1)    =  0.50
         g%s(23)%v(2)    =  0.50
         g%s(23)%v(3)    =  0.50
       
         g%s(24)%m(1,1)  =  1.00
         g%s(24)%m(2,2)  =  1.00
         g%s(24)%m(3,3)  = -1.00
         g%s(24)%v(1)    =  0.50
         g%s(24)%v(2)    =  0.50
         g%s(24)%v(3)    =  0.50
       
       
      case('F m -3', '202')                          
       
         !Space group   # 202
         !Group setting # 228
       
         g%order   =  96
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,3)   =  1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,1)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,3)   = -1.00
         g%s(4)%m(3,1)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,2)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,3)   =  1.00
         g%s(6)%m(3,1)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,3)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,2)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   = -1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   = -1.00
         g%s(9)%m(3,1)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  = -1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,3)  =  1.00
         g%s(16)%m(3,1)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,3)  = -1.00
         g%s(17)%m(2,1)  =  1.00
         g%s(17)%m(3,2)  =  1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,2)  =  1.00
         g%s(18)%m(2,3)  = -1.00
         g%s(18)%m(3,1)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,3)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,2)  = -1.00
         g%s(19)%v(1)    =  0.00
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    =  0.00
       
         g%s(20)%m(1,3)  =  1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,2)  =  1.00
         g%s(20)%v(1)    =  0.00
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    =  0.00
       
         g%s(21)%m(1,2)  = -1.00
         g%s(21)%m(2,3)  =  1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.00
         g%s(21)%v(2)    =  0.00
         g%s(21)%v(3)    =  0.00
       
         g%s(22)%m(1,1)  = -1.00
         g%s(22)%m(2,2)  =  1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.00
         g%s(22)%v(2)    =  0.00
         g%s(22)%v(3)    =  0.00
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(2,2)  = -1.00
         g%s(23)%m(3,3)  =  1.00
         g%s(23)%v(1)    =  0.00
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    =  0.00
       
         g%s(24)%m(1,1)  =  1.00
         g%s(24)%m(2,2)  =  1.00
         g%s(24)%m(3,3)  = -1.00
         g%s(24)%v(1)    =  0.00
         g%s(24)%v(2)    =  0.00
         g%s(24)%v(3)    =  0.00
       
         g%s(25)%m(1,1)  =  1.00
         g%s(25)%m(2,2)  =  1.00
         g%s(25)%m(3,3)  =  1.00
         g%s(25)%v(1)    =  0.00
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  0.50
       
         g%s(26)%m(1,3)  =  1.00
         g%s(26)%m(2,1)  =  1.00
         g%s(26)%m(3,2)  =  1.00
         g%s(26)%v(1)    =  0.00
         g%s(26)%v(2)    =  0.50
         g%s(26)%v(3)    =  0.50
       
         g%s(27)%m(1,2)  =  1.00
         g%s(27)%m(2,3)  =  1.00
         g%s(27)%m(3,1)  =  1.00
         g%s(27)%v(1)    =  0.00
         g%s(27)%v(2)    =  0.50
         g%s(27)%v(3)    =  0.50
       
         g%s(28)%m(1,2)  = -1.00
         g%s(28)%m(2,3)  = -1.00
         g%s(28)%m(3,1)  =  1.00
         g%s(28)%v(1)    =  0.00
         g%s(28)%v(2)    =  0.50
         g%s(28)%v(3)    =  0.50
       
         g%s(29)%m(1,3)  =  1.00
         g%s(29)%m(2,1)  = -1.00
         g%s(29)%m(3,2)  = -1.00
         g%s(29)%v(1)    =  0.00
         g%s(29)%v(2)    =  0.50
         g%s(29)%v(3)    =  0.50
       
         g%s(30)%m(1,2)  = -1.00
         g%s(30)%m(2,3)  =  1.00
         g%s(30)%m(3,1)  = -1.00
         g%s(30)%v(1)    =  0.00
         g%s(30)%v(2)    =  0.50
         g%s(30)%v(3)    =  0.50
       
         g%s(31)%m(1,3)  = -1.00
         g%s(31)%m(2,1)  = -1.00
         g%s(31)%m(3,2)  =  1.00
         g%s(31)%v(1)    =  0.00
         g%s(31)%v(2)    =  0.50
         g%s(31)%v(3)    =  0.50
       
         g%s(32)%m(1,3)  = -1.00
         g%s(32)%m(2,1)  =  1.00
         g%s(32)%m(3,2)  = -1.00
         g%s(32)%v(1)    =  0.00
         g%s(32)%v(2)    =  0.50
         g%s(32)%v(3)    =  0.50
       
         g%s(33)%m(1,2)  =  1.00
         g%s(33)%m(2,3)  = -1.00
         g%s(33)%m(3,1)  = -1.00
         g%s(33)%v(1)    =  0.00
         g%s(33)%v(2)    =  0.50
         g%s(33)%v(3)    =  0.50
       
         g%s(34)%m(1,1)  =  1.00
         g%s(34)%m(2,2)  = -1.00
         g%s(34)%m(3,3)  = -1.00
         g%s(34)%v(1)    =  0.00
         g%s(34)%v(2)    =  0.50
         g%s(34)%v(3)    =  0.50
       
         g%s(35)%m(1,1)  = -1.00
         g%s(35)%m(2,2)  =  1.00
         g%s(35)%m(3,3)  = -1.00
         g%s(35)%v(1)    =  0.00
         g%s(35)%v(2)    =  0.50
         g%s(35)%v(3)    =  0.50
       
         g%s(36)%m(1,1)  = -1.00
         g%s(36)%m(2,2)  = -1.00
         g%s(36)%m(3,3)  =  1.00
         g%s(36)%v(1)    =  0.00
         g%s(36)%v(2)    =  0.50
         g%s(36)%v(3)    =  0.50
       
         g%s(37)%m(1,1)  = -1.00
         g%s(37)%m(2,2)  = -1.00
         g%s(37)%m(3,3)  = -1.00
         g%s(37)%v(1)    =  0.00
         g%s(37)%v(2)    =  0.50
         g%s(37)%v(3)    =  0.50
       
         g%s(38)%m(1,3)  = -1.00
         g%s(38)%m(2,1)  = -1.00
         g%s(38)%m(3,2)  = -1.00
         g%s(38)%v(1)    =  0.00
         g%s(38)%v(2)    =  0.50
         g%s(38)%v(3)    =  0.50
       
         g%s(39)%m(1,2)  = -1.00
         g%s(39)%m(2,3)  = -1.00
         g%s(39)%m(3,1)  = -1.00
         g%s(39)%v(1)    =  0.00
         g%s(39)%v(2)    =  0.50
         g%s(39)%v(3)    =  0.50
       
         g%s(40)%m(1,2)  =  1.00
         g%s(40)%m(2,3)  =  1.00
         g%s(40)%m(3,1)  = -1.00
         g%s(40)%v(1)    =  0.00
         g%s(40)%v(2)    =  0.50
         g%s(40)%v(3)    =  0.50
       
         g%s(41)%m(1,3)  = -1.00
         g%s(41)%m(2,1)  =  1.00
         g%s(41)%m(3,2)  =  1.00
         g%s(41)%v(1)    =  0.00
         g%s(41)%v(2)    =  0.50
         g%s(41)%v(3)    =  0.50
       
         g%s(42)%m(1,2)  =  1.00
         g%s(42)%m(2,3)  = -1.00
         g%s(42)%m(3,1)  =  1.00
         g%s(42)%v(1)    =  0.00
         g%s(42)%v(2)    =  0.50
         g%s(42)%v(3)    =  0.50
       
         g%s(43)%m(1,3)  =  1.00
         g%s(43)%m(2,1)  =  1.00
         g%s(43)%m(3,2)  = -1.00
         g%s(43)%v(1)    =  0.00
         g%s(43)%v(2)    =  0.50
         g%s(43)%v(3)    =  0.50
       
         g%s(44)%m(1,3)  =  1.00
         g%s(44)%m(2,1)  = -1.00
         g%s(44)%m(3,2)  =  1.00
         g%s(44)%v(1)    =  0.00
         g%s(44)%v(2)    =  0.50
         g%s(44)%v(3)    =  0.50
       
         g%s(45)%m(1,2)  = -1.00
         g%s(45)%m(2,3)  =  1.00
         g%s(45)%m(3,1)  =  1.00
         g%s(45)%v(1)    =  0.00
         g%s(45)%v(2)    =  0.50
         g%s(45)%v(3)    =  0.50
       
         g%s(46)%m(1,1)  = -1.00
         g%s(46)%m(2,2)  =  1.00
         g%s(46)%m(3,3)  =  1.00
         g%s(46)%v(1)    =  0.00
         g%s(46)%v(2)    =  0.50
         g%s(46)%v(3)    =  0.50
       
         g%s(47)%m(1,1)  =  1.00
         g%s(47)%m(2,2)  = -1.00
         g%s(47)%m(3,3)  =  1.00
         g%s(47)%v(1)    =  0.00
         g%s(47)%v(2)    =  0.50
         g%s(47)%v(3)    =  0.50
       
         g%s(48)%m(1,1)  =  1.00
         g%s(48)%m(2,2)  =  1.00
         g%s(48)%m(3,3)  = -1.00
         g%s(48)%v(1)    =  0.00
         g%s(48)%v(2)    =  0.50
         g%s(48)%v(3)    =  0.50
       
         g%s(49)%m(1,1)  =  1.00
         g%s(49)%m(2,2)  =  1.00
         g%s(49)%m(3,3)  =  1.00
         g%s(49)%v(1)    =  0.50
         g%s(49)%v(2)    =  0.00
         g%s(49)%v(3)    =  0.50
       
         g%s(50)%m(1,3)  =  1.00
         g%s(50)%m(2,1)  =  1.00
         g%s(50)%m(3,2)  =  1.00
         g%s(50)%v(1)    =  0.50
         g%s(50)%v(2)    =  0.00
         g%s(50)%v(3)    =  0.50
       
         g%s(51)%m(1,2)  =  1.00
         g%s(51)%m(2,3)  =  1.00
         g%s(51)%m(3,1)  =  1.00
         g%s(51)%v(1)    =  0.50
         g%s(51)%v(2)    =  0.00
         g%s(51)%v(3)    =  0.50
       
         g%s(52)%m(1,2)  = -1.00
         g%s(52)%m(2,3)  = -1.00
         g%s(52)%m(3,1)  =  1.00
         g%s(52)%v(1)    =  0.50
         g%s(52)%v(2)    =  0.00
         g%s(52)%v(3)    =  0.50
       
         g%s(53)%m(1,3)  =  1.00
         g%s(53)%m(2,1)  = -1.00
         g%s(53)%m(3,2)  = -1.00
         g%s(53)%v(1)    =  0.50
         g%s(53)%v(2)    =  0.00
         g%s(53)%v(3)    =  0.50
       
         g%s(54)%m(1,2)  = -1.00
         g%s(54)%m(2,3)  =  1.00
         g%s(54)%m(3,1)  = -1.00
         g%s(54)%v(1)    =  0.50
         g%s(54)%v(2)    =  0.00
         g%s(54)%v(3)    =  0.50
       
         g%s(55)%m(1,3)  = -1.00
         g%s(55)%m(2,1)  = -1.00
         g%s(55)%m(3,2)  =  1.00
         g%s(55)%v(1)    =  0.50
         g%s(55)%v(2)    =  0.00
         g%s(55)%v(3)    =  0.50
       
         g%s(56)%m(1,3)  = -1.00
         g%s(56)%m(2,1)  =  1.00
         g%s(56)%m(3,2)  = -1.00
         g%s(56)%v(1)    =  0.50
         g%s(56)%v(2)    =  0.00
         g%s(56)%v(3)    =  0.50
       
         g%s(57)%m(1,2)  =  1.00
         g%s(57)%m(2,3)  = -1.00
         g%s(57)%m(3,1)  = -1.00
         g%s(57)%v(1)    =  0.50
         g%s(57)%v(2)    =  0.00
         g%s(57)%v(3)    =  0.50
       
         g%s(58)%m(1,1)  =  1.00
         g%s(58)%m(2,2)  = -1.00
         g%s(58)%m(3,3)  = -1.00
         g%s(58)%v(1)    =  0.50
         g%s(58)%v(2)    =  0.00
         g%s(58)%v(3)    =  0.50
       
         g%s(59)%m(1,1)  = -1.00
         g%s(59)%m(2,2)  =  1.00
         g%s(59)%m(3,3)  = -1.00
         g%s(59)%v(1)    =  0.50
         g%s(59)%v(2)    =  0.00
         g%s(59)%v(3)    =  0.50
       
         g%s(60)%m(1,1)  = -1.00
         g%s(60)%m(2,2)  = -1.00
         g%s(60)%m(3,3)  =  1.00
         g%s(60)%v(1)    =  0.50
         g%s(60)%v(2)    =  0.00
         g%s(60)%v(3)    =  0.50
       
         g%s(61)%m(1,1)  = -1.00
         g%s(61)%m(2,2)  = -1.00
         g%s(61)%m(3,3)  = -1.00
         g%s(61)%v(1)    =  0.50
         g%s(61)%v(2)    =  0.00
         g%s(61)%v(3)    =  0.50
       
         g%s(62)%m(1,3)  = -1.00
         g%s(62)%m(2,1)  = -1.00
         g%s(62)%m(3,2)  = -1.00
         g%s(62)%v(1)    =  0.50
         g%s(62)%v(2)    =  0.00
         g%s(62)%v(3)    =  0.50
       
         g%s(63)%m(1,2)  = -1.00
         g%s(63)%m(2,3)  = -1.00
         g%s(63)%m(3,1)  = -1.00
         g%s(63)%v(1)    =  0.50
         g%s(63)%v(2)    =  0.00
         g%s(63)%v(3)    =  0.50
       
         g%s(64)%m(1,2)  =  1.00
         g%s(64)%m(2,3)  =  1.00
         g%s(64)%m(3,1)  = -1.00
         g%s(64)%v(1)    =  0.50
         g%s(64)%v(2)    =  0.00
         g%s(64)%v(3)    =  0.50
       
         g%s(65)%m(1,3)  = -1.00
         g%s(65)%m(2,1)  =  1.00
         g%s(65)%m(3,2)  =  1.00
         g%s(65)%v(1)    =  0.50
         g%s(65)%v(2)    =  0.00
         g%s(65)%v(3)    =  0.50
       
         g%s(66)%m(1,2)  =  1.00
         g%s(66)%m(2,3)  = -1.00
         g%s(66)%m(3,1)  =  1.00
         g%s(66)%v(1)    =  0.50
         g%s(66)%v(2)    =  0.00
         g%s(66)%v(3)    =  0.50
       
         g%s(67)%m(1,3)  =  1.00
         g%s(67)%m(2,1)  =  1.00
         g%s(67)%m(3,2)  = -1.00
         g%s(67)%v(1)    =  0.50
         g%s(67)%v(2)    =  0.00
         g%s(67)%v(3)    =  0.50
       
         g%s(68)%m(1,3)  =  1.00
         g%s(68)%m(2,1)  = -1.00
         g%s(68)%m(3,2)  =  1.00
         g%s(68)%v(1)    =  0.50
         g%s(68)%v(2)    =  0.00
         g%s(68)%v(3)    =  0.50
       
         g%s(69)%m(1,2)  = -1.00
         g%s(69)%m(2,3)  =  1.00
         g%s(69)%m(3,1)  =  1.00
         g%s(69)%v(1)    =  0.50
         g%s(69)%v(2)    =  0.00
         g%s(69)%v(3)    =  0.50
       
         g%s(70)%m(1,1)  = -1.00
         g%s(70)%m(2,2)  =  1.00
         g%s(70)%m(3,3)  =  1.00
         g%s(70)%v(1)    =  0.50
         g%s(70)%v(2)    =  0.00
         g%s(70)%v(3)    =  0.50
       
         g%s(71)%m(1,1)  =  1.00
         g%s(71)%m(2,2)  = -1.00
         g%s(71)%m(3,3)  =  1.00
         g%s(71)%v(1)    =  0.50
         g%s(71)%v(2)    =  0.00
         g%s(71)%v(3)    =  0.50
       
         g%s(72)%m(1,1)  =  1.00
         g%s(72)%m(2,2)  =  1.00
         g%s(72)%m(3,3)  = -1.00
         g%s(72)%v(1)    =  0.50
         g%s(72)%v(2)    =  0.00
         g%s(72)%v(3)    =  0.50
       
         g%s(73)%m(1,1)  =  1.00
         g%s(73)%m(2,2)  =  1.00
         g%s(73)%m(3,3)  =  1.00
         g%s(73)%v(1)    =  0.50
         g%s(73)%v(2)    =  0.50
         g%s(73)%v(3)    =  0.00
       
         g%s(74)%m(1,3)  =  1.00
         g%s(74)%m(2,1)  =  1.00
         g%s(74)%m(3,2)  =  1.00
         g%s(74)%v(1)    =  0.50
         g%s(74)%v(2)    =  0.50
         g%s(74)%v(3)    =  0.00
       
         g%s(75)%m(1,2)  =  1.00
         g%s(75)%m(2,3)  =  1.00
         g%s(75)%m(3,1)  =  1.00
         g%s(75)%v(1)    =  0.50
         g%s(75)%v(2)    =  0.50
         g%s(75)%v(3)    =  0.00
       
         g%s(76)%m(1,2)  = -1.00
         g%s(76)%m(2,3)  = -1.00
         g%s(76)%m(3,1)  =  1.00
         g%s(76)%v(1)    =  0.50
         g%s(76)%v(2)    =  0.50
         g%s(76)%v(3)    =  0.00
       
         g%s(77)%m(1,3)  =  1.00
         g%s(77)%m(2,1)  = -1.00
         g%s(77)%m(3,2)  = -1.00
         g%s(77)%v(1)    =  0.50
         g%s(77)%v(2)    =  0.50
         g%s(77)%v(3)    =  0.00
       
         g%s(78)%m(1,2)  = -1.00
         g%s(78)%m(2,3)  =  1.00
         g%s(78)%m(3,1)  = -1.00
         g%s(78)%v(1)    =  0.50
         g%s(78)%v(2)    =  0.50
         g%s(78)%v(3)    =  0.00
       
         g%s(79)%m(1,3)  = -1.00
         g%s(79)%m(2,1)  = -1.00
         g%s(79)%m(3,2)  =  1.00
         g%s(79)%v(1)    =  0.50
         g%s(79)%v(2)    =  0.50
         g%s(79)%v(3)    =  0.00
       
         g%s(80)%m(1,3)  = -1.00
         g%s(80)%m(2,1)  =  1.00
         g%s(80)%m(3,2)  = -1.00
         g%s(80)%v(1)    =  0.50
         g%s(80)%v(2)    =  0.50
         g%s(80)%v(3)    =  0.00
       
         g%s(81)%m(1,2)  =  1.00
         g%s(81)%m(2,3)  = -1.00
         g%s(81)%m(3,1)  = -1.00
         g%s(81)%v(1)    =  0.50
         g%s(81)%v(2)    =  0.50
         g%s(81)%v(3)    =  0.00
       
         g%s(82)%m(1,1)  =  1.00
         g%s(82)%m(2,2)  = -1.00
         g%s(82)%m(3,3)  = -1.00
         g%s(82)%v(1)    =  0.50
         g%s(82)%v(2)    =  0.50
         g%s(82)%v(3)    =  0.00
       
         g%s(83)%m(1,1)  = -1.00
         g%s(83)%m(2,2)  =  1.00
         g%s(83)%m(3,3)  = -1.00
         g%s(83)%v(1)    =  0.50
         g%s(83)%v(2)    =  0.50
         g%s(83)%v(3)    =  0.00
       
         g%s(84)%m(1,1)  = -1.00
         g%s(84)%m(2,2)  = -1.00
         g%s(84)%m(3,3)  =  1.00
         g%s(84)%v(1)    =  0.50
         g%s(84)%v(2)    =  0.50
         g%s(84)%v(3)    =  0.00
       
         g%s(85)%m(1,1)  = -1.00
         g%s(85)%m(2,2)  = -1.00
         g%s(85)%m(3,3)  = -1.00
         g%s(85)%v(1)    =  0.50
         g%s(85)%v(2)    =  0.50
         g%s(85)%v(3)    =  0.00
       
         g%s(86)%m(1,3)  = -1.00
         g%s(86)%m(2,1)  = -1.00
         g%s(86)%m(3,2)  = -1.00
         g%s(86)%v(1)    =  0.50
         g%s(86)%v(2)    =  0.50
         g%s(86)%v(3)    =  0.00
       
         g%s(87)%m(1,2)  = -1.00
         g%s(87)%m(2,3)  = -1.00
         g%s(87)%m(3,1)  = -1.00
         g%s(87)%v(1)    =  0.50
         g%s(87)%v(2)    =  0.50
         g%s(87)%v(3)    =  0.00
       
         g%s(88)%m(1,2)  =  1.00
         g%s(88)%m(2,3)  =  1.00
         g%s(88)%m(3,1)  = -1.00
         g%s(88)%v(1)    =  0.50
         g%s(88)%v(2)    =  0.50
         g%s(88)%v(3)    =  0.00
       
         g%s(89)%m(1,3)  = -1.00
         g%s(89)%m(2,1)  =  1.00
         g%s(89)%m(3,2)  =  1.00
         g%s(89)%v(1)    =  0.50
         g%s(89)%v(2)    =  0.50
         g%s(89)%v(3)    =  0.00
       
         g%s(90)%m(1,2)  =  1.00
         g%s(90)%m(2,3)  = -1.00
         g%s(90)%m(3,1)  =  1.00
         g%s(90)%v(1)    =  0.50
         g%s(90)%v(2)    =  0.50
         g%s(90)%v(3)    =  0.00
       
         g%s(91)%m(1,3)  =  1.00
         g%s(91)%m(2,1)  =  1.00
         g%s(91)%m(3,2)  = -1.00
         g%s(91)%v(1)    =  0.50
         g%s(91)%v(2)    =  0.50
         g%s(91)%v(3)    =  0.00
       
         g%s(92)%m(1,3)  =  1.00
         g%s(92)%m(2,1)  = -1.00
         g%s(92)%m(3,2)  =  1.00
         g%s(92)%v(1)    =  0.50
         g%s(92)%v(2)    =  0.50
         g%s(92)%v(3)    =  0.00
       
         g%s(93)%m(1,2)  = -1.00
         g%s(93)%m(2,3)  =  1.00
         g%s(93)%m(3,1)  =  1.00
         g%s(93)%v(1)    =  0.50
         g%s(93)%v(2)    =  0.50
         g%s(93)%v(3)    =  0.00
       
         g%s(94)%m(1,1)  = -1.00
         g%s(94)%m(2,2)  =  1.00
         g%s(94)%m(3,3)  =  1.00
         g%s(94)%v(1)    =  0.50
         g%s(94)%v(2)    =  0.50
         g%s(94)%v(3)    =  0.00
       
         g%s(95)%m(1,1)  =  1.00
         g%s(95)%m(2,2)  = -1.00
         g%s(95)%m(3,3)  =  1.00
         g%s(95)%v(1)    =  0.50
         g%s(95)%v(2)    =  0.50
         g%s(95)%v(3)    =  0.00
       
         g%s(96)%m(1,1)  =  1.00
         g%s(96)%m(2,2)  =  1.00
         g%s(96)%m(3,3)  = -1.00
         g%s(96)%v(1)    =  0.50
         g%s(96)%v(2)    =  0.50
         g%s(96)%v(3)    =  0.00
       
       
      case('F d -3 : 2', '203')                      
       
         !Space group   # 203
         !Group setting # 229
       
         g%order   =  96
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,3)   =  1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,1)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,3)   = -1.00
         g%s(4)%m(3,1)   =  1.00
         g%s(4)%v(1)     =  0.25
         g%s(4)%v(2)     =  0.25
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,2)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.25
         g%s(5)%v(3)     =  0.25
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,3)   =  1.00
         g%s(6)%m(3,1)   = -1.00
         g%s(6)%v(1)     =  0.25
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.25
       
         g%s(7)%m(1,3)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,2)   =  1.00
         g%s(7)%v(1)     =  0.25
         g%s(7)%v(2)     =  0.25
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   = -1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   = -1.00
         g%s(8)%v(1)     =  0.25
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.25
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   = -1.00
         g%s(9)%m(3,1)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.25
         g%s(9)%v(3)     =  0.25
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.25
         g%s(10)%v(3)    =  0.25
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.25
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.25
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.25
         g%s(12)%v(2)    =  0.25
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  = -1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,3)  =  1.00
         g%s(16)%m(3,1)  = -1.00
         g%s(16)%v(1)    = -0.25
         g%s(16)%v(2)    = -0.25
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,3)  = -1.00
         g%s(17)%m(2,1)  =  1.00
         g%s(17)%m(3,2)  =  1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    = -0.25
         g%s(17)%v(3)    = -0.25
       
         g%s(18)%m(1,2)  =  1.00
         g%s(18)%m(2,3)  = -1.00
         g%s(18)%m(3,1)  =  1.00
         g%s(18)%v(1)    = -0.25
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    = -0.25
       
         g%s(19)%m(1,3)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,2)  = -1.00
         g%s(19)%v(1)    = -0.25
         g%s(19)%v(2)    = -0.25
         g%s(19)%v(3)    =  0.00
       
         g%s(20)%m(1,3)  =  1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,2)  =  1.00
         g%s(20)%v(1)    = -0.25
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    = -0.25
       
         g%s(21)%m(1,2)  = -1.00
         g%s(21)%m(2,3)  =  1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.00
         g%s(21)%v(2)    = -0.25
         g%s(21)%v(3)    = -0.25
       
         g%s(22)%m(1,1)  = -1.00
         g%s(22)%m(2,2)  =  1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.00
         g%s(22)%v(2)    = -0.25
         g%s(22)%v(3)    = -0.25
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(2,2)  = -1.00
         g%s(23)%m(3,3)  =  1.00
         g%s(23)%v(1)    = -0.25
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    = -0.25
       
         g%s(24)%m(1,1)  =  1.00
         g%s(24)%m(2,2)  =  1.00
         g%s(24)%m(3,3)  = -1.00
         g%s(24)%v(1)    = -0.25
         g%s(24)%v(2)    = -0.25
         g%s(24)%v(3)    =  0.00
       
         g%s(25)%m(1,1)  =  1.00
         g%s(25)%m(2,2)  =  1.00
         g%s(25)%m(3,3)  =  1.00
         g%s(25)%v(1)    =  0.00
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  0.50
       
         g%s(26)%m(1,3)  =  1.00
         g%s(26)%m(2,1)  =  1.00
         g%s(26)%m(3,2)  =  1.00
         g%s(26)%v(1)    =  0.00
         g%s(26)%v(2)    =  0.50
         g%s(26)%v(3)    =  0.50
       
         g%s(27)%m(1,2)  =  1.00
         g%s(27)%m(2,3)  =  1.00
         g%s(27)%m(3,1)  =  1.00
         g%s(27)%v(1)    =  0.00
         g%s(27)%v(2)    =  0.50
         g%s(27)%v(3)    =  0.50
       
         g%s(28)%m(1,2)  = -1.00
         g%s(28)%m(2,3)  = -1.00
         g%s(28)%m(3,1)  =  1.00
         g%s(28)%v(1)    =  0.25
         g%s(28)%v(2)    =  0.75000000
         g%s(28)%v(3)    =  0.50
       
         g%s(29)%m(1,3)  =  1.00
         g%s(29)%m(2,1)  = -1.00
         g%s(29)%m(3,2)  = -1.00
         g%s(29)%v(1)    =  0.00
         g%s(29)%v(2)    =  0.75000000
         g%s(29)%v(3)    =  0.75000000
       
         g%s(30)%m(1,2)  = -1.00
         g%s(30)%m(2,3)  =  1.00
         g%s(30)%m(3,1)  = -1.00
         g%s(30)%v(1)    =  0.25
         g%s(30)%v(2)    =  0.50
         g%s(30)%v(3)    =  0.75000000
       
         g%s(31)%m(1,3)  = -1.00
         g%s(31)%m(2,1)  = -1.00
         g%s(31)%m(3,2)  =  1.00
         g%s(31)%v(1)    =  0.25
         g%s(31)%v(2)    =  0.75000000
         g%s(31)%v(3)    =  0.50
       
         g%s(32)%m(1,3)  = -1.00
         g%s(32)%m(2,1)  =  1.00
         g%s(32)%m(3,2)  = -1.00
         g%s(32)%v(1)    =  0.25
         g%s(32)%v(2)    =  0.50
         g%s(32)%v(3)    =  0.75000000
       
         g%s(33)%m(1,2)  =  1.00
         g%s(33)%m(2,3)  = -1.00
         g%s(33)%m(3,1)  = -1.00
         g%s(33)%v(1)    =  0.00
         g%s(33)%v(2)    =  0.75000000
         g%s(33)%v(3)    =  0.75000000
       
         g%s(34)%m(1,1)  =  1.00
         g%s(34)%m(2,2)  = -1.00
         g%s(34)%m(3,3)  = -1.00
         g%s(34)%v(1)    =  0.00
         g%s(34)%v(2)    =  0.75000000
         g%s(34)%v(3)    =  0.75000000
       
         g%s(35)%m(1,1)  = -1.00
         g%s(35)%m(2,2)  =  1.00
         g%s(35)%m(3,3)  = -1.00
         g%s(35)%v(1)    =  0.25
         g%s(35)%v(2)    =  0.50
         g%s(35)%v(3)    =  0.75000000
       
         g%s(36)%m(1,1)  = -1.00
         g%s(36)%m(2,2)  = -1.00
         g%s(36)%m(3,3)  =  1.00
         g%s(36)%v(1)    =  0.25
         g%s(36)%v(2)    =  0.75000000
         g%s(36)%v(3)    =  0.50
       
         g%s(37)%m(1,1)  = -1.00
         g%s(37)%m(2,2)  = -1.00
         g%s(37)%m(3,3)  = -1.00
         g%s(37)%v(1)    =  0.00
         g%s(37)%v(2)    =  0.50
         g%s(37)%v(3)    =  0.50
       
         g%s(38)%m(1,3)  = -1.00
         g%s(38)%m(2,1)  = -1.00
         g%s(38)%m(3,2)  = -1.00
         g%s(38)%v(1)    =  0.00
         g%s(38)%v(2)    =  0.50
         g%s(38)%v(3)    =  0.50
       
         g%s(39)%m(1,2)  = -1.00
         g%s(39)%m(2,3)  = -1.00
         g%s(39)%m(3,1)  = -1.00
         g%s(39)%v(1)    =  0.00
         g%s(39)%v(2)    =  0.50
         g%s(39)%v(3)    =  0.50
       
         g%s(40)%m(1,2)  =  1.00
         g%s(40)%m(2,3)  =  1.00
         g%s(40)%m(3,1)  = -1.00
         g%s(40)%v(1)    = -0.25
         g%s(40)%v(2)    =  0.25
         g%s(40)%v(3)    =  0.50
       
         g%s(41)%m(1,3)  = -1.00
         g%s(41)%m(2,1)  =  1.00
         g%s(41)%m(3,2)  =  1.00
         g%s(41)%v(1)    =  0.00
         g%s(41)%v(2)    =  0.25
         g%s(41)%v(3)    =  0.25
       
         g%s(42)%m(1,2)  =  1.00
         g%s(42)%m(2,3)  = -1.00
         g%s(42)%m(3,1)  =  1.00
         g%s(42)%v(1)    = -0.25
         g%s(42)%v(2)    =  0.50
         g%s(42)%v(3)    =  0.25
       
         g%s(43)%m(1,3)  =  1.00
         g%s(43)%m(2,1)  =  1.00
         g%s(43)%m(3,2)  = -1.00
         g%s(43)%v(1)    = -0.25
         g%s(43)%v(2)    =  0.25
         g%s(43)%v(3)    =  0.50
       
         g%s(44)%m(1,3)  =  1.00
         g%s(44)%m(2,1)  = -1.00
         g%s(44)%m(3,2)  =  1.00
         g%s(44)%v(1)    = -0.25
         g%s(44)%v(2)    =  0.50
         g%s(44)%v(3)    =  0.25
       
         g%s(45)%m(1,2)  = -1.00
         g%s(45)%m(2,3)  =  1.00
         g%s(45)%m(3,1)  =  1.00
         g%s(45)%v(1)    =  0.00
         g%s(45)%v(2)    =  0.25
         g%s(45)%v(3)    =  0.25
       
         g%s(46)%m(1,1)  = -1.00
         g%s(46)%m(2,2)  =  1.00
         g%s(46)%m(3,3)  =  1.00
         g%s(46)%v(1)    =  0.00
         g%s(46)%v(2)    =  0.25
         g%s(46)%v(3)    =  0.25
       
         g%s(47)%m(1,1)  =  1.00
         g%s(47)%m(2,2)  = -1.00
         g%s(47)%m(3,3)  =  1.00
         g%s(47)%v(1)    = -0.25
         g%s(47)%v(2)    =  0.50
         g%s(47)%v(3)    =  0.25
       
         g%s(48)%m(1,1)  =  1.00
         g%s(48)%m(2,2)  =  1.00
         g%s(48)%m(3,3)  = -1.00
         g%s(48)%v(1)    = -0.25
         g%s(48)%v(2)    =  0.25
         g%s(48)%v(3)    =  0.50
       
         g%s(49)%m(1,1)  =  1.00
         g%s(49)%m(2,2)  =  1.00
         g%s(49)%m(3,3)  =  1.00
         g%s(49)%v(1)    =  0.50
         g%s(49)%v(2)    =  0.00
         g%s(49)%v(3)    =  0.50
       
         g%s(50)%m(1,3)  =  1.00
         g%s(50)%m(2,1)  =  1.00
         g%s(50)%m(3,2)  =  1.00
         g%s(50)%v(1)    =  0.50
         g%s(50)%v(2)    =  0.00
         g%s(50)%v(3)    =  0.50
       
         g%s(51)%m(1,2)  =  1.00
         g%s(51)%m(2,3)  =  1.00
         g%s(51)%m(3,1)  =  1.00
         g%s(51)%v(1)    =  0.50
         g%s(51)%v(2)    =  0.00
         g%s(51)%v(3)    =  0.50
       
         g%s(52)%m(1,2)  = -1.00
         g%s(52)%m(2,3)  = -1.00
         g%s(52)%m(3,1)  =  1.00
         g%s(52)%v(1)    =  0.75000000
         g%s(52)%v(2)    =  0.25
         g%s(52)%v(3)    =  0.50
       
         g%s(53)%m(1,3)  =  1.00
         g%s(53)%m(2,1)  = -1.00
         g%s(53)%m(3,2)  = -1.00
         g%s(53)%v(1)    =  0.50
         g%s(53)%v(2)    =  0.25
         g%s(53)%v(3)    =  0.75000000
       
         g%s(54)%m(1,2)  = -1.00
         g%s(54)%m(2,3)  =  1.00
         g%s(54)%m(3,1)  = -1.00
         g%s(54)%v(1)    =  0.75000000
         g%s(54)%v(2)    =  0.00
         g%s(54)%v(3)    =  0.75000000
       
         g%s(55)%m(1,3)  = -1.00
         g%s(55)%m(2,1)  = -1.00
         g%s(55)%m(3,2)  =  1.00
         g%s(55)%v(1)    =  0.75000000
         g%s(55)%v(2)    =  0.25
         g%s(55)%v(3)    =  0.50
       
         g%s(56)%m(1,3)  = -1.00
         g%s(56)%m(2,1)  =  1.00
         g%s(56)%m(3,2)  = -1.00
         g%s(56)%v(1)    =  0.75000000
         g%s(56)%v(2)    =  0.00
         g%s(56)%v(3)    =  0.75000000
       
         g%s(57)%m(1,2)  =  1.00
         g%s(57)%m(2,3)  = -1.00
         g%s(57)%m(3,1)  = -1.00
         g%s(57)%v(1)    =  0.50
         g%s(57)%v(2)    =  0.25
         g%s(57)%v(3)    =  0.75000000
       
         g%s(58)%m(1,1)  =  1.00
         g%s(58)%m(2,2)  = -1.00
         g%s(58)%m(3,3)  = -1.00
         g%s(58)%v(1)    =  0.50
         g%s(58)%v(2)    =  0.25
         g%s(58)%v(3)    =  0.75000000
       
         g%s(59)%m(1,1)  = -1.00
         g%s(59)%m(2,2)  =  1.00
         g%s(59)%m(3,3)  = -1.00
         g%s(59)%v(1)    =  0.75000000
         g%s(59)%v(2)    =  0.00
         g%s(59)%v(3)    =  0.75000000
       
         g%s(60)%m(1,1)  = -1.00
         g%s(60)%m(2,2)  = -1.00
         g%s(60)%m(3,3)  =  1.00
         g%s(60)%v(1)    =  0.75000000
         g%s(60)%v(2)    =  0.25
         g%s(60)%v(3)    =  0.50
       
         g%s(61)%m(1,1)  = -1.00
         g%s(61)%m(2,2)  = -1.00
         g%s(61)%m(3,3)  = -1.00
         g%s(61)%v(1)    =  0.50
         g%s(61)%v(2)    =  0.00
         g%s(61)%v(3)    =  0.50
       
         g%s(62)%m(1,3)  = -1.00
         g%s(62)%m(2,1)  = -1.00
         g%s(62)%m(3,2)  = -1.00
         g%s(62)%v(1)    =  0.50
         g%s(62)%v(2)    =  0.00
         g%s(62)%v(3)    =  0.50
       
         g%s(63)%m(1,2)  = -1.00
         g%s(63)%m(2,3)  = -1.00
         g%s(63)%m(3,1)  = -1.00
         g%s(63)%v(1)    =  0.50
         g%s(63)%v(2)    =  0.00
         g%s(63)%v(3)    =  0.50
       
         g%s(64)%m(1,2)  =  1.00
         g%s(64)%m(2,3)  =  1.00
         g%s(64)%m(3,1)  = -1.00
         g%s(64)%v(1)    =  0.25
         g%s(64)%v(2)    = -0.25
         g%s(64)%v(3)    =  0.50
       
         g%s(65)%m(1,3)  = -1.00
         g%s(65)%m(2,1)  =  1.00
         g%s(65)%m(3,2)  =  1.00
         g%s(65)%v(1)    =  0.50
         g%s(65)%v(2)    = -0.25
         g%s(65)%v(3)    =  0.25
       
         g%s(66)%m(1,2)  =  1.00
         g%s(66)%m(2,3)  = -1.00
         g%s(66)%m(3,1)  =  1.00
         g%s(66)%v(1)    =  0.25
         g%s(66)%v(2)    =  0.00
         g%s(66)%v(3)    =  0.25
       
         g%s(67)%m(1,3)  =  1.00
         g%s(67)%m(2,1)  =  1.00
         g%s(67)%m(3,2)  = -1.00
         g%s(67)%v(1)    =  0.25
         g%s(67)%v(2)    = -0.25
         g%s(67)%v(3)    =  0.50
       
         g%s(68)%m(1,3)  =  1.00
         g%s(68)%m(2,1)  = -1.00
         g%s(68)%m(3,2)  =  1.00
         g%s(68)%v(1)    =  0.25
         g%s(68)%v(2)    =  0.00
         g%s(68)%v(3)    =  0.25
       
         g%s(69)%m(1,2)  = -1.00
         g%s(69)%m(2,3)  =  1.00
         g%s(69)%m(3,1)  =  1.00
         g%s(69)%v(1)    =  0.50
         g%s(69)%v(2)    = -0.25
         g%s(69)%v(3)    =  0.25
       
         g%s(70)%m(1,1)  = -1.00
         g%s(70)%m(2,2)  =  1.00
         g%s(70)%m(3,3)  =  1.00
         g%s(70)%v(1)    =  0.50
         g%s(70)%v(2)    = -0.25
         g%s(70)%v(3)    =  0.25
       
         g%s(71)%m(1,1)  =  1.00
         g%s(71)%m(2,2)  = -1.00
         g%s(71)%m(3,3)  =  1.00
         g%s(71)%v(1)    =  0.25
         g%s(71)%v(2)    =  0.00
         g%s(71)%v(3)    =  0.25
       
         g%s(72)%m(1,1)  =  1.00
         g%s(72)%m(2,2)  =  1.00
         g%s(72)%m(3,3)  = -1.00
         g%s(72)%v(1)    =  0.25
         g%s(72)%v(2)    = -0.25
         g%s(72)%v(3)    =  0.50
       
         g%s(73)%m(1,1)  =  1.00
         g%s(73)%m(2,2)  =  1.00
         g%s(73)%m(3,3)  =  1.00
         g%s(73)%v(1)    =  0.50
         g%s(73)%v(2)    =  0.50
         g%s(73)%v(3)    =  0.00
       
         g%s(74)%m(1,3)  =  1.00
         g%s(74)%m(2,1)  =  1.00
         g%s(74)%m(3,2)  =  1.00
         g%s(74)%v(1)    =  0.50
         g%s(74)%v(2)    =  0.50
         g%s(74)%v(3)    =  0.00
       
         g%s(75)%m(1,2)  =  1.00
         g%s(75)%m(2,3)  =  1.00
         g%s(75)%m(3,1)  =  1.00
         g%s(75)%v(1)    =  0.50
         g%s(75)%v(2)    =  0.50
         g%s(75)%v(3)    =  0.00
       
         g%s(76)%m(1,2)  = -1.00
         g%s(76)%m(2,3)  = -1.00
         g%s(76)%m(3,1)  =  1.00
         g%s(76)%v(1)    =  0.75000000
         g%s(76)%v(2)    =  0.75000000
         g%s(76)%v(3)    =  0.00
       
         g%s(77)%m(1,3)  =  1.00
         g%s(77)%m(2,1)  = -1.00
         g%s(77)%m(3,2)  = -1.00
         g%s(77)%v(1)    =  0.50
         g%s(77)%v(2)    =  0.75000000
         g%s(77)%v(3)    =  0.25
       
         g%s(78)%m(1,2)  = -1.00
         g%s(78)%m(2,3)  =  1.00
         g%s(78)%m(3,1)  = -1.00
         g%s(78)%v(1)    =  0.75000000
         g%s(78)%v(2)    =  0.50
         g%s(78)%v(3)    =  0.25
       
         g%s(79)%m(1,3)  = -1.00
         g%s(79)%m(2,1)  = -1.00
         g%s(79)%m(3,2)  =  1.00
         g%s(79)%v(1)    =  0.75000000
         g%s(79)%v(2)    =  0.75000000
         g%s(79)%v(3)    =  0.00
       
         g%s(80)%m(1,3)  = -1.00
         g%s(80)%m(2,1)  =  1.00
         g%s(80)%m(3,2)  = -1.00
         g%s(80)%v(1)    =  0.75000000
         g%s(80)%v(2)    =  0.50
         g%s(80)%v(3)    =  0.25
       
         g%s(81)%m(1,2)  =  1.00
         g%s(81)%m(2,3)  = -1.00
         g%s(81)%m(3,1)  = -1.00
         g%s(81)%v(1)    =  0.50
         g%s(81)%v(2)    =  0.75000000
         g%s(81)%v(3)    =  0.25
       
         g%s(82)%m(1,1)  =  1.00
         g%s(82)%m(2,2)  = -1.00
         g%s(82)%m(3,3)  = -1.00
         g%s(82)%v(1)    =  0.50
         g%s(82)%v(2)    =  0.75000000
         g%s(82)%v(3)    =  0.25
       
         g%s(83)%m(1,1)  = -1.00
         g%s(83)%m(2,2)  =  1.00
         g%s(83)%m(3,3)  = -1.00
         g%s(83)%v(1)    =  0.75000000
         g%s(83)%v(2)    =  0.50
         g%s(83)%v(3)    =  0.25
       
         g%s(84)%m(1,1)  = -1.00
         g%s(84)%m(2,2)  = -1.00
         g%s(84)%m(3,3)  =  1.00
         g%s(84)%v(1)    =  0.75000000
         g%s(84)%v(2)    =  0.75000000
         g%s(84)%v(3)    =  0.00
       
         g%s(85)%m(1,1)  = -1.00
         g%s(85)%m(2,2)  = -1.00
         g%s(85)%m(3,3)  = -1.00
         g%s(85)%v(1)    =  0.50
         g%s(85)%v(2)    =  0.50
         g%s(85)%v(3)    =  0.00
       
         g%s(86)%m(1,3)  = -1.00
         g%s(86)%m(2,1)  = -1.00
         g%s(86)%m(3,2)  = -1.00
         g%s(86)%v(1)    =  0.50
         g%s(86)%v(2)    =  0.50
         g%s(86)%v(3)    =  0.00
       
         g%s(87)%m(1,2)  = -1.00
         g%s(87)%m(2,3)  = -1.00
         g%s(87)%m(3,1)  = -1.00
         g%s(87)%v(1)    =  0.50
         g%s(87)%v(2)    =  0.50
         g%s(87)%v(3)    =  0.00
       
         g%s(88)%m(1,2)  =  1.00
         g%s(88)%m(2,3)  =  1.00
         g%s(88)%m(3,1)  = -1.00
         g%s(88)%v(1)    =  0.25
         g%s(88)%v(2)    =  0.25
         g%s(88)%v(3)    =  0.00
       
         g%s(89)%m(1,3)  = -1.00
         g%s(89)%m(2,1)  =  1.00
         g%s(89)%m(3,2)  =  1.00
         g%s(89)%v(1)    =  0.50
         g%s(89)%v(2)    =  0.25
         g%s(89)%v(3)    = -0.25
       
         g%s(90)%m(1,2)  =  1.00
         g%s(90)%m(2,3)  = -1.00
         g%s(90)%m(3,1)  =  1.00
         g%s(90)%v(1)    =  0.25
         g%s(90)%v(2)    =  0.50
         g%s(90)%v(3)    = -0.25
       
         g%s(91)%m(1,3)  =  1.00
         g%s(91)%m(2,1)  =  1.00
         g%s(91)%m(3,2)  = -1.00
         g%s(91)%v(1)    =  0.25
         g%s(91)%v(2)    =  0.25
         g%s(91)%v(3)    =  0.00
       
         g%s(92)%m(1,3)  =  1.00
         g%s(92)%m(2,1)  = -1.00
         g%s(92)%m(3,2)  =  1.00
         g%s(92)%v(1)    =  0.25
         g%s(92)%v(2)    =  0.50
         g%s(92)%v(3)    = -0.25
       
         g%s(93)%m(1,2)  = -1.00
         g%s(93)%m(2,3)  =  1.00
         g%s(93)%m(3,1)  =  1.00
         g%s(93)%v(1)    =  0.50
         g%s(93)%v(2)    =  0.25
         g%s(93)%v(3)    = -0.25
       
         g%s(94)%m(1,1)  = -1.00
         g%s(94)%m(2,2)  =  1.00
         g%s(94)%m(3,3)  =  1.00
         g%s(94)%v(1)    =  0.50
         g%s(94)%v(2)    =  0.25
         g%s(94)%v(3)    = -0.25
       
         g%s(95)%m(1,1)  =  1.00
         g%s(95)%m(2,2)  = -1.00
         g%s(95)%m(3,3)  =  1.00
         g%s(95)%v(1)    =  0.25
         g%s(95)%v(2)    =  0.50
         g%s(95)%v(3)    = -0.25
       
         g%s(96)%m(1,1)  =  1.00
         g%s(96)%m(2,2)  =  1.00
         g%s(96)%m(3,3)  = -1.00
         g%s(96)%v(1)    =  0.25
         g%s(96)%v(2)    =  0.25
         g%s(96)%v(3)    =  0.00
       
       
      case('F d -3 : 1')                      
       
         !Space group   # 203
         !Group setting # 230
       
         g%order   =  96
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,3)   =  1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,1)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,3)   = -1.00
         g%s(4)%m(3,1)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,2)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,3)   =  1.00
         g%s(6)%m(3,1)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,3)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,2)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   = -1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   = -1.00
         g%s(9)%m(3,1)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.25
         g%s(13)%v(2)    =  0.25
         g%s(13)%v(3)    =  0.25
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  = -1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.25
         g%s(14)%v(2)    =  0.25
         g%s(14)%v(3)    =  0.25
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.25
         g%s(15)%v(2)    =  0.25
         g%s(15)%v(3)    =  0.25
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,3)  =  1.00
         g%s(16)%m(3,1)  = -1.00
         g%s(16)%v(1)    =  0.25
         g%s(16)%v(2)    =  0.25
         g%s(16)%v(3)    =  0.25
       
         g%s(17)%m(1,3)  = -1.00
         g%s(17)%m(2,1)  =  1.00
         g%s(17)%m(3,2)  =  1.00
         g%s(17)%v(1)    =  0.25
         g%s(17)%v(2)    =  0.25
         g%s(17)%v(3)    =  0.25
       
         g%s(18)%m(1,2)  =  1.00
         g%s(18)%m(2,3)  = -1.00
         g%s(18)%m(3,1)  =  1.00
         g%s(18)%v(1)    =  0.25
         g%s(18)%v(2)    =  0.25
         g%s(18)%v(3)    =  0.25
       
         g%s(19)%m(1,3)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,2)  = -1.00
         g%s(19)%v(1)    =  0.25
         g%s(19)%v(2)    =  0.25
         g%s(19)%v(3)    =  0.25
       
         g%s(20)%m(1,3)  =  1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,2)  =  1.00
         g%s(20)%v(1)    =  0.25
         g%s(20)%v(2)    =  0.25
         g%s(20)%v(3)    =  0.25
       
         g%s(21)%m(1,2)  = -1.00
         g%s(21)%m(2,3)  =  1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.25
         g%s(21)%v(2)    =  0.25
         g%s(21)%v(3)    =  0.25
       
         g%s(22)%m(1,1)  = -1.00
         g%s(22)%m(2,2)  =  1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.25
         g%s(22)%v(2)    =  0.25
         g%s(22)%v(3)    =  0.25
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(2,2)  = -1.00
         g%s(23)%m(3,3)  =  1.00
         g%s(23)%v(1)    =  0.25
         g%s(23)%v(2)    =  0.25
         g%s(23)%v(3)    =  0.25
       
         g%s(24)%m(1,1)  =  1.00
         g%s(24)%m(2,2)  =  1.00
         g%s(24)%m(3,3)  = -1.00
         g%s(24)%v(1)    =  0.25
         g%s(24)%v(2)    =  0.25
         g%s(24)%v(3)    =  0.25
       
         g%s(25)%m(1,1)  =  1.00
         g%s(25)%m(2,2)  =  1.00
         g%s(25)%m(3,3)  =  1.00
         g%s(25)%v(1)    =  0.00
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  0.50
       
         g%s(26)%m(1,3)  =  1.00
         g%s(26)%m(2,1)  =  1.00
         g%s(26)%m(3,2)  =  1.00
         g%s(26)%v(1)    =  0.00
         g%s(26)%v(2)    =  0.50
         g%s(26)%v(3)    =  0.50
       
         g%s(27)%m(1,2)  =  1.00
         g%s(27)%m(2,3)  =  1.00
         g%s(27)%m(3,1)  =  1.00
         g%s(27)%v(1)    =  0.00
         g%s(27)%v(2)    =  0.50
         g%s(27)%v(3)    =  0.50
       
         g%s(28)%m(1,2)  = -1.00
         g%s(28)%m(2,3)  = -1.00
         g%s(28)%m(3,1)  =  1.00
         g%s(28)%v(1)    =  0.00
         g%s(28)%v(2)    =  0.50
         g%s(28)%v(3)    =  0.50
       
         g%s(29)%m(1,3)  =  1.00
         g%s(29)%m(2,1)  = -1.00
         g%s(29)%m(3,2)  = -1.00
         g%s(29)%v(1)    =  0.00
         g%s(29)%v(2)    =  0.50
         g%s(29)%v(3)    =  0.50
       
         g%s(30)%m(1,2)  = -1.00
         g%s(30)%m(2,3)  =  1.00
         g%s(30)%m(3,1)  = -1.00
         g%s(30)%v(1)    =  0.00
         g%s(30)%v(2)    =  0.50
         g%s(30)%v(3)    =  0.50
       
         g%s(31)%m(1,3)  = -1.00
         g%s(31)%m(2,1)  = -1.00
         g%s(31)%m(3,2)  =  1.00
         g%s(31)%v(1)    =  0.00
         g%s(31)%v(2)    =  0.50
         g%s(31)%v(3)    =  0.50
       
         g%s(32)%m(1,3)  = -1.00
         g%s(32)%m(2,1)  =  1.00
         g%s(32)%m(3,2)  = -1.00
         g%s(32)%v(1)    =  0.00
         g%s(32)%v(2)    =  0.50
         g%s(32)%v(3)    =  0.50
       
         g%s(33)%m(1,2)  =  1.00
         g%s(33)%m(2,3)  = -1.00
         g%s(33)%m(3,1)  = -1.00
         g%s(33)%v(1)    =  0.00
         g%s(33)%v(2)    =  0.50
         g%s(33)%v(3)    =  0.50
       
         g%s(34)%m(1,1)  =  1.00
         g%s(34)%m(2,2)  = -1.00
         g%s(34)%m(3,3)  = -1.00
         g%s(34)%v(1)    =  0.00
         g%s(34)%v(2)    =  0.50
         g%s(34)%v(3)    =  0.50
       
         g%s(35)%m(1,1)  = -1.00
         g%s(35)%m(2,2)  =  1.00
         g%s(35)%m(3,3)  = -1.00
         g%s(35)%v(1)    =  0.00
         g%s(35)%v(2)    =  0.50
         g%s(35)%v(3)    =  0.50
       
         g%s(36)%m(1,1)  = -1.00
         g%s(36)%m(2,2)  = -1.00
         g%s(36)%m(3,3)  =  1.00
         g%s(36)%v(1)    =  0.00
         g%s(36)%v(2)    =  0.50
         g%s(36)%v(3)    =  0.50
       
         g%s(37)%m(1,1)  = -1.00
         g%s(37)%m(2,2)  = -1.00
         g%s(37)%m(3,3)  = -1.00
         g%s(37)%v(1)    =  0.25
         g%s(37)%v(2)    =  0.75000000
         g%s(37)%v(3)    =  0.75000000
       
         g%s(38)%m(1,3)  = -1.00
         g%s(38)%m(2,1)  = -1.00
         g%s(38)%m(3,2)  = -1.00
         g%s(38)%v(1)    =  0.25
         g%s(38)%v(2)    =  0.75000000
         g%s(38)%v(3)    =  0.75000000
       
         g%s(39)%m(1,2)  = -1.00
         g%s(39)%m(2,3)  = -1.00
         g%s(39)%m(3,1)  = -1.00
         g%s(39)%v(1)    =  0.25
         g%s(39)%v(2)    =  0.75000000
         g%s(39)%v(3)    =  0.75000000
       
         g%s(40)%m(1,2)  =  1.00
         g%s(40)%m(2,3)  =  1.00
         g%s(40)%m(3,1)  = -1.00
         g%s(40)%v(1)    =  0.25
         g%s(40)%v(2)    =  0.75000000
         g%s(40)%v(3)    =  0.75000000
       
         g%s(41)%m(1,3)  = -1.00
         g%s(41)%m(2,1)  =  1.00
         g%s(41)%m(3,2)  =  1.00
         g%s(41)%v(1)    =  0.25
         g%s(41)%v(2)    =  0.75000000
         g%s(41)%v(3)    =  0.75000000
       
         g%s(42)%m(1,2)  =  1.00
         g%s(42)%m(2,3)  = -1.00
         g%s(42)%m(3,1)  =  1.00
         g%s(42)%v(1)    =  0.25
         g%s(42)%v(2)    =  0.75000000
         g%s(42)%v(3)    =  0.75000000
       
         g%s(43)%m(1,3)  =  1.00
         g%s(43)%m(2,1)  =  1.00
         g%s(43)%m(3,2)  = -1.00
         g%s(43)%v(1)    =  0.25
         g%s(43)%v(2)    =  0.75000000
         g%s(43)%v(3)    =  0.75000000
       
         g%s(44)%m(1,3)  =  1.00
         g%s(44)%m(2,1)  = -1.00
         g%s(44)%m(3,2)  =  1.00
         g%s(44)%v(1)    =  0.25
         g%s(44)%v(2)    =  0.75000000
         g%s(44)%v(3)    =  0.75000000
       
         g%s(45)%m(1,2)  = -1.00
         g%s(45)%m(2,3)  =  1.00
         g%s(45)%m(3,1)  =  1.00
         g%s(45)%v(1)    =  0.25
         g%s(45)%v(2)    =  0.75000000
         g%s(45)%v(3)    =  0.75000000
       
         g%s(46)%m(1,1)  = -1.00
         g%s(46)%m(2,2)  =  1.00
         g%s(46)%m(3,3)  =  1.00
         g%s(46)%v(1)    =  0.25
         g%s(46)%v(2)    =  0.75000000
         g%s(46)%v(3)    =  0.75000000
       
         g%s(47)%m(1,1)  =  1.00
         g%s(47)%m(2,2)  = -1.00
         g%s(47)%m(3,3)  =  1.00
         g%s(47)%v(1)    =  0.25
         g%s(47)%v(2)    =  0.75000000
         g%s(47)%v(3)    =  0.75000000
       
         g%s(48)%m(1,1)  =  1.00
         g%s(48)%m(2,2)  =  1.00
         g%s(48)%m(3,3)  = -1.00
         g%s(48)%v(1)    =  0.25
         g%s(48)%v(2)    =  0.75000000
         g%s(48)%v(3)    =  0.75000000
       
         g%s(49)%m(1,1)  =  1.00
         g%s(49)%m(2,2)  =  1.00
         g%s(49)%m(3,3)  =  1.00
         g%s(49)%v(1)    =  0.50
         g%s(49)%v(2)    =  0.00
         g%s(49)%v(3)    =  0.50
       
         g%s(50)%m(1,3)  =  1.00
         g%s(50)%m(2,1)  =  1.00
         g%s(50)%m(3,2)  =  1.00
         g%s(50)%v(1)    =  0.50
         g%s(50)%v(2)    =  0.00
         g%s(50)%v(3)    =  0.50
       
         g%s(51)%m(1,2)  =  1.00
         g%s(51)%m(2,3)  =  1.00
         g%s(51)%m(3,1)  =  1.00
         g%s(51)%v(1)    =  0.50
         g%s(51)%v(2)    =  0.00
         g%s(51)%v(3)    =  0.50
       
         g%s(52)%m(1,2)  = -1.00
         g%s(52)%m(2,3)  = -1.00
         g%s(52)%m(3,1)  =  1.00
         g%s(52)%v(1)    =  0.50
         g%s(52)%v(2)    =  0.00
         g%s(52)%v(3)    =  0.50
       
         g%s(53)%m(1,3)  =  1.00
         g%s(53)%m(2,1)  = -1.00
         g%s(53)%m(3,2)  = -1.00
         g%s(53)%v(1)    =  0.50
         g%s(53)%v(2)    =  0.00
         g%s(53)%v(3)    =  0.50
       
         g%s(54)%m(1,2)  = -1.00
         g%s(54)%m(2,3)  =  1.00
         g%s(54)%m(3,1)  = -1.00
         g%s(54)%v(1)    =  0.50
         g%s(54)%v(2)    =  0.00
         g%s(54)%v(3)    =  0.50
       
         g%s(55)%m(1,3)  = -1.00
         g%s(55)%m(2,1)  = -1.00
         g%s(55)%m(3,2)  =  1.00
         g%s(55)%v(1)    =  0.50
         g%s(55)%v(2)    =  0.00
         g%s(55)%v(3)    =  0.50
       
         g%s(56)%m(1,3)  = -1.00
         g%s(56)%m(2,1)  =  1.00
         g%s(56)%m(3,2)  = -1.00
         g%s(56)%v(1)    =  0.50
         g%s(56)%v(2)    =  0.00
         g%s(56)%v(3)    =  0.50
       
         g%s(57)%m(1,2)  =  1.00
         g%s(57)%m(2,3)  = -1.00
         g%s(57)%m(3,1)  = -1.00
         g%s(57)%v(1)    =  0.50
         g%s(57)%v(2)    =  0.00
         g%s(57)%v(3)    =  0.50
       
         g%s(58)%m(1,1)  =  1.00
         g%s(58)%m(2,2)  = -1.00
         g%s(58)%m(3,3)  = -1.00
         g%s(58)%v(1)    =  0.50
         g%s(58)%v(2)    =  0.00
         g%s(58)%v(3)    =  0.50
       
         g%s(59)%m(1,1)  = -1.00
         g%s(59)%m(2,2)  =  1.00
         g%s(59)%m(3,3)  = -1.00
         g%s(59)%v(1)    =  0.50
         g%s(59)%v(2)    =  0.00
         g%s(59)%v(3)    =  0.50
       
         g%s(60)%m(1,1)  = -1.00
         g%s(60)%m(2,2)  = -1.00
         g%s(60)%m(3,3)  =  1.00
         g%s(60)%v(1)    =  0.50
         g%s(60)%v(2)    =  0.00
         g%s(60)%v(3)    =  0.50
       
         g%s(61)%m(1,1)  = -1.00
         g%s(61)%m(2,2)  = -1.00
         g%s(61)%m(3,3)  = -1.00
         g%s(61)%v(1)    =  0.75000000
         g%s(61)%v(2)    =  0.25
         g%s(61)%v(3)    =  0.75000000
       
         g%s(62)%m(1,3)  = -1.00
         g%s(62)%m(2,1)  = -1.00
         g%s(62)%m(3,2)  = -1.00
         g%s(62)%v(1)    =  0.75000000
         g%s(62)%v(2)    =  0.25
         g%s(62)%v(3)    =  0.75000000
       
         g%s(63)%m(1,2)  = -1.00
         g%s(63)%m(2,3)  = -1.00
         g%s(63)%m(3,1)  = -1.00
         g%s(63)%v(1)    =  0.75000000
         g%s(63)%v(2)    =  0.25
         g%s(63)%v(3)    =  0.75000000
       
         g%s(64)%m(1,2)  =  1.00
         g%s(64)%m(2,3)  =  1.00
         g%s(64)%m(3,1)  = -1.00
         g%s(64)%v(1)    =  0.75000000
         g%s(64)%v(2)    =  0.25
         g%s(64)%v(3)    =  0.75000000
       
         g%s(65)%m(1,3)  = -1.00
         g%s(65)%m(2,1)  =  1.00
         g%s(65)%m(3,2)  =  1.00
         g%s(65)%v(1)    =  0.75000000
         g%s(65)%v(2)    =  0.25
         g%s(65)%v(3)    =  0.75000000
       
         g%s(66)%m(1,2)  =  1.00
         g%s(66)%m(2,3)  = -1.00
         g%s(66)%m(3,1)  =  1.00
         g%s(66)%v(1)    =  0.75000000
         g%s(66)%v(2)    =  0.25
         g%s(66)%v(3)    =  0.75000000
       
         g%s(67)%m(1,3)  =  1.00
         g%s(67)%m(2,1)  =  1.00
         g%s(67)%m(3,2)  = -1.00
         g%s(67)%v(1)    =  0.75000000
         g%s(67)%v(2)    =  0.25
         g%s(67)%v(3)    =  0.75000000
       
         g%s(68)%m(1,3)  =  1.00
         g%s(68)%m(2,1)  = -1.00
         g%s(68)%m(3,2)  =  1.00
         g%s(68)%v(1)    =  0.75000000
         g%s(68)%v(2)    =  0.25
         g%s(68)%v(3)    =  0.75000000
       
         g%s(69)%m(1,2)  = -1.00
         g%s(69)%m(2,3)  =  1.00
         g%s(69)%m(3,1)  =  1.00
         g%s(69)%v(1)    =  0.75000000
         g%s(69)%v(2)    =  0.25
         g%s(69)%v(3)    =  0.75000000
       
         g%s(70)%m(1,1)  = -1.00
         g%s(70)%m(2,2)  =  1.00
         g%s(70)%m(3,3)  =  1.00
         g%s(70)%v(1)    =  0.75000000
         g%s(70)%v(2)    =  0.25
         g%s(70)%v(3)    =  0.75000000
       
         g%s(71)%m(1,1)  =  1.00
         g%s(71)%m(2,2)  = -1.00
         g%s(71)%m(3,3)  =  1.00
         g%s(71)%v(1)    =  0.75000000
         g%s(71)%v(2)    =  0.25
         g%s(71)%v(3)    =  0.75000000
       
         g%s(72)%m(1,1)  =  1.00
         g%s(72)%m(2,2)  =  1.00
         g%s(72)%m(3,3)  = -1.00
         g%s(72)%v(1)    =  0.75000000
         g%s(72)%v(2)    =  0.25
         g%s(72)%v(3)    =  0.75000000
       
         g%s(73)%m(1,1)  =  1.00
         g%s(73)%m(2,2)  =  1.00
         g%s(73)%m(3,3)  =  1.00
         g%s(73)%v(1)    =  0.50
         g%s(73)%v(2)    =  0.50
         g%s(73)%v(3)    =  0.00
       
         g%s(74)%m(1,3)  =  1.00
         g%s(74)%m(2,1)  =  1.00
         g%s(74)%m(3,2)  =  1.00
         g%s(74)%v(1)    =  0.50
         g%s(74)%v(2)    =  0.50
         g%s(74)%v(3)    =  0.00
       
         g%s(75)%m(1,2)  =  1.00
         g%s(75)%m(2,3)  =  1.00
         g%s(75)%m(3,1)  =  1.00
         g%s(75)%v(1)    =  0.50
         g%s(75)%v(2)    =  0.50
         g%s(75)%v(3)    =  0.00
       
         g%s(76)%m(1,2)  = -1.00
         g%s(76)%m(2,3)  = -1.00
         g%s(76)%m(3,1)  =  1.00
         g%s(76)%v(1)    =  0.50
         g%s(76)%v(2)    =  0.50
         g%s(76)%v(3)    =  0.00
       
         g%s(77)%m(1,3)  =  1.00
         g%s(77)%m(2,1)  = -1.00
         g%s(77)%m(3,2)  = -1.00
         g%s(77)%v(1)    =  0.50
         g%s(77)%v(2)    =  0.50
         g%s(77)%v(3)    =  0.00
       
         g%s(78)%m(1,2)  = -1.00
         g%s(78)%m(2,3)  =  1.00
         g%s(78)%m(3,1)  = -1.00
         g%s(78)%v(1)    =  0.50
         g%s(78)%v(2)    =  0.50
         g%s(78)%v(3)    =  0.00
       
         g%s(79)%m(1,3)  = -1.00
         g%s(79)%m(2,1)  = -1.00
         g%s(79)%m(3,2)  =  1.00
         g%s(79)%v(1)    =  0.50
         g%s(79)%v(2)    =  0.50
         g%s(79)%v(3)    =  0.00
       
         g%s(80)%m(1,3)  = -1.00
         g%s(80)%m(2,1)  =  1.00
         g%s(80)%m(3,2)  = -1.00
         g%s(80)%v(1)    =  0.50
         g%s(80)%v(2)    =  0.50
         g%s(80)%v(3)    =  0.00
       
         g%s(81)%m(1,2)  =  1.00
         g%s(81)%m(2,3)  = -1.00
         g%s(81)%m(3,1)  = -1.00
         g%s(81)%v(1)    =  0.50
         g%s(81)%v(2)    =  0.50
         g%s(81)%v(3)    =  0.00
       
         g%s(82)%m(1,1)  =  1.00
         g%s(82)%m(2,2)  = -1.00
         g%s(82)%m(3,3)  = -1.00
         g%s(82)%v(1)    =  0.50
         g%s(82)%v(2)    =  0.50
         g%s(82)%v(3)    =  0.00
       
         g%s(83)%m(1,1)  = -1.00
         g%s(83)%m(2,2)  =  1.00
         g%s(83)%m(3,3)  = -1.00
         g%s(83)%v(1)    =  0.50
         g%s(83)%v(2)    =  0.50
         g%s(83)%v(3)    =  0.00
       
         g%s(84)%m(1,1)  = -1.00
         g%s(84)%m(2,2)  = -1.00
         g%s(84)%m(3,3)  =  1.00
         g%s(84)%v(1)    =  0.50
         g%s(84)%v(2)    =  0.50
         g%s(84)%v(3)    =  0.00
       
         g%s(85)%m(1,1)  = -1.00
         g%s(85)%m(2,2)  = -1.00
         g%s(85)%m(3,3)  = -1.00
         g%s(85)%v(1)    =  0.75000000
         g%s(85)%v(2)    =  0.75000000
         g%s(85)%v(3)    =  0.25
       
         g%s(86)%m(1,3)  = -1.00
         g%s(86)%m(2,1)  = -1.00
         g%s(86)%m(3,2)  = -1.00
         g%s(86)%v(1)    =  0.75000000
         g%s(86)%v(2)    =  0.75000000
         g%s(86)%v(3)    =  0.25
       
         g%s(87)%m(1,2)  = -1.00
         g%s(87)%m(2,3)  = -1.00
         g%s(87)%m(3,1)  = -1.00
         g%s(87)%v(1)    =  0.75000000
         g%s(87)%v(2)    =  0.75000000
         g%s(87)%v(3)    =  0.25
       
         g%s(88)%m(1,2)  =  1.00
         g%s(88)%m(2,3)  =  1.00
         g%s(88)%m(3,1)  = -1.00
         g%s(88)%v(1)    =  0.75000000
         g%s(88)%v(2)    =  0.75000000
         g%s(88)%v(3)    =  0.25
       
         g%s(89)%m(1,3)  = -1.00
         g%s(89)%m(2,1)  =  1.00
         g%s(89)%m(3,2)  =  1.00
         g%s(89)%v(1)    =  0.75000000
         g%s(89)%v(2)    =  0.75000000
         g%s(89)%v(3)    =  0.25
       
         g%s(90)%m(1,2)  =  1.00
         g%s(90)%m(2,3)  = -1.00
         g%s(90)%m(3,1)  =  1.00
         g%s(90)%v(1)    =  0.75000000
         g%s(90)%v(2)    =  0.75000000
         g%s(90)%v(3)    =  0.25
       
         g%s(91)%m(1,3)  =  1.00
         g%s(91)%m(2,1)  =  1.00
         g%s(91)%m(3,2)  = -1.00
         g%s(91)%v(1)    =  0.75000000
         g%s(91)%v(2)    =  0.75000000
         g%s(91)%v(3)    =  0.25
       
         g%s(92)%m(1,3)  =  1.00
         g%s(92)%m(2,1)  = -1.00
         g%s(92)%m(3,2)  =  1.00
         g%s(92)%v(1)    =  0.75000000
         g%s(92)%v(2)    =  0.75000000
         g%s(92)%v(3)    =  0.25
       
         g%s(93)%m(1,2)  = -1.00
         g%s(93)%m(2,3)  =  1.00
         g%s(93)%m(3,1)  =  1.00
         g%s(93)%v(1)    =  0.75000000
         g%s(93)%v(2)    =  0.75000000
         g%s(93)%v(3)    =  0.25
       
         g%s(94)%m(1,1)  = -1.00
         g%s(94)%m(2,2)  =  1.00
         g%s(94)%m(3,3)  =  1.00
         g%s(94)%v(1)    =  0.75000000
         g%s(94)%v(2)    =  0.75000000
         g%s(94)%v(3)    =  0.25
       
         g%s(95)%m(1,1)  =  1.00
         g%s(95)%m(2,2)  = -1.00
         g%s(95)%m(3,3)  =  1.00
         g%s(95)%v(1)    =  0.75000000
         g%s(95)%v(2)    =  0.75000000
         g%s(95)%v(3)    =  0.25
       
         g%s(96)%m(1,1)  =  1.00
         g%s(96)%m(2,2)  =  1.00
         g%s(96)%m(3,3)  = -1.00
         g%s(96)%v(1)    =  0.75000000
         g%s(96)%v(2)    =  0.75000000
         g%s(96)%v(3)    =  0.25
       
       
      case('I m -3', '204')                          
       
         !Space group   # 204
         !Group setting # 231
       
         g%order   =  48
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,3)   =  1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,1)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,3)   = -1.00
         g%s(4)%m(3,1)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,2)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,3)   =  1.00
         g%s(6)%m(3,1)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,3)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,2)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   = -1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   = -1.00
         g%s(9)%m(3,1)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  = -1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,3)  =  1.00
         g%s(16)%m(3,1)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,3)  = -1.00
         g%s(17)%m(2,1)  =  1.00
         g%s(17)%m(3,2)  =  1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,2)  =  1.00
         g%s(18)%m(2,3)  = -1.00
         g%s(18)%m(3,1)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,3)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,2)  = -1.00
         g%s(19)%v(1)    =  0.00
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    =  0.00
       
         g%s(20)%m(1,3)  =  1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,2)  =  1.00
         g%s(20)%v(1)    =  0.00
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    =  0.00
       
         g%s(21)%m(1,2)  = -1.00
         g%s(21)%m(2,3)  =  1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.00
         g%s(21)%v(2)    =  0.00
         g%s(21)%v(3)    =  0.00
       
         g%s(22)%m(1,1)  = -1.00
         g%s(22)%m(2,2)  =  1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.00
         g%s(22)%v(2)    =  0.00
         g%s(22)%v(3)    =  0.00
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(2,2)  = -1.00
         g%s(23)%m(3,3)  =  1.00
         g%s(23)%v(1)    =  0.00
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    =  0.00
       
         g%s(24)%m(1,1)  =  1.00
         g%s(24)%m(2,2)  =  1.00
         g%s(24)%m(3,3)  = -1.00
         g%s(24)%v(1)    =  0.00
         g%s(24)%v(2)    =  0.00
         g%s(24)%v(3)    =  0.00
       
         g%s(25)%m(1,1)  =  1.00
         g%s(25)%m(2,2)  =  1.00
         g%s(25)%m(3,3)  =  1.00
         g%s(25)%v(1)    =  0.50
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  0.50
       
         g%s(26)%m(1,3)  =  1.00
         g%s(26)%m(2,1)  =  1.00
         g%s(26)%m(3,2)  =  1.00
         g%s(26)%v(1)    =  0.50
         g%s(26)%v(2)    =  0.50
         g%s(26)%v(3)    =  0.50
       
         g%s(27)%m(1,2)  =  1.00
         g%s(27)%m(2,3)  =  1.00
         g%s(27)%m(3,1)  =  1.00
         g%s(27)%v(1)    =  0.50
         g%s(27)%v(2)    =  0.50
         g%s(27)%v(3)    =  0.50
       
         g%s(28)%m(1,2)  = -1.00
         g%s(28)%m(2,3)  = -1.00
         g%s(28)%m(3,1)  =  1.00
         g%s(28)%v(1)    =  0.50
         g%s(28)%v(2)    =  0.50
         g%s(28)%v(3)    =  0.50
       
         g%s(29)%m(1,3)  =  1.00
         g%s(29)%m(2,1)  = -1.00
         g%s(29)%m(3,2)  = -1.00
         g%s(29)%v(1)    =  0.50
         g%s(29)%v(2)    =  0.50
         g%s(29)%v(3)    =  0.50
       
         g%s(30)%m(1,2)  = -1.00
         g%s(30)%m(2,3)  =  1.00
         g%s(30)%m(3,1)  = -1.00
         g%s(30)%v(1)    =  0.50
         g%s(30)%v(2)    =  0.50
         g%s(30)%v(3)    =  0.50
       
         g%s(31)%m(1,3)  = -1.00
         g%s(31)%m(2,1)  = -1.00
         g%s(31)%m(3,2)  =  1.00
         g%s(31)%v(1)    =  0.50
         g%s(31)%v(2)    =  0.50
         g%s(31)%v(3)    =  0.50
       
         g%s(32)%m(1,3)  = -1.00
         g%s(32)%m(2,1)  =  1.00
         g%s(32)%m(3,2)  = -1.00
         g%s(32)%v(1)    =  0.50
         g%s(32)%v(2)    =  0.50
         g%s(32)%v(3)    =  0.50
       
         g%s(33)%m(1,2)  =  1.00
         g%s(33)%m(2,3)  = -1.00
         g%s(33)%m(3,1)  = -1.00
         g%s(33)%v(1)    =  0.50
         g%s(33)%v(2)    =  0.50
         g%s(33)%v(3)    =  0.50
       
         g%s(34)%m(1,1)  =  1.00
         g%s(34)%m(2,2)  = -1.00
         g%s(34)%m(3,3)  = -1.00
         g%s(34)%v(1)    =  0.50
         g%s(34)%v(2)    =  0.50
         g%s(34)%v(3)    =  0.50
       
         g%s(35)%m(1,1)  = -1.00
         g%s(35)%m(2,2)  =  1.00
         g%s(35)%m(3,3)  = -1.00
         g%s(35)%v(1)    =  0.50
         g%s(35)%v(2)    =  0.50
         g%s(35)%v(3)    =  0.50
       
         g%s(36)%m(1,1)  = -1.00
         g%s(36)%m(2,2)  = -1.00
         g%s(36)%m(3,3)  =  1.00
         g%s(36)%v(1)    =  0.50
         g%s(36)%v(2)    =  0.50
         g%s(36)%v(3)    =  0.50
       
         g%s(37)%m(1,1)  = -1.00
         g%s(37)%m(2,2)  = -1.00
         g%s(37)%m(3,3)  = -1.00
         g%s(37)%v(1)    =  0.50
         g%s(37)%v(2)    =  0.50
         g%s(37)%v(3)    =  0.50
       
         g%s(38)%m(1,3)  = -1.00
         g%s(38)%m(2,1)  = -1.00
         g%s(38)%m(3,2)  = -1.00
         g%s(38)%v(1)    =  0.50
         g%s(38)%v(2)    =  0.50
         g%s(38)%v(3)    =  0.50
       
         g%s(39)%m(1,2)  = -1.00
         g%s(39)%m(2,3)  = -1.00
         g%s(39)%m(3,1)  = -1.00
         g%s(39)%v(1)    =  0.50
         g%s(39)%v(2)    =  0.50
         g%s(39)%v(3)    =  0.50
       
         g%s(40)%m(1,2)  =  1.00
         g%s(40)%m(2,3)  =  1.00
         g%s(40)%m(3,1)  = -1.00
         g%s(40)%v(1)    =  0.50
         g%s(40)%v(2)    =  0.50
         g%s(40)%v(3)    =  0.50
       
         g%s(41)%m(1,3)  = -1.00
         g%s(41)%m(2,1)  =  1.00
         g%s(41)%m(3,2)  =  1.00
         g%s(41)%v(1)    =  0.50
         g%s(41)%v(2)    =  0.50
         g%s(41)%v(3)    =  0.50
       
         g%s(42)%m(1,2)  =  1.00
         g%s(42)%m(2,3)  = -1.00
         g%s(42)%m(3,1)  =  1.00
         g%s(42)%v(1)    =  0.50
         g%s(42)%v(2)    =  0.50
         g%s(42)%v(3)    =  0.50
       
         g%s(43)%m(1,3)  =  1.00
         g%s(43)%m(2,1)  =  1.00
         g%s(43)%m(3,2)  = -1.00
         g%s(43)%v(1)    =  0.50
         g%s(43)%v(2)    =  0.50
         g%s(43)%v(3)    =  0.50
       
         g%s(44)%m(1,3)  =  1.00
         g%s(44)%m(2,1)  = -1.00
         g%s(44)%m(3,2)  =  1.00
         g%s(44)%v(1)    =  0.50
         g%s(44)%v(2)    =  0.50
         g%s(44)%v(3)    =  0.50
       
         g%s(45)%m(1,2)  = -1.00
         g%s(45)%m(2,3)  =  1.00
         g%s(45)%m(3,1)  =  1.00
         g%s(45)%v(1)    =  0.50
         g%s(45)%v(2)    =  0.50
         g%s(45)%v(3)    =  0.50
       
         g%s(46)%m(1,1)  = -1.00
         g%s(46)%m(2,2)  =  1.00
         g%s(46)%m(3,3)  =  1.00
         g%s(46)%v(1)    =  0.50
         g%s(46)%v(2)    =  0.50
         g%s(46)%v(3)    =  0.50
       
         g%s(47)%m(1,1)  =  1.00
         g%s(47)%m(2,2)  = -1.00
         g%s(47)%m(3,3)  =  1.00
         g%s(47)%v(1)    =  0.50
         g%s(47)%v(2)    =  0.50
         g%s(47)%v(3)    =  0.50
       
         g%s(48)%m(1,1)  =  1.00
         g%s(48)%m(2,2)  =  1.00
         g%s(48)%m(3,3)  = -1.00
         g%s(48)%v(1)    =  0.50
         g%s(48)%v(2)    =  0.50
         g%s(48)%v(3)    =  0.50
       
       
      case('P a -3', '205')                          
       
         !Space group   # 205
         !Group setting # 232
       
         g%order   =  24
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,3)   =  1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,1)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,3)   = -1.00
         g%s(4)%m(3,1)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,3)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,2)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,3)   =  1.00
         g%s(6)%m(3,1)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,3)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,2)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,3)   = -1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   = -1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.50
         g%s(8)%v(3)     =  0.50
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   = -1.00
         g%s(9)%m(3,1)   = -1.00
         g%s(9)%v(1)     =  0.50
         g%s(9)%v(2)     =  0.50
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  = -1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,3)  =  1.00
         g%s(16)%m(3,1)  = -1.00
         g%s(16)%v(1)    = -0.50
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    = -0.50
       
         g%s(17)%m(1,3)  = -1.00
         g%s(17)%m(2,1)  =  1.00
         g%s(17)%m(3,2)  =  1.00
         g%s(17)%v(1)    = -0.50
         g%s(17)%v(2)    = -0.50
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,2)  =  1.00
         g%s(18)%m(2,3)  = -1.00
         g%s(18)%m(3,1)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    = -0.50
         g%s(18)%v(3)    = -0.50
       
         g%s(19)%m(1,3)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,2)  = -1.00
         g%s(19)%v(1)    = -0.50
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    = -0.50
       
         g%s(20)%m(1,3)  =  1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,2)  =  1.00
         g%s(20)%v(1)    =  0.00
         g%s(20)%v(2)    = -0.50
         g%s(20)%v(3)    = -0.50
       
         g%s(21)%m(1,2)  = -1.00
         g%s(21)%m(2,3)  =  1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    = -0.50
         g%s(21)%v(2)    = -0.50
         g%s(21)%v(3)    =  0.00
       
         g%s(22)%m(1,1)  = -1.00
         g%s(22)%m(2,2)  =  1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    = -0.50
         g%s(22)%v(2)    = -0.50
         g%s(22)%v(3)    =  0.00
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(2,2)  = -1.00
         g%s(23)%m(3,3)  =  1.00
         g%s(23)%v(1)    =  0.00
         g%s(23)%v(2)    = -0.50
         g%s(23)%v(3)    = -0.50
       
         g%s(24)%m(1,1)  =  1.00
         g%s(24)%m(2,2)  =  1.00
         g%s(24)%m(3,3)  = -1.00
         g%s(24)%v(1)    = -0.50
         g%s(24)%v(2)    =  0.00
         g%s(24)%v(3)    = -0.50
       
       
      case('I a -3', '206')                          
       
         !Space group   # 206
         !Group setting # 233
       
         g%order   =  48
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,3)   =  1.00
         g%s(2)%m(2,1)   =  1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,2)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,1)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,2)   = -1.00
         g%s(4)%m(2,3)   = -1.00
         g%s(4)%m(3,1)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   =  1.00
         g%s(5)%m(2,1)   = -1.00
         g%s(5)%m(3,2)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,3)   =  1.00
         g%s(6)%m(3,1)   = -1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,3)   = -1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,2)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   = -1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   = -1.00
         g%s(8)%v(1)     =  0.50
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   = -1.00
         g%s(9)%m(3,1)   = -1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.50
       
         g%s(10)%m(1,1)  =  1.00
         g%s(10)%m(2,2)  = -1.00
         g%s(10)%m(3,3)  = -1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,1)  = -1.00
         g%s(11)%m(2,2)  =  1.00
         g%s(11)%m(3,3)  = -1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,1)  = -1.00
         g%s(12)%m(2,2)  = -1.00
         g%s(12)%m(3,3)  =  1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,1)  = -1.00
         g%s(13)%m(2,2)  = -1.00
         g%s(13)%m(3,3)  = -1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  = -1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  = -1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,2)  =  1.00
         g%s(16)%m(2,3)  =  1.00
         g%s(16)%m(3,1)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    = -0.50
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,3)  = -1.00
         g%s(17)%m(2,1)  =  1.00
         g%s(17)%m(3,2)  =  1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    = -0.50
       
         g%s(18)%m(1,2)  =  1.00
         g%s(18)%m(2,3)  = -1.00
         g%s(18)%m(3,1)  =  1.00
         g%s(18)%v(1)    = -0.50
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,3)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,2)  = -1.00
         g%s(19)%v(1)    =  0.00
         g%s(19)%v(2)    = -0.50
         g%s(19)%v(3)    =  0.00
       
         g%s(20)%m(1,3)  =  1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,2)  =  1.00
         g%s(20)%v(1)    = -0.50
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    =  0.00
       
         g%s(21)%m(1,2)  = -1.00
         g%s(21)%m(2,3)  =  1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.00
         g%s(21)%v(2)    =  0.00
         g%s(21)%v(3)    = -0.50
       
         g%s(22)%m(1,1)  = -1.00
         g%s(22)%m(2,2)  =  1.00
         g%s(22)%m(3,3)  =  1.00
         g%s(22)%v(1)    =  0.00
         g%s(22)%v(2)    =  0.00
         g%s(22)%v(3)    = -0.50
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(2,2)  = -1.00
         g%s(23)%m(3,3)  =  1.00
         g%s(23)%v(1)    = -0.50
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    =  0.00
       
         g%s(24)%m(1,1)  =  1.00
         g%s(24)%m(2,2)  =  1.00
         g%s(24)%m(3,3)  = -1.00
         g%s(24)%v(1)    =  0.00
         g%s(24)%v(2)    = -0.50
         g%s(24)%v(3)    =  0.00
       
         g%s(25)%m(1,1)  =  1.00
         g%s(25)%m(2,2)  =  1.00
         g%s(25)%m(3,3)  =  1.00
         g%s(25)%v(1)    =  0.50
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  0.50
       
         g%s(26)%m(1,3)  =  1.00
         g%s(26)%m(2,1)  =  1.00
         g%s(26)%m(3,2)  =  1.00
         g%s(26)%v(1)    =  0.50
         g%s(26)%v(2)    =  0.50
         g%s(26)%v(3)    =  0.50
       
         g%s(27)%m(1,2)  =  1.00
         g%s(27)%m(2,3)  =  1.00
         g%s(27)%m(3,1)  =  1.00
         g%s(27)%v(1)    =  0.50
         g%s(27)%v(2)    =  0.50
         g%s(27)%v(3)    =  0.50
       
         g%s(28)%m(1,2)  = -1.00
         g%s(28)%m(2,3)  = -1.00
         g%s(28)%m(3,1)  =  1.00
         g%s(28)%v(1)    =  0.50
         g%s(28)%v(2)    =  1.00000000
         g%s(28)%v(3)    =  0.50
       
         g%s(29)%m(1,3)  =  1.00
         g%s(29)%m(2,1)  = -1.00
         g%s(29)%m(3,2)  = -1.00
         g%s(29)%v(1)    =  0.50
         g%s(29)%v(2)    =  0.50
         g%s(29)%v(3)    =  1.00000000
       
         g%s(30)%m(1,2)  = -1.00
         g%s(30)%m(2,3)  =  1.00
         g%s(30)%m(3,1)  = -1.00
         g%s(30)%v(1)    =  1.00000000
         g%s(30)%v(2)    =  0.50
         g%s(30)%v(3)    =  0.50
       
         g%s(31)%m(1,3)  = -1.00
         g%s(31)%m(2,1)  = -1.00
         g%s(31)%m(3,2)  =  1.00
         g%s(31)%v(1)    =  0.50
         g%s(31)%v(2)    =  1.00000000
         g%s(31)%v(3)    =  0.50
       
         g%s(32)%m(1,3)  = -1.00
         g%s(32)%m(2,1)  =  1.00
         g%s(32)%m(3,2)  = -1.00
         g%s(32)%v(1)    =  1.00000000
         g%s(32)%v(2)    =  0.50
         g%s(32)%v(3)    =  0.50
       
         g%s(33)%m(1,2)  =  1.00
         g%s(33)%m(2,3)  = -1.00
         g%s(33)%m(3,1)  = -1.00
         g%s(33)%v(1)    =  0.50
         g%s(33)%v(2)    =  0.50
         g%s(33)%v(3)    =  1.00000000
       
         g%s(34)%m(1,1)  =  1.00
         g%s(34)%m(2,2)  = -1.00
         g%s(34)%m(3,3)  = -1.00
         g%s(34)%v(1)    =  0.50
         g%s(34)%v(2)    =  0.50
         g%s(34)%v(3)    =  1.00000000
       
         g%s(35)%m(1,1)  = -1.00
         g%s(35)%m(2,2)  =  1.00
         g%s(35)%m(3,3)  = -1.00
         g%s(35)%v(1)    =  1.00000000
         g%s(35)%v(2)    =  0.50
         g%s(35)%v(3)    =  0.50
       
         g%s(36)%m(1,1)  = -1.00
         g%s(36)%m(2,2)  = -1.00
         g%s(36)%m(3,3)  =  1.00
         g%s(36)%v(1)    =  0.50
         g%s(36)%v(2)    =  1.00000000
         g%s(36)%v(3)    =  0.50
       
         g%s(37)%m(1,1)  = -1.00
         g%s(37)%m(2,2)  = -1.00
         g%s(37)%m(3,3)  = -1.00
         g%s(37)%v(1)    =  0.50
         g%s(37)%v(2)    =  0.50
         g%s(37)%v(3)    =  0.50
       
         g%s(38)%m(1,3)  = -1.00
         g%s(38)%m(2,1)  = -1.00
         g%s(38)%m(3,2)  = -1.00
         g%s(38)%v(1)    =  0.50
         g%s(38)%v(2)    =  0.50
         g%s(38)%v(3)    =  0.50
       
         g%s(39)%m(1,2)  = -1.00
         g%s(39)%m(2,3)  = -1.00
         g%s(39)%m(3,1)  = -1.00
         g%s(39)%v(1)    =  0.50
         g%s(39)%v(2)    =  0.50
         g%s(39)%v(3)    =  0.50
       
         g%s(40)%m(1,2)  =  1.00
         g%s(40)%m(2,3)  =  1.00
         g%s(40)%m(3,1)  = -1.00
         g%s(40)%v(1)    =  0.50
         g%s(40)%v(2)    =  0.00
         g%s(40)%v(3)    =  0.50
       
         g%s(41)%m(1,3)  = -1.00
         g%s(41)%m(2,1)  =  1.00
         g%s(41)%m(3,2)  =  1.00
         g%s(41)%v(1)    =  0.50
         g%s(41)%v(2)    =  0.50
         g%s(41)%v(3)    =  0.00
       
         g%s(42)%m(1,2)  =  1.00
         g%s(42)%m(2,3)  = -1.00
         g%s(42)%m(3,1)  =  1.00
         g%s(42)%v(1)    =  0.00
         g%s(42)%v(2)    =  0.50
         g%s(42)%v(3)    =  0.50
       
         g%s(43)%m(1,3)  =  1.00
         g%s(43)%m(2,1)  =  1.00
         g%s(43)%m(3,2)  = -1.00
         g%s(43)%v(1)    =  0.50
         g%s(43)%v(2)    =  0.00
         g%s(43)%v(3)    =  0.50
       
         g%s(44)%m(1,3)  =  1.00
         g%s(44)%m(2,1)  = -1.00
         g%s(44)%m(3,2)  =  1.00
         g%s(44)%v(1)    =  0.00
         g%s(44)%v(2)    =  0.50
         g%s(44)%v(3)    =  0.50
       
         g%s(45)%m(1,2)  = -1.00
         g%s(45)%m(2,3)  =  1.00
         g%s(45)%m(3,1)  =  1.00
         g%s(45)%v(1)    =  0.50
         g%s(45)%v(2)    =  0.50
         g%s(45)%v(3)    =  0.00
       
         g%s(46)%m(1,1)  = -1.00
         g%s(46)%m(2,2)  =  1.00
         g%s(46)%m(3,3)  =  1.00
         g%s(46)%v(1)    =  0.50
         g%s(46)%v(2)    =  0.50
         g%s(46)%v(3)    =  0.00
       
         g%s(47)%m(1,1)  =  1.00
         g%s(47)%m(2,2)  = -1.00
         g%s(47)%m(3,3)  =  1.00
         g%s(47)%v(1)    =  0.00
         g%s(47)%v(2)    =  0.50
         g%s(47)%v(3)    =  0.50
       
         g%s(48)%m(1,1)  =  1.00
         g%s(48)%m(2,2)  =  1.00
         g%s(48)%m(3,3)  = -1.00
         g%s(48)%v(1)    =  0.50
         g%s(48)%v(2)    =  0.00
         g%s(48)%v(3)    =  0.50
       
       
      case('P 4 3 2', '207')                         
       
         !Space group   # 207
         !Group setting # 234
       
         g%order   =  24
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,3)   = -1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,2)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,3)   =  1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%m(3,1)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,1)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   =  1.00
         g%s(9)%m(3,1)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,3)  = -1.00
         g%s(10)%m(3,1)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,2)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,1)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,3)  = -1.00
         g%s(13)%m(2,1)  = -1.00
         g%s(13)%m(3,2)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,2)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,3)  = -1.00
         g%s(19)%v(1)    =  0.00
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    =  0.00
       
         g%s(20)%m(1,2)  = -1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.00
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    =  0.00
       
         g%s(21)%m(1,3)  =  1.00
         g%s(21)%m(2,2)  = -1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.00
         g%s(21)%v(2)    =  0.00
         g%s(21)%v(3)    =  0.00
       
         g%s(22)%m(1,3)  = -1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,1)  = -1.00
         g%s(22)%v(1)    =  0.00
         g%s(22)%v(2)    =  0.00
         g%s(22)%v(3)    =  0.00
       
         g%s(23)%m(1,1)  = -1.00
         g%s(23)%m(2,3)  =  1.00
         g%s(23)%m(3,2)  =  1.00
         g%s(23)%v(1)    =  0.00
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    =  0.00
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,3)  = -1.00
         g%s(24)%m(3,2)  = -1.00
         g%s(24)%v(1)    =  0.00
         g%s(24)%v(2)    =  0.00
         g%s(24)%v(3)    =  0.00
       
       
      case('P 42 3 2', '208')                        
       
         !Space group   # 208
         !Group setting # 235
       
         g%order   =  24
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,3)   = -1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,2)   = -1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,3)   =  1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%m(3,1)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,3)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,1)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,3)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   =  1.00
         g%s(9)%m(3,1)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,3)  = -1.00
         g%s(10)%m(3,1)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,2)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,1)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,3)  = -1.00
         g%s(13)%m(2,1)  = -1.00
         g%s(13)%m(3,2)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,2)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,3)  = -1.00
         g%s(19)%v(1)    =  0.50
         g%s(19)%v(2)    =  0.50
         g%s(19)%v(3)    =  0.50
       
         g%s(20)%m(1,2)  = -1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.50
         g%s(20)%v(2)    =  0.50
         g%s(20)%v(3)    =  0.50
       
         g%s(21)%m(1,3)  =  1.00
         g%s(21)%m(2,2)  = -1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.50
         g%s(21)%v(2)    =  0.50
         g%s(21)%v(3)    =  0.50
       
         g%s(22)%m(1,3)  = -1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,1)  = -1.00
         g%s(22)%v(1)    =  0.50
         g%s(22)%v(2)    =  0.50
         g%s(22)%v(3)    =  0.50
       
         g%s(23)%m(1,1)  = -1.00
         g%s(23)%m(2,3)  =  1.00
         g%s(23)%m(3,2)  =  1.00
         g%s(23)%v(1)    =  0.50
         g%s(23)%v(2)    =  0.50
         g%s(23)%v(3)    =  0.50
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,3)  = -1.00
         g%s(24)%m(3,2)  = -1.00
         g%s(24)%v(1)    =  0.50
         g%s(24)%v(2)    =  0.50
         g%s(24)%v(3)    =  0.50
       
       
      case('F 4 3 2', '209')                         
       
         !Space group   # 209
         !Group setting # 236
       
         g%order   =  96
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,3)   = -1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,2)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,3)   =  1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%m(3,1)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,1)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   =  1.00
         g%s(9)%m(3,1)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,3)  = -1.00
         g%s(10)%m(3,1)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,2)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,1)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,3)  = -1.00
         g%s(13)%m(2,1)  = -1.00
         g%s(13)%m(3,2)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,2)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,3)  = -1.00
         g%s(19)%v(1)    =  0.00
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    =  0.00
       
         g%s(20)%m(1,2)  = -1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.00
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    =  0.00
       
         g%s(21)%m(1,3)  =  1.00
         g%s(21)%m(2,2)  = -1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.00
         g%s(21)%v(2)    =  0.00
         g%s(21)%v(3)    =  0.00
       
         g%s(22)%m(1,3)  = -1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,1)  = -1.00
         g%s(22)%v(1)    =  0.00
         g%s(22)%v(2)    =  0.00
         g%s(22)%v(3)    =  0.00
       
         g%s(23)%m(1,1)  = -1.00
         g%s(23)%m(2,3)  =  1.00
         g%s(23)%m(3,2)  =  1.00
         g%s(23)%v(1)    =  0.00
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    =  0.00
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,3)  = -1.00
         g%s(24)%m(3,2)  = -1.00
         g%s(24)%v(1)    =  0.00
         g%s(24)%v(2)    =  0.00
         g%s(24)%v(3)    =  0.00
       
         g%s(25)%m(1,1)  =  1.00
         g%s(25)%m(2,2)  =  1.00
         g%s(25)%m(3,3)  =  1.00
         g%s(25)%v(1)    =  0.00
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  0.50
       
         g%s(26)%m(1,1)  =  1.00
         g%s(26)%m(2,3)  = -1.00
         g%s(26)%m(3,2)  =  1.00
         g%s(26)%v(1)    =  0.00
         g%s(26)%v(2)    =  0.50
         g%s(26)%v(3)    =  0.50
       
         g%s(27)%m(1,1)  =  1.00
         g%s(27)%m(2,3)  =  1.00
         g%s(27)%m(3,2)  = -1.00
         g%s(27)%v(1)    =  0.00
         g%s(27)%v(2)    =  0.50
         g%s(27)%v(3)    =  0.50
       
         g%s(28)%m(1,3)  =  1.00
         g%s(28)%m(2,2)  =  1.00
         g%s(28)%m(3,1)  = -1.00
         g%s(28)%v(1)    =  0.00
         g%s(28)%v(2)    =  0.50
         g%s(28)%v(3)    =  0.50
       
         g%s(29)%m(1,3)  = -1.00
         g%s(29)%m(2,2)  =  1.00
         g%s(29)%m(3,1)  =  1.00
         g%s(29)%v(1)    =  0.00
         g%s(29)%v(2)    =  0.50
         g%s(29)%v(3)    =  0.50
       
         g%s(30)%m(1,2)  = -1.00
         g%s(30)%m(2,1)  =  1.00
         g%s(30)%m(3,3)  =  1.00
         g%s(30)%v(1)    =  0.00
         g%s(30)%v(2)    =  0.50
         g%s(30)%v(3)    =  0.50
       
         g%s(31)%m(1,2)  =  1.00
         g%s(31)%m(2,1)  = -1.00
         g%s(31)%m(3,3)  =  1.00
         g%s(31)%v(1)    =  0.00
         g%s(31)%v(2)    =  0.50
         g%s(31)%v(3)    =  0.50
       
         g%s(32)%m(1,3)  =  1.00
         g%s(32)%m(2,1)  =  1.00
         g%s(32)%m(3,2)  =  1.00
         g%s(32)%v(1)    =  0.00
         g%s(32)%v(2)    =  0.50
         g%s(32)%v(3)    =  0.50
       
         g%s(33)%m(1,2)  =  1.00
         g%s(33)%m(2,3)  =  1.00
         g%s(33)%m(3,1)  =  1.00
         g%s(33)%v(1)    =  0.00
         g%s(33)%v(2)    =  0.50
         g%s(33)%v(3)    =  0.50
       
         g%s(34)%m(1,2)  = -1.00
         g%s(34)%m(2,3)  = -1.00
         g%s(34)%m(3,1)  =  1.00
         g%s(34)%v(1)    =  0.00
         g%s(34)%v(2)    =  0.50
         g%s(34)%v(3)    =  0.50
       
         g%s(35)%m(1,3)  =  1.00
         g%s(35)%m(2,1)  = -1.00
         g%s(35)%m(3,2)  = -1.00
         g%s(35)%v(1)    =  0.00
         g%s(35)%v(2)    =  0.50
         g%s(35)%v(3)    =  0.50
       
         g%s(36)%m(1,2)  = -1.00
         g%s(36)%m(2,3)  =  1.00
         g%s(36)%m(3,1)  = -1.00
         g%s(36)%v(1)    =  0.00
         g%s(36)%v(2)    =  0.50
         g%s(36)%v(3)    =  0.50
       
         g%s(37)%m(1,3)  = -1.00
         g%s(37)%m(2,1)  = -1.00
         g%s(37)%m(3,2)  =  1.00
         g%s(37)%v(1)    =  0.00
         g%s(37)%v(2)    =  0.50
         g%s(37)%v(3)    =  0.50
       
         g%s(38)%m(1,3)  = -1.00
         g%s(38)%m(2,1)  =  1.00
         g%s(38)%m(3,2)  = -1.00
         g%s(38)%v(1)    =  0.00
         g%s(38)%v(2)    =  0.50
         g%s(38)%v(3)    =  0.50
       
         g%s(39)%m(1,2)  =  1.00
         g%s(39)%m(2,3)  = -1.00
         g%s(39)%m(3,1)  = -1.00
         g%s(39)%v(1)    =  0.00
         g%s(39)%v(2)    =  0.50
         g%s(39)%v(3)    =  0.50
       
         g%s(40)%m(1,1)  =  1.00
         g%s(40)%m(2,2)  = -1.00
         g%s(40)%m(3,3)  = -1.00
         g%s(40)%v(1)    =  0.00
         g%s(40)%v(2)    =  0.50
         g%s(40)%v(3)    =  0.50
       
         g%s(41)%m(1,1)  = -1.00
         g%s(41)%m(2,2)  =  1.00
         g%s(41)%m(3,3)  = -1.00
         g%s(41)%v(1)    =  0.00
         g%s(41)%v(2)    =  0.50
         g%s(41)%v(3)    =  0.50
       
         g%s(42)%m(1,1)  = -1.00
         g%s(42)%m(2,2)  = -1.00
         g%s(42)%m(3,3)  =  1.00
         g%s(42)%v(1)    =  0.00
         g%s(42)%v(2)    =  0.50
         g%s(42)%v(3)    =  0.50
       
         g%s(43)%m(1,2)  =  1.00
         g%s(43)%m(2,1)  =  1.00
         g%s(43)%m(3,3)  = -1.00
         g%s(43)%v(1)    =  0.00
         g%s(43)%v(2)    =  0.50
         g%s(43)%v(3)    =  0.50
       
         g%s(44)%m(1,2)  = -1.00
         g%s(44)%m(2,1)  = -1.00
         g%s(44)%m(3,3)  = -1.00
         g%s(44)%v(1)    =  0.00
         g%s(44)%v(2)    =  0.50
         g%s(44)%v(3)    =  0.50
       
         g%s(45)%m(1,3)  =  1.00
         g%s(45)%m(2,2)  = -1.00
         g%s(45)%m(3,1)  =  1.00
         g%s(45)%v(1)    =  0.00
         g%s(45)%v(2)    =  0.50
         g%s(45)%v(3)    =  0.50
       
         g%s(46)%m(1,3)  = -1.00
         g%s(46)%m(2,2)  = -1.00
         g%s(46)%m(3,1)  = -1.00
         g%s(46)%v(1)    =  0.00
         g%s(46)%v(2)    =  0.50
         g%s(46)%v(3)    =  0.50
       
         g%s(47)%m(1,1)  = -1.00
         g%s(47)%m(2,3)  =  1.00
         g%s(47)%m(3,2)  =  1.00
         g%s(47)%v(1)    =  0.00
         g%s(47)%v(2)    =  0.50
         g%s(47)%v(3)    =  0.50
       
         g%s(48)%m(1,1)  = -1.00
         g%s(48)%m(2,3)  = -1.00
         g%s(48)%m(3,2)  = -1.00
         g%s(48)%v(1)    =  0.00
         g%s(48)%v(2)    =  0.50
         g%s(48)%v(3)    =  0.50
       
         g%s(49)%m(1,1)  =  1.00
         g%s(49)%m(2,2)  =  1.00
         g%s(49)%m(3,3)  =  1.00
         g%s(49)%v(1)    =  0.50
         g%s(49)%v(2)    =  0.00
         g%s(49)%v(3)    =  0.50
       
         g%s(50)%m(1,1)  =  1.00
         g%s(50)%m(2,3)  = -1.00
         g%s(50)%m(3,2)  =  1.00
         g%s(50)%v(1)    =  0.50
         g%s(50)%v(2)    =  0.00
         g%s(50)%v(3)    =  0.50
       
         g%s(51)%m(1,1)  =  1.00
         g%s(51)%m(2,3)  =  1.00
         g%s(51)%m(3,2)  = -1.00
         g%s(51)%v(1)    =  0.50
         g%s(51)%v(2)    =  0.00
         g%s(51)%v(3)    =  0.50
       
         g%s(52)%m(1,3)  =  1.00
         g%s(52)%m(2,2)  =  1.00
         g%s(52)%m(3,1)  = -1.00
         g%s(52)%v(1)    =  0.50
         g%s(52)%v(2)    =  0.00
         g%s(52)%v(3)    =  0.50
       
         g%s(53)%m(1,3)  = -1.00
         g%s(53)%m(2,2)  =  1.00
         g%s(53)%m(3,1)  =  1.00
         g%s(53)%v(1)    =  0.50
         g%s(53)%v(2)    =  0.00
         g%s(53)%v(3)    =  0.50
       
         g%s(54)%m(1,2)  = -1.00
         g%s(54)%m(2,1)  =  1.00
         g%s(54)%m(3,3)  =  1.00
         g%s(54)%v(1)    =  0.50
         g%s(54)%v(2)    =  0.00
         g%s(54)%v(3)    =  0.50
       
         g%s(55)%m(1,2)  =  1.00
         g%s(55)%m(2,1)  = -1.00
         g%s(55)%m(3,3)  =  1.00
         g%s(55)%v(1)    =  0.50
         g%s(55)%v(2)    =  0.00
         g%s(55)%v(3)    =  0.50
       
         g%s(56)%m(1,3)  =  1.00
         g%s(56)%m(2,1)  =  1.00
         g%s(56)%m(3,2)  =  1.00
         g%s(56)%v(1)    =  0.50
         g%s(56)%v(2)    =  0.00
         g%s(56)%v(3)    =  0.50
       
         g%s(57)%m(1,2)  =  1.00
         g%s(57)%m(2,3)  =  1.00
         g%s(57)%m(3,1)  =  1.00
         g%s(57)%v(1)    =  0.50
         g%s(57)%v(2)    =  0.00
         g%s(57)%v(3)    =  0.50
       
         g%s(58)%m(1,2)  = -1.00
         g%s(58)%m(2,3)  = -1.00
         g%s(58)%m(3,1)  =  1.00
         g%s(58)%v(1)    =  0.50
         g%s(58)%v(2)    =  0.00
         g%s(58)%v(3)    =  0.50
       
         g%s(59)%m(1,3)  =  1.00
         g%s(59)%m(2,1)  = -1.00
         g%s(59)%m(3,2)  = -1.00
         g%s(59)%v(1)    =  0.50
         g%s(59)%v(2)    =  0.00
         g%s(59)%v(3)    =  0.50
       
         g%s(60)%m(1,2)  = -1.00
         g%s(60)%m(2,3)  =  1.00
         g%s(60)%m(3,1)  = -1.00
         g%s(60)%v(1)    =  0.50
         g%s(60)%v(2)    =  0.00
         g%s(60)%v(3)    =  0.50
       
         g%s(61)%m(1,3)  = -1.00
         g%s(61)%m(2,1)  = -1.00
         g%s(61)%m(3,2)  =  1.00
         g%s(61)%v(1)    =  0.50
         g%s(61)%v(2)    =  0.00
         g%s(61)%v(3)    =  0.50
       
         g%s(62)%m(1,3)  = -1.00
         g%s(62)%m(2,1)  =  1.00
         g%s(62)%m(3,2)  = -1.00
         g%s(62)%v(1)    =  0.50
         g%s(62)%v(2)    =  0.00
         g%s(62)%v(3)    =  0.50
       
         g%s(63)%m(1,2)  =  1.00
         g%s(63)%m(2,3)  = -1.00
         g%s(63)%m(3,1)  = -1.00
         g%s(63)%v(1)    =  0.50
         g%s(63)%v(2)    =  0.00
         g%s(63)%v(3)    =  0.50
       
         g%s(64)%m(1,1)  =  1.00
         g%s(64)%m(2,2)  = -1.00
         g%s(64)%m(3,3)  = -1.00
         g%s(64)%v(1)    =  0.50
         g%s(64)%v(2)    =  0.00
         g%s(64)%v(3)    =  0.50
       
         g%s(65)%m(1,1)  = -1.00
         g%s(65)%m(2,2)  =  1.00
         g%s(65)%m(3,3)  = -1.00
         g%s(65)%v(1)    =  0.50
         g%s(65)%v(2)    =  0.00
         g%s(65)%v(3)    =  0.50
       
         g%s(66)%m(1,1)  = -1.00
         g%s(66)%m(2,2)  = -1.00
         g%s(66)%m(3,3)  =  1.00
         g%s(66)%v(1)    =  0.50
         g%s(66)%v(2)    =  0.00
         g%s(66)%v(3)    =  0.50
       
         g%s(67)%m(1,2)  =  1.00
         g%s(67)%m(2,1)  =  1.00
         g%s(67)%m(3,3)  = -1.00
         g%s(67)%v(1)    =  0.50
         g%s(67)%v(2)    =  0.00
         g%s(67)%v(3)    =  0.50
       
         g%s(68)%m(1,2)  = -1.00
         g%s(68)%m(2,1)  = -1.00
         g%s(68)%m(3,3)  = -1.00
         g%s(68)%v(1)    =  0.50
         g%s(68)%v(2)    =  0.00
         g%s(68)%v(3)    =  0.50
       
         g%s(69)%m(1,3)  =  1.00
         g%s(69)%m(2,2)  = -1.00
         g%s(69)%m(3,1)  =  1.00
         g%s(69)%v(1)    =  0.50
         g%s(69)%v(2)    =  0.00
         g%s(69)%v(3)    =  0.50
       
         g%s(70)%m(1,3)  = -1.00
         g%s(70)%m(2,2)  = -1.00
         g%s(70)%m(3,1)  = -1.00
         g%s(70)%v(1)    =  0.50
         g%s(70)%v(2)    =  0.00
         g%s(70)%v(3)    =  0.50
       
         g%s(71)%m(1,1)  = -1.00
         g%s(71)%m(2,3)  =  1.00
         g%s(71)%m(3,2)  =  1.00
         g%s(71)%v(1)    =  0.50
         g%s(71)%v(2)    =  0.00
         g%s(71)%v(3)    =  0.50
       
         g%s(72)%m(1,1)  = -1.00
         g%s(72)%m(2,3)  = -1.00
         g%s(72)%m(3,2)  = -1.00
         g%s(72)%v(1)    =  0.50
         g%s(72)%v(2)    =  0.00
         g%s(72)%v(3)    =  0.50
       
         g%s(73)%m(1,1)  =  1.00
         g%s(73)%m(2,2)  =  1.00
         g%s(73)%m(3,3)  =  1.00
         g%s(73)%v(1)    =  0.50
         g%s(73)%v(2)    =  0.50
         g%s(73)%v(3)    =  0.00
       
         g%s(74)%m(1,1)  =  1.00
         g%s(74)%m(2,3)  = -1.00
         g%s(74)%m(3,2)  =  1.00
         g%s(74)%v(1)    =  0.50
         g%s(74)%v(2)    =  0.50
         g%s(74)%v(3)    =  0.00
       
         g%s(75)%m(1,1)  =  1.00
         g%s(75)%m(2,3)  =  1.00
         g%s(75)%m(3,2)  = -1.00
         g%s(75)%v(1)    =  0.50
         g%s(75)%v(2)    =  0.50
         g%s(75)%v(3)    =  0.00
       
         g%s(76)%m(1,3)  =  1.00
         g%s(76)%m(2,2)  =  1.00
         g%s(76)%m(3,1)  = -1.00
         g%s(76)%v(1)    =  0.50
         g%s(76)%v(2)    =  0.50
         g%s(76)%v(3)    =  0.00
       
         g%s(77)%m(1,3)  = -1.00
         g%s(77)%m(2,2)  =  1.00
         g%s(77)%m(3,1)  =  1.00
         g%s(77)%v(1)    =  0.50
         g%s(77)%v(2)    =  0.50
         g%s(77)%v(3)    =  0.00
       
         g%s(78)%m(1,2)  = -1.00
         g%s(78)%m(2,1)  =  1.00
         g%s(78)%m(3,3)  =  1.00
         g%s(78)%v(1)    =  0.50
         g%s(78)%v(2)    =  0.50
         g%s(78)%v(3)    =  0.00
       
         g%s(79)%m(1,2)  =  1.00
         g%s(79)%m(2,1)  = -1.00
         g%s(79)%m(3,3)  =  1.00
         g%s(79)%v(1)    =  0.50
         g%s(79)%v(2)    =  0.50
         g%s(79)%v(3)    =  0.00
       
         g%s(80)%m(1,3)  =  1.00
         g%s(80)%m(2,1)  =  1.00
         g%s(80)%m(3,2)  =  1.00
         g%s(80)%v(1)    =  0.50
         g%s(80)%v(2)    =  0.50
         g%s(80)%v(3)    =  0.00
       
         g%s(81)%m(1,2)  =  1.00
         g%s(81)%m(2,3)  =  1.00
         g%s(81)%m(3,1)  =  1.00
         g%s(81)%v(1)    =  0.50
         g%s(81)%v(2)    =  0.50
         g%s(81)%v(3)    =  0.00
       
         g%s(82)%m(1,2)  = -1.00
         g%s(82)%m(2,3)  = -1.00
         g%s(82)%m(3,1)  =  1.00
         g%s(82)%v(1)    =  0.50
         g%s(82)%v(2)    =  0.50
         g%s(82)%v(3)    =  0.00
       
         g%s(83)%m(1,3)  =  1.00
         g%s(83)%m(2,1)  = -1.00
         g%s(83)%m(3,2)  = -1.00
         g%s(83)%v(1)    =  0.50
         g%s(83)%v(2)    =  0.50
         g%s(83)%v(3)    =  0.00
       
         g%s(84)%m(1,2)  = -1.00
         g%s(84)%m(2,3)  =  1.00
         g%s(84)%m(3,1)  = -1.00
         g%s(84)%v(1)    =  0.50
         g%s(84)%v(2)    =  0.50
         g%s(84)%v(3)    =  0.00
       
         g%s(85)%m(1,3)  = -1.00
         g%s(85)%m(2,1)  = -1.00
         g%s(85)%m(3,2)  =  1.00
         g%s(85)%v(1)    =  0.50
         g%s(85)%v(2)    =  0.50
         g%s(85)%v(3)    =  0.00
       
         g%s(86)%m(1,3)  = -1.00
         g%s(86)%m(2,1)  =  1.00
         g%s(86)%m(3,2)  = -1.00
         g%s(86)%v(1)    =  0.50
         g%s(86)%v(2)    =  0.50
         g%s(86)%v(3)    =  0.00
       
         g%s(87)%m(1,2)  =  1.00
         g%s(87)%m(2,3)  = -1.00
         g%s(87)%m(3,1)  = -1.00
         g%s(87)%v(1)    =  0.50
         g%s(87)%v(2)    =  0.50
         g%s(87)%v(3)    =  0.00
       
         g%s(88)%m(1,1)  =  1.00
         g%s(88)%m(2,2)  = -1.00
         g%s(88)%m(3,3)  = -1.00
         g%s(88)%v(1)    =  0.50
         g%s(88)%v(2)    =  0.50
         g%s(88)%v(3)    =  0.00
       
         g%s(89)%m(1,1)  = -1.00
         g%s(89)%m(2,2)  =  1.00
         g%s(89)%m(3,3)  = -1.00
         g%s(89)%v(1)    =  0.50
         g%s(89)%v(2)    =  0.50
         g%s(89)%v(3)    =  0.00
       
         g%s(90)%m(1,1)  = -1.00
         g%s(90)%m(2,2)  = -1.00
         g%s(90)%m(3,3)  =  1.00
         g%s(90)%v(1)    =  0.50
         g%s(90)%v(2)    =  0.50
         g%s(90)%v(3)    =  0.00
       
         g%s(91)%m(1,2)  =  1.00
         g%s(91)%m(2,1)  =  1.00
         g%s(91)%m(3,3)  = -1.00
         g%s(91)%v(1)    =  0.50
         g%s(91)%v(2)    =  0.50
         g%s(91)%v(3)    =  0.00
       
         g%s(92)%m(1,2)  = -1.00
         g%s(92)%m(2,1)  = -1.00
         g%s(92)%m(3,3)  = -1.00
         g%s(92)%v(1)    =  0.50
         g%s(92)%v(2)    =  0.50
         g%s(92)%v(3)    =  0.00
       
         g%s(93)%m(1,3)  =  1.00
         g%s(93)%m(2,2)  = -1.00
         g%s(93)%m(3,1)  =  1.00
         g%s(93)%v(1)    =  0.50
         g%s(93)%v(2)    =  0.50
         g%s(93)%v(3)    =  0.00
       
         g%s(94)%m(1,3)  = -1.00
         g%s(94)%m(2,2)  = -1.00
         g%s(94)%m(3,1)  = -1.00
         g%s(94)%v(1)    =  0.50
         g%s(94)%v(2)    =  0.50
         g%s(94)%v(3)    =  0.00
       
         g%s(95)%m(1,1)  = -1.00
         g%s(95)%m(2,3)  =  1.00
         g%s(95)%m(3,2)  =  1.00
         g%s(95)%v(1)    =  0.50
         g%s(95)%v(2)    =  0.50
         g%s(95)%v(3)    =  0.00
       
         g%s(96)%m(1,1)  = -1.00
         g%s(96)%m(2,3)  = -1.00
         g%s(96)%m(3,2)  = -1.00
         g%s(96)%v(1)    =  0.50
         g%s(96)%v(2)    =  0.50
         g%s(96)%v(3)    =  0.00
       
       
      case('F 41 3 2', '210')                        
       
         !Space group   # 210
         !Group setting # 237
       
         g%order   =  96
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,3)   = -1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.25
         g%s(2)%v(2)     =  0.25
         g%s(2)%v(3)     =  0.25
       
         g%s(3)%m(1,1)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,2)   = -1.00
         g%s(3)%v(1)     =  0.25
         g%s(3)%v(2)     =  0.25
         g%s(3)%v(3)     =  0.25
       
         g%s(4)%m(1,3)   =  1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%m(3,1)   = -1.00
         g%s(4)%v(1)     =  0.25
         g%s(4)%v(2)     =  0.25
         g%s(4)%v(3)     =  0.25
       
         g%s(5)%m(1,3)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,1)   =  1.00
         g%s(5)%v(1)     =  0.25
         g%s(5)%v(2)     =  0.25
         g%s(5)%v(3)     =  0.25
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.25
         g%s(6)%v(2)     =  0.25
         g%s(6)%v(3)     =  0.25
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.25
         g%s(7)%v(2)     =  0.25
         g%s(7)%v(3)     =  0.25
       
         g%s(8)%m(1,3)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   =  1.00
         g%s(9)%m(3,1)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,3)  = -1.00
         g%s(10)%m(3,1)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,2)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,1)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,3)  = -1.00
         g%s(13)%m(2,1)  = -1.00
         g%s(13)%m(3,2)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,2)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,3)  = -1.00
         g%s(19)%v(1)    =  0.25
         g%s(19)%v(2)    =  0.25
         g%s(19)%v(3)    =  0.25
       
         g%s(20)%m(1,2)  = -1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.25
         g%s(20)%v(2)    =  0.25
         g%s(20)%v(3)    =  0.25
       
         g%s(21)%m(1,3)  =  1.00
         g%s(21)%m(2,2)  = -1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.25
         g%s(21)%v(2)    =  0.25
         g%s(21)%v(3)    =  0.25
       
         g%s(22)%m(1,3)  = -1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,1)  = -1.00
         g%s(22)%v(1)    =  0.25
         g%s(22)%v(2)    =  0.25
         g%s(22)%v(3)    =  0.25
       
         g%s(23)%m(1,1)  = -1.00
         g%s(23)%m(2,3)  =  1.00
         g%s(23)%m(3,2)  =  1.00
         g%s(23)%v(1)    =  0.25
         g%s(23)%v(2)    =  0.25
         g%s(23)%v(3)    =  0.25
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,3)  = -1.00
         g%s(24)%m(3,2)  = -1.00
         g%s(24)%v(1)    =  0.25
         g%s(24)%v(2)    =  0.25
         g%s(24)%v(3)    =  0.25
       
         g%s(25)%m(1,1)  =  1.00
         g%s(25)%m(2,2)  =  1.00
         g%s(25)%m(3,3)  =  1.00
         g%s(25)%v(1)    =  0.00
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  0.50
       
         g%s(26)%m(1,1)  =  1.00
         g%s(26)%m(2,3)  = -1.00
         g%s(26)%m(3,2)  =  1.00
         g%s(26)%v(1)    =  0.25
         g%s(26)%v(2)    =  0.75000000
         g%s(26)%v(3)    =  0.75000000
       
         g%s(27)%m(1,1)  =  1.00
         g%s(27)%m(2,3)  =  1.00
         g%s(27)%m(3,2)  = -1.00
         g%s(27)%v(1)    =  0.25
         g%s(27)%v(2)    =  0.75000000
         g%s(27)%v(3)    =  0.75000000
       
         g%s(28)%m(1,3)  =  1.00
         g%s(28)%m(2,2)  =  1.00
         g%s(28)%m(3,1)  = -1.00
         g%s(28)%v(1)    =  0.25
         g%s(28)%v(2)    =  0.75000000
         g%s(28)%v(3)    =  0.75000000
       
         g%s(29)%m(1,3)  = -1.00
         g%s(29)%m(2,2)  =  1.00
         g%s(29)%m(3,1)  =  1.00
         g%s(29)%v(1)    =  0.25
         g%s(29)%v(2)    =  0.75000000
         g%s(29)%v(3)    =  0.75000000
       
         g%s(30)%m(1,2)  = -1.00
         g%s(30)%m(2,1)  =  1.00
         g%s(30)%m(3,3)  =  1.00
         g%s(30)%v(1)    =  0.25
         g%s(30)%v(2)    =  0.75000000
         g%s(30)%v(3)    =  0.75000000
       
         g%s(31)%m(1,2)  =  1.00
         g%s(31)%m(2,1)  = -1.00
         g%s(31)%m(3,3)  =  1.00
         g%s(31)%v(1)    =  0.25
         g%s(31)%v(2)    =  0.75000000
         g%s(31)%v(3)    =  0.75000000
       
         g%s(32)%m(1,3)  =  1.00
         g%s(32)%m(2,1)  =  1.00
         g%s(32)%m(3,2)  =  1.00
         g%s(32)%v(1)    =  0.00
         g%s(32)%v(2)    =  0.50
         g%s(32)%v(3)    =  0.50
       
         g%s(33)%m(1,2)  =  1.00
         g%s(33)%m(2,3)  =  1.00
         g%s(33)%m(3,1)  =  1.00
         g%s(33)%v(1)    =  0.00
         g%s(33)%v(2)    =  0.50
         g%s(33)%v(3)    =  0.50
       
         g%s(34)%m(1,2)  = -1.00
         g%s(34)%m(2,3)  = -1.00
         g%s(34)%m(3,1)  =  1.00
         g%s(34)%v(1)    =  0.00
         g%s(34)%v(2)    =  0.50
         g%s(34)%v(3)    =  0.50
       
         g%s(35)%m(1,3)  =  1.00
         g%s(35)%m(2,1)  = -1.00
         g%s(35)%m(3,2)  = -1.00
         g%s(35)%v(1)    =  0.00
         g%s(35)%v(2)    =  0.50
         g%s(35)%v(3)    =  0.50
       
         g%s(36)%m(1,2)  = -1.00
         g%s(36)%m(2,3)  =  1.00
         g%s(36)%m(3,1)  = -1.00
         g%s(36)%v(1)    =  0.00
         g%s(36)%v(2)    =  0.50
         g%s(36)%v(3)    =  0.50
       
         g%s(37)%m(1,3)  = -1.00
         g%s(37)%m(2,1)  = -1.00
         g%s(37)%m(3,2)  =  1.00
         g%s(37)%v(1)    =  0.00
         g%s(37)%v(2)    =  0.50
         g%s(37)%v(3)    =  0.50
       
         g%s(38)%m(1,3)  = -1.00
         g%s(38)%m(2,1)  =  1.00
         g%s(38)%m(3,2)  = -1.00
         g%s(38)%v(1)    =  0.00
         g%s(38)%v(2)    =  0.50
         g%s(38)%v(3)    =  0.50
       
         g%s(39)%m(1,2)  =  1.00
         g%s(39)%m(2,3)  = -1.00
         g%s(39)%m(3,1)  = -1.00
         g%s(39)%v(1)    =  0.00
         g%s(39)%v(2)    =  0.50
         g%s(39)%v(3)    =  0.50
       
         g%s(40)%m(1,1)  =  1.00
         g%s(40)%m(2,2)  = -1.00
         g%s(40)%m(3,3)  = -1.00
         g%s(40)%v(1)    =  0.00
         g%s(40)%v(2)    =  0.50
         g%s(40)%v(3)    =  0.50
       
         g%s(41)%m(1,1)  = -1.00
         g%s(41)%m(2,2)  =  1.00
         g%s(41)%m(3,3)  = -1.00
         g%s(41)%v(1)    =  0.00
         g%s(41)%v(2)    =  0.50
         g%s(41)%v(3)    =  0.50
       
         g%s(42)%m(1,1)  = -1.00
         g%s(42)%m(2,2)  = -1.00
         g%s(42)%m(3,3)  =  1.00
         g%s(42)%v(1)    =  0.00
         g%s(42)%v(2)    =  0.50
         g%s(42)%v(3)    =  0.50
       
         g%s(43)%m(1,2)  =  1.00
         g%s(43)%m(2,1)  =  1.00
         g%s(43)%m(3,3)  = -1.00
         g%s(43)%v(1)    =  0.25
         g%s(43)%v(2)    =  0.75000000
         g%s(43)%v(3)    =  0.75000000
       
         g%s(44)%m(1,2)  = -1.00
         g%s(44)%m(2,1)  = -1.00
         g%s(44)%m(3,3)  = -1.00
         g%s(44)%v(1)    =  0.25
         g%s(44)%v(2)    =  0.75000000
         g%s(44)%v(3)    =  0.75000000
       
         g%s(45)%m(1,3)  =  1.00
         g%s(45)%m(2,2)  = -1.00
         g%s(45)%m(3,1)  =  1.00
         g%s(45)%v(1)    =  0.25
         g%s(45)%v(2)    =  0.75000000
         g%s(45)%v(3)    =  0.75000000
       
         g%s(46)%m(1,3)  = -1.00
         g%s(46)%m(2,2)  = -1.00
         g%s(46)%m(3,1)  = -1.00
         g%s(46)%v(1)    =  0.25
         g%s(46)%v(2)    =  0.75000000
         g%s(46)%v(3)    =  0.75000000
       
         g%s(47)%m(1,1)  = -1.00
         g%s(47)%m(2,3)  =  1.00
         g%s(47)%m(3,2)  =  1.00
         g%s(47)%v(1)    =  0.25
         g%s(47)%v(2)    =  0.75000000
         g%s(47)%v(3)    =  0.75000000
       
         g%s(48)%m(1,1)  = -1.00
         g%s(48)%m(2,3)  = -1.00
         g%s(48)%m(3,2)  = -1.00
         g%s(48)%v(1)    =  0.25
         g%s(48)%v(2)    =  0.75000000
         g%s(48)%v(3)    =  0.75000000
       
         g%s(49)%m(1,1)  =  1.00
         g%s(49)%m(2,2)  =  1.00
         g%s(49)%m(3,3)  =  1.00
         g%s(49)%v(1)    =  0.50
         g%s(49)%v(2)    =  0.00
         g%s(49)%v(3)    =  0.50
       
         g%s(50)%m(1,1)  =  1.00
         g%s(50)%m(2,3)  = -1.00
         g%s(50)%m(3,2)  =  1.00
         g%s(50)%v(1)    =  0.75000000
         g%s(50)%v(2)    =  0.25
         g%s(50)%v(3)    =  0.75000000
       
         g%s(51)%m(1,1)  =  1.00
         g%s(51)%m(2,3)  =  1.00
         g%s(51)%m(3,2)  = -1.00
         g%s(51)%v(1)    =  0.75000000
         g%s(51)%v(2)    =  0.25
         g%s(51)%v(3)    =  0.75000000
       
         g%s(52)%m(1,3)  =  1.00
         g%s(52)%m(2,2)  =  1.00
         g%s(52)%m(3,1)  = -1.00
         g%s(52)%v(1)    =  0.75000000
         g%s(52)%v(2)    =  0.25
         g%s(52)%v(3)    =  0.75000000
       
         g%s(53)%m(1,3)  = -1.00
         g%s(53)%m(2,2)  =  1.00
         g%s(53)%m(3,1)  =  1.00
         g%s(53)%v(1)    =  0.75000000
         g%s(53)%v(2)    =  0.25
         g%s(53)%v(3)    =  0.75000000
       
         g%s(54)%m(1,2)  = -1.00
         g%s(54)%m(2,1)  =  1.00
         g%s(54)%m(3,3)  =  1.00
         g%s(54)%v(1)    =  0.75000000
         g%s(54)%v(2)    =  0.25
         g%s(54)%v(3)    =  0.75000000
       
         g%s(55)%m(1,2)  =  1.00
         g%s(55)%m(2,1)  = -1.00
         g%s(55)%m(3,3)  =  1.00
         g%s(55)%v(1)    =  0.75000000
         g%s(55)%v(2)    =  0.25
         g%s(55)%v(3)    =  0.75000000
       
         g%s(56)%m(1,3)  =  1.00
         g%s(56)%m(2,1)  =  1.00
         g%s(56)%m(3,2)  =  1.00
         g%s(56)%v(1)    =  0.50
         g%s(56)%v(2)    =  0.00
         g%s(56)%v(3)    =  0.50
       
         g%s(57)%m(1,2)  =  1.00
         g%s(57)%m(2,3)  =  1.00
         g%s(57)%m(3,1)  =  1.00
         g%s(57)%v(1)    =  0.50
         g%s(57)%v(2)    =  0.00
         g%s(57)%v(3)    =  0.50
       
         g%s(58)%m(1,2)  = -1.00
         g%s(58)%m(2,3)  = -1.00
         g%s(58)%m(3,1)  =  1.00
         g%s(58)%v(1)    =  0.50
         g%s(58)%v(2)    =  0.00
         g%s(58)%v(3)    =  0.50
       
         g%s(59)%m(1,3)  =  1.00
         g%s(59)%m(2,1)  = -1.00
         g%s(59)%m(3,2)  = -1.00
         g%s(59)%v(1)    =  0.50
         g%s(59)%v(2)    =  0.00
         g%s(59)%v(3)    =  0.50
       
         g%s(60)%m(1,2)  = -1.00
         g%s(60)%m(2,3)  =  1.00
         g%s(60)%m(3,1)  = -1.00
         g%s(60)%v(1)    =  0.50
         g%s(60)%v(2)    =  0.00
         g%s(60)%v(3)    =  0.50
       
         g%s(61)%m(1,3)  = -1.00
         g%s(61)%m(2,1)  = -1.00
         g%s(61)%m(3,2)  =  1.00
         g%s(61)%v(1)    =  0.50
         g%s(61)%v(2)    =  0.00
         g%s(61)%v(3)    =  0.50
       
         g%s(62)%m(1,3)  = -1.00
         g%s(62)%m(2,1)  =  1.00
         g%s(62)%m(3,2)  = -1.00
         g%s(62)%v(1)    =  0.50
         g%s(62)%v(2)    =  0.00
         g%s(62)%v(3)    =  0.50
       
         g%s(63)%m(1,2)  =  1.00
         g%s(63)%m(2,3)  = -1.00
         g%s(63)%m(3,1)  = -1.00
         g%s(63)%v(1)    =  0.50
         g%s(63)%v(2)    =  0.00
         g%s(63)%v(3)    =  0.50
       
         g%s(64)%m(1,1)  =  1.00
         g%s(64)%m(2,2)  = -1.00
         g%s(64)%m(3,3)  = -1.00
         g%s(64)%v(1)    =  0.50
         g%s(64)%v(2)    =  0.00
         g%s(64)%v(3)    =  0.50
       
         g%s(65)%m(1,1)  = -1.00
         g%s(65)%m(2,2)  =  1.00
         g%s(65)%m(3,3)  = -1.00
         g%s(65)%v(1)    =  0.50
         g%s(65)%v(2)    =  0.00
         g%s(65)%v(3)    =  0.50
       
         g%s(66)%m(1,1)  = -1.00
         g%s(66)%m(2,2)  = -1.00
         g%s(66)%m(3,3)  =  1.00
         g%s(66)%v(1)    =  0.50
         g%s(66)%v(2)    =  0.00
         g%s(66)%v(3)    =  0.50
       
         g%s(67)%m(1,2)  =  1.00
         g%s(67)%m(2,1)  =  1.00
         g%s(67)%m(3,3)  = -1.00
         g%s(67)%v(1)    =  0.75000000
         g%s(67)%v(2)    =  0.25
         g%s(67)%v(3)    =  0.75000000
       
         g%s(68)%m(1,2)  = -1.00
         g%s(68)%m(2,1)  = -1.00
         g%s(68)%m(3,3)  = -1.00
         g%s(68)%v(1)    =  0.75000000
         g%s(68)%v(2)    =  0.25
         g%s(68)%v(3)    =  0.75000000
       
         g%s(69)%m(1,3)  =  1.00
         g%s(69)%m(2,2)  = -1.00
         g%s(69)%m(3,1)  =  1.00
         g%s(69)%v(1)    =  0.75000000
         g%s(69)%v(2)    =  0.25
         g%s(69)%v(3)    =  0.75000000
       
         g%s(70)%m(1,3)  = -1.00
         g%s(70)%m(2,2)  = -1.00
         g%s(70)%m(3,1)  = -1.00
         g%s(70)%v(1)    =  0.75000000
         g%s(70)%v(2)    =  0.25
         g%s(70)%v(3)    =  0.75000000
       
         g%s(71)%m(1,1)  = -1.00
         g%s(71)%m(2,3)  =  1.00
         g%s(71)%m(3,2)  =  1.00
         g%s(71)%v(1)    =  0.75000000
         g%s(71)%v(2)    =  0.25
         g%s(71)%v(3)    =  0.75000000
       
         g%s(72)%m(1,1)  = -1.00
         g%s(72)%m(2,3)  = -1.00
         g%s(72)%m(3,2)  = -1.00
         g%s(72)%v(1)    =  0.75000000
         g%s(72)%v(2)    =  0.25
         g%s(72)%v(3)    =  0.75000000
       
         g%s(73)%m(1,1)  =  1.00
         g%s(73)%m(2,2)  =  1.00
         g%s(73)%m(3,3)  =  1.00
         g%s(73)%v(1)    =  0.50
         g%s(73)%v(2)    =  0.50
         g%s(73)%v(3)    =  0.00
       
         g%s(74)%m(1,1)  =  1.00
         g%s(74)%m(2,3)  = -1.00
         g%s(74)%m(3,2)  =  1.00
         g%s(74)%v(1)    =  0.75000000
         g%s(74)%v(2)    =  0.75000000
         g%s(74)%v(3)    =  0.25
       
         g%s(75)%m(1,1)  =  1.00
         g%s(75)%m(2,3)  =  1.00
         g%s(75)%m(3,2)  = -1.00
         g%s(75)%v(1)    =  0.75000000
         g%s(75)%v(2)    =  0.75000000
         g%s(75)%v(3)    =  0.25
       
         g%s(76)%m(1,3)  =  1.00
         g%s(76)%m(2,2)  =  1.00
         g%s(76)%m(3,1)  = -1.00
         g%s(76)%v(1)    =  0.75000000
         g%s(76)%v(2)    =  0.75000000
         g%s(76)%v(3)    =  0.25
       
         g%s(77)%m(1,3)  = -1.00
         g%s(77)%m(2,2)  =  1.00
         g%s(77)%m(3,1)  =  1.00
         g%s(77)%v(1)    =  0.75000000
         g%s(77)%v(2)    =  0.75000000
         g%s(77)%v(3)    =  0.25
       
         g%s(78)%m(1,2)  = -1.00
         g%s(78)%m(2,1)  =  1.00
         g%s(78)%m(3,3)  =  1.00
         g%s(78)%v(1)    =  0.75000000
         g%s(78)%v(2)    =  0.75000000
         g%s(78)%v(3)    =  0.25
       
         g%s(79)%m(1,2)  =  1.00
         g%s(79)%m(2,1)  = -1.00
         g%s(79)%m(3,3)  =  1.00
         g%s(79)%v(1)    =  0.75000000
         g%s(79)%v(2)    =  0.75000000
         g%s(79)%v(3)    =  0.25
       
         g%s(80)%m(1,3)  =  1.00
         g%s(80)%m(2,1)  =  1.00
         g%s(80)%m(3,2)  =  1.00
         g%s(80)%v(1)    =  0.50
         g%s(80)%v(2)    =  0.50
         g%s(80)%v(3)    =  0.00
       
         g%s(81)%m(1,2)  =  1.00
         g%s(81)%m(2,3)  =  1.00
         g%s(81)%m(3,1)  =  1.00
         g%s(81)%v(1)    =  0.50
         g%s(81)%v(2)    =  0.50
         g%s(81)%v(3)    =  0.00
       
         g%s(82)%m(1,2)  = -1.00
         g%s(82)%m(2,3)  = -1.00
         g%s(82)%m(3,1)  =  1.00
         g%s(82)%v(1)    =  0.50
         g%s(82)%v(2)    =  0.50
         g%s(82)%v(3)    =  0.00
       
         g%s(83)%m(1,3)  =  1.00
         g%s(83)%m(2,1)  = -1.00
         g%s(83)%m(3,2)  = -1.00
         g%s(83)%v(1)    =  0.50
         g%s(83)%v(2)    =  0.50
         g%s(83)%v(3)    =  0.00
       
         g%s(84)%m(1,2)  = -1.00
         g%s(84)%m(2,3)  =  1.00
         g%s(84)%m(3,1)  = -1.00
         g%s(84)%v(1)    =  0.50
         g%s(84)%v(2)    =  0.50
         g%s(84)%v(3)    =  0.00
       
         g%s(85)%m(1,3)  = -1.00
         g%s(85)%m(2,1)  = -1.00
         g%s(85)%m(3,2)  =  1.00
         g%s(85)%v(1)    =  0.50
         g%s(85)%v(2)    =  0.50
         g%s(85)%v(3)    =  0.00
       
         g%s(86)%m(1,3)  = -1.00
         g%s(86)%m(2,1)  =  1.00
         g%s(86)%m(3,2)  = -1.00
         g%s(86)%v(1)    =  0.50
         g%s(86)%v(2)    =  0.50
         g%s(86)%v(3)    =  0.00
       
         g%s(87)%m(1,2)  =  1.00
         g%s(87)%m(2,3)  = -1.00
         g%s(87)%m(3,1)  = -1.00
         g%s(87)%v(1)    =  0.50
         g%s(87)%v(2)    =  0.50
         g%s(87)%v(3)    =  0.00
       
         g%s(88)%m(1,1)  =  1.00
         g%s(88)%m(2,2)  = -1.00
         g%s(88)%m(3,3)  = -1.00
         g%s(88)%v(1)    =  0.50
         g%s(88)%v(2)    =  0.50
         g%s(88)%v(3)    =  0.00
       
         g%s(89)%m(1,1)  = -1.00
         g%s(89)%m(2,2)  =  1.00
         g%s(89)%m(3,3)  = -1.00
         g%s(89)%v(1)    =  0.50
         g%s(89)%v(2)    =  0.50
         g%s(89)%v(3)    =  0.00
       
         g%s(90)%m(1,1)  = -1.00
         g%s(90)%m(2,2)  = -1.00
         g%s(90)%m(3,3)  =  1.00
         g%s(90)%v(1)    =  0.50
         g%s(90)%v(2)    =  0.50
         g%s(90)%v(3)    =  0.00
       
         g%s(91)%m(1,2)  =  1.00
         g%s(91)%m(2,1)  =  1.00
         g%s(91)%m(3,3)  = -1.00
         g%s(91)%v(1)    =  0.75000000
         g%s(91)%v(2)    =  0.75000000
         g%s(91)%v(3)    =  0.25
       
         g%s(92)%m(1,2)  = -1.00
         g%s(92)%m(2,1)  = -1.00
         g%s(92)%m(3,3)  = -1.00
         g%s(92)%v(1)    =  0.75000000
         g%s(92)%v(2)    =  0.75000000
         g%s(92)%v(3)    =  0.25
       
         g%s(93)%m(1,3)  =  1.00
         g%s(93)%m(2,2)  = -1.00
         g%s(93)%m(3,1)  =  1.00
         g%s(93)%v(1)    =  0.75000000
         g%s(93)%v(2)    =  0.75000000
         g%s(93)%v(3)    =  0.25
       
         g%s(94)%m(1,3)  = -1.00
         g%s(94)%m(2,2)  = -1.00
         g%s(94)%m(3,1)  = -1.00
         g%s(94)%v(1)    =  0.75000000
         g%s(94)%v(2)    =  0.75000000
         g%s(94)%v(3)    =  0.25
       
         g%s(95)%m(1,1)  = -1.00
         g%s(95)%m(2,3)  =  1.00
         g%s(95)%m(3,2)  =  1.00
         g%s(95)%v(1)    =  0.75000000
         g%s(95)%v(2)    =  0.75000000
         g%s(95)%v(3)    =  0.25
       
         g%s(96)%m(1,1)  = -1.00
         g%s(96)%m(2,3)  = -1.00
         g%s(96)%m(3,2)  = -1.00
         g%s(96)%v(1)    =  0.75000000
         g%s(96)%v(2)    =  0.75000000
         g%s(96)%v(3)    =  0.25
       
       
      case('I 4 3 2', '211')                         
       
         !Space group   # 211
         !Group setting # 238
       
         g%order   =  48
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,3)   = -1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,2)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,3)   =  1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%m(3,1)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,1)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   =  1.00
         g%s(9)%m(3,1)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,3)  = -1.00
         g%s(10)%m(3,1)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,2)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,1)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,3)  = -1.00
         g%s(13)%m(2,1)  = -1.00
         g%s(13)%m(3,2)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,2)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,3)  = -1.00
         g%s(19)%v(1)    =  0.00
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    =  0.00
       
         g%s(20)%m(1,2)  = -1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.00
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    =  0.00
       
         g%s(21)%m(1,3)  =  1.00
         g%s(21)%m(2,2)  = -1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.00
         g%s(21)%v(2)    =  0.00
         g%s(21)%v(3)    =  0.00
       
         g%s(22)%m(1,3)  = -1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,1)  = -1.00
         g%s(22)%v(1)    =  0.00
         g%s(22)%v(2)    =  0.00
         g%s(22)%v(3)    =  0.00
       
         g%s(23)%m(1,1)  = -1.00
         g%s(23)%m(2,3)  =  1.00
         g%s(23)%m(3,2)  =  1.00
         g%s(23)%v(1)    =  0.00
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    =  0.00
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,3)  = -1.00
         g%s(24)%m(3,2)  = -1.00
         g%s(24)%v(1)    =  0.00
         g%s(24)%v(2)    =  0.00
         g%s(24)%v(3)    =  0.00
       
         g%s(25)%m(1,1)  =  1.00
         g%s(25)%m(2,2)  =  1.00
         g%s(25)%m(3,3)  =  1.00
         g%s(25)%v(1)    =  0.50
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  0.50
       
         g%s(26)%m(1,1)  =  1.00
         g%s(26)%m(2,3)  = -1.00
         g%s(26)%m(3,2)  =  1.00
         g%s(26)%v(1)    =  0.50
         g%s(26)%v(2)    =  0.50
         g%s(26)%v(3)    =  0.50
       
         g%s(27)%m(1,1)  =  1.00
         g%s(27)%m(2,3)  =  1.00
         g%s(27)%m(3,2)  = -1.00
         g%s(27)%v(1)    =  0.50
         g%s(27)%v(2)    =  0.50
         g%s(27)%v(3)    =  0.50
       
         g%s(28)%m(1,3)  =  1.00
         g%s(28)%m(2,2)  =  1.00
         g%s(28)%m(3,1)  = -1.00
         g%s(28)%v(1)    =  0.50
         g%s(28)%v(2)    =  0.50
         g%s(28)%v(3)    =  0.50
       
         g%s(29)%m(1,3)  = -1.00
         g%s(29)%m(2,2)  =  1.00
         g%s(29)%m(3,1)  =  1.00
         g%s(29)%v(1)    =  0.50
         g%s(29)%v(2)    =  0.50
         g%s(29)%v(3)    =  0.50
       
         g%s(30)%m(1,2)  = -1.00
         g%s(30)%m(2,1)  =  1.00
         g%s(30)%m(3,3)  =  1.00
         g%s(30)%v(1)    =  0.50
         g%s(30)%v(2)    =  0.50
         g%s(30)%v(3)    =  0.50
       
         g%s(31)%m(1,2)  =  1.00
         g%s(31)%m(2,1)  = -1.00
         g%s(31)%m(3,3)  =  1.00
         g%s(31)%v(1)    =  0.50
         g%s(31)%v(2)    =  0.50
         g%s(31)%v(3)    =  0.50
       
         g%s(32)%m(1,3)  =  1.00
         g%s(32)%m(2,1)  =  1.00
         g%s(32)%m(3,2)  =  1.00
         g%s(32)%v(1)    =  0.50
         g%s(32)%v(2)    =  0.50
         g%s(32)%v(3)    =  0.50
       
         g%s(33)%m(1,2)  =  1.00
         g%s(33)%m(2,3)  =  1.00
         g%s(33)%m(3,1)  =  1.00
         g%s(33)%v(1)    =  0.50
         g%s(33)%v(2)    =  0.50
         g%s(33)%v(3)    =  0.50
       
         g%s(34)%m(1,2)  = -1.00
         g%s(34)%m(2,3)  = -1.00
         g%s(34)%m(3,1)  =  1.00
         g%s(34)%v(1)    =  0.50
         g%s(34)%v(2)    =  0.50
         g%s(34)%v(3)    =  0.50
       
         g%s(35)%m(1,3)  =  1.00
         g%s(35)%m(2,1)  = -1.00
         g%s(35)%m(3,2)  = -1.00
         g%s(35)%v(1)    =  0.50
         g%s(35)%v(2)    =  0.50
         g%s(35)%v(3)    =  0.50
       
         g%s(36)%m(1,2)  = -1.00
         g%s(36)%m(2,3)  =  1.00
         g%s(36)%m(3,1)  = -1.00
         g%s(36)%v(1)    =  0.50
         g%s(36)%v(2)    =  0.50
         g%s(36)%v(3)    =  0.50
       
         g%s(37)%m(1,3)  = -1.00
         g%s(37)%m(2,1)  = -1.00
         g%s(37)%m(3,2)  =  1.00
         g%s(37)%v(1)    =  0.50
         g%s(37)%v(2)    =  0.50
         g%s(37)%v(3)    =  0.50
       
         g%s(38)%m(1,3)  = -1.00
         g%s(38)%m(2,1)  =  1.00
         g%s(38)%m(3,2)  = -1.00
         g%s(38)%v(1)    =  0.50
         g%s(38)%v(2)    =  0.50
         g%s(38)%v(3)    =  0.50
       
         g%s(39)%m(1,2)  =  1.00
         g%s(39)%m(2,3)  = -1.00
         g%s(39)%m(3,1)  = -1.00
         g%s(39)%v(1)    =  0.50
         g%s(39)%v(2)    =  0.50
         g%s(39)%v(3)    =  0.50
       
         g%s(40)%m(1,1)  =  1.00
         g%s(40)%m(2,2)  = -1.00
         g%s(40)%m(3,3)  = -1.00
         g%s(40)%v(1)    =  0.50
         g%s(40)%v(2)    =  0.50
         g%s(40)%v(3)    =  0.50
       
         g%s(41)%m(1,1)  = -1.00
         g%s(41)%m(2,2)  =  1.00
         g%s(41)%m(3,3)  = -1.00
         g%s(41)%v(1)    =  0.50
         g%s(41)%v(2)    =  0.50
         g%s(41)%v(3)    =  0.50
       
         g%s(42)%m(1,1)  = -1.00
         g%s(42)%m(2,2)  = -1.00
         g%s(42)%m(3,3)  =  1.00
         g%s(42)%v(1)    =  0.50
         g%s(42)%v(2)    =  0.50
         g%s(42)%v(3)    =  0.50
       
         g%s(43)%m(1,2)  =  1.00
         g%s(43)%m(2,1)  =  1.00
         g%s(43)%m(3,3)  = -1.00
         g%s(43)%v(1)    =  0.50
         g%s(43)%v(2)    =  0.50
         g%s(43)%v(3)    =  0.50
       
         g%s(44)%m(1,2)  = -1.00
         g%s(44)%m(2,1)  = -1.00
         g%s(44)%m(3,3)  = -1.00
         g%s(44)%v(1)    =  0.50
         g%s(44)%v(2)    =  0.50
         g%s(44)%v(3)    =  0.50
       
         g%s(45)%m(1,3)  =  1.00
         g%s(45)%m(2,2)  = -1.00
         g%s(45)%m(3,1)  =  1.00
         g%s(45)%v(1)    =  0.50
         g%s(45)%v(2)    =  0.50
         g%s(45)%v(3)    =  0.50
       
         g%s(46)%m(1,3)  = -1.00
         g%s(46)%m(2,2)  = -1.00
         g%s(46)%m(3,1)  = -1.00
         g%s(46)%v(1)    =  0.50
         g%s(46)%v(2)    =  0.50
         g%s(46)%v(3)    =  0.50
       
         g%s(47)%m(1,1)  = -1.00
         g%s(47)%m(2,3)  =  1.00
         g%s(47)%m(3,2)  =  1.00
         g%s(47)%v(1)    =  0.50
         g%s(47)%v(2)    =  0.50
         g%s(47)%v(3)    =  0.50
       
         g%s(48)%m(1,1)  = -1.00
         g%s(48)%m(2,3)  = -1.00
         g%s(48)%m(3,2)  = -1.00
         g%s(48)%v(1)    =  0.50
         g%s(48)%v(2)    =  0.50
         g%s(48)%v(3)    =  0.50
       
       
      case('P 43 3 2', '212')                        
       
         !Space group   # 212
         !Group setting # 239
       
         g%order   =  24
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,3)   = -1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.75000000
         g%s(2)%v(2)     =  0.75000000
         g%s(2)%v(3)     =  0.25
       
         g%s(3)%m(1,1)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,2)   = -1.00
         g%s(3)%v(1)     =  0.25
         g%s(3)%v(2)     =  0.75000000
         g%s(3)%v(3)     =  0.75000000
       
         g%s(4)%m(1,3)   =  1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%m(3,1)   = -1.00
         g%s(4)%v(1)     =  0.25
         g%s(4)%v(2)     =  0.75000000
         g%s(4)%v(3)     =  0.75000000
       
         g%s(5)%m(1,3)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,1)   =  1.00
         g%s(5)%v(1)     =  0.75000000
         g%s(5)%v(2)     =  0.25
         g%s(5)%v(3)     =  0.75000000
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.75000000
         g%s(6)%v(2)     =  0.25
         g%s(6)%v(3)     =  0.75000000
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.75000000
         g%s(7)%v(2)     =  0.75000000
         g%s(7)%v(3)     =  0.25
       
         g%s(8)%m(1,3)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   =  1.00
         g%s(9)%m(3,1)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,3)  = -1.00
         g%s(10)%m(3,1)  =  1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,2)  = -1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,1)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,3)  = -1.00
         g%s(13)%m(2,1)  = -1.00
         g%s(13)%m(3,2)  =  1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.50
         g%s(17)%v(3)    =  0.50
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.50
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.50
       
         g%s(19)%m(1,2)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,3)  = -1.00
         g%s(19)%v(1)    =  0.25
         g%s(19)%v(2)    =  0.75000000
         g%s(19)%v(3)    =  0.75000000
       
         g%s(20)%m(1,2)  = -1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.25
         g%s(20)%v(2)    =  0.25
         g%s(20)%v(3)    =  0.25
       
         g%s(21)%m(1,3)  =  1.00
         g%s(21)%m(2,2)  = -1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.75000000
         g%s(21)%v(2)    =  0.75000000
         g%s(21)%v(3)    =  0.25
       
         g%s(22)%m(1,3)  = -1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,1)  = -1.00
         g%s(22)%v(1)    =  0.25
         g%s(22)%v(2)    =  0.25
         g%s(22)%v(3)    =  0.25
       
         g%s(23)%m(1,1)  = -1.00
         g%s(23)%m(2,3)  =  1.00
         g%s(23)%m(3,2)  =  1.00
         g%s(23)%v(1)    =  0.75000000
         g%s(23)%v(2)    =  0.25
         g%s(23)%v(3)    =  0.75000000
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,3)  = -1.00
         g%s(24)%m(3,2)  = -1.00
         g%s(24)%v(1)    =  0.25
         g%s(24)%v(2)    =  0.25
         g%s(24)%v(3)    =  0.25
       
       
      case('P 41 3 2', '213')                        
       
         !Space group   # 213
         !Group setting # 240
       
         g%order   =  24
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,3)   = -1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.25
         g%s(2)%v(2)     =  0.25
         g%s(2)%v(3)     =  0.75000000
       
         g%s(3)%m(1,1)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,2)   = -1.00
         g%s(3)%v(1)     =  0.75000000
         g%s(3)%v(2)     =  0.25
         g%s(3)%v(3)     =  0.25
       
         g%s(4)%m(1,3)   =  1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%m(3,1)   = -1.00
         g%s(4)%v(1)     =  0.75000000
         g%s(4)%v(2)     =  0.25
         g%s(4)%v(3)     =  0.25
       
         g%s(5)%m(1,3)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,1)   =  1.00
         g%s(5)%v(1)     =  0.25
         g%s(5)%v(2)     =  0.75000000
         g%s(5)%v(3)     =  0.25
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.25
         g%s(6)%v(2)     =  0.75000000
         g%s(6)%v(3)     =  0.25
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.25
         g%s(7)%v(2)     =  0.25
         g%s(7)%v(3)     =  0.75000000
       
         g%s(8)%m(1,3)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   =  1.00
         g%s(9)%m(3,1)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,3)  = -1.00
         g%s(10)%m(3,1)  =  1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.50
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,2)  = -1.00
         g%s(11)%v(1)    =  0.50
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,1)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.50
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,3)  = -1.00
         g%s(13)%m(2,1)  = -1.00
         g%s(13)%m(3,2)  =  1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.50
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.50
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.50
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.50
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.50
         g%s(17)%v(3)    =  0.50
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.50
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.50
       
         g%s(19)%m(1,2)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,3)  = -1.00
         g%s(19)%v(1)    =  0.75000000
         g%s(19)%v(2)    =  0.25
         g%s(19)%v(3)    =  0.25
       
         g%s(20)%m(1,2)  = -1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.75000000
         g%s(20)%v(2)    =  0.75000000
         g%s(20)%v(3)    =  0.75000000
       
         g%s(21)%m(1,3)  =  1.00
         g%s(21)%m(2,2)  = -1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.25
         g%s(21)%v(2)    =  0.25
         g%s(21)%v(3)    =  0.75000000
       
         g%s(22)%m(1,3)  = -1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,1)  = -1.00
         g%s(22)%v(1)    =  0.75000000
         g%s(22)%v(2)    =  0.75000000
         g%s(22)%v(3)    =  0.75000000
       
         g%s(23)%m(1,1)  = -1.00
         g%s(23)%m(2,3)  =  1.00
         g%s(23)%m(3,2)  =  1.00
         g%s(23)%v(1)    =  0.25
         g%s(23)%v(2)    =  0.75000000
         g%s(23)%v(3)    =  0.25
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,3)  = -1.00
         g%s(24)%m(3,2)  = -1.00
         g%s(24)%v(1)    =  0.75000000
         g%s(24)%v(2)    =  0.75000000
         g%s(24)%v(3)    =  0.75000000
       
       
      case('I 41 3 2', '214')                        
       
         !Space group   # 214
         !Group setting # 241
       
         g%order   =  48
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,3)   = -1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.25
         g%s(2)%v(2)     =  0.25
         g%s(2)%v(3)     =  0.75000000
       
         g%s(3)%m(1,1)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,2)   = -1.00
         g%s(3)%v(1)     =  0.25
         g%s(3)%v(2)     =  0.75000000
         g%s(3)%v(3)     =  0.75000000
       
         g%s(4)%m(1,3)   =  1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%m(3,1)   = -1.00
         g%s(4)%v(1)     =  0.25
         g%s(4)%v(2)     =  0.75000000
         g%s(4)%v(3)     =  0.75000000
       
         g%s(5)%m(1,3)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,1)   =  1.00
         g%s(5)%v(1)     =  0.25
         g%s(5)%v(2)     =  0.75000000
         g%s(5)%v(3)     =  0.25
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.25
         g%s(6)%v(2)     =  0.75000000
         g%s(6)%v(3)     =  0.25
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.25
         g%s(7)%v(2)     =  0.25
         g%s(7)%v(3)     =  0.75000000
       
         g%s(8)%m(1,3)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   =  1.00
         g%s(9)%m(3,1)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,3)  = -1.00
         g%s(10)%m(3,1)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,2)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,1)  = -1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,3)  = -1.00
         g%s(13)%m(2,1)  = -1.00
         g%s(13)%m(3,2)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.50
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.50
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.50
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,2)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,3)  = -1.00
         g%s(19)%v(1)    =  0.25
         g%s(19)%v(2)    =  0.75000000
         g%s(19)%v(3)    =  0.75000000
       
         g%s(20)%m(1,2)  = -1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.25
         g%s(20)%v(2)    =  0.25
         g%s(20)%v(3)    =  0.25
       
         g%s(21)%m(1,3)  =  1.00
         g%s(21)%m(2,2)  = -1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.25
         g%s(21)%v(2)    =  0.25
         g%s(21)%v(3)    =  0.75000000
       
         g%s(22)%m(1,3)  = -1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,1)  = -1.00
         g%s(22)%v(1)    =  0.25
         g%s(22)%v(2)    =  0.25
         g%s(22)%v(3)    =  0.25
       
         g%s(23)%m(1,1)  = -1.00
         g%s(23)%m(2,3)  =  1.00
         g%s(23)%m(3,2)  =  1.00
         g%s(23)%v(1)    =  0.25
         g%s(23)%v(2)    =  0.75000000
         g%s(23)%v(3)    =  0.25
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,3)  = -1.00
         g%s(24)%m(3,2)  = -1.00
         g%s(24)%v(1)    =  0.25
         g%s(24)%v(2)    =  0.25
         g%s(24)%v(3)    =  0.25
       
         g%s(25)%m(1,1)  =  1.00
         g%s(25)%m(2,2)  =  1.00
         g%s(25)%m(3,3)  =  1.00
         g%s(25)%v(1)    =  0.50
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  0.50
       
         g%s(26)%m(1,1)  =  1.00
         g%s(26)%m(2,3)  = -1.00
         g%s(26)%m(3,2)  =  1.00
         g%s(26)%v(1)    =  0.75000000
         g%s(26)%v(2)    =  0.75000000
         g%s(26)%v(3)    =  1.25000000
       
         g%s(27)%m(1,1)  =  1.00
         g%s(27)%m(2,3)  =  1.00
         g%s(27)%m(3,2)  = -1.00
         g%s(27)%v(1)    =  0.75000000
         g%s(27)%v(2)    =  1.25000000
         g%s(27)%v(3)    =  1.25000000
       
         g%s(28)%m(1,3)  =  1.00
         g%s(28)%m(2,2)  =  1.00
         g%s(28)%m(3,1)  = -1.00
         g%s(28)%v(1)    =  0.75000000
         g%s(28)%v(2)    =  1.25000000
         g%s(28)%v(3)    =  1.25000000
       
         g%s(29)%m(1,3)  = -1.00
         g%s(29)%m(2,2)  =  1.00
         g%s(29)%m(3,1)  =  1.00
         g%s(29)%v(1)    =  0.75000000
         g%s(29)%v(2)    =  1.25000000
         g%s(29)%v(3)    =  0.75000000
       
         g%s(30)%m(1,2)  = -1.00
         g%s(30)%m(2,1)  =  1.00
         g%s(30)%m(3,3)  =  1.00
         g%s(30)%v(1)    =  0.75000000
         g%s(30)%v(2)    =  1.25000000
         g%s(30)%v(3)    =  0.75000000
       
         g%s(31)%m(1,2)  =  1.00
         g%s(31)%m(2,1)  = -1.00
         g%s(31)%m(3,3)  =  1.00
         g%s(31)%v(1)    =  0.75000000
         g%s(31)%v(2)    =  0.75000000
         g%s(31)%v(3)    =  1.25000000
       
         g%s(32)%m(1,3)  =  1.00
         g%s(32)%m(2,1)  =  1.00
         g%s(32)%m(3,2)  =  1.00
         g%s(32)%v(1)    =  0.50
         g%s(32)%v(2)    =  0.50
         g%s(32)%v(3)    =  0.50
       
         g%s(33)%m(1,2)  =  1.00
         g%s(33)%m(2,3)  =  1.00
         g%s(33)%m(3,1)  =  1.00
         g%s(33)%v(1)    =  0.50
         g%s(33)%v(2)    =  0.50
         g%s(33)%v(3)    =  0.50
       
         g%s(34)%m(1,2)  = -1.00
         g%s(34)%m(2,3)  = -1.00
         g%s(34)%m(3,1)  =  1.00
         g%s(34)%v(1)    =  0.50
         g%s(34)%v(2)    =  1.00000000
         g%s(34)%v(3)    =  0.50
       
         g%s(35)%m(1,3)  =  1.00
         g%s(35)%m(2,1)  = -1.00
         g%s(35)%m(3,2)  = -1.00
         g%s(35)%v(1)    =  0.50
         g%s(35)%v(2)    =  0.50
         g%s(35)%v(3)    =  1.00000000
       
         g%s(36)%m(1,2)  = -1.00
         g%s(36)%m(2,3)  =  1.00
         g%s(36)%m(3,1)  = -1.00
         g%s(36)%v(1)    =  1.00000000
         g%s(36)%v(2)    =  0.50
         g%s(36)%v(3)    =  0.50
       
         g%s(37)%m(1,3)  = -1.00
         g%s(37)%m(2,1)  = -1.00
         g%s(37)%m(3,2)  =  1.00
         g%s(37)%v(1)    =  0.50
         g%s(37)%v(2)    =  1.00000000
         g%s(37)%v(3)    =  0.50
       
         g%s(38)%m(1,3)  = -1.00
         g%s(38)%m(2,1)  =  1.00
         g%s(38)%m(3,2)  = -1.00
         g%s(38)%v(1)    =  1.00000000
         g%s(38)%v(2)    =  0.50
         g%s(38)%v(3)    =  0.50
       
         g%s(39)%m(1,2)  =  1.00
         g%s(39)%m(2,3)  = -1.00
         g%s(39)%m(3,1)  = -1.00
         g%s(39)%v(1)    =  0.50
         g%s(39)%v(2)    =  0.50
         g%s(39)%v(3)    =  1.00000000
       
         g%s(40)%m(1,1)  =  1.00
         g%s(40)%m(2,2)  = -1.00
         g%s(40)%m(3,3)  = -1.00
         g%s(40)%v(1)    =  0.50
         g%s(40)%v(2)    =  0.50
         g%s(40)%v(3)    =  1.00000000
       
         g%s(41)%m(1,1)  = -1.00
         g%s(41)%m(2,2)  =  1.00
         g%s(41)%m(3,3)  = -1.00
         g%s(41)%v(1)    =  1.00000000
         g%s(41)%v(2)    =  0.50
         g%s(41)%v(3)    =  0.50
       
         g%s(42)%m(1,1)  = -1.00
         g%s(42)%m(2,2)  = -1.00
         g%s(42)%m(3,3)  =  1.00
         g%s(42)%v(1)    =  0.50
         g%s(42)%v(2)    =  1.00000000
         g%s(42)%v(3)    =  0.50
       
         g%s(43)%m(1,2)  =  1.00
         g%s(43)%m(2,1)  =  1.00
         g%s(43)%m(3,3)  = -1.00
         g%s(43)%v(1)    =  0.75000000
         g%s(43)%v(2)    =  1.25000000
         g%s(43)%v(3)    =  1.25000000
       
         g%s(44)%m(1,2)  = -1.00
         g%s(44)%m(2,1)  = -1.00
         g%s(44)%m(3,3)  = -1.00
         g%s(44)%v(1)    =  0.75000000
         g%s(44)%v(2)    =  0.75000000
         g%s(44)%v(3)    =  0.75000000
       
         g%s(45)%m(1,3)  =  1.00
         g%s(45)%m(2,2)  = -1.00
         g%s(45)%m(3,1)  =  1.00
         g%s(45)%v(1)    =  0.75000000
         g%s(45)%v(2)    =  0.75000000
         g%s(45)%v(3)    =  1.25000000
       
         g%s(46)%m(1,3)  = -1.00
         g%s(46)%m(2,2)  = -1.00
         g%s(46)%m(3,1)  = -1.00
         g%s(46)%v(1)    =  0.75000000
         g%s(46)%v(2)    =  0.75000000
         g%s(46)%v(3)    =  0.75000000
       
         g%s(47)%m(1,1)  = -1.00
         g%s(47)%m(2,3)  =  1.00
         g%s(47)%m(3,2)  =  1.00
         g%s(47)%v(1)    =  0.75000000
         g%s(47)%v(2)    =  1.25000000
         g%s(47)%v(3)    =  0.75000000
       
         g%s(48)%m(1,1)  = -1.00
         g%s(48)%m(2,3)  = -1.00
         g%s(48)%m(3,2)  = -1.00
         g%s(48)%v(1)    =  0.75000000
         g%s(48)%v(2)    =  0.75000000
         g%s(48)%v(3)    =  0.75000000
       
       
      case('P -4 3 m', '215')                        
       
         !Space group   # 215
         !Group setting # 242
       
         g%order   =  24
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,3)   =  1.00
         g%s(2)%m(3,2)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,3)   = -1.00
         g%s(3)%m(3,2)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,3)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,1)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   =  1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,1)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   =  1.00
         g%s(9)%m(3,1)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,3)  = -1.00
         g%s(10)%m(3,1)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,2)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,1)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,3)  = -1.00
         g%s(13)%m(2,1)  = -1.00
         g%s(13)%m(3,2)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,2)  = -1.00
         g%s(19)%m(2,1)  = -1.00
         g%s(19)%m(3,3)  =  1.00
         g%s(19)%v(1)    =  0.00
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    =  0.00
       
         g%s(20)%m(1,2)  =  1.00
         g%s(20)%m(2,1)  =  1.00
         g%s(20)%m(3,3)  =  1.00
         g%s(20)%v(1)    =  0.00
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    =  0.00
       
         g%s(21)%m(1,3)  = -1.00
         g%s(21)%m(2,2)  =  1.00
         g%s(21)%m(3,1)  = -1.00
         g%s(21)%v(1)    =  0.00
         g%s(21)%v(2)    =  0.00
         g%s(21)%v(3)    =  0.00
       
         g%s(22)%m(1,3)  =  1.00
         g%s(22)%m(2,2)  =  1.00
         g%s(22)%m(3,1)  =  1.00
         g%s(22)%v(1)    =  0.00
         g%s(22)%v(2)    =  0.00
         g%s(22)%v(3)    =  0.00
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(2,3)  = -1.00
         g%s(23)%m(3,2)  = -1.00
         g%s(23)%v(1)    =  0.00
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    =  0.00
       
         g%s(24)%m(1,1)  =  1.00
         g%s(24)%m(2,3)  =  1.00
         g%s(24)%m(3,2)  =  1.00
         g%s(24)%v(1)    =  0.00
         g%s(24)%v(2)    =  0.00
         g%s(24)%v(3)    =  0.00
       
       
      case('F -4 3 m', '216')                        
       
         !Space group   # 216
         !Group setting # 243
       
         g%order   =  96
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,3)   =  1.00
         g%s(2)%m(3,2)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,3)   = -1.00
         g%s(3)%m(3,2)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,3)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,1)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   =  1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,1)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   =  1.00
         g%s(9)%m(3,1)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,3)  = -1.00
         g%s(10)%m(3,1)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,2)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,1)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,3)  = -1.00
         g%s(13)%m(2,1)  = -1.00
         g%s(13)%m(3,2)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,2)  = -1.00
         g%s(19)%m(2,1)  = -1.00
         g%s(19)%m(3,3)  =  1.00
         g%s(19)%v(1)    =  0.00
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    =  0.00
       
         g%s(20)%m(1,2)  =  1.00
         g%s(20)%m(2,1)  =  1.00
         g%s(20)%m(3,3)  =  1.00
         g%s(20)%v(1)    =  0.00
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    =  0.00
       
         g%s(21)%m(1,3)  = -1.00
         g%s(21)%m(2,2)  =  1.00
         g%s(21)%m(3,1)  = -1.00
         g%s(21)%v(1)    =  0.00
         g%s(21)%v(2)    =  0.00
         g%s(21)%v(3)    =  0.00
       
         g%s(22)%m(1,3)  =  1.00
         g%s(22)%m(2,2)  =  1.00
         g%s(22)%m(3,1)  =  1.00
         g%s(22)%v(1)    =  0.00
         g%s(22)%v(2)    =  0.00
         g%s(22)%v(3)    =  0.00
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(2,3)  = -1.00
         g%s(23)%m(3,2)  = -1.00
         g%s(23)%v(1)    =  0.00
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    =  0.00
       
         g%s(24)%m(1,1)  =  1.00
         g%s(24)%m(2,3)  =  1.00
         g%s(24)%m(3,2)  =  1.00
         g%s(24)%v(1)    =  0.00
         g%s(24)%v(2)    =  0.00
         g%s(24)%v(3)    =  0.00
       
         g%s(25)%m(1,1)  =  1.00
         g%s(25)%m(2,2)  =  1.00
         g%s(25)%m(3,3)  =  1.00
         g%s(25)%v(1)    =  0.00
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  0.50
       
         g%s(26)%m(1,1)  = -1.00
         g%s(26)%m(2,3)  =  1.00
         g%s(26)%m(3,2)  = -1.00
         g%s(26)%v(1)    =  0.00
         g%s(26)%v(2)    =  0.50
         g%s(26)%v(3)    =  0.50
       
         g%s(27)%m(1,1)  = -1.00
         g%s(27)%m(2,3)  = -1.00
         g%s(27)%m(3,2)  =  1.00
         g%s(27)%v(1)    =  0.00
         g%s(27)%v(2)    =  0.50
         g%s(27)%v(3)    =  0.50
       
         g%s(28)%m(1,3)  = -1.00
         g%s(28)%m(2,2)  = -1.00
         g%s(28)%m(3,1)  =  1.00
         g%s(28)%v(1)    =  0.00
         g%s(28)%v(2)    =  0.50
         g%s(28)%v(3)    =  0.50
       
         g%s(29)%m(1,3)  =  1.00
         g%s(29)%m(2,2)  = -1.00
         g%s(29)%m(3,1)  = -1.00
         g%s(29)%v(1)    =  0.00
         g%s(29)%v(2)    =  0.50
         g%s(29)%v(3)    =  0.50
       
         g%s(30)%m(1,2)  =  1.00
         g%s(30)%m(2,1)  = -1.00
         g%s(30)%m(3,3)  = -1.00
         g%s(30)%v(1)    =  0.00
         g%s(30)%v(2)    =  0.50
         g%s(30)%v(3)    =  0.50
       
         g%s(31)%m(1,2)  = -1.00
         g%s(31)%m(2,1)  =  1.00
         g%s(31)%m(3,3)  = -1.00
         g%s(31)%v(1)    =  0.00
         g%s(31)%v(2)    =  0.50
         g%s(31)%v(3)    =  0.50
       
         g%s(32)%m(1,3)  =  1.00
         g%s(32)%m(2,1)  =  1.00
         g%s(32)%m(3,2)  =  1.00
         g%s(32)%v(1)    =  0.00
         g%s(32)%v(2)    =  0.50
         g%s(32)%v(3)    =  0.50
       
         g%s(33)%m(1,2)  =  1.00
         g%s(33)%m(2,3)  =  1.00
         g%s(33)%m(3,1)  =  1.00
         g%s(33)%v(1)    =  0.00
         g%s(33)%v(2)    =  0.50
         g%s(33)%v(3)    =  0.50
       
         g%s(34)%m(1,2)  = -1.00
         g%s(34)%m(2,3)  = -1.00
         g%s(34)%m(3,1)  =  1.00
         g%s(34)%v(1)    =  0.00
         g%s(34)%v(2)    =  0.50
         g%s(34)%v(3)    =  0.50
       
         g%s(35)%m(1,3)  =  1.00
         g%s(35)%m(2,1)  = -1.00
         g%s(35)%m(3,2)  = -1.00
         g%s(35)%v(1)    =  0.00
         g%s(35)%v(2)    =  0.50
         g%s(35)%v(3)    =  0.50
       
         g%s(36)%m(1,2)  = -1.00
         g%s(36)%m(2,3)  =  1.00
         g%s(36)%m(3,1)  = -1.00
         g%s(36)%v(1)    =  0.00
         g%s(36)%v(2)    =  0.50
         g%s(36)%v(3)    =  0.50
       
         g%s(37)%m(1,3)  = -1.00
         g%s(37)%m(2,1)  = -1.00
         g%s(37)%m(3,2)  =  1.00
         g%s(37)%v(1)    =  0.00
         g%s(37)%v(2)    =  0.50
         g%s(37)%v(3)    =  0.50
       
         g%s(38)%m(1,3)  = -1.00
         g%s(38)%m(2,1)  =  1.00
         g%s(38)%m(3,2)  = -1.00
         g%s(38)%v(1)    =  0.00
         g%s(38)%v(2)    =  0.50
         g%s(38)%v(3)    =  0.50
       
         g%s(39)%m(1,2)  =  1.00
         g%s(39)%m(2,3)  = -1.00
         g%s(39)%m(3,1)  = -1.00
         g%s(39)%v(1)    =  0.00
         g%s(39)%v(2)    =  0.50
         g%s(39)%v(3)    =  0.50
       
         g%s(40)%m(1,1)  =  1.00
         g%s(40)%m(2,2)  = -1.00
         g%s(40)%m(3,3)  = -1.00
         g%s(40)%v(1)    =  0.00
         g%s(40)%v(2)    =  0.50
         g%s(40)%v(3)    =  0.50
       
         g%s(41)%m(1,1)  = -1.00
         g%s(41)%m(2,2)  =  1.00
         g%s(41)%m(3,3)  = -1.00
         g%s(41)%v(1)    =  0.00
         g%s(41)%v(2)    =  0.50
         g%s(41)%v(3)    =  0.50
       
         g%s(42)%m(1,1)  = -1.00
         g%s(42)%m(2,2)  = -1.00
         g%s(42)%m(3,3)  =  1.00
         g%s(42)%v(1)    =  0.00
         g%s(42)%v(2)    =  0.50
         g%s(42)%v(3)    =  0.50
       
         g%s(43)%m(1,2)  = -1.00
         g%s(43)%m(2,1)  = -1.00
         g%s(43)%m(3,3)  =  1.00
         g%s(43)%v(1)    =  0.00
         g%s(43)%v(2)    =  0.50
         g%s(43)%v(3)    =  0.50
       
         g%s(44)%m(1,2)  =  1.00
         g%s(44)%m(2,1)  =  1.00
         g%s(44)%m(3,3)  =  1.00
         g%s(44)%v(1)    =  0.00
         g%s(44)%v(2)    =  0.50
         g%s(44)%v(3)    =  0.50
       
         g%s(45)%m(1,3)  = -1.00
         g%s(45)%m(2,2)  =  1.00
         g%s(45)%m(3,1)  = -1.00
         g%s(45)%v(1)    =  0.00
         g%s(45)%v(2)    =  0.50
         g%s(45)%v(3)    =  0.50
       
         g%s(46)%m(1,3)  =  1.00
         g%s(46)%m(2,2)  =  1.00
         g%s(46)%m(3,1)  =  1.00
         g%s(46)%v(1)    =  0.00
         g%s(46)%v(2)    =  0.50
         g%s(46)%v(3)    =  0.50
       
         g%s(47)%m(1,1)  =  1.00
         g%s(47)%m(2,3)  = -1.00
         g%s(47)%m(3,2)  = -1.00
         g%s(47)%v(1)    =  0.00
         g%s(47)%v(2)    =  0.50
         g%s(47)%v(3)    =  0.50
       
         g%s(48)%m(1,1)  =  1.00
         g%s(48)%m(2,3)  =  1.00
         g%s(48)%m(3,2)  =  1.00
         g%s(48)%v(1)    =  0.00
         g%s(48)%v(2)    =  0.50
         g%s(48)%v(3)    =  0.50
       
         g%s(49)%m(1,1)  =  1.00
         g%s(49)%m(2,2)  =  1.00
         g%s(49)%m(3,3)  =  1.00
         g%s(49)%v(1)    =  0.50
         g%s(49)%v(2)    =  0.00
         g%s(49)%v(3)    =  0.50
       
         g%s(50)%m(1,1)  = -1.00
         g%s(50)%m(2,3)  =  1.00
         g%s(50)%m(3,2)  = -1.00
         g%s(50)%v(1)    =  0.50
         g%s(50)%v(2)    =  0.00
         g%s(50)%v(3)    =  0.50
       
         g%s(51)%m(1,1)  = -1.00
         g%s(51)%m(2,3)  = -1.00
         g%s(51)%m(3,2)  =  1.00
         g%s(51)%v(1)    =  0.50
         g%s(51)%v(2)    =  0.00
         g%s(51)%v(3)    =  0.50
       
         g%s(52)%m(1,3)  = -1.00
         g%s(52)%m(2,2)  = -1.00
         g%s(52)%m(3,1)  =  1.00
         g%s(52)%v(1)    =  0.50
         g%s(52)%v(2)    =  0.00
         g%s(52)%v(3)    =  0.50
       
         g%s(53)%m(1,3)  =  1.00
         g%s(53)%m(2,2)  = -1.00
         g%s(53)%m(3,1)  = -1.00
         g%s(53)%v(1)    =  0.50
         g%s(53)%v(2)    =  0.00
         g%s(53)%v(3)    =  0.50
       
         g%s(54)%m(1,2)  =  1.00
         g%s(54)%m(2,1)  = -1.00
         g%s(54)%m(3,3)  = -1.00
         g%s(54)%v(1)    =  0.50
         g%s(54)%v(2)    =  0.00
         g%s(54)%v(3)    =  0.50
       
         g%s(55)%m(1,2)  = -1.00
         g%s(55)%m(2,1)  =  1.00
         g%s(55)%m(3,3)  = -1.00
         g%s(55)%v(1)    =  0.50
         g%s(55)%v(2)    =  0.00
         g%s(55)%v(3)    =  0.50
       
         g%s(56)%m(1,3)  =  1.00
         g%s(56)%m(2,1)  =  1.00
         g%s(56)%m(3,2)  =  1.00
         g%s(56)%v(1)    =  0.50
         g%s(56)%v(2)    =  0.00
         g%s(56)%v(3)    =  0.50
       
         g%s(57)%m(1,2)  =  1.00
         g%s(57)%m(2,3)  =  1.00
         g%s(57)%m(3,1)  =  1.00
         g%s(57)%v(1)    =  0.50
         g%s(57)%v(2)    =  0.00
         g%s(57)%v(3)    =  0.50
       
         g%s(58)%m(1,2)  = -1.00
         g%s(58)%m(2,3)  = -1.00
         g%s(58)%m(3,1)  =  1.00
         g%s(58)%v(1)    =  0.50
         g%s(58)%v(2)    =  0.00
         g%s(58)%v(3)    =  0.50
       
         g%s(59)%m(1,3)  =  1.00
         g%s(59)%m(2,1)  = -1.00
         g%s(59)%m(3,2)  = -1.00
         g%s(59)%v(1)    =  0.50
         g%s(59)%v(2)    =  0.00
         g%s(59)%v(3)    =  0.50
       
         g%s(60)%m(1,2)  = -1.00
         g%s(60)%m(2,3)  =  1.00
         g%s(60)%m(3,1)  = -1.00
         g%s(60)%v(1)    =  0.50
         g%s(60)%v(2)    =  0.00
         g%s(60)%v(3)    =  0.50
       
         g%s(61)%m(1,3)  = -1.00
         g%s(61)%m(2,1)  = -1.00
         g%s(61)%m(3,2)  =  1.00
         g%s(61)%v(1)    =  0.50
         g%s(61)%v(2)    =  0.00
         g%s(61)%v(3)    =  0.50
       
         g%s(62)%m(1,3)  = -1.00
         g%s(62)%m(2,1)  =  1.00
         g%s(62)%m(3,2)  = -1.00
         g%s(62)%v(1)    =  0.50
         g%s(62)%v(2)    =  0.00
         g%s(62)%v(3)    =  0.50
       
         g%s(63)%m(1,2)  =  1.00
         g%s(63)%m(2,3)  = -1.00
         g%s(63)%m(3,1)  = -1.00
         g%s(63)%v(1)    =  0.50
         g%s(63)%v(2)    =  0.00
         g%s(63)%v(3)    =  0.50
       
         g%s(64)%m(1,1)  =  1.00
         g%s(64)%m(2,2)  = -1.00
         g%s(64)%m(3,3)  = -1.00
         g%s(64)%v(1)    =  0.50
         g%s(64)%v(2)    =  0.00
         g%s(64)%v(3)    =  0.50
       
         g%s(65)%m(1,1)  = -1.00
         g%s(65)%m(2,2)  =  1.00
         g%s(65)%m(3,3)  = -1.00
         g%s(65)%v(1)    =  0.50
         g%s(65)%v(2)    =  0.00
         g%s(65)%v(3)    =  0.50
       
         g%s(66)%m(1,1)  = -1.00
         g%s(66)%m(2,2)  = -1.00
         g%s(66)%m(3,3)  =  1.00
         g%s(66)%v(1)    =  0.50
         g%s(66)%v(2)    =  0.00
         g%s(66)%v(3)    =  0.50
       
         g%s(67)%m(1,2)  = -1.00
         g%s(67)%m(2,1)  = -1.00
         g%s(67)%m(3,3)  =  1.00
         g%s(67)%v(1)    =  0.50
         g%s(67)%v(2)    =  0.00
         g%s(67)%v(3)    =  0.50
       
         g%s(68)%m(1,2)  =  1.00
         g%s(68)%m(2,1)  =  1.00
         g%s(68)%m(3,3)  =  1.00
         g%s(68)%v(1)    =  0.50
         g%s(68)%v(2)    =  0.00
         g%s(68)%v(3)    =  0.50
       
         g%s(69)%m(1,3)  = -1.00
         g%s(69)%m(2,2)  =  1.00
         g%s(69)%m(3,1)  = -1.00
         g%s(69)%v(1)    =  0.50
         g%s(69)%v(2)    =  0.00
         g%s(69)%v(3)    =  0.50
       
         g%s(70)%m(1,3)  =  1.00
         g%s(70)%m(2,2)  =  1.00
         g%s(70)%m(3,1)  =  1.00
         g%s(70)%v(1)    =  0.50
         g%s(70)%v(2)    =  0.00
         g%s(70)%v(3)    =  0.50
       
         g%s(71)%m(1,1)  =  1.00
         g%s(71)%m(2,3)  = -1.00
         g%s(71)%m(3,2)  = -1.00
         g%s(71)%v(1)    =  0.50
         g%s(71)%v(2)    =  0.00
         g%s(71)%v(3)    =  0.50
       
         g%s(72)%m(1,1)  =  1.00
         g%s(72)%m(2,3)  =  1.00
         g%s(72)%m(3,2)  =  1.00
         g%s(72)%v(1)    =  0.50
         g%s(72)%v(2)    =  0.00
         g%s(72)%v(3)    =  0.50
       
         g%s(73)%m(1,1)  =  1.00
         g%s(73)%m(2,2)  =  1.00
         g%s(73)%m(3,3)  =  1.00
         g%s(73)%v(1)    =  0.50
         g%s(73)%v(2)    =  0.50
         g%s(73)%v(3)    =  0.00
       
         g%s(74)%m(1,1)  = -1.00
         g%s(74)%m(2,3)  =  1.00
         g%s(74)%m(3,2)  = -1.00
         g%s(74)%v(1)    =  0.50
         g%s(74)%v(2)    =  0.50
         g%s(74)%v(3)    =  0.00
       
         g%s(75)%m(1,1)  = -1.00
         g%s(75)%m(2,3)  = -1.00
         g%s(75)%m(3,2)  =  1.00
         g%s(75)%v(1)    =  0.50
         g%s(75)%v(2)    =  0.50
         g%s(75)%v(3)    =  0.00
       
         g%s(76)%m(1,3)  = -1.00
         g%s(76)%m(2,2)  = -1.00
         g%s(76)%m(3,1)  =  1.00
         g%s(76)%v(1)    =  0.50
         g%s(76)%v(2)    =  0.50
         g%s(76)%v(3)    =  0.00
       
         g%s(77)%m(1,3)  =  1.00
         g%s(77)%m(2,2)  = -1.00
         g%s(77)%m(3,1)  = -1.00
         g%s(77)%v(1)    =  0.50
         g%s(77)%v(2)    =  0.50
         g%s(77)%v(3)    =  0.00
       
         g%s(78)%m(1,2)  =  1.00
         g%s(78)%m(2,1)  = -1.00
         g%s(78)%m(3,3)  = -1.00
         g%s(78)%v(1)    =  0.50
         g%s(78)%v(2)    =  0.50
         g%s(78)%v(3)    =  0.00
       
         g%s(79)%m(1,2)  = -1.00
         g%s(79)%m(2,1)  =  1.00
         g%s(79)%m(3,3)  = -1.00
         g%s(79)%v(1)    =  0.50
         g%s(79)%v(2)    =  0.50
         g%s(79)%v(3)    =  0.00
       
         g%s(80)%m(1,3)  =  1.00
         g%s(80)%m(2,1)  =  1.00
         g%s(80)%m(3,2)  =  1.00
         g%s(80)%v(1)    =  0.50
         g%s(80)%v(2)    =  0.50
         g%s(80)%v(3)    =  0.00
       
         g%s(81)%m(1,2)  =  1.00
         g%s(81)%m(2,3)  =  1.00
         g%s(81)%m(3,1)  =  1.00
         g%s(81)%v(1)    =  0.50
         g%s(81)%v(2)    =  0.50
         g%s(81)%v(3)    =  0.00
       
         g%s(82)%m(1,2)  = -1.00
         g%s(82)%m(2,3)  = -1.00
         g%s(82)%m(3,1)  =  1.00
         g%s(82)%v(1)    =  0.50
         g%s(82)%v(2)    =  0.50
         g%s(82)%v(3)    =  0.00
       
         g%s(83)%m(1,3)  =  1.00
         g%s(83)%m(2,1)  = -1.00
         g%s(83)%m(3,2)  = -1.00
         g%s(83)%v(1)    =  0.50
         g%s(83)%v(2)    =  0.50
         g%s(83)%v(3)    =  0.00
       
         g%s(84)%m(1,2)  = -1.00
         g%s(84)%m(2,3)  =  1.00
         g%s(84)%m(3,1)  = -1.00
         g%s(84)%v(1)    =  0.50
         g%s(84)%v(2)    =  0.50
         g%s(84)%v(3)    =  0.00
       
         g%s(85)%m(1,3)  = -1.00
         g%s(85)%m(2,1)  = -1.00
         g%s(85)%m(3,2)  =  1.00
         g%s(85)%v(1)    =  0.50
         g%s(85)%v(2)    =  0.50
         g%s(85)%v(3)    =  0.00
       
         g%s(86)%m(1,3)  = -1.00
         g%s(86)%m(2,1)  =  1.00
         g%s(86)%m(3,2)  = -1.00
         g%s(86)%v(1)    =  0.50
         g%s(86)%v(2)    =  0.50
         g%s(86)%v(3)    =  0.00
       
         g%s(87)%m(1,2)  =  1.00
         g%s(87)%m(2,3)  = -1.00
         g%s(87)%m(3,1)  = -1.00
         g%s(87)%v(1)    =  0.50
         g%s(87)%v(2)    =  0.50
         g%s(87)%v(3)    =  0.00
       
         g%s(88)%m(1,1)  =  1.00
         g%s(88)%m(2,2)  = -1.00
         g%s(88)%m(3,3)  = -1.00
         g%s(88)%v(1)    =  0.50
         g%s(88)%v(2)    =  0.50
         g%s(88)%v(3)    =  0.00
       
         g%s(89)%m(1,1)  = -1.00
         g%s(89)%m(2,2)  =  1.00
         g%s(89)%m(3,3)  = -1.00
         g%s(89)%v(1)    =  0.50
         g%s(89)%v(2)    =  0.50
         g%s(89)%v(3)    =  0.00
       
         g%s(90)%m(1,1)  = -1.00
         g%s(90)%m(2,2)  = -1.00
         g%s(90)%m(3,3)  =  1.00
         g%s(90)%v(1)    =  0.50
         g%s(90)%v(2)    =  0.50
         g%s(90)%v(3)    =  0.00
       
         g%s(91)%m(1,2)  = -1.00
         g%s(91)%m(2,1)  = -1.00
         g%s(91)%m(3,3)  =  1.00
         g%s(91)%v(1)    =  0.50
         g%s(91)%v(2)    =  0.50
         g%s(91)%v(3)    =  0.00
       
         g%s(92)%m(1,2)  =  1.00
         g%s(92)%m(2,1)  =  1.00
         g%s(92)%m(3,3)  =  1.00
         g%s(92)%v(1)    =  0.50
         g%s(92)%v(2)    =  0.50
         g%s(92)%v(3)    =  0.00
       
         g%s(93)%m(1,3)  = -1.00
         g%s(93)%m(2,2)  =  1.00
         g%s(93)%m(3,1)  = -1.00
         g%s(93)%v(1)    =  0.50
         g%s(93)%v(2)    =  0.50
         g%s(93)%v(3)    =  0.00
       
         g%s(94)%m(1,3)  =  1.00
         g%s(94)%m(2,2)  =  1.00
         g%s(94)%m(3,1)  =  1.00
         g%s(94)%v(1)    =  0.50
         g%s(94)%v(2)    =  0.50
         g%s(94)%v(3)    =  0.00
       
         g%s(95)%m(1,1)  =  1.00
         g%s(95)%m(2,3)  = -1.00
         g%s(95)%m(3,2)  = -1.00
         g%s(95)%v(1)    =  0.50
         g%s(95)%v(2)    =  0.50
         g%s(95)%v(3)    =  0.00
       
         g%s(96)%m(1,1)  =  1.00
         g%s(96)%m(2,3)  =  1.00
         g%s(96)%m(3,2)  =  1.00
         g%s(96)%v(1)    =  0.50
         g%s(96)%v(2)    =  0.50
         g%s(96)%v(3)    =  0.00
       
       
      case('I -4 3 m', '217')                        
       
         !Space group   # 217
         !Group setting # 244
       
         g%order   =  48
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,3)   =  1.00
         g%s(2)%m(3,2)   = -1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,3)   = -1.00
         g%s(3)%m(3,2)   =  1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,3)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,1)   =  1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   =  1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,1)   = -1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   =  1.00
         g%s(9)%m(3,1)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,3)  = -1.00
         g%s(10)%m(3,1)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,2)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,1)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,3)  = -1.00
         g%s(13)%m(2,1)  = -1.00
         g%s(13)%m(3,2)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,2)  = -1.00
         g%s(19)%m(2,1)  = -1.00
         g%s(19)%m(3,3)  =  1.00
         g%s(19)%v(1)    =  0.00
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    =  0.00
       
         g%s(20)%m(1,2)  =  1.00
         g%s(20)%m(2,1)  =  1.00
         g%s(20)%m(3,3)  =  1.00
         g%s(20)%v(1)    =  0.00
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    =  0.00
       
         g%s(21)%m(1,3)  = -1.00
         g%s(21)%m(2,2)  =  1.00
         g%s(21)%m(3,1)  = -1.00
         g%s(21)%v(1)    =  0.00
         g%s(21)%v(2)    =  0.00
         g%s(21)%v(3)    =  0.00
       
         g%s(22)%m(1,3)  =  1.00
         g%s(22)%m(2,2)  =  1.00
         g%s(22)%m(3,1)  =  1.00
         g%s(22)%v(1)    =  0.00
         g%s(22)%v(2)    =  0.00
         g%s(22)%v(3)    =  0.00
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(2,3)  = -1.00
         g%s(23)%m(3,2)  = -1.00
         g%s(23)%v(1)    =  0.00
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    =  0.00
       
         g%s(24)%m(1,1)  =  1.00
         g%s(24)%m(2,3)  =  1.00
         g%s(24)%m(3,2)  =  1.00
         g%s(24)%v(1)    =  0.00
         g%s(24)%v(2)    =  0.00
         g%s(24)%v(3)    =  0.00
       
         g%s(25)%m(1,1)  =  1.00
         g%s(25)%m(2,2)  =  1.00
         g%s(25)%m(3,3)  =  1.00
         g%s(25)%v(1)    =  0.50
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  0.50
       
         g%s(26)%m(1,1)  = -1.00
         g%s(26)%m(2,3)  =  1.00
         g%s(26)%m(3,2)  = -1.00
         g%s(26)%v(1)    =  0.50
         g%s(26)%v(2)    =  0.50
         g%s(26)%v(3)    =  0.50
       
         g%s(27)%m(1,1)  = -1.00
         g%s(27)%m(2,3)  = -1.00
         g%s(27)%m(3,2)  =  1.00
         g%s(27)%v(1)    =  0.50
         g%s(27)%v(2)    =  0.50
         g%s(27)%v(3)    =  0.50
       
         g%s(28)%m(1,3)  = -1.00
         g%s(28)%m(2,2)  = -1.00
         g%s(28)%m(3,1)  =  1.00
         g%s(28)%v(1)    =  0.50
         g%s(28)%v(2)    =  0.50
         g%s(28)%v(3)    =  0.50
       
         g%s(29)%m(1,3)  =  1.00
         g%s(29)%m(2,2)  = -1.00
         g%s(29)%m(3,1)  = -1.00
         g%s(29)%v(1)    =  0.50
         g%s(29)%v(2)    =  0.50
         g%s(29)%v(3)    =  0.50
       
         g%s(30)%m(1,2)  =  1.00
         g%s(30)%m(2,1)  = -1.00
         g%s(30)%m(3,3)  = -1.00
         g%s(30)%v(1)    =  0.50
         g%s(30)%v(2)    =  0.50
         g%s(30)%v(3)    =  0.50
       
         g%s(31)%m(1,2)  = -1.00
         g%s(31)%m(2,1)  =  1.00
         g%s(31)%m(3,3)  = -1.00
         g%s(31)%v(1)    =  0.50
         g%s(31)%v(2)    =  0.50
         g%s(31)%v(3)    =  0.50
       
         g%s(32)%m(1,3)  =  1.00
         g%s(32)%m(2,1)  =  1.00
         g%s(32)%m(3,2)  =  1.00
         g%s(32)%v(1)    =  0.50
         g%s(32)%v(2)    =  0.50
         g%s(32)%v(3)    =  0.50
       
         g%s(33)%m(1,2)  =  1.00
         g%s(33)%m(2,3)  =  1.00
         g%s(33)%m(3,1)  =  1.00
         g%s(33)%v(1)    =  0.50
         g%s(33)%v(2)    =  0.50
         g%s(33)%v(3)    =  0.50
       
         g%s(34)%m(1,2)  = -1.00
         g%s(34)%m(2,3)  = -1.00
         g%s(34)%m(3,1)  =  1.00
         g%s(34)%v(1)    =  0.50
         g%s(34)%v(2)    =  0.50
         g%s(34)%v(3)    =  0.50
       
         g%s(35)%m(1,3)  =  1.00
         g%s(35)%m(2,1)  = -1.00
         g%s(35)%m(3,2)  = -1.00
         g%s(35)%v(1)    =  0.50
         g%s(35)%v(2)    =  0.50
         g%s(35)%v(3)    =  0.50
       
         g%s(36)%m(1,2)  = -1.00
         g%s(36)%m(2,3)  =  1.00
         g%s(36)%m(3,1)  = -1.00
         g%s(36)%v(1)    =  0.50
         g%s(36)%v(2)    =  0.50
         g%s(36)%v(3)    =  0.50
       
         g%s(37)%m(1,3)  = -1.00
         g%s(37)%m(2,1)  = -1.00
         g%s(37)%m(3,2)  =  1.00
         g%s(37)%v(1)    =  0.50
         g%s(37)%v(2)    =  0.50
         g%s(37)%v(3)    =  0.50
       
         g%s(38)%m(1,3)  = -1.00
         g%s(38)%m(2,1)  =  1.00
         g%s(38)%m(3,2)  = -1.00
         g%s(38)%v(1)    =  0.50
         g%s(38)%v(2)    =  0.50
         g%s(38)%v(3)    =  0.50
       
         g%s(39)%m(1,2)  =  1.00
         g%s(39)%m(2,3)  = -1.00
         g%s(39)%m(3,1)  = -1.00
         g%s(39)%v(1)    =  0.50
         g%s(39)%v(2)    =  0.50
         g%s(39)%v(3)    =  0.50
       
         g%s(40)%m(1,1)  =  1.00
         g%s(40)%m(2,2)  = -1.00
         g%s(40)%m(3,3)  = -1.00
         g%s(40)%v(1)    =  0.50
         g%s(40)%v(2)    =  0.50
         g%s(40)%v(3)    =  0.50
       
         g%s(41)%m(1,1)  = -1.00
         g%s(41)%m(2,2)  =  1.00
         g%s(41)%m(3,3)  = -1.00
         g%s(41)%v(1)    =  0.50
         g%s(41)%v(2)    =  0.50
         g%s(41)%v(3)    =  0.50
       
         g%s(42)%m(1,1)  = -1.00
         g%s(42)%m(2,2)  = -1.00
         g%s(42)%m(3,3)  =  1.00
         g%s(42)%v(1)    =  0.50
         g%s(42)%v(2)    =  0.50
         g%s(42)%v(3)    =  0.50
       
         g%s(43)%m(1,2)  = -1.00
         g%s(43)%m(2,1)  = -1.00
         g%s(43)%m(3,3)  =  1.00
         g%s(43)%v(1)    =  0.50
         g%s(43)%v(2)    =  0.50
         g%s(43)%v(3)    =  0.50
       
         g%s(44)%m(1,2)  =  1.00
         g%s(44)%m(2,1)  =  1.00
         g%s(44)%m(3,3)  =  1.00
         g%s(44)%v(1)    =  0.50
         g%s(44)%v(2)    =  0.50
         g%s(44)%v(3)    =  0.50
       
         g%s(45)%m(1,3)  = -1.00
         g%s(45)%m(2,2)  =  1.00
         g%s(45)%m(3,1)  = -1.00
         g%s(45)%v(1)    =  0.50
         g%s(45)%v(2)    =  0.50
         g%s(45)%v(3)    =  0.50
       
         g%s(46)%m(1,3)  =  1.00
         g%s(46)%m(2,2)  =  1.00
         g%s(46)%m(3,1)  =  1.00
         g%s(46)%v(1)    =  0.50
         g%s(46)%v(2)    =  0.50
         g%s(46)%v(3)    =  0.50
       
         g%s(47)%m(1,1)  =  1.00
         g%s(47)%m(2,3)  = -1.00
         g%s(47)%m(3,2)  = -1.00
         g%s(47)%v(1)    =  0.50
         g%s(47)%v(2)    =  0.50
         g%s(47)%v(3)    =  0.50
       
         g%s(48)%m(1,1)  =  1.00
         g%s(48)%m(2,3)  =  1.00
         g%s(48)%m(3,2)  =  1.00
         g%s(48)%v(1)    =  0.50
         g%s(48)%v(2)    =  0.50
         g%s(48)%v(3)    =  0.50
       
       
      case('P -4 3 n', '218')                        
       
         !Space group   # 218
         !Group setting # 245
       
         g%order   =  24
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,3)   =  1.00
         g%s(2)%m(3,2)   = -1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,3)   = -1.00
         g%s(3)%m(3,2)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,3)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,1)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,3)   =  1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,1)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,3)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   =  1.00
         g%s(9)%m(3,1)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,3)  = -1.00
         g%s(10)%m(3,1)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,2)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,1)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,3)  = -1.00
         g%s(13)%m(2,1)  = -1.00
         g%s(13)%m(3,2)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,2)  = -1.00
         g%s(19)%m(2,1)  = -1.00
         g%s(19)%m(3,3)  =  1.00
         g%s(19)%v(1)    =  0.50
         g%s(19)%v(2)    =  0.50
         g%s(19)%v(3)    =  0.50
       
         g%s(20)%m(1,2)  =  1.00
         g%s(20)%m(2,1)  =  1.00
         g%s(20)%m(3,3)  =  1.00
         g%s(20)%v(1)    =  0.50
         g%s(20)%v(2)    =  0.50
         g%s(20)%v(3)    =  0.50
       
         g%s(21)%m(1,3)  = -1.00
         g%s(21)%m(2,2)  =  1.00
         g%s(21)%m(3,1)  = -1.00
         g%s(21)%v(1)    =  0.50
         g%s(21)%v(2)    =  0.50
         g%s(21)%v(3)    =  0.50
       
         g%s(22)%m(1,3)  =  1.00
         g%s(22)%m(2,2)  =  1.00
         g%s(22)%m(3,1)  =  1.00
         g%s(22)%v(1)    =  0.50
         g%s(22)%v(2)    =  0.50
         g%s(22)%v(3)    =  0.50
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(2,3)  = -1.00
         g%s(23)%m(3,2)  = -1.00
         g%s(23)%v(1)    =  0.50
         g%s(23)%v(2)    =  0.50
         g%s(23)%v(3)    =  0.50
       
         g%s(24)%m(1,1)  =  1.00
         g%s(24)%m(2,3)  =  1.00
         g%s(24)%m(3,2)  =  1.00
         g%s(24)%v(1)    =  0.50
         g%s(24)%v(2)    =  0.50
         g%s(24)%v(3)    =  0.50
       
       
      case('F -4 3 c', '219')                        
       
         !Space group   # 219
         !Group setting # 246
       
         g%order   =  96
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,3)   =  1.00
         g%s(2)%m(3,2)   = -1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,3)   = -1.00
         g%s(3)%m(3,2)   =  1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,3)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,1)   =  1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   =  1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,1)   = -1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   =  1.00
         g%s(9)%m(3,1)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,3)  = -1.00
         g%s(10)%m(3,1)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,2)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,1)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,3)  = -1.00
         g%s(13)%m(2,1)  = -1.00
         g%s(13)%m(3,2)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,2)  = -1.00
         g%s(19)%m(2,1)  = -1.00
         g%s(19)%m(3,3)  =  1.00
         g%s(19)%v(1)    =  0.50
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    =  0.00
       
         g%s(20)%m(1,2)  =  1.00
         g%s(20)%m(2,1)  =  1.00
         g%s(20)%m(3,3)  =  1.00
         g%s(20)%v(1)    =  0.50
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    =  0.00
       
         g%s(21)%m(1,3)  = -1.00
         g%s(21)%m(2,2)  =  1.00
         g%s(21)%m(3,1)  = -1.00
         g%s(21)%v(1)    =  0.50
         g%s(21)%v(2)    =  0.00
         g%s(21)%v(3)    =  0.00
       
         g%s(22)%m(1,3)  =  1.00
         g%s(22)%m(2,2)  =  1.00
         g%s(22)%m(3,1)  =  1.00
         g%s(22)%v(1)    =  0.50
         g%s(22)%v(2)    =  0.00
         g%s(22)%v(3)    =  0.00
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(2,3)  = -1.00
         g%s(23)%m(3,2)  = -1.00
         g%s(23)%v(1)    =  0.50
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    =  0.00
       
         g%s(24)%m(1,1)  =  1.00
         g%s(24)%m(2,3)  =  1.00
         g%s(24)%m(3,2)  =  1.00
         g%s(24)%v(1)    =  0.50
         g%s(24)%v(2)    =  0.00
         g%s(24)%v(3)    =  0.00
       
         g%s(25)%m(1,1)  =  1.00
         g%s(25)%m(2,2)  =  1.00
         g%s(25)%m(3,3)  =  1.00
         g%s(25)%v(1)    =  0.00
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  0.50
       
         g%s(26)%m(1,1)  = -1.00
         g%s(26)%m(2,3)  =  1.00
         g%s(26)%m(3,2)  = -1.00
         g%s(26)%v(1)    =  0.50
         g%s(26)%v(2)    =  0.50
         g%s(26)%v(3)    =  0.50
       
         g%s(27)%m(1,1)  = -1.00
         g%s(27)%m(2,3)  = -1.00
         g%s(27)%m(3,2)  =  1.00
         g%s(27)%v(1)    =  0.50
         g%s(27)%v(2)    =  0.50
         g%s(27)%v(3)    =  0.50
       
         g%s(28)%m(1,3)  = -1.00
         g%s(28)%m(2,2)  = -1.00
         g%s(28)%m(3,1)  =  1.00
         g%s(28)%v(1)    =  0.50
         g%s(28)%v(2)    =  0.50
         g%s(28)%v(3)    =  0.50
       
         g%s(29)%m(1,3)  =  1.00
         g%s(29)%m(2,2)  = -1.00
         g%s(29)%m(3,1)  = -1.00
         g%s(29)%v(1)    =  0.50
         g%s(29)%v(2)    =  0.50
         g%s(29)%v(3)    =  0.50
       
         g%s(30)%m(1,2)  =  1.00
         g%s(30)%m(2,1)  = -1.00
         g%s(30)%m(3,3)  = -1.00
         g%s(30)%v(1)    =  0.50
         g%s(30)%v(2)    =  0.50
         g%s(30)%v(3)    =  0.50
       
         g%s(31)%m(1,2)  = -1.00
         g%s(31)%m(2,1)  =  1.00
         g%s(31)%m(3,3)  = -1.00
         g%s(31)%v(1)    =  0.50
         g%s(31)%v(2)    =  0.50
         g%s(31)%v(3)    =  0.50
       
         g%s(32)%m(1,3)  =  1.00
         g%s(32)%m(2,1)  =  1.00
         g%s(32)%m(3,2)  =  1.00
         g%s(32)%v(1)    =  0.00
         g%s(32)%v(2)    =  0.50
         g%s(32)%v(3)    =  0.50
       
         g%s(33)%m(1,2)  =  1.00
         g%s(33)%m(2,3)  =  1.00
         g%s(33)%m(3,1)  =  1.00
         g%s(33)%v(1)    =  0.00
         g%s(33)%v(2)    =  0.50
         g%s(33)%v(3)    =  0.50
       
         g%s(34)%m(1,2)  = -1.00
         g%s(34)%m(2,3)  = -1.00
         g%s(34)%m(3,1)  =  1.00
         g%s(34)%v(1)    =  0.00
         g%s(34)%v(2)    =  0.50
         g%s(34)%v(3)    =  0.50
       
         g%s(35)%m(1,3)  =  1.00
         g%s(35)%m(2,1)  = -1.00
         g%s(35)%m(3,2)  = -1.00
         g%s(35)%v(1)    =  0.00
         g%s(35)%v(2)    =  0.50
         g%s(35)%v(3)    =  0.50
       
         g%s(36)%m(1,2)  = -1.00
         g%s(36)%m(2,3)  =  1.00
         g%s(36)%m(3,1)  = -1.00
         g%s(36)%v(1)    =  0.00
         g%s(36)%v(2)    =  0.50
         g%s(36)%v(3)    =  0.50
       
         g%s(37)%m(1,3)  = -1.00
         g%s(37)%m(2,1)  = -1.00
         g%s(37)%m(3,2)  =  1.00
         g%s(37)%v(1)    =  0.00
         g%s(37)%v(2)    =  0.50
         g%s(37)%v(3)    =  0.50
       
         g%s(38)%m(1,3)  = -1.00
         g%s(38)%m(2,1)  =  1.00
         g%s(38)%m(3,2)  = -1.00
         g%s(38)%v(1)    =  0.00
         g%s(38)%v(2)    =  0.50
         g%s(38)%v(3)    =  0.50
       
         g%s(39)%m(1,2)  =  1.00
         g%s(39)%m(2,3)  = -1.00
         g%s(39)%m(3,1)  = -1.00
         g%s(39)%v(1)    =  0.00
         g%s(39)%v(2)    =  0.50
         g%s(39)%v(3)    =  0.50
       
         g%s(40)%m(1,1)  =  1.00
         g%s(40)%m(2,2)  = -1.00
         g%s(40)%m(3,3)  = -1.00
         g%s(40)%v(1)    =  0.00
         g%s(40)%v(2)    =  0.50
         g%s(40)%v(3)    =  0.50
       
         g%s(41)%m(1,1)  = -1.00
         g%s(41)%m(2,2)  =  1.00
         g%s(41)%m(3,3)  = -1.00
         g%s(41)%v(1)    =  0.00
         g%s(41)%v(2)    =  0.50
         g%s(41)%v(3)    =  0.50
       
         g%s(42)%m(1,1)  = -1.00
         g%s(42)%m(2,2)  = -1.00
         g%s(42)%m(3,3)  =  1.00
         g%s(42)%v(1)    =  0.00
         g%s(42)%v(2)    =  0.50
         g%s(42)%v(3)    =  0.50
       
         g%s(43)%m(1,2)  = -1.00
         g%s(43)%m(2,1)  = -1.00
         g%s(43)%m(3,3)  =  1.00
         g%s(43)%v(1)    =  0.50
         g%s(43)%v(2)    =  0.50
         g%s(43)%v(3)    =  0.50
       
         g%s(44)%m(1,2)  =  1.00
         g%s(44)%m(2,1)  =  1.00
         g%s(44)%m(3,3)  =  1.00
         g%s(44)%v(1)    =  0.50
         g%s(44)%v(2)    =  0.50
         g%s(44)%v(3)    =  0.50
       
         g%s(45)%m(1,3)  = -1.00
         g%s(45)%m(2,2)  =  1.00
         g%s(45)%m(3,1)  = -1.00
         g%s(45)%v(1)    =  0.50
         g%s(45)%v(2)    =  0.50
         g%s(45)%v(3)    =  0.50
       
         g%s(46)%m(1,3)  =  1.00
         g%s(46)%m(2,2)  =  1.00
         g%s(46)%m(3,1)  =  1.00
         g%s(46)%v(1)    =  0.50
         g%s(46)%v(2)    =  0.50
         g%s(46)%v(3)    =  0.50
       
         g%s(47)%m(1,1)  =  1.00
         g%s(47)%m(2,3)  = -1.00
         g%s(47)%m(3,2)  = -1.00
         g%s(47)%v(1)    =  0.50
         g%s(47)%v(2)    =  0.50
         g%s(47)%v(3)    =  0.50
       
         g%s(48)%m(1,1)  =  1.00
         g%s(48)%m(2,3)  =  1.00
         g%s(48)%m(3,2)  =  1.00
         g%s(48)%v(1)    =  0.50
         g%s(48)%v(2)    =  0.50
         g%s(48)%v(3)    =  0.50
       
         g%s(49)%m(1,1)  =  1.00
         g%s(49)%m(2,2)  =  1.00
         g%s(49)%m(3,3)  =  1.00
         g%s(49)%v(1)    =  0.50
         g%s(49)%v(2)    =  0.00
         g%s(49)%v(3)    =  0.50
       
         g%s(50)%m(1,1)  = -1.00
         g%s(50)%m(2,3)  =  1.00
         g%s(50)%m(3,2)  = -1.00
         g%s(50)%v(1)    =  1.00000000
         g%s(50)%v(2)    =  0.00
         g%s(50)%v(3)    =  0.50
       
         g%s(51)%m(1,1)  = -1.00
         g%s(51)%m(2,3)  = -1.00
         g%s(51)%m(3,2)  =  1.00
         g%s(51)%v(1)    =  1.00000000
         g%s(51)%v(2)    =  0.00
         g%s(51)%v(3)    =  0.50
       
         g%s(52)%m(1,3)  = -1.00
         g%s(52)%m(2,2)  = -1.00
         g%s(52)%m(3,1)  =  1.00
         g%s(52)%v(1)    =  1.00000000
         g%s(52)%v(2)    =  0.00
         g%s(52)%v(3)    =  0.50
       
         g%s(53)%m(1,3)  =  1.00
         g%s(53)%m(2,2)  = -1.00
         g%s(53)%m(3,1)  = -1.00
         g%s(53)%v(1)    =  1.00000000
         g%s(53)%v(2)    =  0.00
         g%s(53)%v(3)    =  0.50
       
         g%s(54)%m(1,2)  =  1.00
         g%s(54)%m(2,1)  = -1.00
         g%s(54)%m(3,3)  = -1.00
         g%s(54)%v(1)    =  1.00000000
         g%s(54)%v(2)    =  0.00
         g%s(54)%v(3)    =  0.50
       
         g%s(55)%m(1,2)  = -1.00
         g%s(55)%m(2,1)  =  1.00
         g%s(55)%m(3,3)  = -1.00
         g%s(55)%v(1)    =  1.00000000
         g%s(55)%v(2)    =  0.00
         g%s(55)%v(3)    =  0.50
       
         g%s(56)%m(1,3)  =  1.00
         g%s(56)%m(2,1)  =  1.00
         g%s(56)%m(3,2)  =  1.00
         g%s(56)%v(1)    =  0.50
         g%s(56)%v(2)    =  0.00
         g%s(56)%v(3)    =  0.50
       
         g%s(57)%m(1,2)  =  1.00
         g%s(57)%m(2,3)  =  1.00
         g%s(57)%m(3,1)  =  1.00
         g%s(57)%v(1)    =  0.50
         g%s(57)%v(2)    =  0.00
         g%s(57)%v(3)    =  0.50
       
         g%s(58)%m(1,2)  = -1.00
         g%s(58)%m(2,3)  = -1.00
         g%s(58)%m(3,1)  =  1.00
         g%s(58)%v(1)    =  0.50
         g%s(58)%v(2)    =  0.00
         g%s(58)%v(3)    =  0.50
       
         g%s(59)%m(1,3)  =  1.00
         g%s(59)%m(2,1)  = -1.00
         g%s(59)%m(3,2)  = -1.00
         g%s(59)%v(1)    =  0.50
         g%s(59)%v(2)    =  0.00
         g%s(59)%v(3)    =  0.50
       
         g%s(60)%m(1,2)  = -1.00
         g%s(60)%m(2,3)  =  1.00
         g%s(60)%m(3,1)  = -1.00
         g%s(60)%v(1)    =  0.50
         g%s(60)%v(2)    =  0.00
         g%s(60)%v(3)    =  0.50
       
         g%s(61)%m(1,3)  = -1.00
         g%s(61)%m(2,1)  = -1.00
         g%s(61)%m(3,2)  =  1.00
         g%s(61)%v(1)    =  0.50
         g%s(61)%v(2)    =  0.00
         g%s(61)%v(3)    =  0.50
       
         g%s(62)%m(1,3)  = -1.00
         g%s(62)%m(2,1)  =  1.00
         g%s(62)%m(3,2)  = -1.00
         g%s(62)%v(1)    =  0.50
         g%s(62)%v(2)    =  0.00
         g%s(62)%v(3)    =  0.50
       
         g%s(63)%m(1,2)  =  1.00
         g%s(63)%m(2,3)  = -1.00
         g%s(63)%m(3,1)  = -1.00
         g%s(63)%v(1)    =  0.50
         g%s(63)%v(2)    =  0.00
         g%s(63)%v(3)    =  0.50
       
         g%s(64)%m(1,1)  =  1.00
         g%s(64)%m(2,2)  = -1.00
         g%s(64)%m(3,3)  = -1.00
         g%s(64)%v(1)    =  0.50
         g%s(64)%v(2)    =  0.00
         g%s(64)%v(3)    =  0.50
       
         g%s(65)%m(1,1)  = -1.00
         g%s(65)%m(2,2)  =  1.00
         g%s(65)%m(3,3)  = -1.00
         g%s(65)%v(1)    =  0.50
         g%s(65)%v(2)    =  0.00
         g%s(65)%v(3)    =  0.50
       
         g%s(66)%m(1,1)  = -1.00
         g%s(66)%m(2,2)  = -1.00
         g%s(66)%m(3,3)  =  1.00
         g%s(66)%v(1)    =  0.50
         g%s(66)%v(2)    =  0.00
         g%s(66)%v(3)    =  0.50
       
         g%s(67)%m(1,2)  = -1.00
         g%s(67)%m(2,1)  = -1.00
         g%s(67)%m(3,3)  =  1.00
         g%s(67)%v(1)    =  1.00000000
         g%s(67)%v(2)    =  0.00
         g%s(67)%v(3)    =  0.50
       
         g%s(68)%m(1,2)  =  1.00
         g%s(68)%m(2,1)  =  1.00
         g%s(68)%m(3,3)  =  1.00
         g%s(68)%v(1)    =  1.00000000
         g%s(68)%v(2)    =  0.00
         g%s(68)%v(3)    =  0.50
       
         g%s(69)%m(1,3)  = -1.00
         g%s(69)%m(2,2)  =  1.00
         g%s(69)%m(3,1)  = -1.00
         g%s(69)%v(1)    =  1.00000000
         g%s(69)%v(2)    =  0.00
         g%s(69)%v(3)    =  0.50
       
         g%s(70)%m(1,3)  =  1.00
         g%s(70)%m(2,2)  =  1.00
         g%s(70)%m(3,1)  =  1.00
         g%s(70)%v(1)    =  1.00000000
         g%s(70)%v(2)    =  0.00
         g%s(70)%v(3)    =  0.50
       
         g%s(71)%m(1,1)  =  1.00
         g%s(71)%m(2,3)  = -1.00
         g%s(71)%m(3,2)  = -1.00
         g%s(71)%v(1)    =  1.00000000
         g%s(71)%v(2)    =  0.00
         g%s(71)%v(3)    =  0.50
       
         g%s(72)%m(1,1)  =  1.00
         g%s(72)%m(2,3)  =  1.00
         g%s(72)%m(3,2)  =  1.00
         g%s(72)%v(1)    =  1.00000000
         g%s(72)%v(2)    =  0.00
         g%s(72)%v(3)    =  0.50
       
         g%s(73)%m(1,1)  =  1.00
         g%s(73)%m(2,2)  =  1.00
         g%s(73)%m(3,3)  =  1.00
         g%s(73)%v(1)    =  0.50
         g%s(73)%v(2)    =  0.50
         g%s(73)%v(3)    =  0.00
       
         g%s(74)%m(1,1)  = -1.00
         g%s(74)%m(2,3)  =  1.00
         g%s(74)%m(3,2)  = -1.00
         g%s(74)%v(1)    =  1.00000000
         g%s(74)%v(2)    =  0.50
         g%s(74)%v(3)    =  0.00
       
         g%s(75)%m(1,1)  = -1.00
         g%s(75)%m(2,3)  = -1.00
         g%s(75)%m(3,2)  =  1.00
         g%s(75)%v(1)    =  1.00000000
         g%s(75)%v(2)    =  0.50
         g%s(75)%v(3)    =  0.00
       
         g%s(76)%m(1,3)  = -1.00
         g%s(76)%m(2,2)  = -1.00
         g%s(76)%m(3,1)  =  1.00
         g%s(76)%v(1)    =  1.00000000
         g%s(76)%v(2)    =  0.50
         g%s(76)%v(3)    =  0.00
       
         g%s(77)%m(1,3)  =  1.00
         g%s(77)%m(2,2)  = -1.00
         g%s(77)%m(3,1)  = -1.00
         g%s(77)%v(1)    =  1.00000000
         g%s(77)%v(2)    =  0.50
         g%s(77)%v(3)    =  0.00
       
         g%s(78)%m(1,2)  =  1.00
         g%s(78)%m(2,1)  = -1.00
         g%s(78)%m(3,3)  = -1.00
         g%s(78)%v(1)    =  1.00000000
         g%s(78)%v(2)    =  0.50
         g%s(78)%v(3)    =  0.00
       
         g%s(79)%m(1,2)  = -1.00
         g%s(79)%m(2,1)  =  1.00
         g%s(79)%m(3,3)  = -1.00
         g%s(79)%v(1)    =  1.00000000
         g%s(79)%v(2)    =  0.50
         g%s(79)%v(3)    =  0.00
       
         g%s(80)%m(1,3)  =  1.00
         g%s(80)%m(2,1)  =  1.00
         g%s(80)%m(3,2)  =  1.00
         g%s(80)%v(1)    =  0.50
         g%s(80)%v(2)    =  0.50
         g%s(80)%v(3)    =  0.00
       
         g%s(81)%m(1,2)  =  1.00
         g%s(81)%m(2,3)  =  1.00
         g%s(81)%m(3,1)  =  1.00
         g%s(81)%v(1)    =  0.50
         g%s(81)%v(2)    =  0.50
         g%s(81)%v(3)    =  0.00
       
         g%s(82)%m(1,2)  = -1.00
         g%s(82)%m(2,3)  = -1.00
         g%s(82)%m(3,1)  =  1.00
         g%s(82)%v(1)    =  0.50
         g%s(82)%v(2)    =  0.50
         g%s(82)%v(3)    =  0.00
       
         g%s(83)%m(1,3)  =  1.00
         g%s(83)%m(2,1)  = -1.00
         g%s(83)%m(3,2)  = -1.00
         g%s(83)%v(1)    =  0.50
         g%s(83)%v(2)    =  0.50
         g%s(83)%v(3)    =  0.00
       
         g%s(84)%m(1,2)  = -1.00
         g%s(84)%m(2,3)  =  1.00
         g%s(84)%m(3,1)  = -1.00
         g%s(84)%v(1)    =  0.50
         g%s(84)%v(2)    =  0.50
         g%s(84)%v(3)    =  0.00
       
         g%s(85)%m(1,3)  = -1.00
         g%s(85)%m(2,1)  = -1.00
         g%s(85)%m(3,2)  =  1.00
         g%s(85)%v(1)    =  0.50
         g%s(85)%v(2)    =  0.50
         g%s(85)%v(3)    =  0.00
       
         g%s(86)%m(1,3)  = -1.00
         g%s(86)%m(2,1)  =  1.00
         g%s(86)%m(3,2)  = -1.00
         g%s(86)%v(1)    =  0.50
         g%s(86)%v(2)    =  0.50
         g%s(86)%v(3)    =  0.00
       
         g%s(87)%m(1,2)  =  1.00
         g%s(87)%m(2,3)  = -1.00
         g%s(87)%m(3,1)  = -1.00
         g%s(87)%v(1)    =  0.50
         g%s(87)%v(2)    =  0.50
         g%s(87)%v(3)    =  0.00
       
         g%s(88)%m(1,1)  =  1.00
         g%s(88)%m(2,2)  = -1.00
         g%s(88)%m(3,3)  = -1.00
         g%s(88)%v(1)    =  0.50
         g%s(88)%v(2)    =  0.50
         g%s(88)%v(3)    =  0.00
       
         g%s(89)%m(1,1)  = -1.00
         g%s(89)%m(2,2)  =  1.00
         g%s(89)%m(3,3)  = -1.00
         g%s(89)%v(1)    =  0.50
         g%s(89)%v(2)    =  0.50
         g%s(89)%v(3)    =  0.00
       
         g%s(90)%m(1,1)  = -1.00
         g%s(90)%m(2,2)  = -1.00
         g%s(90)%m(3,3)  =  1.00
         g%s(90)%v(1)    =  0.50
         g%s(90)%v(2)    =  0.50
         g%s(90)%v(3)    =  0.00
       
         g%s(91)%m(1,2)  = -1.00
         g%s(91)%m(2,1)  = -1.00
         g%s(91)%m(3,3)  =  1.00
         g%s(91)%v(1)    =  1.00000000
         g%s(91)%v(2)    =  0.50
         g%s(91)%v(3)    =  0.00
       
         g%s(92)%m(1,2)  =  1.00
         g%s(92)%m(2,1)  =  1.00
         g%s(92)%m(3,3)  =  1.00
         g%s(92)%v(1)    =  1.00000000
         g%s(92)%v(2)    =  0.50
         g%s(92)%v(3)    =  0.00
       
         g%s(93)%m(1,3)  = -1.00
         g%s(93)%m(2,2)  =  1.00
         g%s(93)%m(3,1)  = -1.00
         g%s(93)%v(1)    =  1.00000000
         g%s(93)%v(2)    =  0.50
         g%s(93)%v(3)    =  0.00
       
         g%s(94)%m(1,3)  =  1.00
         g%s(94)%m(2,2)  =  1.00
         g%s(94)%m(3,1)  =  1.00
         g%s(94)%v(1)    =  1.00000000
         g%s(94)%v(2)    =  0.50
         g%s(94)%v(3)    =  0.00
       
         g%s(95)%m(1,1)  =  1.00
         g%s(95)%m(2,3)  = -1.00
         g%s(95)%m(3,2)  = -1.00
         g%s(95)%v(1)    =  1.00000000
         g%s(95)%v(2)    =  0.50
         g%s(95)%v(3)    =  0.00
       
         g%s(96)%m(1,1)  =  1.00
         g%s(96)%m(2,3)  =  1.00
         g%s(96)%m(3,2)  =  1.00
         g%s(96)%v(1)    =  1.00000000
         g%s(96)%v(2)    =  0.50
         g%s(96)%v(3)    =  0.00
       
       
      case('I -4 3 d', '220')                        
       
         !Space group   # 220
         !Group setting # 247
       
         g%order   =  48
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   = -1.00
         g%s(2)%m(2,3)   =  1.00
         g%s(2)%m(3,2)   = -1.00
         g%s(2)%v(1)     =  0.25
         g%s(2)%v(2)     =  0.25
         g%s(2)%v(3)     =  0.75000000
       
         g%s(3)%m(1,1)   = -1.00
         g%s(3)%m(2,3)   = -1.00
         g%s(3)%m(3,2)   =  1.00
         g%s(3)%v(1)     =  0.25
         g%s(3)%v(2)     =  0.75000000
         g%s(3)%v(3)     =  0.75000000
       
         g%s(4)%m(1,3)   = -1.00
         g%s(4)%m(2,2)   = -1.00
         g%s(4)%m(3,1)   =  1.00
         g%s(4)%v(1)     =  0.25
         g%s(4)%v(2)     =  0.75000000
         g%s(4)%v(3)     =  0.75000000
       
         g%s(5)%m(1,3)   =  1.00
         g%s(5)%m(2,2)   = -1.00
         g%s(5)%m(3,1)   = -1.00
         g%s(5)%v(1)     =  0.25
         g%s(5)%v(2)     =  0.75000000
         g%s(5)%v(3)     =  0.25
       
         g%s(6)%m(1,2)   =  1.00
         g%s(6)%m(2,1)   = -1.00
         g%s(6)%m(3,3)   = -1.00
         g%s(6)%v(1)     =  0.25
         g%s(6)%v(2)     =  0.75000000
         g%s(6)%v(3)     =  0.25
       
         g%s(7)%m(1,2)   = -1.00
         g%s(7)%m(2,1)   =  1.00
         g%s(7)%m(3,3)   = -1.00
         g%s(7)%v(1)     =  0.25
         g%s(7)%v(2)     =  0.25
         g%s(7)%v(3)     =  0.75000000
       
         g%s(8)%m(1,3)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   =  1.00
         g%s(9)%m(3,1)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,3)  = -1.00
         g%s(10)%m(3,1)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,2)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,1)  = -1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,3)  = -1.00
         g%s(13)%m(2,1)  = -1.00
         g%s(13)%m(3,2)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.50
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.50
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.50
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,2)  = -1.00
         g%s(19)%m(2,1)  = -1.00
         g%s(19)%m(3,3)  =  1.00
         g%s(19)%v(1)    =  0.25
         g%s(19)%v(2)    =  0.75000000
         g%s(19)%v(3)    =  0.75000000
       
         g%s(20)%m(1,2)  =  1.00
         g%s(20)%m(2,1)  =  1.00
         g%s(20)%m(3,3)  =  1.00
         g%s(20)%v(1)    =  0.25
         g%s(20)%v(2)    =  0.25
         g%s(20)%v(3)    =  0.25
       
         g%s(21)%m(1,3)  = -1.00
         g%s(21)%m(2,2)  =  1.00
         g%s(21)%m(3,1)  = -1.00
         g%s(21)%v(1)    =  0.25
         g%s(21)%v(2)    =  0.25
         g%s(21)%v(3)    =  0.75000000
       
         g%s(22)%m(1,3)  =  1.00
         g%s(22)%m(2,2)  =  1.00
         g%s(22)%m(3,1)  =  1.00
         g%s(22)%v(1)    =  0.25
         g%s(22)%v(2)    =  0.25
         g%s(22)%v(3)    =  0.25
       
         g%s(23)%m(1,1)  =  1.00
         g%s(23)%m(2,3)  = -1.00
         g%s(23)%m(3,2)  = -1.00
         g%s(23)%v(1)    =  0.25
         g%s(23)%v(2)    =  0.75000000
         g%s(23)%v(3)    =  0.25
       
         g%s(24)%m(1,1)  =  1.00
         g%s(24)%m(2,3)  =  1.00
         g%s(24)%m(3,2)  =  1.00
         g%s(24)%v(1)    =  0.25
         g%s(24)%v(2)    =  0.25
         g%s(24)%v(3)    =  0.25
       
         g%s(25)%m(1,1)  =  1.00
         g%s(25)%m(2,2)  =  1.00
         g%s(25)%m(3,3)  =  1.00
         g%s(25)%v(1)    =  0.50
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  0.50
       
         g%s(26)%m(1,1)  = -1.00
         g%s(26)%m(2,3)  =  1.00
         g%s(26)%m(3,2)  = -1.00
         g%s(26)%v(1)    =  0.75000000
         g%s(26)%v(2)    =  0.75000000
         g%s(26)%v(3)    =  1.25000000
       
         g%s(27)%m(1,1)  = -1.00
         g%s(27)%m(2,3)  = -1.00
         g%s(27)%m(3,2)  =  1.00
         g%s(27)%v(1)    =  0.75000000
         g%s(27)%v(2)    =  1.25000000
         g%s(27)%v(3)    =  1.25000000
       
         g%s(28)%m(1,3)  = -1.00
         g%s(28)%m(2,2)  = -1.00
         g%s(28)%m(3,1)  =  1.00
         g%s(28)%v(1)    =  0.75000000
         g%s(28)%v(2)    =  1.25000000
         g%s(28)%v(3)    =  1.25000000
       
         g%s(29)%m(1,3)  =  1.00
         g%s(29)%m(2,2)  = -1.00
         g%s(29)%m(3,1)  = -1.00
         g%s(29)%v(1)    =  0.75000000
         g%s(29)%v(2)    =  1.25000000
         g%s(29)%v(3)    =  0.75000000
       
         g%s(30)%m(1,2)  =  1.00
         g%s(30)%m(2,1)  = -1.00
         g%s(30)%m(3,3)  = -1.00
         g%s(30)%v(1)    =  0.75000000
         g%s(30)%v(2)    =  1.25000000
         g%s(30)%v(3)    =  0.75000000
       
         g%s(31)%m(1,2)  = -1.00
         g%s(31)%m(2,1)  =  1.00
         g%s(31)%m(3,3)  = -1.00
         g%s(31)%v(1)    =  0.75000000
         g%s(31)%v(2)    =  0.75000000
         g%s(31)%v(3)    =  1.25000000
       
         g%s(32)%m(1,3)  =  1.00
         g%s(32)%m(2,1)  =  1.00
         g%s(32)%m(3,2)  =  1.00
         g%s(32)%v(1)    =  0.50
         g%s(32)%v(2)    =  0.50
         g%s(32)%v(3)    =  0.50
       
         g%s(33)%m(1,2)  =  1.00
         g%s(33)%m(2,3)  =  1.00
         g%s(33)%m(3,1)  =  1.00
         g%s(33)%v(1)    =  0.50
         g%s(33)%v(2)    =  0.50
         g%s(33)%v(3)    =  0.50
       
         g%s(34)%m(1,2)  = -1.00
         g%s(34)%m(2,3)  = -1.00
         g%s(34)%m(3,1)  =  1.00
         g%s(34)%v(1)    =  0.50
         g%s(34)%v(2)    =  1.00000000
         g%s(34)%v(3)    =  0.50
       
         g%s(35)%m(1,3)  =  1.00
         g%s(35)%m(2,1)  = -1.00
         g%s(35)%m(3,2)  = -1.00
         g%s(35)%v(1)    =  0.50
         g%s(35)%v(2)    =  0.50
         g%s(35)%v(3)    =  1.00000000
       
         g%s(36)%m(1,2)  = -1.00
         g%s(36)%m(2,3)  =  1.00
         g%s(36)%m(3,1)  = -1.00
         g%s(36)%v(1)    =  1.00000000
         g%s(36)%v(2)    =  0.50
         g%s(36)%v(3)    =  0.50
       
         g%s(37)%m(1,3)  = -1.00
         g%s(37)%m(2,1)  = -1.00
         g%s(37)%m(3,2)  =  1.00
         g%s(37)%v(1)    =  0.50
         g%s(37)%v(2)    =  1.00000000
         g%s(37)%v(3)    =  0.50
       
         g%s(38)%m(1,3)  = -1.00
         g%s(38)%m(2,1)  =  1.00
         g%s(38)%m(3,2)  = -1.00
         g%s(38)%v(1)    =  1.00000000
         g%s(38)%v(2)    =  0.50
         g%s(38)%v(3)    =  0.50
       
         g%s(39)%m(1,2)  =  1.00
         g%s(39)%m(2,3)  = -1.00
         g%s(39)%m(3,1)  = -1.00
         g%s(39)%v(1)    =  0.50
         g%s(39)%v(2)    =  0.50
         g%s(39)%v(3)    =  1.00000000
       
         g%s(40)%m(1,1)  =  1.00
         g%s(40)%m(2,2)  = -1.00
         g%s(40)%m(3,3)  = -1.00
         g%s(40)%v(1)    =  0.50
         g%s(40)%v(2)    =  0.50
         g%s(40)%v(3)    =  1.00000000
       
         g%s(41)%m(1,1)  = -1.00
         g%s(41)%m(2,2)  =  1.00
         g%s(41)%m(3,3)  = -1.00
         g%s(41)%v(1)    =  1.00000000
         g%s(41)%v(2)    =  0.50
         g%s(41)%v(3)    =  0.50
       
         g%s(42)%m(1,1)  = -1.00
         g%s(42)%m(2,2)  = -1.00
         g%s(42)%m(3,3)  =  1.00
         g%s(42)%v(1)    =  0.50
         g%s(42)%v(2)    =  1.00000000
         g%s(42)%v(3)    =  0.50
       
         g%s(43)%m(1,2)  = -1.00
         g%s(43)%m(2,1)  = -1.00
         g%s(43)%m(3,3)  =  1.00
         g%s(43)%v(1)    =  0.75000000
         g%s(43)%v(2)    =  1.25000000
         g%s(43)%v(3)    =  1.25000000
       
         g%s(44)%m(1,2)  =  1.00
         g%s(44)%m(2,1)  =  1.00
         g%s(44)%m(3,3)  =  1.00
         g%s(44)%v(1)    =  0.75000000
         g%s(44)%v(2)    =  0.75000000
         g%s(44)%v(3)    =  0.75000000
       
         g%s(45)%m(1,3)  = -1.00
         g%s(45)%m(2,2)  =  1.00
         g%s(45)%m(3,1)  = -1.00
         g%s(45)%v(1)    =  0.75000000
         g%s(45)%v(2)    =  0.75000000
         g%s(45)%v(3)    =  1.25000000
       
         g%s(46)%m(1,3)  =  1.00
         g%s(46)%m(2,2)  =  1.00
         g%s(46)%m(3,1)  =  1.00
         g%s(46)%v(1)    =  0.75000000
         g%s(46)%v(2)    =  0.75000000
         g%s(46)%v(3)    =  0.75000000
       
         g%s(47)%m(1,1)  =  1.00
         g%s(47)%m(2,3)  = -1.00
         g%s(47)%m(3,2)  = -1.00
         g%s(47)%v(1)    =  0.75000000
         g%s(47)%v(2)    =  1.25000000
         g%s(47)%v(3)    =  0.75000000
       
         g%s(48)%m(1,1)  =  1.00
         g%s(48)%m(2,3)  =  1.00
         g%s(48)%m(3,2)  =  1.00
         g%s(48)%v(1)    =  0.75000000
         g%s(48)%v(2)    =  0.75000000
         g%s(48)%v(3)    =  0.75000000
       
       
      case('P m -3 m', '221')                        
       
         !Space group   # 221
         !Group setting # 248
       
         g%order   =  48
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,3)   = -1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,2)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,3)   =  1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%m(3,1)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,1)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   =  1.00
         g%s(9)%m(3,1)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,3)  = -1.00
         g%s(10)%m(3,1)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,2)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,1)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,3)  = -1.00
         g%s(13)%m(2,1)  = -1.00
         g%s(13)%m(3,2)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,2)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,3)  = -1.00
         g%s(19)%v(1)    =  0.00
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    =  0.00
       
         g%s(20)%m(1,2)  = -1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.00
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    =  0.00
       
         g%s(21)%m(1,3)  =  1.00
         g%s(21)%m(2,2)  = -1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.00
         g%s(21)%v(2)    =  0.00
         g%s(21)%v(3)    =  0.00
       
         g%s(22)%m(1,3)  = -1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,1)  = -1.00
         g%s(22)%v(1)    =  0.00
         g%s(22)%v(2)    =  0.00
         g%s(22)%v(3)    =  0.00
       
         g%s(23)%m(1,1)  = -1.00
         g%s(23)%m(2,3)  =  1.00
         g%s(23)%m(3,2)  =  1.00
         g%s(23)%v(1)    =  0.00
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    =  0.00
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,3)  = -1.00
         g%s(24)%m(3,2)  = -1.00
         g%s(24)%v(1)    =  0.00
         g%s(24)%v(2)    =  0.00
         g%s(24)%v(3)    =  0.00
       
         g%s(25)%m(1,1)  = -1.00
         g%s(25)%m(2,2)  = -1.00
         g%s(25)%m(3,3)  = -1.00
         g%s(25)%v(1)    =  0.00
         g%s(25)%v(2)    =  0.00
         g%s(25)%v(3)    =  0.00
       
         g%s(26)%m(1,1)  = -1.00
         g%s(26)%m(2,3)  =  1.00
         g%s(26)%m(3,2)  = -1.00
         g%s(26)%v(1)    =  0.00
         g%s(26)%v(2)    =  0.00
         g%s(26)%v(3)    =  0.00
       
         g%s(27)%m(1,1)  = -1.00
         g%s(27)%m(2,3)  = -1.00
         g%s(27)%m(3,2)  =  1.00
         g%s(27)%v(1)    =  0.00
         g%s(27)%v(2)    =  0.00
         g%s(27)%v(3)    =  0.00
       
         g%s(28)%m(1,3)  = -1.00
         g%s(28)%m(2,2)  = -1.00
         g%s(28)%m(3,1)  =  1.00
         g%s(28)%v(1)    =  0.00
         g%s(28)%v(2)    =  0.00
         g%s(28)%v(3)    =  0.00
       
         g%s(29)%m(1,3)  =  1.00
         g%s(29)%m(2,2)  = -1.00
         g%s(29)%m(3,1)  = -1.00
         g%s(29)%v(1)    =  0.00
         g%s(29)%v(2)    =  0.00
         g%s(29)%v(3)    =  0.00
       
         g%s(30)%m(1,2)  =  1.00
         g%s(30)%m(2,1)  = -1.00
         g%s(30)%m(3,3)  = -1.00
         g%s(30)%v(1)    =  0.00
         g%s(30)%v(2)    =  0.00
         g%s(30)%v(3)    =  0.00
       
         g%s(31)%m(1,2)  = -1.00
         g%s(31)%m(2,1)  =  1.00
         g%s(31)%m(3,3)  = -1.00
         g%s(31)%v(1)    =  0.00
         g%s(31)%v(2)    =  0.00
         g%s(31)%v(3)    =  0.00
       
         g%s(32)%m(1,3)  = -1.00
         g%s(32)%m(2,1)  = -1.00
         g%s(32)%m(3,2)  = -1.00
         g%s(32)%v(1)    =  0.00
         g%s(32)%v(2)    =  0.00
         g%s(32)%v(3)    =  0.00
       
         g%s(33)%m(1,2)  = -1.00
         g%s(33)%m(2,3)  = -1.00
         g%s(33)%m(3,1)  = -1.00
         g%s(33)%v(1)    =  0.00
         g%s(33)%v(2)    =  0.00
         g%s(33)%v(3)    =  0.00
       
         g%s(34)%m(1,2)  =  1.00
         g%s(34)%m(2,3)  =  1.00
         g%s(34)%m(3,1)  = -1.00
         g%s(34)%v(1)    =  0.00
         g%s(34)%v(2)    =  0.00
         g%s(34)%v(3)    =  0.00
       
         g%s(35)%m(1,3)  = -1.00
         g%s(35)%m(2,1)  =  1.00
         g%s(35)%m(3,2)  =  1.00
         g%s(35)%v(1)    =  0.00
         g%s(35)%v(2)    =  0.00
         g%s(35)%v(3)    =  0.00
       
         g%s(36)%m(1,2)  =  1.00
         g%s(36)%m(2,3)  = -1.00
         g%s(36)%m(3,1)  =  1.00
         g%s(36)%v(1)    =  0.00
         g%s(36)%v(2)    =  0.00
         g%s(36)%v(3)    =  0.00
       
         g%s(37)%m(1,3)  =  1.00
         g%s(37)%m(2,1)  =  1.00
         g%s(37)%m(3,2)  = -1.00
         g%s(37)%v(1)    =  0.00
         g%s(37)%v(2)    =  0.00
         g%s(37)%v(3)    =  0.00
       
         g%s(38)%m(1,3)  =  1.00
         g%s(38)%m(2,1)  = -1.00
         g%s(38)%m(3,2)  =  1.00
         g%s(38)%v(1)    =  0.00
         g%s(38)%v(2)    =  0.00
         g%s(38)%v(3)    =  0.00
       
         g%s(39)%m(1,2)  = -1.00
         g%s(39)%m(2,3)  =  1.00
         g%s(39)%m(3,1)  =  1.00
         g%s(39)%v(1)    =  0.00
         g%s(39)%v(2)    =  0.00
         g%s(39)%v(3)    =  0.00
       
         g%s(40)%m(1,1)  = -1.00
         g%s(40)%m(2,2)  =  1.00
         g%s(40)%m(3,3)  =  1.00
         g%s(40)%v(1)    =  0.00
         g%s(40)%v(2)    =  0.00
         g%s(40)%v(3)    =  0.00
       
         g%s(41)%m(1,1)  =  1.00
         g%s(41)%m(2,2)  = -1.00
         g%s(41)%m(3,3)  =  1.00
         g%s(41)%v(1)    =  0.00
         g%s(41)%v(2)    =  0.00
         g%s(41)%v(3)    =  0.00
       
         g%s(42)%m(1,1)  =  1.00
         g%s(42)%m(2,2)  =  1.00
         g%s(42)%m(3,3)  = -1.00
         g%s(42)%v(1)    =  0.00
         g%s(42)%v(2)    =  0.00
         g%s(42)%v(3)    =  0.00
       
         g%s(43)%m(1,2)  = -1.00
         g%s(43)%m(2,1)  = -1.00
         g%s(43)%m(3,3)  =  1.00
         g%s(43)%v(1)    =  0.00
         g%s(43)%v(2)    =  0.00
         g%s(43)%v(3)    =  0.00
       
         g%s(44)%m(1,2)  =  1.00
         g%s(44)%m(2,1)  =  1.00
         g%s(44)%m(3,3)  =  1.00
         g%s(44)%v(1)    =  0.00
         g%s(44)%v(2)    =  0.00
         g%s(44)%v(3)    =  0.00
       
         g%s(45)%m(1,3)  = -1.00
         g%s(45)%m(2,2)  =  1.00
         g%s(45)%m(3,1)  = -1.00
         g%s(45)%v(1)    =  0.00
         g%s(45)%v(2)    =  0.00
         g%s(45)%v(3)    =  0.00
       
         g%s(46)%m(1,3)  =  1.00
         g%s(46)%m(2,2)  =  1.00
         g%s(46)%m(3,1)  =  1.00
         g%s(46)%v(1)    =  0.00
         g%s(46)%v(2)    =  0.00
         g%s(46)%v(3)    =  0.00
       
         g%s(47)%m(1,1)  =  1.00
         g%s(47)%m(2,3)  = -1.00
         g%s(47)%m(3,2)  = -1.00
         g%s(47)%v(1)    =  0.00
         g%s(47)%v(2)    =  0.00
         g%s(47)%v(3)    =  0.00
       
         g%s(48)%m(1,1)  =  1.00
         g%s(48)%m(2,3)  =  1.00
         g%s(48)%m(3,2)  =  1.00
         g%s(48)%v(1)    =  0.00
         g%s(48)%v(2)    =  0.00
         g%s(48)%v(3)    =  0.00
       
       
      case('P n -3 n : 2', '222')                    
       
         !Space group   # 222
         !Group setting # 249
       
         g%order   =  48
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,3)   = -1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,2)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,3)   =  1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%m(3,1)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,3)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,1)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   =  1.00
         g%s(9)%m(3,1)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,3)  = -1.00
         g%s(10)%m(3,1)  =  1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,2)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,1)  = -1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,3)  = -1.00
         g%s(13)%m(2,1)  = -1.00
         g%s(13)%m(3,2)  =  1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.50
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.50
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.50
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.50
         g%s(18)%v(2)    =  0.50
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,2)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,3)  = -1.00
         g%s(19)%v(1)    =  0.00
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    =  0.50
       
         g%s(20)%m(1,2)  = -1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.50
         g%s(20)%v(2)    =  0.50
         g%s(20)%v(3)    =  0.50
       
         g%s(21)%m(1,3)  =  1.00
         g%s(21)%m(2,2)  = -1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.00
         g%s(21)%v(2)    =  0.50
         g%s(21)%v(3)    =  0.00
       
         g%s(22)%m(1,3)  = -1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,1)  = -1.00
         g%s(22)%v(1)    =  0.50
         g%s(22)%v(2)    =  0.50
         g%s(22)%v(3)    =  0.50
       
         g%s(23)%m(1,1)  = -1.00
         g%s(23)%m(2,3)  =  1.00
         g%s(23)%m(3,2)  =  1.00
         g%s(23)%v(1)    =  0.50
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    =  0.00
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,3)  = -1.00
         g%s(24)%m(3,2)  = -1.00
         g%s(24)%v(1)    =  0.50
         g%s(24)%v(2)    =  0.50
         g%s(24)%v(3)    =  0.50
       
         g%s(25)%m(1,1)  = -1.00
         g%s(25)%m(2,2)  = -1.00
         g%s(25)%m(3,3)  = -1.00
         g%s(25)%v(1)    =  0.00
         g%s(25)%v(2)    =  0.00
         g%s(25)%v(3)    =  0.00
       
         g%s(26)%m(1,1)  = -1.00
         g%s(26)%m(2,3)  =  1.00
         g%s(26)%m(3,2)  = -1.00
         g%s(26)%v(1)    =  0.00
         g%s(26)%v(2)    = -0.50
         g%s(26)%v(3)    =  0.00
       
         g%s(27)%m(1,1)  = -1.00
         g%s(27)%m(2,3)  = -1.00
         g%s(27)%m(3,2)  =  1.00
         g%s(27)%v(1)    =  0.00
         g%s(27)%v(2)    =  0.00
         g%s(27)%v(3)    = -0.50
       
         g%s(28)%m(1,3)  = -1.00
         g%s(28)%m(2,2)  = -1.00
         g%s(28)%m(3,1)  =  1.00
         g%s(28)%v(1)    =  0.00
         g%s(28)%v(2)    =  0.00
         g%s(28)%v(3)    = -0.50
       
         g%s(29)%m(1,3)  =  1.00
         g%s(29)%m(2,2)  = -1.00
         g%s(29)%m(3,1)  = -1.00
         g%s(29)%v(1)    = -0.50
         g%s(29)%v(2)    =  0.00
         g%s(29)%v(3)    =  0.00
       
         g%s(30)%m(1,2)  =  1.00
         g%s(30)%m(2,1)  = -1.00
         g%s(30)%m(3,3)  = -1.00
         g%s(30)%v(1)    = -0.50
         g%s(30)%v(2)    =  0.00
         g%s(30)%v(3)    =  0.00
       
         g%s(31)%m(1,2)  = -1.00
         g%s(31)%m(2,1)  =  1.00
         g%s(31)%m(3,3)  = -1.00
         g%s(31)%v(1)    =  0.00
         g%s(31)%v(2)    = -0.50
         g%s(31)%v(3)    =  0.00
       
         g%s(32)%m(1,3)  = -1.00
         g%s(32)%m(2,1)  = -1.00
         g%s(32)%m(3,2)  = -1.00
         g%s(32)%v(1)    =  0.00
         g%s(32)%v(2)    =  0.00
         g%s(32)%v(3)    =  0.00
       
         g%s(33)%m(1,2)  = -1.00
         g%s(33)%m(2,3)  = -1.00
         g%s(33)%m(3,1)  = -1.00
         g%s(33)%v(1)    =  0.00
         g%s(33)%v(2)    =  0.00
         g%s(33)%v(3)    =  0.00
       
         g%s(34)%m(1,2)  =  1.00
         g%s(34)%m(2,3)  =  1.00
         g%s(34)%m(3,1)  = -1.00
         g%s(34)%v(1)    = -0.50
         g%s(34)%v(2)    = -0.50
         g%s(34)%v(3)    =  0.00
       
         g%s(35)%m(1,3)  = -1.00
         g%s(35)%m(2,1)  =  1.00
         g%s(35)%m(3,2)  =  1.00
         g%s(35)%v(1)    =  0.00
         g%s(35)%v(2)    = -0.50
         g%s(35)%v(3)    = -0.50
       
         g%s(36)%m(1,2)  =  1.00
         g%s(36)%m(2,3)  = -1.00
         g%s(36)%m(3,1)  =  1.00
         g%s(36)%v(1)    = -0.50
         g%s(36)%v(2)    =  0.00
         g%s(36)%v(3)    = -0.50
       
         g%s(37)%m(1,3)  =  1.00
         g%s(37)%m(2,1)  =  1.00
         g%s(37)%m(3,2)  = -1.00
         g%s(37)%v(1)    = -0.50
         g%s(37)%v(2)    = -0.50
         g%s(37)%v(3)    =  0.00
       
         g%s(38)%m(1,3)  =  1.00
         g%s(38)%m(2,1)  = -1.00
         g%s(38)%m(3,2)  =  1.00
         g%s(38)%v(1)    = -0.50
         g%s(38)%v(2)    =  0.00
         g%s(38)%v(3)    = -0.50
       
         g%s(39)%m(1,2)  = -1.00
         g%s(39)%m(2,3)  =  1.00
         g%s(39)%m(3,1)  =  1.00
         g%s(39)%v(1)    =  0.00
         g%s(39)%v(2)    = -0.50
         g%s(39)%v(3)    = -0.50
       
         g%s(40)%m(1,1)  = -1.00
         g%s(40)%m(2,2)  =  1.00
         g%s(40)%m(3,3)  =  1.00
         g%s(40)%v(1)    =  0.00
         g%s(40)%v(2)    = -0.50
         g%s(40)%v(3)    = -0.50
       
         g%s(41)%m(1,1)  =  1.00
         g%s(41)%m(2,2)  = -1.00
         g%s(41)%m(3,3)  =  1.00
         g%s(41)%v(1)    = -0.50
         g%s(41)%v(2)    =  0.00
         g%s(41)%v(3)    = -0.50
       
         g%s(42)%m(1,1)  =  1.00
         g%s(42)%m(2,2)  =  1.00
         g%s(42)%m(3,3)  = -1.00
         g%s(42)%v(1)    = -0.50
         g%s(42)%v(2)    = -0.50
         g%s(42)%v(3)    =  0.00
       
         g%s(43)%m(1,2)  = -1.00
         g%s(43)%m(2,1)  = -1.00
         g%s(43)%m(3,3)  =  1.00
         g%s(43)%v(1)    =  0.00
         g%s(43)%v(2)    =  0.00
         g%s(43)%v(3)    = -0.50
       
         g%s(44)%m(1,2)  =  1.00
         g%s(44)%m(2,1)  =  1.00
         g%s(44)%m(3,3)  =  1.00
         g%s(44)%v(1)    = -0.50
         g%s(44)%v(2)    = -0.50
         g%s(44)%v(3)    = -0.50
       
         g%s(45)%m(1,3)  = -1.00
         g%s(45)%m(2,2)  =  1.00
         g%s(45)%m(3,1)  = -1.00
         g%s(45)%v(1)    =  0.00
         g%s(45)%v(2)    = -0.50
         g%s(45)%v(3)    =  0.00
       
         g%s(46)%m(1,3)  =  1.00
         g%s(46)%m(2,2)  =  1.00
         g%s(46)%m(3,1)  =  1.00
         g%s(46)%v(1)    = -0.50
         g%s(46)%v(2)    = -0.50
         g%s(46)%v(3)    = -0.50
       
         g%s(47)%m(1,1)  =  1.00
         g%s(47)%m(2,3)  = -1.00
         g%s(47)%m(3,2)  = -1.00
         g%s(47)%v(1)    = -0.50
         g%s(47)%v(2)    =  0.00
         g%s(47)%v(3)    =  0.00
       
         g%s(48)%m(1,1)  =  1.00
         g%s(48)%m(2,3)  =  1.00
         g%s(48)%m(3,2)  =  1.00
         g%s(48)%v(1)    = -0.50
         g%s(48)%v(2)    = -0.50
         g%s(48)%v(3)    = -0.50
       
       
      case('P n -3 n : 1')                    
       
         !Space group   # 222
         !Group setting # 250
       
         g%order   =  48
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,3)   = -1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,2)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,3)   =  1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%m(3,1)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,1)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   =  1.00
         g%s(9)%m(3,1)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,3)  = -1.00
         g%s(10)%m(3,1)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,2)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,1)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,3)  = -1.00
         g%s(13)%m(2,1)  = -1.00
         g%s(13)%m(3,2)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,2)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,3)  = -1.00
         g%s(19)%v(1)    =  0.00
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    =  0.00
       
         g%s(20)%m(1,2)  = -1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.00
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    =  0.00
       
         g%s(21)%m(1,3)  =  1.00
         g%s(21)%m(2,2)  = -1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.00
         g%s(21)%v(2)    =  0.00
         g%s(21)%v(3)    =  0.00
       
         g%s(22)%m(1,3)  = -1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,1)  = -1.00
         g%s(22)%v(1)    =  0.00
         g%s(22)%v(2)    =  0.00
         g%s(22)%v(3)    =  0.00
       
         g%s(23)%m(1,1)  = -1.00
         g%s(23)%m(2,3)  =  1.00
         g%s(23)%m(3,2)  =  1.00
         g%s(23)%v(1)    =  0.00
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    =  0.00
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,3)  = -1.00
         g%s(24)%m(3,2)  = -1.00
         g%s(24)%v(1)    =  0.00
         g%s(24)%v(2)    =  0.00
         g%s(24)%v(3)    =  0.00
       
         g%s(25)%m(1,1)  = -1.00
         g%s(25)%m(2,2)  = -1.00
         g%s(25)%m(3,3)  = -1.00
         g%s(25)%v(1)    =  0.50
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  0.50
       
         g%s(26)%m(1,1)  = -1.00
         g%s(26)%m(2,3)  =  1.00
         g%s(26)%m(3,2)  = -1.00
         g%s(26)%v(1)    =  0.50
         g%s(26)%v(2)    =  0.50
         g%s(26)%v(3)    =  0.50
       
         g%s(27)%m(1,1)  = -1.00
         g%s(27)%m(2,3)  = -1.00
         g%s(27)%m(3,2)  =  1.00
         g%s(27)%v(1)    =  0.50
         g%s(27)%v(2)    =  0.50
         g%s(27)%v(3)    =  0.50
       
         g%s(28)%m(1,3)  = -1.00
         g%s(28)%m(2,2)  = -1.00
         g%s(28)%m(3,1)  =  1.00
         g%s(28)%v(1)    =  0.50
         g%s(28)%v(2)    =  0.50
         g%s(28)%v(3)    =  0.50
       
         g%s(29)%m(1,3)  =  1.00
         g%s(29)%m(2,2)  = -1.00
         g%s(29)%m(3,1)  = -1.00
         g%s(29)%v(1)    =  0.50
         g%s(29)%v(2)    =  0.50
         g%s(29)%v(3)    =  0.50
       
         g%s(30)%m(1,2)  =  1.00
         g%s(30)%m(2,1)  = -1.00
         g%s(30)%m(3,3)  = -1.00
         g%s(30)%v(1)    =  0.50
         g%s(30)%v(2)    =  0.50
         g%s(30)%v(3)    =  0.50
       
         g%s(31)%m(1,2)  = -1.00
         g%s(31)%m(2,1)  =  1.00
         g%s(31)%m(3,3)  = -1.00
         g%s(31)%v(1)    =  0.50
         g%s(31)%v(2)    =  0.50
         g%s(31)%v(3)    =  0.50
       
         g%s(32)%m(1,3)  = -1.00
         g%s(32)%m(2,1)  = -1.00
         g%s(32)%m(3,2)  = -1.00
         g%s(32)%v(1)    =  0.50
         g%s(32)%v(2)    =  0.50
         g%s(32)%v(3)    =  0.50
       
         g%s(33)%m(1,2)  = -1.00
         g%s(33)%m(2,3)  = -1.00
         g%s(33)%m(3,1)  = -1.00
         g%s(33)%v(1)    =  0.50
         g%s(33)%v(2)    =  0.50
         g%s(33)%v(3)    =  0.50
       
         g%s(34)%m(1,2)  =  1.00
         g%s(34)%m(2,3)  =  1.00
         g%s(34)%m(3,1)  = -1.00
         g%s(34)%v(1)    =  0.50
         g%s(34)%v(2)    =  0.50
         g%s(34)%v(3)    =  0.50
       
         g%s(35)%m(1,3)  = -1.00
         g%s(35)%m(2,1)  =  1.00
         g%s(35)%m(3,2)  =  1.00
         g%s(35)%v(1)    =  0.50
         g%s(35)%v(2)    =  0.50
         g%s(35)%v(3)    =  0.50
       
         g%s(36)%m(1,2)  =  1.00
         g%s(36)%m(2,3)  = -1.00
         g%s(36)%m(3,1)  =  1.00
         g%s(36)%v(1)    =  0.50
         g%s(36)%v(2)    =  0.50
         g%s(36)%v(3)    =  0.50
       
         g%s(37)%m(1,3)  =  1.00
         g%s(37)%m(2,1)  =  1.00
         g%s(37)%m(3,2)  = -1.00
         g%s(37)%v(1)    =  0.50
         g%s(37)%v(2)    =  0.50
         g%s(37)%v(3)    =  0.50
       
         g%s(38)%m(1,3)  =  1.00
         g%s(38)%m(2,1)  = -1.00
         g%s(38)%m(3,2)  =  1.00
         g%s(38)%v(1)    =  0.50
         g%s(38)%v(2)    =  0.50
         g%s(38)%v(3)    =  0.50
       
         g%s(39)%m(1,2)  = -1.00
         g%s(39)%m(2,3)  =  1.00
         g%s(39)%m(3,1)  =  1.00
         g%s(39)%v(1)    =  0.50
         g%s(39)%v(2)    =  0.50
         g%s(39)%v(3)    =  0.50
       
         g%s(40)%m(1,1)  = -1.00
         g%s(40)%m(2,2)  =  1.00
         g%s(40)%m(3,3)  =  1.00
         g%s(40)%v(1)    =  0.50
         g%s(40)%v(2)    =  0.50
         g%s(40)%v(3)    =  0.50
       
         g%s(41)%m(1,1)  =  1.00
         g%s(41)%m(2,2)  = -1.00
         g%s(41)%m(3,3)  =  1.00
         g%s(41)%v(1)    =  0.50
         g%s(41)%v(2)    =  0.50
         g%s(41)%v(3)    =  0.50
       
         g%s(42)%m(1,1)  =  1.00
         g%s(42)%m(2,2)  =  1.00
         g%s(42)%m(3,3)  = -1.00
         g%s(42)%v(1)    =  0.50
         g%s(42)%v(2)    =  0.50
         g%s(42)%v(3)    =  0.50
       
         g%s(43)%m(1,2)  = -1.00
         g%s(43)%m(2,1)  = -1.00
         g%s(43)%m(3,3)  =  1.00
         g%s(43)%v(1)    =  0.50
         g%s(43)%v(2)    =  0.50
         g%s(43)%v(3)    =  0.50
       
         g%s(44)%m(1,2)  =  1.00
         g%s(44)%m(2,1)  =  1.00
         g%s(44)%m(3,3)  =  1.00
         g%s(44)%v(1)    =  0.50
         g%s(44)%v(2)    =  0.50
         g%s(44)%v(3)    =  0.50
       
         g%s(45)%m(1,3)  = -1.00
         g%s(45)%m(2,2)  =  1.00
         g%s(45)%m(3,1)  = -1.00
         g%s(45)%v(1)    =  0.50
         g%s(45)%v(2)    =  0.50
         g%s(45)%v(3)    =  0.50
       
         g%s(46)%m(1,3)  =  1.00
         g%s(46)%m(2,2)  =  1.00
         g%s(46)%m(3,1)  =  1.00
         g%s(46)%v(1)    =  0.50
         g%s(46)%v(2)    =  0.50
         g%s(46)%v(3)    =  0.50
       
         g%s(47)%m(1,1)  =  1.00
         g%s(47)%m(2,3)  = -1.00
         g%s(47)%m(3,2)  = -1.00
         g%s(47)%v(1)    =  0.50
         g%s(47)%v(2)    =  0.50
         g%s(47)%v(3)    =  0.50
       
         g%s(48)%m(1,1)  =  1.00
         g%s(48)%m(2,3)  =  1.00
         g%s(48)%m(3,2)  =  1.00
         g%s(48)%v(1)    =  0.50
         g%s(48)%v(2)    =  0.50
         g%s(48)%v(3)    =  0.50
       
       
      case('P m -3 n', '223')                        
       
         !Space group   # 223
         !Group setting # 251
       
         g%order   =  48
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,3)   = -1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,2)   = -1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,3)   =  1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%m(3,1)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,3)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,1)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,3)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   =  1.00
         g%s(9)%m(3,1)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,3)  = -1.00
         g%s(10)%m(3,1)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,2)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,1)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,3)  = -1.00
         g%s(13)%m(2,1)  = -1.00
         g%s(13)%m(3,2)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,2)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,3)  = -1.00
         g%s(19)%v(1)    =  0.50
         g%s(19)%v(2)    =  0.50
         g%s(19)%v(3)    =  0.50
       
         g%s(20)%m(1,2)  = -1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.50
         g%s(20)%v(2)    =  0.50
         g%s(20)%v(3)    =  0.50
       
         g%s(21)%m(1,3)  =  1.00
         g%s(21)%m(2,2)  = -1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.50
         g%s(21)%v(2)    =  0.50
         g%s(21)%v(3)    =  0.50
       
         g%s(22)%m(1,3)  = -1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,1)  = -1.00
         g%s(22)%v(1)    =  0.50
         g%s(22)%v(2)    =  0.50
         g%s(22)%v(3)    =  0.50
       
         g%s(23)%m(1,1)  = -1.00
         g%s(23)%m(2,3)  =  1.00
         g%s(23)%m(3,2)  =  1.00
         g%s(23)%v(1)    =  0.50
         g%s(23)%v(2)    =  0.50
         g%s(23)%v(3)    =  0.50
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,3)  = -1.00
         g%s(24)%m(3,2)  = -1.00
         g%s(24)%v(1)    =  0.50
         g%s(24)%v(2)    =  0.50
         g%s(24)%v(3)    =  0.50
       
         g%s(25)%m(1,1)  = -1.00
         g%s(25)%m(2,2)  = -1.00
         g%s(25)%m(3,3)  = -1.00
         g%s(25)%v(1)    =  0.00
         g%s(25)%v(2)    =  0.00
         g%s(25)%v(3)    =  0.00
       
         g%s(26)%m(1,1)  = -1.00
         g%s(26)%m(2,3)  =  1.00
         g%s(26)%m(3,2)  = -1.00
         g%s(26)%v(1)    = -0.50
         g%s(26)%v(2)    = -0.50
         g%s(26)%v(3)    = -0.50
       
         g%s(27)%m(1,1)  = -1.00
         g%s(27)%m(2,3)  = -1.00
         g%s(27)%m(3,2)  =  1.00
         g%s(27)%v(1)    = -0.50
         g%s(27)%v(2)    = -0.50
         g%s(27)%v(3)    = -0.50
       
         g%s(28)%m(1,3)  = -1.00
         g%s(28)%m(2,2)  = -1.00
         g%s(28)%m(3,1)  =  1.00
         g%s(28)%v(1)    = -0.50
         g%s(28)%v(2)    = -0.50
         g%s(28)%v(3)    = -0.50
       
         g%s(29)%m(1,3)  =  1.00
         g%s(29)%m(2,2)  = -1.00
         g%s(29)%m(3,1)  = -1.00
         g%s(29)%v(1)    = -0.50
         g%s(29)%v(2)    = -0.50
         g%s(29)%v(3)    = -0.50
       
         g%s(30)%m(1,2)  =  1.00
         g%s(30)%m(2,1)  = -1.00
         g%s(30)%m(3,3)  = -1.00
         g%s(30)%v(1)    = -0.50
         g%s(30)%v(2)    = -0.50
         g%s(30)%v(3)    = -0.50
       
         g%s(31)%m(1,2)  = -1.00
         g%s(31)%m(2,1)  =  1.00
         g%s(31)%m(3,3)  = -1.00
         g%s(31)%v(1)    = -0.50
         g%s(31)%v(2)    = -0.50
         g%s(31)%v(3)    = -0.50
       
         g%s(32)%m(1,3)  = -1.00
         g%s(32)%m(2,1)  = -1.00
         g%s(32)%m(3,2)  = -1.00
         g%s(32)%v(1)    =  0.00
         g%s(32)%v(2)    =  0.00
         g%s(32)%v(3)    =  0.00
       
         g%s(33)%m(1,2)  = -1.00
         g%s(33)%m(2,3)  = -1.00
         g%s(33)%m(3,1)  = -1.00
         g%s(33)%v(1)    =  0.00
         g%s(33)%v(2)    =  0.00
         g%s(33)%v(3)    =  0.00
       
         g%s(34)%m(1,2)  =  1.00
         g%s(34)%m(2,3)  =  1.00
         g%s(34)%m(3,1)  = -1.00
         g%s(34)%v(1)    =  0.00
         g%s(34)%v(2)    =  0.00
         g%s(34)%v(3)    =  0.00
       
         g%s(35)%m(1,3)  = -1.00
         g%s(35)%m(2,1)  =  1.00
         g%s(35)%m(3,2)  =  1.00
         g%s(35)%v(1)    =  0.00
         g%s(35)%v(2)    =  0.00
         g%s(35)%v(3)    =  0.00
       
         g%s(36)%m(1,2)  =  1.00
         g%s(36)%m(2,3)  = -1.00
         g%s(36)%m(3,1)  =  1.00
         g%s(36)%v(1)    =  0.00
         g%s(36)%v(2)    =  0.00
         g%s(36)%v(3)    =  0.00
       
         g%s(37)%m(1,3)  =  1.00
         g%s(37)%m(2,1)  =  1.00
         g%s(37)%m(3,2)  = -1.00
         g%s(37)%v(1)    =  0.00
         g%s(37)%v(2)    =  0.00
         g%s(37)%v(3)    =  0.00
       
         g%s(38)%m(1,3)  =  1.00
         g%s(38)%m(2,1)  = -1.00
         g%s(38)%m(3,2)  =  1.00
         g%s(38)%v(1)    =  0.00
         g%s(38)%v(2)    =  0.00
         g%s(38)%v(3)    =  0.00
       
         g%s(39)%m(1,2)  = -1.00
         g%s(39)%m(2,3)  =  1.00
         g%s(39)%m(3,1)  =  1.00
         g%s(39)%v(1)    =  0.00
         g%s(39)%v(2)    =  0.00
         g%s(39)%v(3)    =  0.00
       
         g%s(40)%m(1,1)  = -1.00
         g%s(40)%m(2,2)  =  1.00
         g%s(40)%m(3,3)  =  1.00
         g%s(40)%v(1)    =  0.00
         g%s(40)%v(2)    =  0.00
         g%s(40)%v(3)    =  0.00
       
         g%s(41)%m(1,1)  =  1.00
         g%s(41)%m(2,2)  = -1.00
         g%s(41)%m(3,3)  =  1.00
         g%s(41)%v(1)    =  0.00
         g%s(41)%v(2)    =  0.00
         g%s(41)%v(3)    =  0.00
       
         g%s(42)%m(1,1)  =  1.00
         g%s(42)%m(2,2)  =  1.00
         g%s(42)%m(3,3)  = -1.00
         g%s(42)%v(1)    =  0.00
         g%s(42)%v(2)    =  0.00
         g%s(42)%v(3)    =  0.00
       
         g%s(43)%m(1,2)  = -1.00
         g%s(43)%m(2,1)  = -1.00
         g%s(43)%m(3,3)  =  1.00
         g%s(43)%v(1)    = -0.50
         g%s(43)%v(2)    = -0.50
         g%s(43)%v(3)    = -0.50
       
         g%s(44)%m(1,2)  =  1.00
         g%s(44)%m(2,1)  =  1.00
         g%s(44)%m(3,3)  =  1.00
         g%s(44)%v(1)    = -0.50
         g%s(44)%v(2)    = -0.50
         g%s(44)%v(3)    = -0.50
       
         g%s(45)%m(1,3)  = -1.00
         g%s(45)%m(2,2)  =  1.00
         g%s(45)%m(3,1)  = -1.00
         g%s(45)%v(1)    = -0.50
         g%s(45)%v(2)    = -0.50
         g%s(45)%v(3)    = -0.50
       
         g%s(46)%m(1,3)  =  1.00
         g%s(46)%m(2,2)  =  1.00
         g%s(46)%m(3,1)  =  1.00
         g%s(46)%v(1)    = -0.50
         g%s(46)%v(2)    = -0.50
         g%s(46)%v(3)    = -0.50
       
         g%s(47)%m(1,1)  =  1.00
         g%s(47)%m(2,3)  = -1.00
         g%s(47)%m(3,2)  = -1.00
         g%s(47)%v(1)    = -0.50
         g%s(47)%v(2)    = -0.50
         g%s(47)%v(3)    = -0.50
       
         g%s(48)%m(1,1)  =  1.00
         g%s(48)%m(2,3)  =  1.00
         g%s(48)%m(3,2)  =  1.00
         g%s(48)%v(1)    = -0.50
         g%s(48)%v(2)    = -0.50
         g%s(48)%v(3)    = -0.50
       
       
      case('P n -3 m : 2', '224')                    
       
         !Space group   # 224
         !Group setting # 252
       
         g%order   =  48
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,3)   = -1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,2)   = -1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,3)   =  1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%m(3,1)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,1)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,3)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   =  1.00
         g%s(9)%m(3,1)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,3)  = -1.00
         g%s(10)%m(3,1)  =  1.00
         g%s(10)%v(1)    =  0.50
         g%s(10)%v(2)    =  0.50
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,2)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.50
         g%s(11)%v(3)    =  0.50
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,1)  = -1.00
         g%s(12)%v(1)    =  0.50
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.50
       
         g%s(13)%m(1,3)  = -1.00
         g%s(13)%m(2,1)  = -1.00
         g%s(13)%m(3,2)  =  1.00
         g%s(13)%v(1)    =  0.50
         g%s(13)%v(2)    =  0.50
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.50
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.50
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.50
         g%s(15)%v(3)    =  0.50
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.50
         g%s(16)%v(3)    =  0.50
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.50
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.50
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.50
         g%s(18)%v(2)    =  0.50
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,2)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,3)  = -1.00
         g%s(19)%v(1)    =  0.50
         g%s(19)%v(2)    =  0.50
         g%s(19)%v(3)    =  0.00
       
         g%s(20)%m(1,2)  = -1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.00
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    =  0.00
       
         g%s(21)%m(1,3)  =  1.00
         g%s(21)%m(2,2)  = -1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.50
         g%s(21)%v(2)    =  0.00
         g%s(21)%v(3)    =  0.50
       
         g%s(22)%m(1,3)  = -1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,1)  = -1.00
         g%s(22)%v(1)    =  0.00
         g%s(22)%v(2)    =  0.00
         g%s(22)%v(3)    =  0.00
       
         g%s(23)%m(1,1)  = -1.00
         g%s(23)%m(2,3)  =  1.00
         g%s(23)%m(3,2)  =  1.00
         g%s(23)%v(1)    =  0.00
         g%s(23)%v(2)    =  0.50
         g%s(23)%v(3)    =  0.50
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,3)  = -1.00
         g%s(24)%m(3,2)  = -1.00
         g%s(24)%v(1)    =  0.00
         g%s(24)%v(2)    =  0.00
         g%s(24)%v(3)    =  0.00
       
         g%s(25)%m(1,1)  = -1.00
         g%s(25)%m(2,2)  = -1.00
         g%s(25)%m(3,3)  = -1.00
         g%s(25)%v(1)    =  0.00
         g%s(25)%v(2)    =  0.00
         g%s(25)%v(3)    =  0.00
       
         g%s(26)%m(1,1)  = -1.00
         g%s(26)%m(2,3)  =  1.00
         g%s(26)%m(3,2)  = -1.00
         g%s(26)%v(1)    = -0.50
         g%s(26)%v(2)    =  0.00
         g%s(26)%v(3)    = -0.50
       
         g%s(27)%m(1,1)  = -1.00
         g%s(27)%m(2,3)  = -1.00
         g%s(27)%m(3,2)  =  1.00
         g%s(27)%v(1)    = -0.50
         g%s(27)%v(2)    = -0.50
         g%s(27)%v(3)    =  0.00
       
         g%s(28)%m(1,3)  = -1.00
         g%s(28)%m(2,2)  = -1.00
         g%s(28)%m(3,1)  =  1.00
         g%s(28)%v(1)    = -0.50
         g%s(28)%v(2)    = -0.50
         g%s(28)%v(3)    =  0.00
       
         g%s(29)%m(1,3)  =  1.00
         g%s(29)%m(2,2)  = -1.00
         g%s(29)%m(3,1)  = -1.00
         g%s(29)%v(1)    =  0.00
         g%s(29)%v(2)    = -0.50
         g%s(29)%v(3)    = -0.50
       
         g%s(30)%m(1,2)  =  1.00
         g%s(30)%m(2,1)  = -1.00
         g%s(30)%m(3,3)  = -1.00
         g%s(30)%v(1)    =  0.00
         g%s(30)%v(2)    = -0.50
         g%s(30)%v(3)    = -0.50
       
         g%s(31)%m(1,2)  = -1.00
         g%s(31)%m(2,1)  =  1.00
         g%s(31)%m(3,3)  = -1.00
         g%s(31)%v(1)    = -0.50
         g%s(31)%v(2)    =  0.00
         g%s(31)%v(3)    = -0.50
       
         g%s(32)%m(1,3)  = -1.00
         g%s(32)%m(2,1)  = -1.00
         g%s(32)%m(3,2)  = -1.00
         g%s(32)%v(1)    =  0.00
         g%s(32)%v(2)    =  0.00
         g%s(32)%v(3)    =  0.00
       
         g%s(33)%m(1,2)  = -1.00
         g%s(33)%m(2,3)  = -1.00
         g%s(33)%m(3,1)  = -1.00
         g%s(33)%v(1)    =  0.00
         g%s(33)%v(2)    =  0.00
         g%s(33)%v(3)    =  0.00
       
         g%s(34)%m(1,2)  =  1.00
         g%s(34)%m(2,3)  =  1.00
         g%s(34)%m(3,1)  = -1.00
         g%s(34)%v(1)    = -0.50
         g%s(34)%v(2)    = -0.50
         g%s(34)%v(3)    =  0.00
       
         g%s(35)%m(1,3)  = -1.00
         g%s(35)%m(2,1)  =  1.00
         g%s(35)%m(3,2)  =  1.00
         g%s(35)%v(1)    =  0.00
         g%s(35)%v(2)    = -0.50
         g%s(35)%v(3)    = -0.50
       
         g%s(36)%m(1,2)  =  1.00
         g%s(36)%m(2,3)  = -1.00
         g%s(36)%m(3,1)  =  1.00
         g%s(36)%v(1)    = -0.50
         g%s(36)%v(2)    =  0.00
         g%s(36)%v(3)    = -0.50
       
         g%s(37)%m(1,3)  =  1.00
         g%s(37)%m(2,1)  =  1.00
         g%s(37)%m(3,2)  = -1.00
         g%s(37)%v(1)    = -0.50
         g%s(37)%v(2)    = -0.50
         g%s(37)%v(3)    =  0.00
       
         g%s(38)%m(1,3)  =  1.00
         g%s(38)%m(2,1)  = -1.00
         g%s(38)%m(3,2)  =  1.00
         g%s(38)%v(1)    = -0.50
         g%s(38)%v(2)    =  0.00
         g%s(38)%v(3)    = -0.50
       
         g%s(39)%m(1,2)  = -1.00
         g%s(39)%m(2,3)  =  1.00
         g%s(39)%m(3,1)  =  1.00
         g%s(39)%v(1)    =  0.00
         g%s(39)%v(2)    = -0.50
         g%s(39)%v(3)    = -0.50
       
         g%s(40)%m(1,1)  = -1.00
         g%s(40)%m(2,2)  =  1.00
         g%s(40)%m(3,3)  =  1.00
         g%s(40)%v(1)    =  0.00
         g%s(40)%v(2)    = -0.50
         g%s(40)%v(3)    = -0.50
       
         g%s(41)%m(1,1)  =  1.00
         g%s(41)%m(2,2)  = -1.00
         g%s(41)%m(3,3)  =  1.00
         g%s(41)%v(1)    = -0.50
         g%s(41)%v(2)    =  0.00
         g%s(41)%v(3)    = -0.50
       
         g%s(42)%m(1,1)  =  1.00
         g%s(42)%m(2,2)  =  1.00
         g%s(42)%m(3,3)  = -1.00
         g%s(42)%v(1)    = -0.50
         g%s(42)%v(2)    = -0.50
         g%s(42)%v(3)    =  0.00
       
         g%s(43)%m(1,2)  = -1.00
         g%s(43)%m(2,1)  = -1.00
         g%s(43)%m(3,3)  =  1.00
         g%s(43)%v(1)    = -0.50
         g%s(43)%v(2)    = -0.50
         g%s(43)%v(3)    =  0.00
       
         g%s(44)%m(1,2)  =  1.00
         g%s(44)%m(2,1)  =  1.00
         g%s(44)%m(3,3)  =  1.00
         g%s(44)%v(1)    =  0.00
         g%s(44)%v(2)    =  0.00
         g%s(44)%v(3)    =  0.00
       
         g%s(45)%m(1,3)  = -1.00
         g%s(45)%m(2,2)  =  1.00
         g%s(45)%m(3,1)  = -1.00
         g%s(45)%v(1)    = -0.50
         g%s(45)%v(2)    =  0.00
         g%s(45)%v(3)    = -0.50
       
         g%s(46)%m(1,3)  =  1.00
         g%s(46)%m(2,2)  =  1.00
         g%s(46)%m(3,1)  =  1.00
         g%s(46)%v(1)    =  0.00
         g%s(46)%v(2)    =  0.00
         g%s(46)%v(3)    =  0.00
       
         g%s(47)%m(1,1)  =  1.00
         g%s(47)%m(2,3)  = -1.00
         g%s(47)%m(3,2)  = -1.00
         g%s(47)%v(1)    =  0.00
         g%s(47)%v(2)    = -0.50
         g%s(47)%v(3)    = -0.50
       
         g%s(48)%m(1,1)  =  1.00
         g%s(48)%m(2,3)  =  1.00
         g%s(48)%m(3,2)  =  1.00
         g%s(48)%v(1)    =  0.00
         g%s(48)%v(2)    =  0.00
         g%s(48)%v(3)    =  0.00
       
       
      case('P n -3 m : 1')                    
       
         !Space group   # 224
         !Group setting # 253
       
         g%order   =  48
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,3)   = -1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.50
         g%s(2)%v(3)     =  0.50
       
         g%s(3)%m(1,1)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,2)   = -1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.50
         g%s(3)%v(3)     =  0.50
       
         g%s(4)%m(1,3)   =  1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%m(3,1)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.50
         g%s(4)%v(3)     =  0.50
       
         g%s(5)%m(1,3)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,1)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.50
         g%s(5)%v(3)     =  0.50
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.50
         g%s(6)%v(3)     =  0.50
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.50
         g%s(7)%v(3)     =  0.50
       
         g%s(8)%m(1,3)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   =  1.00
         g%s(9)%m(3,1)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,3)  = -1.00
         g%s(10)%m(3,1)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,2)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,1)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,3)  = -1.00
         g%s(13)%m(2,1)  = -1.00
         g%s(13)%m(3,2)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,2)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,3)  = -1.00
         g%s(19)%v(1)    =  0.50
         g%s(19)%v(2)    =  0.50
         g%s(19)%v(3)    =  0.50
       
         g%s(20)%m(1,2)  = -1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.50
         g%s(20)%v(2)    =  0.50
         g%s(20)%v(3)    =  0.50
       
         g%s(21)%m(1,3)  =  1.00
         g%s(21)%m(2,2)  = -1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.50
         g%s(21)%v(2)    =  0.50
         g%s(21)%v(3)    =  0.50
       
         g%s(22)%m(1,3)  = -1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,1)  = -1.00
         g%s(22)%v(1)    =  0.50
         g%s(22)%v(2)    =  0.50
         g%s(22)%v(3)    =  0.50
       
         g%s(23)%m(1,1)  = -1.00
         g%s(23)%m(2,3)  =  1.00
         g%s(23)%m(3,2)  =  1.00
         g%s(23)%v(1)    =  0.50
         g%s(23)%v(2)    =  0.50
         g%s(23)%v(3)    =  0.50
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,3)  = -1.00
         g%s(24)%m(3,2)  = -1.00
         g%s(24)%v(1)    =  0.50
         g%s(24)%v(2)    =  0.50
         g%s(24)%v(3)    =  0.50
       
         g%s(25)%m(1,1)  = -1.00
         g%s(25)%m(2,2)  = -1.00
         g%s(25)%m(3,3)  = -1.00
         g%s(25)%v(1)    =  0.50
         g%s(25)%v(2)    =  0.50
         g%s(25)%v(3)    =  0.50
       
         g%s(26)%m(1,1)  = -1.00
         g%s(26)%m(2,3)  =  1.00
         g%s(26)%m(3,2)  = -1.00
         g%s(26)%v(1)    =  0.00
         g%s(26)%v(2)    =  0.00
         g%s(26)%v(3)    =  0.00
       
         g%s(27)%m(1,1)  = -1.00
         g%s(27)%m(2,3)  = -1.00
         g%s(27)%m(3,2)  =  1.00
         g%s(27)%v(1)    =  0.00
         g%s(27)%v(2)    =  0.00
         g%s(27)%v(3)    =  0.00
       
         g%s(28)%m(1,3)  = -1.00
         g%s(28)%m(2,2)  = -1.00
         g%s(28)%m(3,1)  =  1.00
         g%s(28)%v(1)    =  0.00
         g%s(28)%v(2)    =  0.00
         g%s(28)%v(3)    =  0.00
       
         g%s(29)%m(1,3)  =  1.00
         g%s(29)%m(2,2)  = -1.00
         g%s(29)%m(3,1)  = -1.00
         g%s(29)%v(1)    =  0.00
         g%s(29)%v(2)    =  0.00
         g%s(29)%v(3)    =  0.00
       
         g%s(30)%m(1,2)  =  1.00
         g%s(30)%m(2,1)  = -1.00
         g%s(30)%m(3,3)  = -1.00
         g%s(30)%v(1)    =  0.00
         g%s(30)%v(2)    =  0.00
         g%s(30)%v(3)    =  0.00
       
         g%s(31)%m(1,2)  = -1.00
         g%s(31)%m(2,1)  =  1.00
         g%s(31)%m(3,3)  = -1.00
         g%s(31)%v(1)    =  0.00
         g%s(31)%v(2)    =  0.00
         g%s(31)%v(3)    =  0.00
       
         g%s(32)%m(1,3)  = -1.00
         g%s(32)%m(2,1)  = -1.00
         g%s(32)%m(3,2)  = -1.00
         g%s(32)%v(1)    =  0.50
         g%s(32)%v(2)    =  0.50
         g%s(32)%v(3)    =  0.50
       
         g%s(33)%m(1,2)  = -1.00
         g%s(33)%m(2,3)  = -1.00
         g%s(33)%m(3,1)  = -1.00
         g%s(33)%v(1)    =  0.50
         g%s(33)%v(2)    =  0.50
         g%s(33)%v(3)    =  0.50
       
         g%s(34)%m(1,2)  =  1.00
         g%s(34)%m(2,3)  =  1.00
         g%s(34)%m(3,1)  = -1.00
         g%s(34)%v(1)    =  0.50
         g%s(34)%v(2)    =  0.50
         g%s(34)%v(3)    =  0.50
       
         g%s(35)%m(1,3)  = -1.00
         g%s(35)%m(2,1)  =  1.00
         g%s(35)%m(3,2)  =  1.00
         g%s(35)%v(1)    =  0.50
         g%s(35)%v(2)    =  0.50
         g%s(35)%v(3)    =  0.50
       
         g%s(36)%m(1,2)  =  1.00
         g%s(36)%m(2,3)  = -1.00
         g%s(36)%m(3,1)  =  1.00
         g%s(36)%v(1)    =  0.50
         g%s(36)%v(2)    =  0.50
         g%s(36)%v(3)    =  0.50
       
         g%s(37)%m(1,3)  =  1.00
         g%s(37)%m(2,1)  =  1.00
         g%s(37)%m(3,2)  = -1.00
         g%s(37)%v(1)    =  0.50
         g%s(37)%v(2)    =  0.50
         g%s(37)%v(3)    =  0.50
       
         g%s(38)%m(1,3)  =  1.00
         g%s(38)%m(2,1)  = -1.00
         g%s(38)%m(3,2)  =  1.00
         g%s(38)%v(1)    =  0.50
         g%s(38)%v(2)    =  0.50
         g%s(38)%v(3)    =  0.50
       
         g%s(39)%m(1,2)  = -1.00
         g%s(39)%m(2,3)  =  1.00
         g%s(39)%m(3,1)  =  1.00
         g%s(39)%v(1)    =  0.50
         g%s(39)%v(2)    =  0.50
         g%s(39)%v(3)    =  0.50
       
         g%s(40)%m(1,1)  = -1.00
         g%s(40)%m(2,2)  =  1.00
         g%s(40)%m(3,3)  =  1.00
         g%s(40)%v(1)    =  0.50
         g%s(40)%v(2)    =  0.50
         g%s(40)%v(3)    =  0.50
       
         g%s(41)%m(1,1)  =  1.00
         g%s(41)%m(2,2)  = -1.00
         g%s(41)%m(3,3)  =  1.00
         g%s(41)%v(1)    =  0.50
         g%s(41)%v(2)    =  0.50
         g%s(41)%v(3)    =  0.50
       
         g%s(42)%m(1,1)  =  1.00
         g%s(42)%m(2,2)  =  1.00
         g%s(42)%m(3,3)  = -1.00
         g%s(42)%v(1)    =  0.50
         g%s(42)%v(2)    =  0.50
         g%s(42)%v(3)    =  0.50
       
         g%s(43)%m(1,2)  = -1.00
         g%s(43)%m(2,1)  = -1.00
         g%s(43)%m(3,3)  =  1.00
         g%s(43)%v(1)    =  0.00
         g%s(43)%v(2)    =  0.00
         g%s(43)%v(3)    =  0.00
       
         g%s(44)%m(1,2)  =  1.00
         g%s(44)%m(2,1)  =  1.00
         g%s(44)%m(3,3)  =  1.00
         g%s(44)%v(1)    =  0.00
         g%s(44)%v(2)    =  0.00
         g%s(44)%v(3)    =  0.00
       
         g%s(45)%m(1,3)  = -1.00
         g%s(45)%m(2,2)  =  1.00
         g%s(45)%m(3,1)  = -1.00
         g%s(45)%v(1)    =  0.00
         g%s(45)%v(2)    =  0.00
         g%s(45)%v(3)    =  0.00
       
         g%s(46)%m(1,3)  =  1.00
         g%s(46)%m(2,2)  =  1.00
         g%s(46)%m(3,1)  =  1.00
         g%s(46)%v(1)    =  0.00
         g%s(46)%v(2)    =  0.00
         g%s(46)%v(3)    =  0.00
       
         g%s(47)%m(1,1)  =  1.00
         g%s(47)%m(2,3)  = -1.00
         g%s(47)%m(3,2)  = -1.00
         g%s(47)%v(1)    =  0.00
         g%s(47)%v(2)    =  0.00
         g%s(47)%v(3)    =  0.00
       
         g%s(48)%m(1,1)  =  1.00
         g%s(48)%m(2,3)  =  1.00
         g%s(48)%m(3,2)  =  1.00
         g%s(48)%v(1)    =  0.00
         g%s(48)%v(2)    =  0.00
         g%s(48)%v(3)    =  0.00
       
       
      case('F m -3 m', '225')                        
       
         !Space group   # 225
         !Group setting # 254
       
         g%order   = 192
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,3)   = -1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.00
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,2)   = -1.00
         g%s(3)%v(1)     =  0.00
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,3)   =  1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%m(3,1)   = -1.00
         g%s(4)%v(1)     =  0.00
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,1)   =  1.00
         g%s(5)%v(1)     =  0.00
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.00
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.00
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   =  1.00
         g%s(9)%m(3,1)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,3)  = -1.00
         g%s(10)%m(3,1)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,2)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,1)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,3)  = -1.00
         g%s(13)%m(2,1)  = -1.00
         g%s(13)%m(3,2)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,2)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,3)  = -1.00
         g%s(19)%v(1)    =  0.00
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    =  0.00
       
         g%s(20)%m(1,2)  = -1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.00
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    =  0.00
       
         g%s(21)%m(1,3)  =  1.00
         g%s(21)%m(2,2)  = -1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.00
         g%s(21)%v(2)    =  0.00
         g%s(21)%v(3)    =  0.00
       
         g%s(22)%m(1,3)  = -1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,1)  = -1.00
         g%s(22)%v(1)    =  0.00
         g%s(22)%v(2)    =  0.00
         g%s(22)%v(3)    =  0.00
       
         g%s(23)%m(1,1)  = -1.00
         g%s(23)%m(2,3)  =  1.00
         g%s(23)%m(3,2)  =  1.00
         g%s(23)%v(1)    =  0.00
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    =  0.00
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,3)  = -1.00
         g%s(24)%m(3,2)  = -1.00
         g%s(24)%v(1)    =  0.00
         g%s(24)%v(2)    =  0.00
         g%s(24)%v(3)    =  0.00
       
         g%s(25)%m(1,1)  = -1.00
         g%s(25)%m(2,2)  = -1.00
         g%s(25)%m(3,3)  = -1.00
         g%s(25)%v(1)    =  0.00
         g%s(25)%v(2)    =  0.00
         g%s(25)%v(3)    =  0.00
       
         g%s(26)%m(1,1)  = -1.00
         g%s(26)%m(2,3)  =  1.00
         g%s(26)%m(3,2)  = -1.00
         g%s(26)%v(1)    =  0.00
         g%s(26)%v(2)    =  0.00
         g%s(26)%v(3)    =  0.00
       
         g%s(27)%m(1,1)  = -1.00
         g%s(27)%m(2,3)  = -1.00
         g%s(27)%m(3,2)  =  1.00
         g%s(27)%v(1)    =  0.00
         g%s(27)%v(2)    =  0.00
         g%s(27)%v(3)    =  0.00
       
         g%s(28)%m(1,3)  = -1.00
         g%s(28)%m(2,2)  = -1.00
         g%s(28)%m(3,1)  =  1.00
         g%s(28)%v(1)    =  0.00
         g%s(28)%v(2)    =  0.00
         g%s(28)%v(3)    =  0.00
       
         g%s(29)%m(1,3)  =  1.00
         g%s(29)%m(2,2)  = -1.00
         g%s(29)%m(3,1)  = -1.00
         g%s(29)%v(1)    =  0.00
         g%s(29)%v(2)    =  0.00
         g%s(29)%v(3)    =  0.00
       
         g%s(30)%m(1,2)  =  1.00
         g%s(30)%m(2,1)  = -1.00
         g%s(30)%m(3,3)  = -1.00
         g%s(30)%v(1)    =  0.00
         g%s(30)%v(2)    =  0.00
         g%s(30)%v(3)    =  0.00
       
         g%s(31)%m(1,2)  = -1.00
         g%s(31)%m(2,1)  =  1.00
         g%s(31)%m(3,3)  = -1.00
         g%s(31)%v(1)    =  0.00
         g%s(31)%v(2)    =  0.00
         g%s(31)%v(3)    =  0.00
       
         g%s(32)%m(1,3)  = -1.00
         g%s(32)%m(2,1)  = -1.00
         g%s(32)%m(3,2)  = -1.00
         g%s(32)%v(1)    =  0.00
         g%s(32)%v(2)    =  0.00
         g%s(32)%v(3)    =  0.00
       
         g%s(33)%m(1,2)  = -1.00
         g%s(33)%m(2,3)  = -1.00
         g%s(33)%m(3,1)  = -1.00
         g%s(33)%v(1)    =  0.00
         g%s(33)%v(2)    =  0.00
         g%s(33)%v(3)    =  0.00
       
         g%s(34)%m(1,2)  =  1.00
         g%s(34)%m(2,3)  =  1.00
         g%s(34)%m(3,1)  = -1.00
         g%s(34)%v(1)    =  0.00
         g%s(34)%v(2)    =  0.00
         g%s(34)%v(3)    =  0.00
       
         g%s(35)%m(1,3)  = -1.00
         g%s(35)%m(2,1)  =  1.00
         g%s(35)%m(3,2)  =  1.00
         g%s(35)%v(1)    =  0.00
         g%s(35)%v(2)    =  0.00
         g%s(35)%v(3)    =  0.00
       
         g%s(36)%m(1,2)  =  1.00
         g%s(36)%m(2,3)  = -1.00
         g%s(36)%m(3,1)  =  1.00
         g%s(36)%v(1)    =  0.00
         g%s(36)%v(2)    =  0.00
         g%s(36)%v(3)    =  0.00
       
         g%s(37)%m(1,3)  =  1.00
         g%s(37)%m(2,1)  =  1.00
         g%s(37)%m(3,2)  = -1.00
         g%s(37)%v(1)    =  0.00
         g%s(37)%v(2)    =  0.00
         g%s(37)%v(3)    =  0.00
       
         g%s(38)%m(1,3)  =  1.00
         g%s(38)%m(2,1)  = -1.00
         g%s(38)%m(3,2)  =  1.00
         g%s(38)%v(1)    =  0.00
         g%s(38)%v(2)    =  0.00
         g%s(38)%v(3)    =  0.00
       
         g%s(39)%m(1,2)  = -1.00
         g%s(39)%m(2,3)  =  1.00
         g%s(39)%m(3,1)  =  1.00
         g%s(39)%v(1)    =  0.00
         g%s(39)%v(2)    =  0.00
         g%s(39)%v(3)    =  0.00
       
         g%s(40)%m(1,1)  = -1.00
         g%s(40)%m(2,2)  =  1.00
         g%s(40)%m(3,3)  =  1.00
         g%s(40)%v(1)    =  0.00
         g%s(40)%v(2)    =  0.00
         g%s(40)%v(3)    =  0.00
       
         g%s(41)%m(1,1)  =  1.00
         g%s(41)%m(2,2)  = -1.00
         g%s(41)%m(3,3)  =  1.00
         g%s(41)%v(1)    =  0.00
         g%s(41)%v(2)    =  0.00
         g%s(41)%v(3)    =  0.00
       
         g%s(42)%m(1,1)  =  1.00
         g%s(42)%m(2,2)  =  1.00
         g%s(42)%m(3,3)  = -1.00
         g%s(42)%v(1)    =  0.00
         g%s(42)%v(2)    =  0.00
         g%s(42)%v(3)    =  0.00
       
         g%s(43)%m(1,2)  = -1.00
         g%s(43)%m(2,1)  = -1.00
         g%s(43)%m(3,3)  =  1.00
         g%s(43)%v(1)    =  0.00
         g%s(43)%v(2)    =  0.00
         g%s(43)%v(3)    =  0.00
       
         g%s(44)%m(1,2)  =  1.00
         g%s(44)%m(2,1)  =  1.00
         g%s(44)%m(3,3)  =  1.00
         g%s(44)%v(1)    =  0.00
         g%s(44)%v(2)    =  0.00
         g%s(44)%v(3)    =  0.00
       
         g%s(45)%m(1,3)  = -1.00
         g%s(45)%m(2,2)  =  1.00
         g%s(45)%m(3,1)  = -1.00
         g%s(45)%v(1)    =  0.00
         g%s(45)%v(2)    =  0.00
         g%s(45)%v(3)    =  0.00
       
         g%s(46)%m(1,3)  =  1.00
         g%s(46)%m(2,2)  =  1.00
         g%s(46)%m(3,1)  =  1.00
         g%s(46)%v(1)    =  0.00
         g%s(46)%v(2)    =  0.00
         g%s(46)%v(3)    =  0.00
       
         g%s(47)%m(1,1)  =  1.00
         g%s(47)%m(2,3)  = -1.00
         g%s(47)%m(3,2)  = -1.00
         g%s(47)%v(1)    =  0.00
         g%s(47)%v(2)    =  0.00
         g%s(47)%v(3)    =  0.00
       
         g%s(48)%m(1,1)  =  1.00
         g%s(48)%m(2,3)  =  1.00
         g%s(48)%m(3,2)  =  1.00
         g%s(48)%v(1)    =  0.00
         g%s(48)%v(2)    =  0.00
         g%s(48)%v(3)    =  0.00
       
         g%s(49)%m(1,1)  =  1.00
         g%s(49)%m(2,2)  =  1.00
         g%s(49)%m(3,3)  =  1.00
         g%s(49)%v(1)    =  0.00
         g%s(49)%v(2)    =  0.50
         g%s(49)%v(3)    =  0.50
       
         g%s(50)%m(1,1)  =  1.00
         g%s(50)%m(2,3)  = -1.00
         g%s(50)%m(3,2)  =  1.00
         g%s(50)%v(1)    =  0.00
         g%s(50)%v(2)    =  0.50
         g%s(50)%v(3)    =  0.50
       
         g%s(51)%m(1,1)  =  1.00
         g%s(51)%m(2,3)  =  1.00
         g%s(51)%m(3,2)  = -1.00
         g%s(51)%v(1)    =  0.00
         g%s(51)%v(2)    =  0.50
         g%s(51)%v(3)    =  0.50
       
         g%s(52)%m(1,3)  =  1.00
         g%s(52)%m(2,2)  =  1.00
         g%s(52)%m(3,1)  = -1.00
         g%s(52)%v(1)    =  0.00
         g%s(52)%v(2)    =  0.50
         g%s(52)%v(3)    =  0.50
       
         g%s(53)%m(1,3)  = -1.00
         g%s(53)%m(2,2)  =  1.00
         g%s(53)%m(3,1)  =  1.00
         g%s(53)%v(1)    =  0.00
         g%s(53)%v(2)    =  0.50
         g%s(53)%v(3)    =  0.50
       
         g%s(54)%m(1,2)  = -1.00
         g%s(54)%m(2,1)  =  1.00
         g%s(54)%m(3,3)  =  1.00
         g%s(54)%v(1)    =  0.00
         g%s(54)%v(2)    =  0.50
         g%s(54)%v(3)    =  0.50
       
         g%s(55)%m(1,2)  =  1.00
         g%s(55)%m(2,1)  = -1.00
         g%s(55)%m(3,3)  =  1.00
         g%s(55)%v(1)    =  0.00
         g%s(55)%v(2)    =  0.50
         g%s(55)%v(3)    =  0.50
       
         g%s(56)%m(1,3)  =  1.00
         g%s(56)%m(2,1)  =  1.00
         g%s(56)%m(3,2)  =  1.00
         g%s(56)%v(1)    =  0.00
         g%s(56)%v(2)    =  0.50
         g%s(56)%v(3)    =  0.50
       
         g%s(57)%m(1,2)  =  1.00
         g%s(57)%m(2,3)  =  1.00
         g%s(57)%m(3,1)  =  1.00
         g%s(57)%v(1)    =  0.00
         g%s(57)%v(2)    =  0.50
         g%s(57)%v(3)    =  0.50
       
         g%s(58)%m(1,2)  = -1.00
         g%s(58)%m(2,3)  = -1.00
         g%s(58)%m(3,1)  =  1.00
         g%s(58)%v(1)    =  0.00
         g%s(58)%v(2)    =  0.50
         g%s(58)%v(3)    =  0.50
       
         g%s(59)%m(1,3)  =  1.00
         g%s(59)%m(2,1)  = -1.00
         g%s(59)%m(3,2)  = -1.00
         g%s(59)%v(1)    =  0.00
         g%s(59)%v(2)    =  0.50
         g%s(59)%v(3)    =  0.50
       
         g%s(60)%m(1,2)  = -1.00
         g%s(60)%m(2,3)  =  1.00
         g%s(60)%m(3,1)  = -1.00
         g%s(60)%v(1)    =  0.00
         g%s(60)%v(2)    =  0.50
         g%s(60)%v(3)    =  0.50
       
         g%s(61)%m(1,3)  = -1.00
         g%s(61)%m(2,1)  = -1.00
         g%s(61)%m(3,2)  =  1.00
         g%s(61)%v(1)    =  0.00
         g%s(61)%v(2)    =  0.50
         g%s(61)%v(3)    =  0.50
       
         g%s(62)%m(1,3)  = -1.00
         g%s(62)%m(2,1)  =  1.00
         g%s(62)%m(3,2)  = -1.00
         g%s(62)%v(1)    =  0.00
         g%s(62)%v(2)    =  0.50
         g%s(62)%v(3)    =  0.50
       
         g%s(63)%m(1,2)  =  1.00
         g%s(63)%m(2,3)  = -1.00
         g%s(63)%m(3,1)  = -1.00
         g%s(63)%v(1)    =  0.00
         g%s(63)%v(2)    =  0.50
         g%s(63)%v(3)    =  0.50
       
         g%s(64)%m(1,1)  =  1.00
         g%s(64)%m(2,2)  = -1.00
         g%s(64)%m(3,3)  = -1.00
         g%s(64)%v(1)    =  0.00
         g%s(64)%v(2)    =  0.50
         g%s(64)%v(3)    =  0.50
       
         g%s(65)%m(1,1)  = -1.00
         g%s(65)%m(2,2)  =  1.00
         g%s(65)%m(3,3)  = -1.00
         g%s(65)%v(1)    =  0.00
         g%s(65)%v(2)    =  0.50
         g%s(65)%v(3)    =  0.50
       
         g%s(66)%m(1,1)  = -1.00
         g%s(66)%m(2,2)  = -1.00
         g%s(66)%m(3,3)  =  1.00
         g%s(66)%v(1)    =  0.00
         g%s(66)%v(2)    =  0.50
         g%s(66)%v(3)    =  0.50
       
         g%s(67)%m(1,2)  =  1.00
         g%s(67)%m(2,1)  =  1.00
         g%s(67)%m(3,3)  = -1.00
         g%s(67)%v(1)    =  0.00
         g%s(67)%v(2)    =  0.50
         g%s(67)%v(3)    =  0.50
       
         g%s(68)%m(1,2)  = -1.00
         g%s(68)%m(2,1)  = -1.00
         g%s(68)%m(3,3)  = -1.00
         g%s(68)%v(1)    =  0.00
         g%s(68)%v(2)    =  0.50
         g%s(68)%v(3)    =  0.50
       
         g%s(69)%m(1,3)  =  1.00
         g%s(69)%m(2,2)  = -1.00
         g%s(69)%m(3,1)  =  1.00
         g%s(69)%v(1)    =  0.00
         g%s(69)%v(2)    =  0.50
         g%s(69)%v(3)    =  0.50
       
         g%s(70)%m(1,3)  = -1.00
         g%s(70)%m(2,2)  = -1.00
         g%s(70)%m(3,1)  = -1.00
         g%s(70)%v(1)    =  0.00
         g%s(70)%v(2)    =  0.50
         g%s(70)%v(3)    =  0.50
       
         g%s(71)%m(1,1)  = -1.00
         g%s(71)%m(2,3)  =  1.00
         g%s(71)%m(3,2)  =  1.00
         g%s(71)%v(1)    =  0.00
         g%s(71)%v(2)    =  0.50
         g%s(71)%v(3)    =  0.50
       
         g%s(72)%m(1,1)  = -1.00
         g%s(72)%m(2,3)  = -1.00
         g%s(72)%m(3,2)  = -1.00
         g%s(72)%v(1)    =  0.00
         g%s(72)%v(2)    =  0.50
         g%s(72)%v(3)    =  0.50
       
         g%s(73)%m(1,1)  = -1.00
         g%s(73)%m(2,2)  = -1.00
         g%s(73)%m(3,3)  = -1.00
         g%s(73)%v(1)    =  0.00
         g%s(73)%v(2)    =  0.50
         g%s(73)%v(3)    =  0.50
       
         g%s(74)%m(1,1)  = -1.00
         g%s(74)%m(2,3)  =  1.00
         g%s(74)%m(3,2)  = -1.00
         g%s(74)%v(1)    =  0.00
         g%s(74)%v(2)    =  0.50
         g%s(74)%v(3)    =  0.50
       
         g%s(75)%m(1,1)  = -1.00
         g%s(75)%m(2,3)  = -1.00
         g%s(75)%m(3,2)  =  1.00
         g%s(75)%v(1)    =  0.00
         g%s(75)%v(2)    =  0.50
         g%s(75)%v(3)    =  0.50
       
         g%s(76)%m(1,3)  = -1.00
         g%s(76)%m(2,2)  = -1.00
         g%s(76)%m(3,1)  =  1.00
         g%s(76)%v(1)    =  0.00
         g%s(76)%v(2)    =  0.50
         g%s(76)%v(3)    =  0.50
       
         g%s(77)%m(1,3)  =  1.00
         g%s(77)%m(2,2)  = -1.00
         g%s(77)%m(3,1)  = -1.00
         g%s(77)%v(1)    =  0.00
         g%s(77)%v(2)    =  0.50
         g%s(77)%v(3)    =  0.50
       
         g%s(78)%m(1,2)  =  1.00
         g%s(78)%m(2,1)  = -1.00
         g%s(78)%m(3,3)  = -1.00
         g%s(78)%v(1)    =  0.00
         g%s(78)%v(2)    =  0.50
         g%s(78)%v(3)    =  0.50
       
         g%s(79)%m(1,2)  = -1.00
         g%s(79)%m(2,1)  =  1.00
         g%s(79)%m(3,3)  = -1.00
         g%s(79)%v(1)    =  0.00
         g%s(79)%v(2)    =  0.50
         g%s(79)%v(3)    =  0.50
       
         g%s(80)%m(1,3)  = -1.00
         g%s(80)%m(2,1)  = -1.00
         g%s(80)%m(3,2)  = -1.00
         g%s(80)%v(1)    =  0.00
         g%s(80)%v(2)    =  0.50
         g%s(80)%v(3)    =  0.50
       
         g%s(81)%m(1,2)  = -1.00
         g%s(81)%m(2,3)  = -1.00
         g%s(81)%m(3,1)  = -1.00
         g%s(81)%v(1)    =  0.00
         g%s(81)%v(2)    =  0.50
         g%s(81)%v(3)    =  0.50
       
         g%s(82)%m(1,2)  =  1.00
         g%s(82)%m(2,3)  =  1.00
         g%s(82)%m(3,1)  = -1.00
         g%s(82)%v(1)    =  0.00
         g%s(82)%v(2)    =  0.50
         g%s(82)%v(3)    =  0.50
       
         g%s(83)%m(1,3)  = -1.00
         g%s(83)%m(2,1)  =  1.00
         g%s(83)%m(3,2)  =  1.00
         g%s(83)%v(1)    =  0.00
         g%s(83)%v(2)    =  0.50
         g%s(83)%v(3)    =  0.50
       
         g%s(84)%m(1,2)  =  1.00
         g%s(84)%m(2,3)  = -1.00
         g%s(84)%m(3,1)  =  1.00
         g%s(84)%v(1)    =  0.00
         g%s(84)%v(2)    =  0.50
         g%s(84)%v(3)    =  0.50
       
         g%s(85)%m(1,3)  =  1.00
         g%s(85)%m(2,1)  =  1.00
         g%s(85)%m(3,2)  = -1.00
         g%s(85)%v(1)    =  0.00
         g%s(85)%v(2)    =  0.50
         g%s(85)%v(3)    =  0.50
       
         g%s(86)%m(1,3)  =  1.00
         g%s(86)%m(2,1)  = -1.00
         g%s(86)%m(3,2)  =  1.00
         g%s(86)%v(1)    =  0.00
         g%s(86)%v(2)    =  0.50
         g%s(86)%v(3)    =  0.50
       
         g%s(87)%m(1,2)  = -1.00
         g%s(87)%m(2,3)  =  1.00
         g%s(87)%m(3,1)  =  1.00
         g%s(87)%v(1)    =  0.00
         g%s(87)%v(2)    =  0.50
         g%s(87)%v(3)    =  0.50
       
         g%s(88)%m(1,1)  = -1.00
         g%s(88)%m(2,2)  =  1.00
         g%s(88)%m(3,3)  =  1.00
         g%s(88)%v(1)    =  0.00
         g%s(88)%v(2)    =  0.50
         g%s(88)%v(3)    =  0.50
       
         g%s(89)%m(1,1)  =  1.00
         g%s(89)%m(2,2)  = -1.00
         g%s(89)%m(3,3)  =  1.00
         g%s(89)%v(1)    =  0.00
         g%s(89)%v(2)    =  0.50
         g%s(89)%v(3)    =  0.50
       
         g%s(90)%m(1,1)  =  1.00
         g%s(90)%m(2,2)  =  1.00
         g%s(90)%m(3,3)  = -1.00
         g%s(90)%v(1)    =  0.00
         g%s(90)%v(2)    =  0.50
         g%s(90)%v(3)    =  0.50
       
         g%s(91)%m(1,2)  = -1.00
         g%s(91)%m(2,1)  = -1.00
         g%s(91)%m(3,3)  =  1.00
         g%s(91)%v(1)    =  0.00
         g%s(91)%v(2)    =  0.50
         g%s(91)%v(3)    =  0.50
       
         g%s(92)%m(1,2)  =  1.00
         g%s(92)%m(2,1)  =  1.00
         g%s(92)%m(3,3)  =  1.00
         g%s(92)%v(1)    =  0.00
         g%s(92)%v(2)    =  0.50
         g%s(92)%v(3)    =  0.50
       
         g%s(93)%m(1,3)  = -1.00
         g%s(93)%m(2,2)  =  1.00
         g%s(93)%m(3,1)  = -1.00
         g%s(93)%v(1)    =  0.00
         g%s(93)%v(2)    =  0.50
         g%s(93)%v(3)    =  0.50
       
         g%s(94)%m(1,3)  =  1.00
         g%s(94)%m(2,2)  =  1.00
         g%s(94)%m(3,1)  =  1.00
         g%s(94)%v(1)    =  0.00
         g%s(94)%v(2)    =  0.50
         g%s(94)%v(3)    =  0.50
       
         g%s(95)%m(1,1)  =  1.00
         g%s(95)%m(2,3)  = -1.00
         g%s(95)%m(3,2)  = -1.00
         g%s(95)%v(1)    =  0.00
         g%s(95)%v(2)    =  0.50
         g%s(95)%v(3)    =  0.50
       
         g%s(96)%m(1,1)  =  1.00
         g%s(96)%m(2,3)  =  1.00
         g%s(96)%m(3,2)  =  1.00
         g%s(96)%v(1)    =  0.00
         g%s(96)%v(2)    =  0.50
         g%s(96)%v(3)    =  0.50
       
         g%s(97)%m(1,1)  =  1.00
         g%s(97)%m(2,2)  =  1.00
         g%s(97)%m(3,3)  =  1.00
         g%s(97)%v(1)    =  0.50
         g%s(97)%v(2)    =  0.00
         g%s(97)%v(3)    =  0.50
       
         g%s(98)%m(1,1)  =  1.00
         g%s(98)%m(2,3)  = -1.00
         g%s(98)%m(3,2)  =  1.00
         g%s(98)%v(1)    =  0.50
         g%s(98)%v(2)    =  0.00
         g%s(98)%v(3)    =  0.50
       
         g%s(99)%m(1,1)  =  1.00
         g%s(99)%m(2,3)  =  1.00
         g%s(99)%m(3,2)  = -1.00
         g%s(99)%v(1)    =  0.50
         g%s(99)%v(2)    =  0.00
         g%s(99)%v(3)    =  0.50
       
         g%s(100)%m(1,3) =  1.00
         g%s(100)%m(2,2) =  1.00
         g%s(100)%m(3,1) = -1.00
         g%s(100)%v(1)   =  0.50
         g%s(100)%v(2)   =  0.00
         g%s(100)%v(3)   =  0.50
       
         g%s(101)%m(1,3) = -1.00
         g%s(101)%m(2,2) =  1.00
         g%s(101)%m(3,1) =  1.00
         g%s(101)%v(1)   =  0.50
         g%s(101)%v(2)   =  0.00
         g%s(101)%v(3)   =  0.50
       
         g%s(102)%m(1,2) = -1.00
         g%s(102)%m(2,1) =  1.00
         g%s(102)%m(3,3) =  1.00
         g%s(102)%v(1)   =  0.50
         g%s(102)%v(2)   =  0.00
         g%s(102)%v(3)   =  0.50
       
         g%s(103)%m(1,2) =  1.00
         g%s(103)%m(2,1) = -1.00
         g%s(103)%m(3,3) =  1.00
         g%s(103)%v(1)   =  0.50
         g%s(103)%v(2)   =  0.00
         g%s(103)%v(3)   =  0.50
       
         g%s(104)%m(1,3) =  1.00
         g%s(104)%m(2,1) =  1.00
         g%s(104)%m(3,2) =  1.00
         g%s(104)%v(1)   =  0.50
         g%s(104)%v(2)   =  0.00
         g%s(104)%v(3)   =  0.50
       
         g%s(105)%m(1,2) =  1.00
         g%s(105)%m(2,3) =  1.00
         g%s(105)%m(3,1) =  1.00
         g%s(105)%v(1)   =  0.50
         g%s(105)%v(2)   =  0.00
         g%s(105)%v(3)   =  0.50
       
         g%s(106)%m(1,2) = -1.00
         g%s(106)%m(2,3) = -1.00
         g%s(106)%m(3,1) =  1.00
         g%s(106)%v(1)   =  0.50
         g%s(106)%v(2)   =  0.00
         g%s(106)%v(3)   =  0.50
       
         g%s(107)%m(1,3) =  1.00
         g%s(107)%m(2,1) = -1.00
         g%s(107)%m(3,2) = -1.00
         g%s(107)%v(1)   =  0.50
         g%s(107)%v(2)   =  0.00
         g%s(107)%v(3)   =  0.50
       
         g%s(108)%m(1,2) = -1.00
         g%s(108)%m(2,3) =  1.00
         g%s(108)%m(3,1) = -1.00
         g%s(108)%v(1)   =  0.50
         g%s(108)%v(2)   =  0.00
         g%s(108)%v(3)   =  0.50
       
         g%s(109)%m(1,3) = -1.00
         g%s(109)%m(2,1) = -1.00
         g%s(109)%m(3,2) =  1.00
         g%s(109)%v(1)   =  0.50
         g%s(109)%v(2)   =  0.00
         g%s(109)%v(3)   =  0.50
       
         g%s(110)%m(1,3) = -1.00
         g%s(110)%m(2,1) =  1.00
         g%s(110)%m(3,2) = -1.00
         g%s(110)%v(1)   =  0.50
         g%s(110)%v(2)   =  0.00
         g%s(110)%v(3)   =  0.50
       
         g%s(111)%m(1,2) =  1.00
         g%s(111)%m(2,3) = -1.00
         g%s(111)%m(3,1) = -1.00
         g%s(111)%v(1)   =  0.50
         g%s(111)%v(2)   =  0.00
         g%s(111)%v(3)   =  0.50
       
         g%s(112)%m(1,1) =  1.00
         g%s(112)%m(2,2) = -1.00
         g%s(112)%m(3,3) = -1.00
         g%s(112)%v(1)   =  0.50
         g%s(112)%v(2)   =  0.00
         g%s(112)%v(3)   =  0.50
       
         g%s(113)%m(1,1) = -1.00
         g%s(113)%m(2,2) =  1.00
         g%s(113)%m(3,3) = -1.00
         g%s(113)%v(1)   =  0.50
         g%s(113)%v(2)   =  0.00
         g%s(113)%v(3)   =  0.50
       
         g%s(114)%m(1,1) = -1.00
         g%s(114)%m(2,2) = -1.00
         g%s(114)%m(3,3) =  1.00
         g%s(114)%v(1)   =  0.50
         g%s(114)%v(2)   =  0.00
         g%s(114)%v(3)   =  0.50
       
         g%s(115)%m(1,2) =  1.00
         g%s(115)%m(2,1) =  1.00
         g%s(115)%m(3,3) = -1.00
         g%s(115)%v(1)   =  0.50
         g%s(115)%v(2)   =  0.00
         g%s(115)%v(3)   =  0.50
       
         g%s(116)%m(1,2) = -1.00
         g%s(116)%m(2,1) = -1.00
         g%s(116)%m(3,3) = -1.00
         g%s(116)%v(1)   =  0.50
         g%s(116)%v(2)   =  0.00
         g%s(116)%v(3)   =  0.50
       
         g%s(117)%m(1,3) =  1.00
         g%s(117)%m(2,2) = -1.00
         g%s(117)%m(3,1) =  1.00
         g%s(117)%v(1)   =  0.50
         g%s(117)%v(2)   =  0.00
         g%s(117)%v(3)   =  0.50
       
         g%s(118)%m(1,3) = -1.00
         g%s(118)%m(2,2) = -1.00
         g%s(118)%m(3,1) = -1.00
         g%s(118)%v(1)   =  0.50
         g%s(118)%v(2)   =  0.00
         g%s(118)%v(3)   =  0.50
       
         g%s(119)%m(1,1) = -1.00
         g%s(119)%m(2,3) =  1.00
         g%s(119)%m(3,2) =  1.00
         g%s(119)%v(1)   =  0.50
         g%s(119)%v(2)   =  0.00
         g%s(119)%v(3)   =  0.50
       
         g%s(120)%m(1,1) = -1.00
         g%s(120)%m(2,3) = -1.00
         g%s(120)%m(3,2) = -1.00
         g%s(120)%v(1)   =  0.50
         g%s(120)%v(2)   =  0.00
         g%s(120)%v(3)   =  0.50
       
         g%s(121)%m(1,1) = -1.00
         g%s(121)%m(2,2) = -1.00
         g%s(121)%m(3,3) = -1.00
         g%s(121)%v(1)   =  0.50
         g%s(121)%v(2)   =  0.00
         g%s(121)%v(3)   =  0.50
       
         g%s(122)%m(1,1) = -1.00
         g%s(122)%m(2,3) =  1.00
         g%s(122)%m(3,2) = -1.00
         g%s(122)%v(1)   =  0.50
         g%s(122)%v(2)   =  0.00
         g%s(122)%v(3)   =  0.50
       
         g%s(123)%m(1,1) = -1.00
         g%s(123)%m(2,3) = -1.00
         g%s(123)%m(3,2) =  1.00
         g%s(123)%v(1)   =  0.50
         g%s(123)%v(2)   =  0.00
         g%s(123)%v(3)   =  0.50
       
         g%s(124)%m(1,3) = -1.00
         g%s(124)%m(2,2) = -1.00
         g%s(124)%m(3,1) =  1.00
         g%s(124)%v(1)   =  0.50
         g%s(124)%v(2)   =  0.00
         g%s(124)%v(3)   =  0.50
       
         g%s(125)%m(1,3) =  1.00
         g%s(125)%m(2,2) = -1.00
         g%s(125)%m(3,1) = -1.00
         g%s(125)%v(1)   =  0.50
         g%s(125)%v(2)   =  0.00
         g%s(125)%v(3)   =  0.50
       
         g%s(126)%m(1,2) =  1.00
         g%s(126)%m(2,1) = -1.00
         g%s(126)%m(3,3) = -1.00
         g%s(126)%v(1)   =  0.50
         g%s(126)%v(2)   =  0.00
         g%s(126)%v(3)   =  0.50
       
         g%s(127)%m(1,2) = -1.00
         g%s(127)%m(2,1) =  1.00
         g%s(127)%m(3,3) = -1.00
         g%s(127)%v(1)   =  0.50
         g%s(127)%v(2)   =  0.00
         g%s(127)%v(3)   =  0.50
       
         g%s(128)%m(1,3) = -1.00
         g%s(128)%m(2,1) = -1.00
         g%s(128)%m(3,2) = -1.00
         g%s(128)%v(1)   =  0.50
         g%s(128)%v(2)   =  0.00
         g%s(128)%v(3)   =  0.50
       
         g%s(129)%m(1,2) = -1.00
         g%s(129)%m(2,3) = -1.00
         g%s(129)%m(3,1) = -1.00
         g%s(129)%v(1)   =  0.50
         g%s(129)%v(2)   =  0.00
         g%s(129)%v(3)   =  0.50
       
         g%s(130)%m(1,2) =  1.00
         g%s(130)%m(2,3) =  1.00
         g%s(130)%m(3,1) = -1.00
         g%s(130)%v(1)   =  0.50
         g%s(130)%v(2)   =  0.00
         g%s(130)%v(3)   =  0.50
       
         g%s(131)%m(1,3) = -1.00
         g%s(131)%m(2,1) =  1.00
         g%s(131)%m(3,2) =  1.00
         g%s(131)%v(1)   =  0.50
         g%s(131)%v(2)   =  0.00
         g%s(131)%v(3)   =  0.50
       
         g%s(132)%m(1,2) =  1.00
         g%s(132)%m(2,3) = -1.00
         g%s(132)%m(3,1) =  1.00
         g%s(132)%v(1)   =  0.50
         g%s(132)%v(2)   =  0.00
         g%s(132)%v(3)   =  0.50
       
         g%s(133)%m(1,3) =  1.00
         g%s(133)%m(2,1) =  1.00
         g%s(133)%m(3,2) = -1.00
         g%s(133)%v(1)   =  0.50
         g%s(133)%v(2)   =  0.00
         g%s(133)%v(3)   =  0.50
       
         g%s(134)%m(1,3) =  1.00
         g%s(134)%m(2,1) = -1.00
         g%s(134)%m(3,2) =  1.00
         g%s(134)%v(1)   =  0.50
         g%s(134)%v(2)   =  0.00
         g%s(134)%v(3)   =  0.50
       
         g%s(135)%m(1,2) = -1.00
         g%s(135)%m(2,3) =  1.00
         g%s(135)%m(3,1) =  1.00
         g%s(135)%v(1)   =  0.50
         g%s(135)%v(2)   =  0.00
         g%s(135)%v(3)   =  0.50
       
         g%s(136)%m(1,1) = -1.00
         g%s(136)%m(2,2) =  1.00
         g%s(136)%m(3,3) =  1.00
         g%s(136)%v(1)   =  0.50
         g%s(136)%v(2)   =  0.00
         g%s(136)%v(3)   =  0.50
       
         g%s(137)%m(1,1) =  1.00
         g%s(137)%m(2,2) = -1.00
         g%s(137)%m(3,3) =  1.00
         g%s(137)%v(1)   =  0.50
         g%s(137)%v(2)   =  0.00
         g%s(137)%v(3)   =  0.50
       
         g%s(138)%m(1,1) =  1.00
         g%s(138)%m(2,2) =  1.00
         g%s(138)%m(3,3) = -1.00
         g%s(138)%v(1)   =  0.50
         g%s(138)%v(2)   =  0.00
         g%s(138)%v(3)   =  0.50
       
         g%s(139)%m(1,2) = -1.00
         g%s(139)%m(2,1) = -1.00
         g%s(139)%m(3,3) =  1.00
         g%s(139)%v(1)   =  0.50
         g%s(139)%v(2)   =  0.00
         g%s(139)%v(3)   =  0.50
       
         g%s(140)%m(1,2) =  1.00
         g%s(140)%m(2,1) =  1.00
         g%s(140)%m(3,3) =  1.00
         g%s(140)%v(1)   =  0.50
         g%s(140)%v(2)   =  0.00
         g%s(140)%v(3)   =  0.50
       
         g%s(141)%m(1,3) = -1.00
         g%s(141)%m(2,2) =  1.00
         g%s(141)%m(3,1) = -1.00
         g%s(141)%v(1)   =  0.50
         g%s(141)%v(2)   =  0.00
         g%s(141)%v(3)   =  0.50
       
         g%s(142)%m(1,3) =  1.00
         g%s(142)%m(2,2) =  1.00
         g%s(142)%m(3,1) =  1.00
         g%s(142)%v(1)   =  0.50
         g%s(142)%v(2)   =  0.00
         g%s(142)%v(3)   =  0.50
       
         g%s(143)%m(1,1) =  1.00
         g%s(143)%m(2,3) = -1.00
         g%s(143)%m(3,2) = -1.00
         g%s(143)%v(1)   =  0.50
         g%s(143)%v(2)   =  0.00
         g%s(143)%v(3)   =  0.50
       
         g%s(144)%m(1,1) =  1.00
         g%s(144)%m(2,3) =  1.00
         g%s(144)%m(3,2) =  1.00
         g%s(144)%v(1)   =  0.50
         g%s(144)%v(2)   =  0.00
         g%s(144)%v(3)   =  0.50
       
         g%s(145)%m(1,1) =  1.00
         g%s(145)%m(2,2) =  1.00
         g%s(145)%m(3,3) =  1.00
         g%s(145)%v(1)   =  0.50
         g%s(145)%v(2)   =  0.50
         g%s(145)%v(3)   =  0.00
       
         g%s(146)%m(1,1) =  1.00
         g%s(146)%m(2,3) = -1.00
         g%s(146)%m(3,2) =  1.00
         g%s(146)%v(1)   =  0.50
         g%s(146)%v(2)   =  0.50
         g%s(146)%v(3)   =  0.00
       
         g%s(147)%m(1,1) =  1.00
         g%s(147)%m(2,3) =  1.00
         g%s(147)%m(3,2) = -1.00
         g%s(147)%v(1)   =  0.50
         g%s(147)%v(2)   =  0.50
         g%s(147)%v(3)   =  0.00
       
         g%s(148)%m(1,3) =  1.00
         g%s(148)%m(2,2) =  1.00
         g%s(148)%m(3,1) = -1.00
         g%s(148)%v(1)   =  0.50
         g%s(148)%v(2)   =  0.50
         g%s(148)%v(3)   =  0.00
       
         g%s(149)%m(1,3) = -1.00
         g%s(149)%m(2,2) =  1.00
         g%s(149)%m(3,1) =  1.00
         g%s(149)%v(1)   =  0.50
         g%s(149)%v(2)   =  0.50
         g%s(149)%v(3)   =  0.00
       
         g%s(150)%m(1,2) = -1.00
         g%s(150)%m(2,1) =  1.00
         g%s(150)%m(3,3) =  1.00
         g%s(150)%v(1)   =  0.50
         g%s(150)%v(2)   =  0.50
         g%s(150)%v(3)   =  0.00
       
         g%s(151)%m(1,2) =  1.00
         g%s(151)%m(2,1) = -1.00
         g%s(151)%m(3,3) =  1.00
         g%s(151)%v(1)   =  0.50
         g%s(151)%v(2)   =  0.50
         g%s(151)%v(3)   =  0.00
       
         g%s(152)%m(1,3) =  1.00
         g%s(152)%m(2,1) =  1.00
         g%s(152)%m(3,2) =  1.00
         g%s(152)%v(1)   =  0.50
         g%s(152)%v(2)   =  0.50
         g%s(152)%v(3)   =  0.00
       
         g%s(153)%m(1,2) =  1.00
         g%s(153)%m(2,3) =  1.00
         g%s(153)%m(3,1) =  1.00
         g%s(153)%v(1)   =  0.50
         g%s(153)%v(2)   =  0.50
         g%s(153)%v(3)   =  0.00
       
         g%s(154)%m(1,2) = -1.00
         g%s(154)%m(2,3) = -1.00
         g%s(154)%m(3,1) =  1.00
         g%s(154)%v(1)   =  0.50
         g%s(154)%v(2)   =  0.50
         g%s(154)%v(3)   =  0.00
       
         g%s(155)%m(1,3) =  1.00
         g%s(155)%m(2,1) = -1.00
         g%s(155)%m(3,2) = -1.00
         g%s(155)%v(1)   =  0.50
         g%s(155)%v(2)   =  0.50
         g%s(155)%v(3)   =  0.00
       
         g%s(156)%m(1,2) = -1.00
         g%s(156)%m(2,3) =  1.00
         g%s(156)%m(3,1) = -1.00
         g%s(156)%v(1)   =  0.50
         g%s(156)%v(2)   =  0.50
         g%s(156)%v(3)   =  0.00
       
         g%s(157)%m(1,3) = -1.00
         g%s(157)%m(2,1) = -1.00
         g%s(157)%m(3,2) =  1.00
         g%s(157)%v(1)   =  0.50
         g%s(157)%v(2)   =  0.50
         g%s(157)%v(3)   =  0.00
       
         g%s(158)%m(1,3) = -1.00
         g%s(158)%m(2,1) =  1.00
         g%s(158)%m(3,2) = -1.00
         g%s(158)%v(1)   =  0.50
         g%s(158)%v(2)   =  0.50
         g%s(158)%v(3)   =  0.00
       
         g%s(159)%m(1,2) =  1.00
         g%s(159)%m(2,3) = -1.00
         g%s(159)%m(3,1) = -1.00
         g%s(159)%v(1)   =  0.50
         g%s(159)%v(2)   =  0.50
         g%s(159)%v(3)   =  0.00
       
         g%s(160)%m(1,1) =  1.00
         g%s(160)%m(2,2) = -1.00
         g%s(160)%m(3,3) = -1.00
         g%s(160)%v(1)   =  0.50
         g%s(160)%v(2)   =  0.50
         g%s(160)%v(3)   =  0.00
       
         g%s(161)%m(1,1) = -1.00
         g%s(161)%m(2,2) =  1.00
         g%s(161)%m(3,3) = -1.00
         g%s(161)%v(1)   =  0.50
         g%s(161)%v(2)   =  0.50
         g%s(161)%v(3)   =  0.00
       
         g%s(162)%m(1,1) = -1.00
         g%s(162)%m(2,2) = -1.00
         g%s(162)%m(3,3) =  1.00
         g%s(162)%v(1)   =  0.50
         g%s(162)%v(2)   =  0.50
         g%s(162)%v(3)   =  0.00
       
         g%s(163)%m(1,2) =  1.00
         g%s(163)%m(2,1) =  1.00
         g%s(163)%m(3,3) = -1.00
         g%s(163)%v(1)   =  0.50
         g%s(163)%v(2)   =  0.50
         g%s(163)%v(3)   =  0.00
       
         g%s(164)%m(1,2) = -1.00
         g%s(164)%m(2,1) = -1.00
         g%s(164)%m(3,3) = -1.00
         g%s(164)%v(1)   =  0.50
         g%s(164)%v(2)   =  0.50
         g%s(164)%v(3)   =  0.00
       
         g%s(165)%m(1,3) =  1.00
         g%s(165)%m(2,2) = -1.00
         g%s(165)%m(3,1) =  1.00
         g%s(165)%v(1)   =  0.50
         g%s(165)%v(2)   =  0.50
         g%s(165)%v(3)   =  0.00
       
         g%s(166)%m(1,3) = -1.00
         g%s(166)%m(2,2) = -1.00
         g%s(166)%m(3,1) = -1.00
         g%s(166)%v(1)   =  0.50
         g%s(166)%v(2)   =  0.50
         g%s(166)%v(3)   =  0.00
       
         g%s(167)%m(1,1) = -1.00
         g%s(167)%m(2,3) =  1.00
         g%s(167)%m(3,2) =  1.00
         g%s(167)%v(1)   =  0.50
         g%s(167)%v(2)   =  0.50
         g%s(167)%v(3)   =  0.00
       
         g%s(168)%m(1,1) = -1.00
         g%s(168)%m(2,3) = -1.00
         g%s(168)%m(3,2) = -1.00
         g%s(168)%v(1)   =  0.50
         g%s(168)%v(2)   =  0.50
         g%s(168)%v(3)   =  0.00
       
         g%s(169)%m(1,1) = -1.00
         g%s(169)%m(2,2) = -1.00
         g%s(169)%m(3,3) = -1.00
         g%s(169)%v(1)   =  0.50
         g%s(169)%v(2)   =  0.50
         g%s(169)%v(3)   =  0.00
       
         g%s(170)%m(1,1) = -1.00
         g%s(170)%m(2,3) =  1.00
         g%s(170)%m(3,2) = -1.00
         g%s(170)%v(1)   =  0.50
         g%s(170)%v(2)   =  0.50
         g%s(170)%v(3)   =  0.00
       
         g%s(171)%m(1,1) = -1.00
         g%s(171)%m(2,3) = -1.00
         g%s(171)%m(3,2) =  1.00
         g%s(171)%v(1)   =  0.50
         g%s(171)%v(2)   =  0.50
         g%s(171)%v(3)   =  0.00
       
         g%s(172)%m(1,3) = -1.00
         g%s(172)%m(2,2) = -1.00
         g%s(172)%m(3,1) =  1.00
         g%s(172)%v(1)   =  0.50
         g%s(172)%v(2)   =  0.50
         g%s(172)%v(3)   =  0.00
       
         g%s(173)%m(1,3) =  1.00
         g%s(173)%m(2,2) = -1.00
         g%s(173)%m(3,1) = -1.00
         g%s(173)%v(1)   =  0.50
         g%s(173)%v(2)   =  0.50
         g%s(173)%v(3)   =  0.00
       
         g%s(174)%m(1,2) =  1.00
         g%s(174)%m(2,1) = -1.00
         g%s(174)%m(3,3) = -1.00
         g%s(174)%v(1)   =  0.50
         g%s(174)%v(2)   =  0.50
         g%s(174)%v(3)   =  0.00
       
         g%s(175)%m(1,2) = -1.00
         g%s(175)%m(2,1) =  1.00
         g%s(175)%m(3,3) = -1.00
         g%s(175)%v(1)   =  0.50
         g%s(175)%v(2)   =  0.50
         g%s(175)%v(3)   =  0.00
       
         g%s(176)%m(1,3) = -1.00
         g%s(176)%m(2,1) = -1.00
         g%s(176)%m(3,2) = -1.00
         g%s(176)%v(1)   =  0.50
         g%s(176)%v(2)   =  0.50
         g%s(176)%v(3)   =  0.00
       
         g%s(177)%m(1,2) = -1.00
         g%s(177)%m(2,3) = -1.00
         g%s(177)%m(3,1) = -1.00
         g%s(177)%v(1)   =  0.50
         g%s(177)%v(2)   =  0.50
         g%s(177)%v(3)   =  0.00
       
         g%s(178)%m(1,2) =  1.00
         g%s(178)%m(2,3) =  1.00
         g%s(178)%m(3,1) = -1.00
         g%s(178)%v(1)   =  0.50
         g%s(178)%v(2)   =  0.50
         g%s(178)%v(3)   =  0.00
       
         g%s(179)%m(1,3) = -1.00
         g%s(179)%m(2,1) =  1.00
         g%s(179)%m(3,2) =  1.00
         g%s(179)%v(1)   =  0.50
         g%s(179)%v(2)   =  0.50
         g%s(179)%v(3)   =  0.00
       
         g%s(180)%m(1,2) =  1.00
         g%s(180)%m(2,3) = -1.00
         g%s(180)%m(3,1) =  1.00
         g%s(180)%v(1)   =  0.50
         g%s(180)%v(2)   =  0.50
         g%s(180)%v(3)   =  0.00
       
         g%s(181)%m(1,3) =  1.00
         g%s(181)%m(2,1) =  1.00
         g%s(181)%m(3,2) = -1.00
         g%s(181)%v(1)   =  0.50
         g%s(181)%v(2)   =  0.50
         g%s(181)%v(3)   =  0.00
       
         g%s(182)%m(1,3) =  1.00
         g%s(182)%m(2,1) = -1.00
         g%s(182)%m(3,2) =  1.00
         g%s(182)%v(1)   =  0.50
         g%s(182)%v(2)   =  0.50
         g%s(182)%v(3)   =  0.00
       
         g%s(183)%m(1,2) = -1.00
         g%s(183)%m(2,3) =  1.00
         g%s(183)%m(3,1) =  1.00
         g%s(183)%v(1)   =  0.50
         g%s(183)%v(2)   =  0.50
         g%s(183)%v(3)   =  0.00
       
         g%s(184)%m(1,1) = -1.00
         g%s(184)%m(2,2) =  1.00
         g%s(184)%m(3,3) =  1.00
         g%s(184)%v(1)   =  0.50
         g%s(184)%v(2)   =  0.50
         g%s(184)%v(3)   =  0.00
       
         g%s(185)%m(1,1) =  1.00
         g%s(185)%m(2,2) = -1.00
         g%s(185)%m(3,3) =  1.00
         g%s(185)%v(1)   =  0.50
         g%s(185)%v(2)   =  0.50
         g%s(185)%v(3)   =  0.00
       
         g%s(186)%m(1,1) =  1.00
         g%s(186)%m(2,2) =  1.00
         g%s(186)%m(3,3) = -1.00
         g%s(186)%v(1)   =  0.50
         g%s(186)%v(2)   =  0.50
         g%s(186)%v(3)   =  0.00
       
         g%s(187)%m(1,2) = -1.00
         g%s(187)%m(2,1) = -1.00
         g%s(187)%m(3,3) =  1.00
         g%s(187)%v(1)   =  0.50
         g%s(187)%v(2)   =  0.50
         g%s(187)%v(3)   =  0.00
       
         g%s(188)%m(1,2) =  1.00
         g%s(188)%m(2,1) =  1.00
         g%s(188)%m(3,3) =  1.00
         g%s(188)%v(1)   =  0.50
         g%s(188)%v(2)   =  0.50
         g%s(188)%v(3)   =  0.00
       
         g%s(189)%m(1,3) = -1.00
         g%s(189)%m(2,2) =  1.00
         g%s(189)%m(3,1) = -1.00
         g%s(189)%v(1)   =  0.50
         g%s(189)%v(2)   =  0.50
         g%s(189)%v(3)   =  0.00
       
         g%s(190)%m(1,3) =  1.00
         g%s(190)%m(2,2) =  1.00
         g%s(190)%m(3,1) =  1.00
         g%s(190)%v(1)   =  0.50
         g%s(190)%v(2)   =  0.50
         g%s(190)%v(3)   =  0.00
       
         g%s(191)%m(1,1) =  1.00
         g%s(191)%m(2,3) = -1.00
         g%s(191)%m(3,2) = -1.00
         g%s(191)%v(1)   =  0.50
         g%s(191)%v(2)   =  0.50
         g%s(191)%v(3)   =  0.00
       
         g%s(192)%m(1,1) =  1.00
         g%s(192)%m(2,3) =  1.00
         g%s(192)%m(3,2) =  1.00
         g%s(192)%v(1)   =  0.50
         g%s(192)%v(2)   =  0.50
         g%s(192)%v(3)   =  0.00
       
       
      case('F m -3 c', '226')                        
       
         !Space group   # 226
         !Group setting # 255
       
         g%order   = 192
         do i=1, g%order
            g%s(i)%m = 0.0_long
         enddo
       
         g%s(1)%m(1,1)   =  1.00
         g%s(1)%m(2,2)   =  1.00
         g%s(1)%m(3,3)   =  1.00
         g%s(1)%v(1)     =  0.00
         g%s(1)%v(2)     =  0.00
         g%s(1)%v(3)     =  0.00
       
         g%s(2)%m(1,1)   =  1.00
         g%s(2)%m(2,3)   = -1.00
         g%s(2)%m(3,2)   =  1.00
         g%s(2)%v(1)     =  0.50
         g%s(2)%v(2)     =  0.00
         g%s(2)%v(3)     =  0.00
       
         g%s(3)%m(1,1)   =  1.00
         g%s(3)%m(2,3)   =  1.00
         g%s(3)%m(3,2)   = -1.00
         g%s(3)%v(1)     =  0.50
         g%s(3)%v(2)     =  0.00
         g%s(3)%v(3)     =  0.00
       
         g%s(4)%m(1,3)   =  1.00
         g%s(4)%m(2,2)   =  1.00
         g%s(4)%m(3,1)   = -1.00
         g%s(4)%v(1)     =  0.50
         g%s(4)%v(2)     =  0.00
         g%s(4)%v(3)     =  0.00
       
         g%s(5)%m(1,3)   = -1.00
         g%s(5)%m(2,2)   =  1.00
         g%s(5)%m(3,1)   =  1.00
         g%s(5)%v(1)     =  0.50
         g%s(5)%v(2)     =  0.00
         g%s(5)%v(3)     =  0.00
       
         g%s(6)%m(1,2)   = -1.00
         g%s(6)%m(2,1)   =  1.00
         g%s(6)%m(3,3)   =  1.00
         g%s(6)%v(1)     =  0.50
         g%s(6)%v(2)     =  0.00
         g%s(6)%v(3)     =  0.00
       
         g%s(7)%m(1,2)   =  1.00
         g%s(7)%m(2,1)   = -1.00
         g%s(7)%m(3,3)   =  1.00
         g%s(7)%v(1)     =  0.50
         g%s(7)%v(2)     =  0.00
         g%s(7)%v(3)     =  0.00
       
         g%s(8)%m(1,3)   =  1.00
         g%s(8)%m(2,1)   =  1.00
         g%s(8)%m(3,2)   =  1.00
         g%s(8)%v(1)     =  0.00
         g%s(8)%v(2)     =  0.00
         g%s(8)%v(3)     =  0.00
       
         g%s(9)%m(1,2)   =  1.00
         g%s(9)%m(2,3)   =  1.00
         g%s(9)%m(3,1)   =  1.00
         g%s(9)%v(1)     =  0.00
         g%s(9)%v(2)     =  0.00
         g%s(9)%v(3)     =  0.00
       
         g%s(10)%m(1,2)  = -1.00
         g%s(10)%m(2,3)  = -1.00
         g%s(10)%m(3,1)  =  1.00
         g%s(10)%v(1)    =  0.00
         g%s(10)%v(2)    =  0.00
         g%s(10)%v(3)    =  0.00
       
         g%s(11)%m(1,3)  =  1.00
         g%s(11)%m(2,1)  = -1.00
         g%s(11)%m(3,2)  = -1.00
         g%s(11)%v(1)    =  0.00
         g%s(11)%v(2)    =  0.00
         g%s(11)%v(3)    =  0.00
       
         g%s(12)%m(1,2)  = -1.00
         g%s(12)%m(2,3)  =  1.00
         g%s(12)%m(3,1)  = -1.00
         g%s(12)%v(1)    =  0.00
         g%s(12)%v(2)    =  0.00
         g%s(12)%v(3)    =  0.00
       
         g%s(13)%m(1,3)  = -1.00
         g%s(13)%m(2,1)  = -1.00
         g%s(13)%m(3,2)  =  1.00
         g%s(13)%v(1)    =  0.00
         g%s(13)%v(2)    =  0.00
         g%s(13)%v(3)    =  0.00
       
         g%s(14)%m(1,3)  = -1.00
         g%s(14)%m(2,1)  =  1.00
         g%s(14)%m(3,2)  = -1.00
         g%s(14)%v(1)    =  0.00
         g%s(14)%v(2)    =  0.00
         g%s(14)%v(3)    =  0.00
       
         g%s(15)%m(1,2)  =  1.00
         g%s(15)%m(2,3)  = -1.00
         g%s(15)%m(3,1)  = -1.00
         g%s(15)%v(1)    =  0.00
         g%s(15)%v(2)    =  0.00
         g%s(15)%v(3)    =  0.00
       
         g%s(16)%m(1,1)  =  1.00
         g%s(16)%m(2,2)  = -1.00
         g%s(16)%m(3,3)  = -1.00
         g%s(16)%v(1)    =  0.00
         g%s(16)%v(2)    =  0.00
         g%s(16)%v(3)    =  0.00
       
         g%s(17)%m(1,1)  = -1.00
         g%s(17)%m(2,2)  =  1.00
         g%s(17)%m(3,3)  = -1.00
         g%s(17)%v(1)    =  0.00
         g%s(17)%v(2)    =  0.00
         g%s(17)%v(3)    =  0.00
       
         g%s(18)%m(1,1)  = -1.00
         g%s(18)%m(2,2)  = -1.00
         g%s(18)%m(3,3)  =  1.00
         g%s(18)%v(1)    =  0.00
         g%s(18)%v(2)    =  0.00
         g%s(18)%v(3)    =  0.00
       
         g%s(19)%m(1,2)  =  1.00
         g%s(19)%m(2,1)  =  1.00
         g%s(19)%m(3,3)  = -1.00
         g%s(19)%v(1)    =  0.50
         g%s(19)%v(2)    =  0.00
         g%s(19)%v(3)    =  0.00
       
         g%s(20)%m(1,2)  = -1.00
         g%s(20)%m(2,1)  = -1.00
         g%s(20)%m(3,3)  = -1.00
         g%s(20)%v(1)    =  0.50
         g%s(20)%v(2)    =  0.00
         g%s(20)%v(3)    =  0.00
       
         g%s(21)%m(1,3)  =  1.00
         g%s(21)%m(2,2)  = -1.00
         g%s(21)%m(3,1)  =  1.00
         g%s(21)%v(1)    =  0.50
         g%s(21)%v(2)    =  0.00
         g%s(21)%v(3)    =  0.00
       
         g%s(22)%m(1,3)  = -1.00
         g%s(22)%m(2,2)  = -1.00
         g%s(22)%m(3,1)  = -1.00
         g%s(22)%v(1)    =  0.50
         g%s(22)%v(2)    =  0.00
         g%s(22)%v(3)    =  0.00
       
         g%s(23)%m(1,1)  = -1.00
         g%s(23)%m(2,3)  =  1.00
         g%s(23)%m(3,2)  =  1.00
         g%s(23)%v(1)    =  0.50
         g%s(23)%v(2)    =  0.00
         g%s(23)%v(3)    =  0.00
       
         g%s(24)%m(1,1)  = -1.00
         g%s(24)%m(2,3)  = -1.00
         g%s(24)%m(3,2)  = -1.00
         g%s(24)%v(1)    =  0.50
         g%s(24)%v(2)    =  0.00
         g%s(24)%v(3)    =  0.00
       
         g%s(25)%m(1,1)  = -1.00
         g%s(25)%m(2,2)  = -1.00
         g%s(25)%m(3,3)  = -1.00
         g%s(25)%v(1)    =  0.00
         g%s(25)%v(2)    =  0.00
         g%s(25)%v(3)    =  0.00
       
         g%s(26)%m(1,1)  = -1.00
         g%s(26)%m(2,3)  =  1.00
         g%s(26)%m(3,2)  = -1.00
         g%s(26)%v(1)    = -0.50
         g%s(26)%v(2)    =  0.00
         g%s(26)%v(3)    =  0.00
       
         g%s(27)%m(1,1)  = -1.00
         g%s(27)%m(2,3)  = -1.00
         g%s(27)%m(3,2)  =  1.00
         g%s(27)%v(1)    = -0.50
         g%s(27)%v(2)    =  0.00
         g%s(27)%v(3)    =  0.00
       
         g%s(28)%m(1,3)  = -1.00
         g%s(28)%m(2,2)  = -1.00
         g%s(28)%m(3,1)  =  1.00
         g%s(28)%v(1)    = -0.50
         g%s(28)%v(2)    =  0.00
         g%s(28)%v(3)    =  0.00
       
         g%s(29)%m(1,3)  =  1.00
         g%s(29)%m(2,2)  = -1.00
         g%s(29)%m(3,1)  = -1.00
         g%s(29)%v(1)    = -0.50
         g%s(29)%v(2)    =  0.00
         g%s(29)%v(3)    =  0.00
       
         g%s(30)%m(1,2)  =  1.00
         g%s(30)%m(2,1)  = -1.00
         g%s(30)%m(3,3)  = -1.00
         g%s(30)%v(1)    = -0.50
         g%s(30)%v(2)    =  0.00
         g%s(30)%v(3)    =  0.00
       
         g%s(31)%m(1,2)  = -1.00
         g%s(31)%m(2,1)  =  1.00
         g%s(31)%m(3,3)  = -1.00
         g%s(31)%v(1)    = -0.50
         g%s(31)%v(2)    =  0.00
         g%s(31)%v(3)    =  0.00
       
         g%s(32)%m(1,3)  = -1.00
         g%s(32)%m(2,1)  = -1.00
         g%s(32)%m(3,2)  = -1.00
         g%s(32)%v(1)    =  0.00
         g%s(32)%v(2)    =  0.00
         g%s(32)%v(3)    =  0.00
       
         g%s(33)%m(1,2)  = -1.00
         g%s(33)%m(2,3)  = -1.00
         g%s(33)%m(3,1)  = -1.00
         g%s(33)%v(1)    =  0.00
         g%s(33)%v(2)    =  0.00
         g%s(33)%v(3)    =  0.00
       
         g%s(34)%m(1,2)  =  1.00
         g%s(34)%m(2,3)  =  1.00
         g%s(34)%m(3,1)  = -1.00
         g%s(34)%v(1)    =  0.00
         g%s(34)%v(2)    =  0.00
         g%s(34)%v(3)    =  0.00
       
         g%s(35)%m(1,3)  = -1.00
         g%s(35)%m(2,1)  =  1.00
         g%s(35)%m(3,2)  =  1.00
         g%s(35)%v(1)    =  0.00
         g%s(35)%v(2)    =  0.00
         g%s(35)%v(3)    =  0.00
       
         g%s(36)%m(1,2)  =  1.00
         g%s(36)%m(2,3)  = -1.00
         g%s(36)%m(3,1)  =  1.00
         g%s(36)%v(1)    =  0.00
         g%s(36)%v(2)    =  0.00
         g%s(36)%v(3)    =  0.00
       
         g%s(37)%m(1,3)  =  1.00
         g%s(37)%m(2,1)  =  1.00
         g%s(37)%m(3,2)  = -1.00
         g%s(37)%v(1)    =  0.00
         g%s(37)%v(2)    =  0.00
         g%s(37)%v(3)    =  0.00
       
         g%s(38)%m(1,3)  =  1.00
         g%s(38)%m(2,1)  = -1.00
         g%s(38)%m(3,2)  =  1.00
         g%s(38)%v(1)    =  0.00
         g%s(38)%v(2)    =  0.00
         g%s(38)%v(3)    =  0.00
       
         g%s(39)%m(1,2)  = -1.00
         g%s(39)%m(2,3)  =  1.00
         g%s(39)%m(3,1)  =  1.00
         g%s(39)%v(1)    =  0.00
         g%s(39)%v(2)    =  0.00
         g%s(39)%v(3)    =  0.00
       
         g%s(40)%m(1,1)  = -1.00
         g%s(40)%m(2,2)  =  1.00
         g%s(40)%m(3,3)  =  1.00
         g%s(40)%v(1)    =  0.00
         g%s(40)%v(2)    =  0.00
         g%s(40)%v(3)    =  0.00
       
         g%s(41)%m(1,1)  =  1.00
         g%s(41)%m(2,2)  = -1.00
         g%s(41)%m(3,3)  =  1.00
         g%s(41)%v(1)    =  0.00
         g%s(41)%v(2)    =  0.00
         g%s(41)%v(3)    =  0.00
       
         g%s(42)%m(1,1)  =  1.00
         g%s(42)%m(2,2)  =  1.00
         g%s(42)%m(3,3)  = -1.00
         g%s(42)%v(1)    =  0.00
         g%s(42)%v(2)    =  0.00
         g%s(42)%v(3)    =  0.00
       
         g%s(43)%m(1,2)  = -1.00
         g%s(43)%m(2,1)  = -1.00
         g%s(43)%m(3,3)  =  1.00
         g%s(43)%v(1)    = -0.50
         g%s(43)%v(2)    =  0.00
         g%s(43)%v(3)    =  0.00
       
         g%s(44)%m(1,2)  =  1.00
         g%s(44)%m(2,1)  =  1.00
         g%s(44)%m(3,3)  =  1.00
         g%s(44)%v(1)    = -0.50
         g%s(44)%v(2)    =  0.00
         g%s(44)%v(3)    =  0.00
       
         g%s(45)%m(1,3)  = -1.00
         g%s(45)%m(2,2)  =  1.00
         g%s(45)%m(3,1)  = -1.00
         g%s(45)%v(1)    = -0.50
         g%s(45)%v(2)    =  0.00
         g%s(45)%v(3)    =  0.00
       
         g%s(46)%m(1,3)  =  1.00
         g%s(46)%m(2,2)  =  1.00
         g%s(46)%m(3,1)  =  1.00
         g%s(46)%v(1)    = -0.50
         g%s(46)%v(2)    =  0.00
         g%s(46)%v(3)    =  0.00
       
         g%s(47)%m(1,1)  =  1.00
         g%s(47)%m(2,3)  = -1.00
         g%s(47)%m(3,2)  = -1.00
         g%s(47)%v(1)    = -0.50
         g%s(47)%v(2)    =  0.00
         g%s(47)%v(3)    =  0.00
       
         g%s(48)%m(1,1)  =  1.00
         g%s(48)%m(2,3)  =  1.00
         g%s(48)%m(3,2)  =  1.00
         g%s(48)%v(1)    = -0.50
         g%s(48)%v(2)    =  0.00
         g%s(48)%v(3)    =  0.00
       
         g%s(49)%m(1,1)  =  1.00
         g%s(49)%m(2,2)  =  1.00
         g%s(49)%m(3,3)  =  1.00
         g%s(49)%v(1)    =  0.00
         g%s(49)%v(2)    =  0.50
         g%s(49)%v(3)    =  0.50
       
         g%s(50)%m(1,1)  =  1.00
         g%s(50)%m(2,3)  = -1.00
         g%s(50)%m(3,2)  =  1.00
         g%s(50)%v(1)    =  0.50
         g%s(50)%v(2)    =  0.50
         g%s(50)%v(3)    =  0.50
       
         g%s(51)%m(1,1)  =  1.00
         g%s(51)%m(2,3)  =  1.00
         g%s(51)%m(3,2)  = -1.00
         g%s(51)%v(1)    =  0.50
         g%s(51)%v(2)    =  0.50
         g%s(51)%v(3)    =  0.50
       
         g%s(52)%m(1,3)  =  1.00
         g%s(52)%m(2,2)  =  1.00
         g%s(52)%m(3,1)  = -1.00
         g%s(52)%v(1)    =  0.50
         g%s(52)%v(2)    =  0.50
         g%s(52)%v(3)    =  0.50
       
         g%s(53)%m(1,3)  = -1.00
         g%s(53)%m(2,2)  =  1.00
         g%s(53)%m(3,1)  =  1.00
         g%s(53)%v(1)    =  0.50
         g%s(53)%v(2)    =  0.50
         g%s(53)%v(3)    =  0.50
       
         g%s(54)%m(1,2)  = -1.00
         g%s(54)%m(2,1)  =  1.00
         g%s(54)%m(3,3)  =  1.00
         g%s(54)%v(1)    =  0.50
         g%s(54)%v(2)    =  0.50
         g%s(54)%v(3)    =  0.50
       
         g%s(55)%m(1,2)  =  1.00
         g%s(55)%m(2,1)  = -1.00
         g%s(55)%m(3,3)  =  1.00
         g%s(55)%v(1)    =  0.50
         g%s(55)%v(2)    =  0.50
         g%s(55)%v(3)    =  0.50
       
         g%s(56)%m(1,3)  =  1.00
         g%s(56)%m(2,1)  =  1.00
         g%s(56)%m(3,2)  =  1.00
         g%s(56)%v(1)    =  0.00
         g%s(56)%v(2)    =  0.50
         g%s(56)%v(3)    =  0.50
       
         g%s(57)%m(1,2)  =  1.00
         g%s(57)%m(2,3)  =  1.00
         g%s(57)%m(3,1)  =  1.00
         g%s(57)%v(1)    =  0.00
         g%s(57)%v(2)    =  0.50
         g%s(57)%v(3)    =  0.50
       
         g%s(58)%m(1,2)  = -1.00
         g%s(58)%m(2,3)  = -1.00
         g%s(58)%m(3,1)  =  1.00
         g%s(58)%v(1)    =  0.00
         g%s(58)%v(2)    =  0.50
         g%s(58)%v(3)    =  0.50
       
         g%s(59)%m(1,3)  =  1.00
         g%s(59)%m(2,1)  = -1.00
         g%s(59)%m(3,2)  = -1.00
         g%s(59)%v(1)    =  0.00
         g%s(59)%v(2)    =  0.50
         g%s(59)%v(3)    =  0.50
       
         g%s(60)%m(1,2)  = -1.00
         g%s(60)%m(2,3)  =  1.00
         g%s(60)%m(3,1)  = -1.00
         g%s(60)%v(1)    =  0.00
         g%s(60)%v(2)    =  0.50
         g%s(60)%v(3)    =  0.50
       
         g%s(61)%m(1,3)  = -1.00
         g%s(61)%m(2,1)  = -1.00
         g%s(61)%m(3,2)  =  1.00
         g%s(61)%v(1)    =  0.00
         g%s(61)%v(2)    =  0.50
         g%s(61)%v(3)    =  0.50
       
         g%s(62)%m(1,3)  = -1.00
         g%s(62)%m(2,1)  =  1.00
         g%s(62)%m(3,2)  = -1.00
         g%s(62)%v(1)    =  0.00
         g%s(62)%v(2)    =  0.50
         g%s(62)%v(3)    =  0.50
       
         g%s(63)%m(1,2)  =  1.00
         g%s(63)%m(2,3)  = -1.00
         g%s(63)%m(3,1)  = -1.00
         g%s(63)%v(1)    =  0.00
         g%s(63)%v(2)    =  0.50
         g%s(63)%v(3)    =  0.50
       
         g%s(64)%m(1,1)  =  1.00
         g%s(64)%m(2,2)  = -1.00
         g%s(64)%m(3,3)  = -1.00
         g%s(64)%v(1)    =  0.00
         g%s(64)%v(2)    =  0.50
         g%s(64)%v(3)    =  0.50
       
         g%s(65)%m(1,1)  = -1.00
         g%s(65)%m(2,2)  =  1.00
         g%s(65)%m(3,3)  = -1.00
         g%s(65)%v(1)    =  0.00
         g%s(65)%v(2)    =  0.50
         g%s(65)%v(3)    =  0.50
       
         g%s(66)%m(1,1)  = -1.00
         g%s(66)%m(2,2)  = -1.00
         g%s(66)%m(3,3)  =  1.00
         g%s(66)%v(1)    =  0.00
         g%s(66)%v(2)    =  0.50
         g%s(66)%v(3)    =  0.50
       
         g%s(67)%m(1,2)  =  1.00
         g%s(67)%m(2,1)  =  1.00
         g%s(67)%m(3,3)  = -1.00
         g%s(67)%v(1)    =  0.50
         g%s(67)%v(2)    =  0.50
         g%s(67)%v(3)    =  0.50
       
         g%s(68)%m(1,2)  = -1.00
         g%s(68)%m(2,1)  = -1.00
         g%s(68)%m(3,3)  = -1.00
         g%s(68)%v(1)    =  0.50
         g%s(68)%v(2)    =  0.50
         g%s(68)%v(3)    =  0.50
       
         g%s(69)%m(1,3)  =  1.00
         g%s(69)%m(2,2)  = -1.00
         g%s(69)%m(3,1)  =  1.00
         g%s(69)%v(1)    =  0.50
         g%s(69)%v(2)    =  0.50
         g%s(69)%v(3)    =  0.50
       
         g%s(70)%m(1,3)  = -1.00
         g%s(70)%m(2,2)  = -1.00
         g%s(70)%m(3,1)  = -1.00
         g%s(70)%v(1)    =  0.50
         g%s(70)%v(2)    =  0.50
         g%s(70)%v(3)    =  0.50
       
         g%s(71)%m(1,1)  = -1.00
         g%s(71)%m(2,3)  =  1.00
         g%s(71)%m(3,2)  =  1.00
         g%s(71)%v(1)    =  0.50
         g%s(71)%v(2)    =  0.50
         g%s(71)%v(3)    =  0.50
       
         g%s(72)%m(1,1)  = -1.00
         g%s(72)%m(2,3)  = -1.00
         g%s(72)%m(3,2)  = -1.00
         g%s(72)%v(1)    =  0.50
         g%s(72)%v(2)    =  0.50
         g%s(72)%v(3)    =  0.50
       
         g%s(73)%m(1,1)  = -1.00
         g%s(73)%m(2,2)  = -1.00
         g%s(73)%m(3,3)  = -1.00
         g%s(73)%v(1)    =  0.00
         g%s(73)%v(2)    =  0.50
         g%s(73)%v(3)    =  0.50
       
         g%s(74)%m(1,1)  = -1.00
         g%s(74)%m(2,3)  =  1.00
         g%s(74)%m(3,2)  = -1.00
         g%s(74)%v(1)    = -0.50
         g%s(74)%v(2)    =  0.50
         g%s(74)%v(3)    =  0.50
       
         g%s(75)%m(1,1)  = -1.00
         g%s(75)%m(2,3)  = -1.00
         g%s(75)%m(3,2)  =  1.00
         g%s(75)%v(1)    = -0.50
         g%s(75)%v(2)    =  0.50
         g%s(75)%v(3)    =  0.50
       
         g%s(76)%m(1,3)  = -1.00
         g%s(76)%m(2,2)  = -1.00
         g%s(76)%m(3,1)  =  1.00
         g%s(76)%v(1)    = -0.50
         g%s(76)%v(2)    =  0.50
         g%s(76)%v(3)    =  0.50
       
         g%s(77)%m(1,3)  =  1.00
         g%s(77)%m(2,2)  = -1.00
         g%s(77)%m(3,1)  = -1.00
         g%s(77)%v(1)    = -0.50
         g%s(77)%v(2)    =  0.50
         g%s(77)%v(3)    =  0.50
       
         g%s(78)%m(1,2)  =  1.00
         g%s(78)%m(2,1)  = -1.00
         g%s(78)%m(3,3)  = -1.00
         g%s(78)%v(1)    = -0.50
         g%s(78)%v(2)    =  0.50
         g%s(78)%v(3)    =  0.50
       
         g%s(79)%m(1,2)  = -1.00
         g%s(79)%m(2,1)  =  1.00
         g%s(79)%m(3,3)  = -1.00
         g%s(79)%v(1)    = -0.50
         g%s(79)%v(2)    =  0.50
         g%s(79)%v(3)    =  0.50
       
         g%s(80)%m(1,3)  = -1.00
         g%s(80)%m(2,1)  = -1.00
         g%s(80)%m(3,2)  = -1.00
         g%s(80)%v(1)    =  0.00
         g%s(80)%v(2)    =  0.50
         g%s(80)%v(3)    =  0.50
       
         g%s(81)%m(1,2)  = -1.00
         g%s(81)%m(2,3)  = -1.00
         g%s(81)%m(3,1)  = -1.00
         g%s(81)%v(1)    =  0.00
         g%s(81)%v(2)    =  0.50
         g%s(81)%v(3)    =  0.50
       
         g%s(82)%m(1,2)  =  1.00
         g%s(82)%m(2,3)  =  1.00
         g%s(82)%m(3,1)  = -1.00
         g%s(82)%v(1)    =  0.00
         g%s(82)%v(2)    =  0.50
         g%s(82)%v(3)    =  0.50
       
         g%s(83)%m(1,3)  = -1.00
         g%s(83)%m(2,1)  =  1.00
         g%s(83)%m(3,2)  =  1.00
         g%s(83)%v(1)    =  0.00
         g%s(83)%v(2)    =  0.50
         g%s(83)%v(3)    =  0.50
       
         g%s(84)%m(1,2)  =  1.00
         g%s(84)%m(2,3)  = -1.00
         g%s(84)%m(3,1)  =  1.00
         g%s(84)%v(1)    =  0.00
         g%s(84)%v(2)    =  0.50
         g%s(84)%v(3)    =  0.50
       
         g%s(85)%m(1,3)  =  1.00
         g%s(85)%m(2,1)  =  1.00
         g%s(85)%m(3,2)  = -1.00
         g%s(85)%v(1)    =  0.00
         g%s(85)%v(2)    =  0.50
         g%s(85)%v(3)    =  0.50
       
         g%s(86)%m(1,3)  =  1.00
         g%s(86)%m(2,1)  = -1.00
         g%s(86)%m(3,2)  =  1.00
         g%s(86)%v(1)    =  0.00
         g%s(86)%v(2)    =  0.50
         g%s(86)%v(3)    =  0.50
       
         g%s(87)%m(1,2)  = -1.00
         g%s(87)%m(2,3)  =  1.00
         g%s(87)%m(3,1)  =  1.00
         g%s(87)%v(1)    =  0.00
         g%s(87)%v(2)    =  0.50
         g%s(87)%v(3)    =  0.50
       
         g%s(88)%m(1,1)  = -1.00
         g%s(88)%m(2,2)  =  1.00
         g%s(88)%m(3,3)  =  1.00
         g%s(88)%v(1)    =  0.00
         g%s(88)%v(2)    =  0.50
         g%s(88)%v(3)    =  0.50
       
         g%s(89)%m(1,1)  =  1.00
         g%s(89)%m(2,2)  = -1.00
         g%s(89)%m(3,3)  =  1.00
         g%s(89)%v(1)    =  0.00
         g%s(89)%v(2)    =  0.50
         g%s(89)%v(3)    =  0.50
       
         g%s(90)%m(1,1)  =  1.00
         g%s(90)%m(2,2)  =  1.00
         g%s(90)%m(3,3)  = -1.00
         g%s(90)%v(1)    =  0.00
         g%s(90)%v(2)    =  0.50
         g%s(90)%v(3)    =  0.50
       
         g%s(91)%m(1,2)  = -1.00
         g%s(91)%m(2,1)  = -1.00
         g%s(91)%m(3,3)  =  1.00
         g%s(91)%v(1)    = -0.50
         g%s(91)%v(2)    =  0.50
         g%s(91)%v(3)    =  0.50
       
         g%s(92)%m(1,2)  =  1.00
         g%s(92)%m(2,1)  =  1.00
         g%s(92)%m(3,3)  =  1.00
         g%s(92)%v(1)    = -0.50
         g%s(92)%v(2)    =  0.50
         g%s(92)%v(3)    =  0.50
       
         g%s(93)%m(1,3)  = -1.00
         g%s(93)%m(2,2)  =  1.00
         g%s(93)%m(3,1)  = -1.00
         g%s(93)%v(1)    = -0.50
         g%s(93)%v(2)    =  0.50
         g%s(93)%v(3)    =  0.50
       
         g%s(94)%m(1,3)  =  1.00
         g%s(94)%m(2,2)  =  1.00
         g%s(94)%m(3,1)  =  1.00
         g%s(94)%v(1)    = -0.50
         g%s(94)%v(2)    =  0.50
         g%s(94)%v(3)    =  0.50
       
         g%s(95)%m(1,1)  =  1.00
         g%s(95)%m(2,3)  = -1.00
         g%s(95)%m(3,2)  = -1.00
         g%s(95)%v(1)    = -0.50
         g%s(95)%v(2)    =  0.50
         g%s(95)%v(3)    =  0.50
       
         g%s(96)%m(1,1)  =  1.00
         g%s(96)%m(2,3)  =  1.00
         g%s(96)%m(3,2)  =  1.00
         g%s(96)%v(1)    = -0.50
         g%s(96)%v(2)    =  0.50
         g%s(96)%v(3)    =  0.50
       
         g%s(97)%m(1,1)  =  1.00
         g%s(97)%m(2,2)  =  1.00
         g%s(97)%m(3,3)  =  1.00
         g%s(97)%v(1)    =  0.50
         g%s(97)%v(2)    =  0.00
         g%s(97)%v(3)    =  0.50
       
         g%s(98)%m(1,1)  =  1.00
         g%s(98)%m(2,3)  = -1.00
         g%s(98)%m(3,2)  =  1.00
         g%s(98)%v(1)    =  1.00000000
         g%s(98)%v(2)    =  0.00
         g%s(98)%v(3)    =  0.50
       
         g%s(99)%m(1,1)  =  1.00
         g%s(99)%m(2,3)  =  1.00
         g%s(99)%m(3,2)  = -1.00
         g%s(99)%v(1)    =  1.00000000
         g%s(99)%v(2)    =  0.00
         g%s(99)%v(3)    =  0.50
       
         g%s(100)%m(1,3) =  1.00
         g%s(100)%m(2,2) =  1.00
         g%s(100)%m(3,1) = -1.00
         g%s(100)%v(1)   =  1.00000000
         g%s(100)%v(2)   =  0.00
         g%s(100)%v(3)   =  0.50
       
         g%s(101)%m(1,3) = -1.00
         g%s(101)%m(2,2) =  1.00
         g%s(101)%m(3,1) =  1.00
         g%s(101)%v(1)   =  1.00000000
         g%s(101)%v(2)   =  0.00
         g%s(101)%v(3)   =  0.50
       
         g%s(102)%m(1,2) = -1.00
         g%s(102)%m(2,1) =  1.00
         g%s(102)%m(3,3) =  1.00
         g%s(102)%v(1)   =  1.00000000
         g%s(102)%v(2)   =  0.00
         g%s(102)%v(3)   =  0.50
       
         g%s(103)%m(1,2) =  1.00
         g%s(103)%m(2,1) = -1.00
         g%s(103)%m(3,3) =  1.00
         g%s(103)%v(1)   =  1.00000000
         g%s(103)%v(2)   =  0.00
         g%s(103)%v(3)   =  0.50
       
         g%s(104)%m(1,3) =  1.00
         g%s(104)%m(2,1) =  1.00
         g%s(104)%m(3,2) =  1.00
         g%s(104)%v(1)   =  0.50
         g%s(104)%v(2)   =  0.00
         g%s(104)%v(3)   =  0.50
       
         g%s(105)%m(1,2) =  1.00
         g%s(105)%m(2,3) =  1.00
         g%s(105)%m(3,1) =  1.00
         g%s(105)%v(1)   =  0.50
         g%s(105)%v(2)   =  0.00
         g%s(105)%v(3)   =  0.50
       
         g%s(106)%m(1,2) = -1.00
         g%s(106)%m(2,3) = -1.00
         g%s(106)%m(3,1) =  1.00
         g%s(106)%v(1)   =  0.50
         g%s(106)%v(2)   =  0.00
         g%s(106)%v(3)   =  0.50
       
         g%s(107)%m(1,3) =  1.00
         g%s(107)%m(2,1) = -1.00
         g%s(107)%m(3,2) = -1.00
         g%s(107)%v(1)   =  0.50
         g%s(107)%v(2)   =  0.00
         g%s(107)%v(3)   =  0.50
       
         g%s(108)%m(1,2) = -1.00
         g%s(108)%m(2,3) =  1.00
         g%s(108)%m(3,1) = -1.00
         g%s(108)%v(1)   =  0.50
         g%s(108)%v(2)   =  0.00
         g%s(108)%v(3)   =  0.50
       
         g%s(109)%m(1,3) = -1.00
         g%s(109)%m(2,1) = -1.00
         g%s(109)%m(3,2) =  1.00
         g%s(109)%v(1)   =  0.50
         g%s(109)%v(2)   =  0.00
         g%s(109)%v(3)   =  0.50
       
         g%s(110)%m(1,3) = -1.00
         g%s(110)%m(2,1) =  1.00
         g%s(110)%m(3,2) = -1.00
         g%s(110)%v(1)   =  0.50
         g%s(110)%v(2)   =  0.00
         g%s(110)%v(3)   =  0.50
       
         g%s(111)%m(1,2) =  1.00
         g%s(111)%m(2,3) = -1.00
         g%s(111)%m(3,1) = -1.00
         g%s(111)%v(1)   =  0.50
         g%s(111)%v(2)   =  0.00
         g%s(111)%v(3)   =  0.50
       
         g%s(112)%m(1,1) =  1.00
         g%s(112)%m(2,2) = -1.00
         g%s(112)%m(3,3) = -1.00
         g%s(112)%v(1)   =  0.50
         g%s(112)%v(2)   =  0.00
         g%s(112)%v(3)   =  0.50
       
         g%s(113)%m(1,1) = -1.00
         g%s(113)%m(2,2) =  1.00
         g%s(113)%m(3,3) = -1.00
         g%s(113)%v(1)   =  0.50
         g%s(113)%v(2)   =  0.00
         g%s(113)%v(3)   =  0.50
       
         g%s(114)%m(1,1) = -1.00
         g%s(114)%m(2,2) = -1.00
         g%s(114)%m(3,3) =  1.00
         g%s(114)%v(1)   =  0.50
         g%s(114)%v(2)   =  0.00
         g%s(114)%v(3)   =  0.50
       
         g%s(115)%m(1,2) =  1.00
         g%s(115)%m(2,1) =  1.00
         g%s(115)%m(3,3) = -1.00
         g%s(115)%v(1)   =  1.00000000
         g%s(115)%v(2)   =  0.00
         g%s(115)%v(3)   =  0.50
       
         g%s(116)%m(1,2) = -1.00
         g%s(116)%m(2,1) = -1.00
         g%s(116)%m(3,3) = -1.00
         g%s(116)%v(1)   =  1.00000000
         g%s(116)%v(2)   =  0.00
         g%s(116)%v(3)   =  0.50
       
         g%s(117)%m(1,3) =  1.00
         g%s(117)%m(2,2) = -1.00
         g%s(117)%m(3,1) =  1.00
         g%s(117)%v(1)   =  1.00000000
         g%s(117)%v(2)   =  0.00
         g%s(117)%v(3)   =  0.50
       
         g%s(118)%m(1,3) = -1.00
         g%s(118)%m(2,2) = -1.00
         g%s(118)%m(3,1) = -1.00
         g%s(118)%v(1)   =  1.00000000
         g%s(118)%v(2)   =  0.00
         g%s(118)%v(3)   =  0.50
       
         g%s(119)%m(1,1) = -1.00
         g%s(119)%m(2,3) =  1.00
         g%s(119)%m(3,2) =  1.00
         g%s(119)%v(1)   =  1.00000000
         g%s(119)%v(2)   =  0.00
         g%s(119)%v(3)   =  0.50
       
         g%s(120)%m(1,1) = -1.00
         g%s(120)%m(2,3) = -1.00
         g%s(120)%m(3,2) = -1.00
         g%s(120)%v(1)   =  1.00000000
         g%s(120)%v(2)   =  0.00
         g%s(120)%v(3)   =  0.50
       
         g%s(121)%m(1,1) = -1.00
         g%s(121)%m(2,2) = -1.00
         g%s(121)%m(3,3) = -1.00
         g%s(121)%v(1)   =  0.50
         g%s(121)%v(2)   =  0.00
         g%s(121)%v(3)   =  0.50
       
         g%s(122)%m(1,1) = -1.00
         g%s(122)%m(2,3) =  1.00
         g%s(122)%m(3,2) = -1.00
         g%s(122)%v(1)   =  0.00
         g%s(122)%v(2)   =  0.00
         g%s(122)%v(3)   =  0.50
       
         g%s(123)%m(1,1) = -1.00
         g%s(123)%m(2,3) = -1.00
         g%s(123)%m(3,2) =  1.00
         g%s(123)%v(1)   =  0.00
         g%s(123)%v(2)   =  0.00
         g%s(123)%v(3)   =  0.50
       
         g%s(124)%m(1,3) = -1.00
         g%s(124)%m(2,2) = -1.00
         g%s(124)%m(3,1) =  1.00
         g%s(124)%v(1)   =  0.00
         g%s(124)%v(2)   =  0.00
         g%s(124)%v(3)   =  0.50
       
         g%s(125)%m(1,3) =  1.00
         g%s(125)%m(2,2) = -1.00
         g%s(125)%m(3,1) = -1.00
         g%s(125)%v(1)   =  0.00
         g%s(125)%v(2)   =  0.00
         g%s(125)%v(3)   =  0.50
       
         g%s(126)%m(1,2) =  1.00
         g%s(126)%m(2,1) = -1.00
         g%s(126)%m(3,3) = -1.00
         g%s(126)%v(1)   =  0.00
         g%s(126)%v(2)   =  0.00
         g%s(126)%v(3)   =  0.50
       
         g%s(127)%m(1,2) = -1.00
         g%s(127)%m(2,1) =  1.00
         g%s(127)%m(3,3) = -1.00
         g%s(127)%v(1)   =  0.00
         g%s(127)%v(2)   =  0.00
         g%s(127)%v(3)   =  0.50
       
         g%s(128)%m(1,3) = -1.00
         g%s(128)%m(2,1) = -1.00
         g%s(128)%m(3,2) = -1.00
         g%s(128)%v(1)   =  0.50
         g%s(128)%v(2)   =  0.00
         g%s(128)%v(3)   =  0.50
       
         g%s(129)%m(1,2) = -1.00
         g%s(129)%m(2,3) = -1.00
         g%s(129)%m(3,1) = -1.00
         g%s(129)%v(1)   =  0.50
         g%s(129)%v(2)   =  0.00
         g%s(129)%v(3)   =  0.50
       
         g%s(130)%m(1,2) =  1.00
         g%s(130)%m(2,3) =  1.00
         g%s(130)%m(3,1) = -1.00
         g%s(130)%v(1)   =  0.50
         g%s(130)%v(2)   =  0.00
         g%s(130)%v(3)   =  0.50
       
         g%s(131)%m(1,3) = -1.00
         g%s(131)%m(2,1) =  1.00
         g%s(131)%m(3,2) =  1.00
         g%s(131)%v(1)   =  0.50
         g%s(131)%v(2)   =  0.00
         g%s(131)%v(3)   =  0.50
       
         g%s(132)%m(1,2) =  1.00
         g%s(132)%m(2,3) = -1.00
         g%s(132)%m(3,1) =  1.00
         g%s(132)%v(1)   =  0.50
         g%s(132)%v(2)   =  0.00
         g%s(132)%v(3)   =  0.50
       
         g%s(133)%m(1,3) =  1.00
         g%s(133)%m(2,1) =  1.00
         g%s(133)%m(3,2) = -1.00
         g%s(133)%v(1)   =  0.50
         g%s(133)%v(2)   =  0.00
         g%s(133)%v(3)   =  0.50
       
         g%s(134)%m(1,3) =  1.00
         g%s(134)%m(2,1) = -1.00
         g%s(134)%m(3,2) =  1.00
         g%s(134)%v(1)   =  0.50
         g%s(134)%v(2)   =  0.00
         g%s(134)%v(3)   =  0.50
       
         g%s(135)%m(1,2) = -1.00
         g%s(135)%m(2,3) =  1.00
         g%s(135)%m(3,1) =  1.00
         g%s(135)%v(1)   =  0.50
         g%s(135)%v(2)   =  0.00
         g%s(135)%v(3)   =  0.50
       
         g%s(136)%m(1,1) = -1.00
         g%s(136)%m(2,2) =  1.00
         g%s(136)%m(3,3) =  1.00
         g%s(136)%v(1)   =  0.50
         g%s(136)%v(2)   =  0.00
         g%s(136)%v(3)   =  0.50
       
         g%s(137)%m(1,1) =  1.00
         g%s(137)%m(2,2) = -1.00
         g%s(137)%m(3,3) =  1.00
         g%s(137)%v(1)   =  0.50
         g%s(137)%v(2)   =  0.00
         g%s(137)%v(3)   =  0.50
       
         g%s(138)%m(1,1) =  1.00
         g%s(138)%m(2,2) =  1.00
         g%s(138)%m(3,3) = -1.00
         g%s(138)%v(1)   =  0.50
         g%s(138)%v(2)   =  0.00
         g%s(138)%v(3)   =  0.50
       
         g%s(139)%m(1,2) = -1.00
         g%s(139)%m(2,1) = -1.00
         g%s(139)%m(3,3) =  1.00
         g%s(139)%v(1)   =  0.00
         g%s(139)%v(2)   =  0.00
         g%s(139)%v(3)   =  0.50
       
         g%s(140)%m(1,2) =  1.00
         g%s(140)%m(2,1) =  1.00
         g%s(140)%m(3,3) =  1.00
         g%s(140)%v(1)   =  0.00
         g%s(140)%v(2)   =  0.00
         g%s(140)%v(3)   =  0.50
       
         g%s(141