% MATLAB software that solves the standard one-agent neoclassical stochastic
% growth model using an extended function path (EFP) algorithm, as described  
% in the article "A Tractable Framework for Analyzing a Class of Nonstationary 
% Markov Models" by Lilia Maliar, Serguei Maliar, John B. Taylor and Inna 
% Tsener (2020), forthcoming in Quantitative Economics (earlier 2015 version 
% is available from NBER working paper # 21155). 
%
% First version: April 12, 2014.
%
% This version: April 4, 2020. 
%
% ------------------------------------------------------------------------
% This software includes the following files:
% ------------------------------------------------------------------------
%
% 1. "Main_EFP.m"          computes an EFP solution to the nonstationary
%                          neoclassical stochastic growth model with labor 
%                          augmenting technological progress and compares 
%                          it with a balanced growth path solution
% 2. "GH_Quadrature.m"     constructs integration nodes and weights for
%                          Gauss-Hermite quadrature integration; written 
%                          by Lilia Maliar and Serguei Maliar for the
%                          article "Numerically Stable and Accurate
%                          Stochastic Simulation Approaches for Solving
%                          Dynamic Economic Models", Quantitative Economics
%                          2/2, 173-210 by Kenneth L. Judd, Lilia Maliar and
%                          Serguei Maliar (henceforth, JMM (2011))
% 3. "Ord_Polynomial_N.m"  constructs the basis functions of complete ordinary
%                          polynomial of the degrees from one to five for
%                          N variables; written by Lilia Maliar and
%                          Serguei Maliar for the JMM (2011) article
% 4. "smolyak_grid_2d.txt" lists 2-dimensional Smolyak grid points for the
%                          2nd level of approximation; see the software
%                          accompanying the article "Smolyak Method for Solving
%                          Dynamic Economic Models: Lagrange
%                          Interpolation, Anisotropic Grid and Adaptive
%                          Domain", Journal of Economic Dynamics and
%                          Control 44(C), 92-123 by Kenneth L. Judd, Lilia 
%                          Maliar, Serguei Maliar and Rafael Valero, (2014) 
%                          (henceforth, JMMV (2014))
% 5. "epsi.mat"            contains the series of shocks of length 1,000 that 
%                          are used for simulation and comparison of EFP and 
%                          balanced growth solutions
% ------------------------------------------------------------------------
% Copyright  2014, 2015, 2020 by Lilia Maliar, Serguei Maliar, John B. Taylor 
% and Inna Tsener. All rights reserved. The code may be used, modified and
% redistributed under the terms provided in the file "License_Agreement.txt".
% ------------------------------------------------------------------------

clc;
clear all;

fprintf('EXTENDED FUNCTION PATH (EFP) method \n')
fprintf('for solving a nonstationary growth model \n')
fprintf(' \n')

fprintf('EFP INITIALIZATION ... ' )

% 1. Model's parameters
% ---------------------
gam     = 0.1;         % Utility-function parameter
alpha   = 0.36;      % Capital share in output
beta    = 0.99;      % Discount factor
delta   = 0.025;     % Depreciation rate 
rho     = 0.95;      % Persistence of the log of productivity level
sigma   = 0.03;      % Standard deviation of shocks to the log of 
                     % productivity level
A0      = 1;         % Initial value of the process for labor augmenting 
                     % technological progress
gamA    = 1.01;      % Growth rate of labor augmenting technological progress

% 2. Algorithm's parameter
% ------------------------
T       = 200;       % The EFP constructs a path of time-dependent decision 
                     % functions for t=1,...,T  

% 3. Steady state of capital
% --------------------------
ks = ( (gamA^gam-beta+beta*delta) / (alpha*beta) )^(1/(alpha-1) );
                     % Steady state capital in the balanced growth model

% 4. Labor augmenting technological progress
% ------------------------------------------
At = A0*gamA.^(0:1:T)'; % Compute the level of labor augmenting technological 
                        % progress in each period of time      
                                                                       
% 5. Load the Smolyak grid points
% -------------------------------
load smolyak_grid_2d.txt            % Load the Smolyak grid for the 2-dimensional 
                                    % case (two state variables are capital, 
                                    % k, and productivity, z)
grids = smolyak_grid_2d;            % Call the matrix of grid points "grids"

J = size(grids,1);                  % Number of Smolyak grid points 
                       
% 6. The matrices of the polynomial coefficients
% ----------------------------------------------
Degree_polyn = 3;      % The decision functions are approximated by ordinary 
                       % polynomial of degree 3; to compute polynomial 
                       % approximations of higher degrees, the grid in 
                       % "smolyak_grid_2d.txt" must be modified to include 
                       % more points; this must be done, for example, for a
                       % 3rd level Smolyak grid; see JMMV (2014)
Number_coeff = 10;     % Number of coefficients in ordinary polynomial with 
                       % two state variables is 
                       % 1 + Degree_polyn + Degree_polyn(Degree_polyn+1)/2
BK = zeros(Number_coeff,T); 
                       % Matrix of polynomial coefficients of the path for 
                       % capital decision function; Number_coeff-by-T 
              
% 7. Choose an integration method for computing solutions
% -------------------------------------------------------                                              
IM  = 10;        % 1,2,...,10 = Gauss-Hermite quadrature rules with 1,2,...,10 
                 % nodes, respectively
[n_nodes,epsi_nodes,weight_nodes] = GH_Quadrature(IM,1,sigma^2);
                 % n_nodes is the number of integration nodes; epsi_nodes  
                 % are integration nodes; weight_nodes are integration 
                 % weights for Gauss-Hermite quadrature integration rule
                 % with IM nodes; for a one-dimensional integral, n_nodes  
                 % coincides with IM
                       
% 8. Construct a growth path and a sequence of domains on which the capital 
% decision functions will be constructed
% ----------------------------------------------------------------------- 
ks_path = ks*gamA.^[0:1:T]';    
            % Compute the deterministic growth path for capital that starts 
            % from the steady state of the associated balanced growth model 
            % and that grows at the rate of technological progress, gamA; 
            % see Section 4.1 in MMTT (2020) for an analytical derivation 
            % of the growth path for the balanced growth model
    
vk_min = 0.95*ks_path;   vk_max = 1.05*ks_path;        
            % Set the limits of a grid for capital k around its 
            % deterministic path to be in the interval [0.95k, 1.05k];
            % (T+1)-by-1
    
vz_min = 0.95*ones(T+1,1); vz_max = 1.05*ones(T+1,1);
            % Set the limits of a grid for productivity level, z, to be 
            % in the interval [0.95, 1.05]; in the steady state, z=1 for
            % all t; (T+1)-by-1

fprintf('completed!\n')  % End the initialization 
fprintf(' \n')
                                                           
% _________________________________________________________________________
%
% EFP: STEP 1
% 
% Compute a solution to the T-period stationary economy, i.e., assume that
% the economy stops growing in period T and remains stationary forever and 
% construct a stationary Markov solution using the conventional (Smolyak) 
% projection method
% _________________________________________________________________________
%   

fprintf('EFP STEP 1: Solving the T-period stationary economy ... \n')

% 9.1 Algorithm's parameters  
% --------------------------
kdamp = 0.05;      % Damping parameter for (fixed-point) iteration on 
                   % the coefficients of the capital decision function;
                   % To attain convergence under some parameterizations
                   % the damping parameter must be reduced to kdamp = 0.01
                                   
% 9.2 Initialize the capital series
% ---------------------------------
k1_old = zeros(J,1);   % Initialize the capital decision function on the 
                       % grid; this function is used to check the convergence 
                       % on the subsequent iteration; J-by-1

% 9.3 Compute the solution to a T-period stationary economy
% ---------------------------------------------------------
    
for t = T              % We compute the T-period stationary solution; 
                       % we will also use it to construct initial guess for 
                       % the sequence of the decision functions for periods
                       % t=1,...,T-1 in order to initialize STEP 2 of EFP 
        
    dif1 = 1e+10;      % Set the initial difference between the grid values 
                       % from two iterations in the convergence criterion to 
                       % a very large number                               

    gridk = (grids(:,1).*(vk_max(t)-vk_min(t))+vk_max(t)+vk_min(t))/2;
    gridz = (grids(:,2).*(vz_max(t)-vz_min(t))+vz_max(t)+vz_min(t))/2;
                 % Adjust the Smolyak grid on a square [-1,1]x[-1,1] to
                 % that on a rectangle [k_min,k_max]x[z_min,z_max]
    
    z1 = gridz.^rho*exp(epsi_nodes');
                 % Compute next period productivity levels corresponding to
                 % the quadrature integration nodes; J-by-n_nodes
                 
    bk = [0; 0.95; zeros(Number_coeff-2,1)];
                 % Set an initial guess on the coefficients of approximating 
                 % polynomials to match the deterministic  steady state
                 

    % An iterative cycle
    % ------------------
    
    while dif1 > 1e-8;   % Check the convergence of decision function
         
         % Compute the values of capital on the grid
         % -----------------------------------------
         X = Ord_Polynomial_N([gridk gridz],Degree_polyn); 
                                 % Construct the matrix of basis 
                                 % functions at t=T; J-by-Number_coeff
         k1 = X*bk;              % Compute next period capital using bk


         % Compute the values of consumption, c, using the budget
         % ------------------------------------------------------
         c    = At(t)^(1-alpha)*gridk.^alpha.*gridz  + (1-delta)*gridk-k1;

         % Approximate conditional expectation, y, using the chosen 
         % Gauss-Hermite quadrature rule
         % ----------------------------------------------------------
         for i = 1:J    % For each grid point,...
              
              k2 = Ord_Polynomial_N([k1(i)*ones(n_nodes,1) z1(i,:)'],Degree_polyn)*bk;
              % k2 is state-contingent capital of  period t+2 (chosen at t+1); 
              % n_nodes-by-1
                   
              c1(i,:) = At(t+1)^(1-alpha)*k1(i).^alpha.*z1(i,:)'  + (1-delta)*k1(i)-k2;
              % c1 is state-contingent next-period consumption; J-by-n_nodes

              k1_dupl = k1(i)*ones(1,n_nodes);  
              % Duplicate k1 n_nodes times to create a matrix with n_nodes 
              % identical columns; 1-by-n_nodes
              
              c_dupl = c(i)*ones(1,n_nodes);
              % Duplicate c n_nodes times to create a matrix with n_nodes 
              % identical columns; 1-by-n_nodes 
            
              y(i,1)  =  beta*c1(i,:).^(-gam)./c_dupl.^(-gam).*(1-delta+At(t+1)^(1-alpha)*alpha*k1_dupl.^(alpha-1).*z1(i,:)).*k1_dupl*weight_nodes;
              % Compute the values of capital decision funcion using the 
              % Euler equation which we rewrite as f(*)=1 and premultiply 
              % by k(t+1) so that y(t)=f(*)*k(t+1) has expected value k(t+1) 
         
         end

         % Evaluate the average percentage (unit-free) difference between  
         % the capital decision functions on two subsequent iterations
         % across all grid points
         % -------------------------------------------------------------

         dif1 = mean(abs(1-k1./k1_old))/kdamp;
                           % The convergence criterion is adjusted to the
                           % degree of damping
                  
         fprintf('Convergence in T-period decision function K(T) = %6.7f\n', dif1)  
                           % Display the difference between capital stocks
                           % in two subsequent iterations

         % Compute and update the coefficients of the capital 
         % decision function 
         % --------------------------------------------------
         bk_hat = X\y;   % Compute new coefficients of the capital decision 
                         % function using a MATLAB's backslash operator
                      
         bk = kdamp*bk_hat + (1-kdamp)*bk; 
                         % Update the coefficients of the capital decision  
                         % function using damping (partial updating) 

         % Store the grid for future capital
         % ---------------------------------
         k1_old = k1;     % The stored capital grid will be used for checking 
                          % the  convergence on the subsequent iteration                

    end;

    % Store the coefficients of the decision function
    % -----------------------------------------------
    BK(:,t) = bk;
    
end

fprintf('STEP 1 is completed!\n')  
fprintf(' \n')              % End STEP 1
                            
% _________________________________________________________________________
%
% EFP: STEP 2
% 
% Compute a sequence of decision functions (function path) for the periods
% t = 1,...T-1 that matches the given terminal condition (i.e., the decision 
% function of the T-period stationary economy computed in STEP 1)
% _________________________________________________________________________
%      
fprintf('EFP STEP 2: Constructing function path for t=1,...,T-1 ... \n')

% 10.1 Algorithm's parameters  
% ----------------------------

kdamp = 0.01;        % Damping parameter for (fixed-point) iteration on 
                     % the coefficients of the capital decision function
                     % To attain convergence under some parameterizations
                     % the damping parameter must be reduced to kdamp = 0.01
                                   
% 10.2 Initialize the capital series
% ----------------------------------
K_path_old = zeros(T-1,J);   % Initialize the next-period capital values on
                             % the grid; these values are used to check  
                             % convergence on the subsequent iteration; 
                             % (T-1)-by-J
                              
% 10.3 Compute backwards a path for capital decision functions for t=1,...,T-1
% ----------------------------------------------------------------------------
    
for t=T-1:-1:1         % iterate backwards from T-1 to 1

    dif2 = 1e+10;      % Set the initial difference between the grid values 
                       % from two iterations in the convergence criterion to 
                       % a very large number                               

    gridk = (grids(:,1).*(vk_max(t)-vk_min(t))+vk_max(t)+vk_min(t))/2;
    gridz = (grids(:,2).*(vz_max(t)-vz_min(t))+vz_max(t)+vz_min(t))/2;
                       % Adjust the Smolyak grid on a square [-1,1]x[-1,1] to
                       % that on a rectangle [k_min,k_max]x[z_min,z_max]
    
    bk = BK(:,t+1);    % Use the previous period solution as an initial 
                       % guess for the coefficients of the decision function 
                       % in period t        
                
    % An iterative cycle
    % ------------------
    while dif2 > 1e-10 % Check the convergence of decision function

         % Construct values of capital on the grid
         % ---------------------------------------
         X = Ord_Polynomial_N([gridk gridz],Degree_polyn); 
                                 % Construct the matrix of basis 
                                 % functions at t; J-by-Number_coeff
         k1 = X*bk;              % Compute next period capital using bk


         % Compute consumption, c, using the budget constraint 
         % ----------------------------------------------------

         c    = At(t)^(1-alpha)*gridk.^alpha.*gridz  + (1-delta)*gridk-k1;

         % Approximate conditional expectation, y, using the chosen 
         % Gauss-Hermite quadrature rule
         % ---------------------------------------------------------
          
         for i = 1:J  % For each grid point,...
              
              k2 = Ord_Polynomial_N([k1(i)*ones(n_nodes,1) z1(i,:)'],Degree_polyn)*BK(:,t+1);
              % k2 is state-contingent capital of period t+2 (chosen at t+1);
              % n_nodes-by-1
              
              % Note that the state-contingent capital "k2" is computed using
              % the future period coefficients BK(:,t+1) and not the current
              % period coefficients bk that is used to compute k1 (used by 
              % the conventional projection method for stationary problems
                   
              c1(i,:) = At(t+1)^(1-alpha)*k1(i).^alpha.*z1(i,:)'  + (1-delta)*k1(i)-k2;
              % c1 is state-contingent, next-period consumption; J-by-n_nodes

              k1_dupl = k1(i)*ones(1,n_nodes);  
              % Duplicate k1 n_nodes times to create a matrix with n_nodes 
              % identical columns; 1-by-n_nodes
              
              c_dupl = c(i)*ones(1,n_nodes);
              % Duplicate c n_nodes times to create a matrix with n_nodes 
              % identical columns; 1-by-n_nodes 
            
              y(i,1)  =  beta*c1(i,:).^(-gam)./c_dupl.^(-gam).*(1-delta+At(t+1)^(1-alpha)*alpha*k1_dupl.^(alpha-1).*z1(i,:)).*k1_dupl*weight_nodes;
              % Compute the values of capital decision funcion using the 
              % Euler equation which we rewrite as f(*)=1 and premultiply 
              % by k(t+1) so that y(t)=f(*)*k(t+1) has expected value k(t+1) 

         end

         % Evaluate the average percentage (unit-free) difference between
         % the capital decision function on the two subsequent iterations 
         % across all grid points
         % ---------------------------------------------------------------

         dif2 = mean(abs(1-k1./k1_old))/kdamp;
                            % The convergence criterion is adjusted to the
                            % degree of damping

         % Compute and update the coefficients of the capital 
         % decision function 
         % --------------------------------------------------
         bk_hat = X\y;   % Compute new coefficients of the capital decision 
                         % function using a MATLAB's backslash operator
         bk = kdamp*bk_hat + (1-kdamp)*bk; 
                         % Update the coefficients of the capital decision  
                         % function using damping (partial updating) 

         % Store the grid for future capital
         % ---------------------------------
         k1_old = k1;     % The stored values of capital on the grid will 
                           % be used for checking the  convergence on the 
                           % subsequent iteration
                           
    end;

     % Store the t-period coefficients
     % -------------------------------
     BK(:,t) = bk;
     t
end

bk_efp = BK;               % Store the nonstationary solution for periods 

fprintf('STEP 2 is completed!\n')  
fprintf(' \n')             % End STEP 2

% _________________________________________________________________________
%
% CONSTRUCTING BALANCED GROWTH PATH SOLUTION (FOR COMPARISON WITH EFP)
%
% Compute an accurate solution to the model by constructing a balanced growth 
% path and by converting the model into the stationary one and use this accurate 
% balanced growth solution for a comparison with the nonstationary EFP solution; 
% the balanced growth path solution is referred to as an "exact solution" 
% in MMTT (2020) 
% _________________________________________________________________________
%      
fprintf('Solving the nonstationary model by converting it into stationary ... \n')

% 11.1 Parameters of the balanced growth model
% --------------------------------------------

betas = beta*gamA^(1-gam);  % Discount rate for the balanced growth model

% 11.2 Set the initial guess on the coefficients of the polynomial
% function
% -----------------------------------------------------------------
 
bk_s = [0; 0.95; zeros(Number_coeff-2,1)];
                    % An initial guess for the coefficients of approximating 
                    % polynomials that matches the deterministic steady
                    % state of stationary economy
                 
% 11.3 Algorithm's parameters  
% --------------------------    

dif3 = 1e+10;        % Set the initial difference between the grid values from
                     % two iterations in the convergence criterion to a 
                     % very large number   
kdamp = 0.05;        % Damping parameter for (fixed-point) iteration on 
                     % the coefficients of the capital decision function                     
                     % To attain convergence under some parameterizations
                     % the damping parameter must be reduced to kdamp = 0.01
                     
% 11.4 Initialize the capital series
% ---------------------------------
k1_old2 = zeros(J,1);  % Initialize the grid of the next-period capital; 
                       % this grid is used to check the convergence on the 
                       % subsequent iteration; J-by-1                     

while dif3 > 1e-10;     % Check the convergence
      
        gridk = (grids(:,1).*(vk_max(1)-vk_min(1))+vk_max(1)+vk_min(1))/2;
        gridz = (grids(:,2).*(vz_max(1)-vz_min(1))+vz_max(1)+vz_min(1))/2;          
                 % Adjust the Smolyak grid on square [-1,1]x[-1,1] to that
                 % on a rectangle [k_min,k_max]x[z_min,z_max]
                 
        % Generate the values of capital
        % --------------------------
           
        X = Ord_Polynomial_N([gridk gridz],Degree_polyn); 
                                          % Constuct the matrix of basis 
                                          % functions of the polynomial; 
                                          % J-by-Number_coeff
        gridk1 = X*bk_s;               % Compute next period capital using bk_s

   
        % Compute consumption, c, using the budget constraint 
        % ----------------------------------------------------

        c    = gridk.^alpha.*gridz  + (1-delta)*gridk-gamA*gridk1;
                    
        % Approximate conditional expectation, y, using the chosen Gauss-Hermite 
        % quadrature rule
        % -----------------------------------------------------------------
        
        for i = 1:J   % For each grid point, ... 
                 
                 k2 = Ord_Polynomial_N([gridk1(i)*ones(n_nodes,1) z1(i,:)'],Degree_polyn)*bk_s;
                 % k2 is capital of period t+2 (chosen at t+1); use the same 
                 % decision function for stationary economy; n_nodes-by-1
                     
                 c1(i,:) = gridk1(i).^alpha.*z1(i,:)'  + (1-delta)*gridk1(i)-gamA*k2;
                 % c1 is next-period consumption; J-by-n_nodes
     
                 k1_dupl = gridk1(i)*ones(1,n_nodes);  
                 % Duplicate k1 n_nodes times to create a matrix with 
                 % n_nodes identical columns; 1-by-n_nodes
                 
                 c_dupl = c(i)*ones(1,n_nodes);
                 % Duplicate c n_nodes times to create a matrix with 
                 % n_nodes identical columns; 1-by-n_nodes 
                         
                 y(i,1)  =  betas*c1(i,:).^(-gam)./c_dupl.^(-gam).*(1-delta+alpha*k1_dupl.^(alpha-1).*z1(i,:)).*k1_dupl/gamA*weight_nodes;
                 % Compute the values of capital decision funcion using the 
                 % Euler equation which we rewrite as f(*)=1 and premultiply 
                 % by k(t+1) so that y(t)=f(*)*k(t+1) has expected value k(t+1) 
    
        end
                 
        % Evaluate the average percentage (unit-free) difference between 
        % the capital decision function on two subsequent iterations
        % across all grid points
        % ----------------------------------------------------------------
      
        dif3 = mean(abs(1-gridk1./k1_old2))/kdamp;                          
        fprintf('Convergence in stationary decision function K(t) = %6.10f\n', dif3)  
                            % Display the difference                         

                           
        % Compute and update the coefficients of the capital 
        % decision function 
        % --------------------------------------------------
         
        bk_hat_s = X\y;        % Compute new coefficients of the capital decision 
                               % function using a MATLAB's backslash
                               % operator
        bk_s = kdamp*bk_hat_s + (1-kdamp)*bk_s; 
                               % Update the coefficients of the capital decision  
                               % function using damping 
        % Store the grid for future capital
        % --------------------------------
        k1_old2 = gridk1;      % The stored values of capital on the grid 
                               % will be used for checking the  convergence 
                               % on the subsequent iteration           
             
end;

fprintf('The balanced growth model is completed!\n') 
                                % End solving the balanced growth model 
fprintf(' \n')

% _________________________________________________________________________
%
% SIMULATION OF THE EFP AND BALANCED GROWTH SOLUTIONS
% 
% Simulate the EFP and balanced growth solutions of the given length Tsim
% _________________________________________________________________________
%

fprintf('Simulating the EFP and balanced growth solutions ... ')

% 12.1 Load a sequence of shocks
% ------------------------------
load('epsi.mat'); % Load the sequence of shocks for which the solutions are 
                  % going to be simulated; 1000-by-1
            
Tsim = 200;       % Simulation length must be lower than or equal to T, i.e.,
                  % the length of the EFP path
            
            
% 12.2 Construct the productivity levels, z, for simulation
% ---------------------------------------------------------
z = 1;                              % Initial condition for the productivity 
                                    % level (is equal to steady state)
epsi = epsi(1:T,1)*sigma;           % Scale the errors in the process for 
                                    % productivity level by the standard
                                    % deviation
for t = 2:T; 
    
    z(t,1) = z(t-1,1).^rho.*exp(epsi(t,1));
                            % Compute the next-period productivity levels
end;


% 12.3 Simulate the EFP solution for capital 
% ------------------------------------------
k_efp = ks;                 % Initial condition for capital for the EFP 
                            % solution is equal to the steady state
                            
for t=1:Tsim
    k_efp(t+1,1) = Ord_Polynomial_N([k_efp(t,1) z(t,1)],Degree_polyn)*BK(:,t);
                            % Simulate the nonstationary EFP time series 
                            % solution using the sequence (path) of t-period 
                            % coefficients of the capital decision functions
end

% 12.4 Simulate the balanced growth solution for capital 
% ------------------------------------------------------
k_s = ks;                   % Initial condition for capital for the balanced 
                            % growth solution is equal to the steady state

for t=1:Tsim
    k_s(t+1,1)   = Ord_Polynomial_N([k_s(t,1) z(t,1)],Degree_polyn)*bk_s;
                            % Simulate the stationary time series solution 
                            % to the balanced growth model using the time-
                            % invariant capital decision function
end

k_bgrowth = At(1:Tsim+1).*k_s;% Recover nonstationary time series solution to
                            % the balanced growth model by multiplying the 
                            % stationary simulated series by the level of 
                            % labor augmenting technological progress

fprintf('completed!\n')      % End simulation 
fprintf(' \n')

% _________________________________________________________________________
%
% ACCURACY ASSESSMENT OF THE EFP SOLUTION
% 
% Compute a unit-free absolute difference between the EFP and balanced 
% growth time series solutions of length Tsim
% _________________________________________________________________________
%

fprintf('Assessing accuracy of the EFP solution ... ')

% 13.1 Unit-free difference between the EFP and balanced growth solutions
% -----------------------------------------------------------------------
error_dif = log10(abs(((k_efp-k_bgrowth(1:Tsim+1,:)))./(k_bgrowth(1:Tsim+1,:))));
                            % EFP and balanced growth solutions in log10
                            % units; for example, -6 means the difference 
                            % of the size 10^-6 = 0.000001

                            
% 13.2 Mean differences between EFP and the balanced growth solutions 
% --------------------------------------------------------------------                            
mean50 = mean(mean(error_dif(2:50,1)));   % Mean difference between the two
                                          % time series across t=1,...,50 periods
mean100 = mean(mean(error_dif(2:100,1))); % Mean difference between the two
                                          % time series across t=1,...,100 periods
mean150 = mean(mean(error_dif(2:150,1))); % Mean difference between the two
                                          % time series across t=1,...,150 periods
mean175 = mean(mean(error_dif(2:175,1))); % Mean difference between the two
                                          % time series across t=1,...,175 periods
mean200 = mean(mean(error_dif(2:200,1))); % Mean difference between the two
                                          % time series across t=1,...,200 periods                                          
                                          
                                          
% 13.3 Maximum differences between EFP and the balanced growth solutions
% ----------------------------------------------------------------------
max50 = max(max(error_dif(2:50,1)));      % Maximum difference between the two
                                          % time series across t=1,...,50 periods
max100 = max(max(error_dif(2:100,1)));    % Maximum difference between the two
                                          % time series across t=1,...,100 periods
max150 = max(max(error_dif(2:150,1)));    % Maximum difference between the two
                                          % time series across t=1,...,150 periods
max175 = max(max(error_dif(2:175,1)));    % Maximum difference between the two
                                          % time series across t=1,...,175 periods
max200 = max(max(error_dif(2:200,1)));    % Maximum difference between the two
                                          % time series across t=1,...,200 periods

fprintf('completed!\n')      % End accuracy assessment 
fprintf(' \n')

% 13.4 Display the difference between the EFP and balanced growth solutions
% -------------------------------------------------------------------------
disp('Table 1. Differences between the EFP and balanced growth solutions in log10 units:'); disp('');
Table_headers = char('           Mean      Maximum');

time_intervals = char('[0,50]  ','[0,100]  ','[0,150]  ','[0,175]  ','[0,200]  ');
                               % Time period over which the differences in
                               % the two solutions are computed                                          
Difference_mean = [mean50 mean100 mean150 mean175 mean200]';
                               % Mean differences in the two solutions over
                               % the given time horizon 
Difference_max  = [max50 max100 max150 max175 max200]'; 
                               % Maximum differences in the two solutions over
                               % the given time horizon 
Table = [time_intervals,num2str([Difference_mean,Difference_max])];
                               % Filling in the table
disp(Table_headers); disp(''); % Displaying the headers of the table 
disp(Table);                   % Displaying the table
fprintf(' \n')


% _________________________________________________________________________
%
% PLOTTING THE EFP AND BALANCED GROWTH SOLUTIONS
%
% Compare the EFP and balanced growth solutions of length Tsim on a graph
%
% _________________________________________________________________________
%

% 14. Plot the simulated capital series for the nonstationary EFP and 
% balanced growth solutions
% -------------------------------------------------------------------

fprintf('Plotting the EFP and balanced growth solutions ... ')

% 14.1 Growing EFP and balanced growth solutions
% ----------------------------------------------
figure(1)
subplot(1,2,1)                  
plot(1:Tsim+1,[k_efp(1:Tsim+1),k_bgrowth(1:Tsim+1)],'LineWidth',2)
xlim([0, Tsim+1])
ylim([0.97*min(min([k_efp(1:Tsim+1),k_bgrowth(1:Tsim+1)])) 1.05*max(max([k_efp(1:Tsim+1),k_bgrowth(1:Tsim+1)]))])
title('Comparison of the nonstationary EFP and balanced growth solutions')
xlabel('time');
ylabel('Growing capital');
legend('EFP','Balanced growth')

% 14.2 Detrended EFP and balanced growth solutions
% ------------------------------------------------
subplot(1,2,2)                
plot(1:Tsim+1,[k_efp(1:Tsim+1)./At(1:Tsim+1),k_s(1:Tsim+1)],'LineWidth',2)
xlim([0, Tsim+1])
ylim([0.97*min(min([k_efp(1:Tsim+1)./At(1:Tsim+1),k_s(1:Tsim+1)])) 1.05*max(max([k_efp(1:Tsim+1)./At(1:Tsim+1),k_s(1:Tsim+1)]))])
title('Comparison of the detrended EFP and balanced growth solutions')
xlabel('time');
ylabel('Detrended capital');
legend('EFP','Balanced growth')

fprintf('completed!\n')      % End plotting time series solutions
fprintf(' \n')

